<?php

namespace cBuilder\Classes\Analytics;

use cBuilder\Classes\Database\Orders;
use cBuilder\Classes\Database\AnalyticsViews;
use cBuilder\Classes\Database\AnalyticsInteractions;
use cBuilder\Classes\Analytics\CCBFieldAnalytics;
use cBuilder\Classes\CCBSettingsData;

class CCBAnalyticsHelper {
	public static function get_dashboard_data() {
		$dashboard_data = array(
			'carts'  => array(
				'revenue'      => array(
					'title'   => __( 'Revenue', 'cost-calculator-builder-pro' ),
					'sub'     => __( 'Avg. per Order', 'cost-calculator-builder-pro' ),
					'value'   => 0,
					'series'  => array(),
					'average' => 0,
				),
				'orders'       => array(
					'title'   => __( 'Orders', 'cost-calculator-builder-pro' ),
					'sub'     => __( 'Avg. Orders/day', 'cost-calculator-builder-pro' ),
					'value'   => 0,
					'series'  => array(),
					'average' => 0,
				),
				'views'        => array(
					'title'   => __( 'Views', 'cost-calculator-builder-pro' ),
					'sub'     => __( 'Avg. Views/day', 'cost-calculator-builder-pro' ),
					'value'   => 0,
					'series'  => array(),
					'average' => 0,
				),
				'interactions' => array(
					'title'   => __( 'Interactions', 'cost-calculator-builder-pro' ),
					'sub'     => __( 'Avg. Interactions/day', 'cost-calculator-builder-pro' ),
					'value'   => 0,
					'series'  => array(),
					'average' => 0,
				),
			),
			'charts' => array(
				'interactions'       => array(
					'title'       => __( 'Orders vs Interactions', 'cost-calculator-builder-pro' ),
					'type'        => 'line',
					'series'      => array(),
					'categories'  => array(),
					'minValue'    => 0,
					'extraSeries' => array(),
					'key'         => 'interactions_line_chart',
				),
				'revenue_calclators' => array(
					'title'       => __( 'Revenue by Calculator', 'cost-calculator-builder-pro' ),
					'type'        => 'bar',
					'series'      => array(),
					'categories'  => array(),
					'minValue'    => 0,
					'extraSeries' => array(),
					'key'         => 'revenue_bar_chart',
				),
				'weekdays'           => array(
					'title'       => __( 'Best order days', 'cost-calculator-builder-pro' ),
					'type'        => 'columns',
					'series'      => array(),
					'categories'  => array(),
					'minValue'    => 0,
					'extraSeries' => array(),
					'key'         => 'weekdays_columns_chart',
				),
				'revenue_line_chart' => array(
					'title'       => __( 'Revenue', 'cost-calculator-builder-pro' ),
					'type'        => 'line',
					'series'      => array(),
					'categories'  => array(),
					'minValue'    => 0,
					'extraSeries' => array(),
					'key'         => 'revenue_line_chart_chart',
				),
			),
		);

		$status   = isset( $_GET['status'] ) ? sanitize_text_field( $_GET['status'] ) : 'all';
		$period   = sanitize_text_field( $_GET['period'] ) ?? 'all';
		$grouping = array(
			'interactions_line_chart'  => sanitize_text_field( $_GET['interactions_line_chart'] ) ?? 'week',
			'weekdays_columns_chart'   => sanitize_text_field( $_GET['weekdays_columns_chart'] ) ?? 'week',
			'revenue_line_chart_chart' => sanitize_text_field( $_GET['revenue_line_chart_chart'] ) ?? 'week',
		);

		$grouping = self::parse_grouping( $grouping, $period );

		$dashboard_data = self::get_dashboard_data_handler( $dashboard_data, $period, $grouping, $status );

		return $dashboard_data;
	}

	public static function get_dashboard_data_handler( $dashboard_data, $period, $grouping, $status ) {
		$orders_table_name       = Orders::_table();
		$views_table_name        = AnalyticsViews::_table();
		$interactions_table_name = AnalyticsInteractions::_table();

		$orders = array();

		if ( in_array( $period, array( 'today', 'yesterday' ), true ) ) {
			$cart_grouping = 'hour';
		} elseif ( 'custom' === $period ) {
			$custom_date   = isset( $_GET['extra'] ) ? sanitize_text_field( $_GET['extra'] ) : '';
			$periods       = ccb_get_date_range_for_period_pro( $period, $custom_date );
			$cart_grouping = ccb_get_optimal_grouping_for_custom_period( $periods['from'], $periods['to'] );
		} else {
			$cart_grouping = 'week';
		}

		$carts_data = array(
			array(
				'key'   => 'orders',
				'label' => __( 'Orders', 'cost-calculator-builder-pro' ),
				'table' => $orders_table_name,
			),
			array(
				'key'   => 'views',
				'label' => __( 'Views', 'cost-calculator-builder-pro' ),
				'table' => $views_table_name,
			),
			array(
				'key'   => 'interactions',
				'label' => __( 'Interactions', 'cost-calculator-builder-pro' ),
				'table' => $interactions_table_name,
			),
		);

		// Cart data for orders, views, interactions
		foreach ( $carts_data as $cart_data ) {
			if ( 'orders' === $cart_data['key'] ) {
				$_data        = self::get_data_by_period( $period, $cart_data['table'], null, $status );
				$_series_data = self::get_grouped_data_by_period( $period, $cart_data['table'], $cart_grouping, null, $status );
			} else {
				$_data        = self::get_data_by_period( $period, $cart_data['table'] );
				$_series_data = self::get_grouped_data_by_period( $period, $cart_data['table'], $cart_grouping );
			}

			$_series = array_values( $_series_data );

			if ( 'orders' === $cart_data['key'] ) {
				$orders = $_data;
			}

			$cart_total_and_average = self::get_avarage_and_total( $cart_data['table'], $period );

			$dashboard_data['carts'][ $cart_data['key'] ]['value']   = $cart_total_and_average['total'];
			$dashboard_data['carts'][ $cart_data['key'] ]['average'] = ccb_smart_format( $cart_total_and_average['average'] );
			$dashboard_data['carts'][ $cart_data['key'] ]['extra']   = '<span style="color:#27ae60;font-weight:bold;"> ' . ccb_smart_format( $cart_total_and_average['average'] ) . ' </span>';
			$dashboard_data['carts'][ $cart_data['key'] ]['series']  = array(
				array(
					'name' => $cart_data['label'],
					'data' => ccb_get_cart_series( $_series ),
				),
			);
		}

		// Cart data for revenue
		$revenue_calculators = array();
		$grouped_revenue     = array();

		$revenue             = 0;
		$revenue_data        = array();
		$orders_revenue_data = array();

		$interactions      = self::get_grouped_data_by_period( $period, $interactions_table_name, $grouping['interactions_line_chart'] );
		$interaction_extra = self::get_grouped_data_by_period( $period, $orders_table_name, $grouping['interactions_line_chart'], null, $status );
		$interactions_cats = array_keys( $interactions ) >= array_keys( $interaction_extra ) ? array_keys( $interactions ) : array_keys( $interaction_extra );
		$order_best_days   = self::get_grouped_data_by_period( $period, $orders_table_name, $grouping['weekdays_columns_chart'], null, $status );

		foreach ( $orders as $order ) {
			$order_meta = get_option( 'calc_meta_data_order_' . $order->id );
			$totals     = json_decode( $order_meta['totals'], true ); // phpcs:ignore

			$current_revenue = 0;

			foreach ( $totals as $total ) {
				$revenue         += $total['total'];
				$current_revenue += $total['total'];
			}

			$order->calc_title = empty( $order->calc_title ) ? 'Untitled' : $order->calc_title;

			$revenue_data[] = array(
				'order_id'   => $order->id,
				'calc_id'    => $order->calc_id,
				'name'       => $order->calc_title,
				'revenue'    => $current_revenue,
				'created_at' => $order->created_at,
			);
		}

		foreach ( $revenue_data as $revenue_item ) {
			$calc_id = $revenue_item['calc_id'];
			if ( ! isset( $grouped_revenue[ $calc_id ] ) ) {
				$grouped_revenue[ $calc_id ] = array(
					'name'    => $revenue_item['name'],
					'revenue' => 0,
				);
			}
			$grouped_revenue[ $calc_id ]['revenue'] += $revenue_item['revenue'];
		}

		foreach ( $grouped_revenue as $calc_id => $data ) {
			if ( 10 === count( $revenue_calculators ) ) {
				break;
			}

			$revenue_calculators[] = array(
				'name'  => $data['name'],
				'value' => $data['revenue'],
			);
		}

		// phpcs:disable
		usort( $revenue_calculators, function( $a, $b ) {
			$a_value = floatval( preg_replace('/[^0-9.]/', '', $a['value']) );
			$b_value = floatval( preg_replace('/[^0-9.]/', '', $b['value']) );
			return $b_value - $a_value;
		});
		// phpcs:enable

		$orders_revenue_data       = self::group_revenue_by_period( $revenue_data, $period, $grouping['revenue_line_chart_chart'] );
		$orders_revenue_chart_data = self::group_revenue_by_period( $revenue_data, $period, 'week' );

		$revenue_names  = array();
		$revenue_values = array();

		foreach ( $revenue_calculators as $revenue_calculator ) {
			$revenue_names[]  = $revenue_calculator['name'];
			$revenue_values[] = intval( $revenue_calculator['value'] );
		}

		$orders_count = count( $orders );
		$revenue_avg  = $orders_count > 0 ? $revenue / $orders_count : 0;

		$dashboard_data['carts']['revenue']['value']   = $revenue;
		$dashboard_data['carts']['revenue']['average'] = ccb_smart_format( $revenue_avg );
		$dashboard_data['carts']['revenue']['extra']   = '<span style="color:#27ae60;font-weight:bold;"> ' . ccb_smart_format( $revenue_avg ) . ' </span>';
		$dashboard_data['carts']['revenue']['series']  = array(
			array(
				'name' => __( 'Revenue', 'cost-calculator-builder-pro' ),
				'data' => array_values( $orders_revenue_chart_data ),
			),
		);

		// Chart data
		$dashboard_data['charts']['revenue_calclators']['series']     = $revenue_values;
		$dashboard_data['charts']['revenue_calclators']['categories'] = $revenue_names;

		$dashboard_data['charts']['weekdays']['series']     = array_values( $order_best_days );
		$dashboard_data['charts']['weekdays']['categories'] = array_keys( $order_best_days );

		$dashboard_data['charts']['interactions']['series']      = array_values( $interactions );
		$dashboard_data['charts']['interactions']['categories']  = $interactions_cats;
		$dashboard_data['charts']['interactions']['extraSeries'] = array_values( $interaction_extra );

		$dashboard_data['charts']['revenue_line_chart']['series']     = array_values( $orders_revenue_data );
		$dashboard_data['charts']['revenue_line_chart']['categories'] = array_keys( $orders_revenue_data );

		return array(
			'carts'  => array_values( $dashboard_data['carts'] ),
			'charts' => array_values( $dashboard_data['charts'] ),
		);
	}

	public static function get_grouped_data_by_period( $period, $table_name, $grouping = 'week', $calc_id = null, $status = 'all' ) {
		global $wpdb;

		$custom_date = null;
		if ( 'custom' === $period && isset( $_GET['extra'] ) ) {
			$custom_date = sanitize_text_field( $_GET['extra'] );
		}

		$periods = ccb_get_date_range_for_period_pro( $period, $custom_date );
		$from    = $periods['from'];
		$to      = $periods['to'];

		$whereClause = '';
		if ( $from && $to ) {
			$whereClause = sprintf( 'WHERE created_at BETWEEN "%s" AND "%s" + INTERVAL 1 DAY', $from->format( 'Y-m-d H:i:s' ), $to->format( 'Y-m-d H:i:s' ) );

			if ( ! is_null( $calc_id ) ) {
				$whereClause .= sprintf( ' AND calc_id = %d', $calc_id );
			}
		}

		if ( empty( $whereClause ) && ! is_null( $calc_id ) ) {
			$whereClause = sprintf(
				'WHERE calc_id = %d',
				$calc_id
			);
		}

		$orders_table_name = Orders::_table();
		if ( $table_name === $orders_table_name && 'all' !== $status ) {
			$status_value = 'completed' === $status ? 'complete' : $status;
			if ( empty( $whereClause ) ) {
				$whereClause = sprintf( 'WHERE status = "%s"', $status_value );
			} else {
				$whereClause .= sprintf( ' AND status = "%s"', $status_value );
			}
		}

		if ( 'hour' === $grouping ) {
			// phpcs:disable
			$sql = sprintf(
			"SELECT FLOOR(HOUR(created_at) / 3) AS bucket, COUNT(*) AS total
				FROM %s
				%s
				GROUP BY bucket
				ORDER BY bucket",
				$table_name,
				$whereClause
			);
			// phpcs:enable

			$db_results = $wpdb->get_results( $sql, ARRAY_A ); // phpcs:ignore

			$results = array();

			for ( $i = 0; $i < 8; $i++ ) {
				$results[ $i ] = array(
					'label' => sprintf( '%02d:00', $i * 3 ),
					'total' => 0,
				);
			}

			foreach ( $db_results as $row ) {
				$bucket = (int) $row['bucket'];

				$results[ $bucket ]['total'] = (int) $row['total'];
			}

			$results = array_values( $results );

			$result = array();
			foreach ( $results as $res ) {
				$result[ $res['label'] ] = $res['total'];
			}

			return $result;
		}

		if ( 'custom' === $period ) {
			$optimal_grouping = ccb_get_optimal_grouping_for_custom_period( $from, $to );

			if ( $optimal_grouping !== $grouping ) {
				$grouping = $optimal_grouping;
			}
		}

		if ( 'day' === $grouping ) {
			$result = array();

			if ( $from && $to ) {
				$current_date = $from;
				while ( $current_date <= $to ) {
					$day_label            = $current_date->format( 'M j' );
					$result[ $day_label ] = 0;
					$current_date         = $current_date->modify( '+1 day' );
				}

				// phpcs:disable
				$sql = sprintf(
					"SELECT DATE_FORMAT(created_at, '%%b %%e') AS label, COUNT(*) AS total
					FROM %s
					%s
					GROUP BY DATE(created_at)
					ORDER BY DATE(created_at)",
					$table_name,
					$whereClause
				);
				// phpcs:enable

				$rows = $wpdb->get_results( $sql ); // phpcs:ignore
				foreach ( $rows as $row ) {
					$day_label = trim( $row->label );
					if ( isset( $result[ $day_label ] ) ) {
						$result[ $day_label ] = (int) $row->total;
					}
				}
			}

			return $result;
		}

		if ( 'week' === $grouping && in_array( $period, array( 'all', 'last_7_days', 'last_30_days', 'last_90_days', 'last_year' ), true ) ) {
			$weekdays = array( 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun' );
			$result   = array_fill_keys( $weekdays, 0 );

			// phpcs:disable
			$sql = sprintf(
			"SELECT DATE_FORMAT(created_at, '%%a') AS label, COUNT(*) AS total
				FROM %s
				%s
				GROUP BY label
				ORDER BY FIELD(label, 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun')",
				$table_name,
				$whereClause
			);
			// phpcs:enable

			$rows = $wpdb->get_results( $sql ); // phpcs:ignore
			foreach ( $rows as $row ) {
				$weekday = $row->label;
				if ( isset( $result[ $weekday ] ) ) {
					$result[ $weekday ] = (int) $row->total;
				}
			}

			return $result;
		}

		if ( 'month' === $grouping ) {
			$month_names = array(
				'01' => 'Jan',
				'02' => 'Feb',
				'03' => 'Mar',
				'04' => 'Apr',
				'05' => 'May',
				'06' => 'Jun',
				'07' => 'Jul',
				'08' => 'Aug',
				'09' => 'Sep',
				'10' => 'Oct',
				'11' => 'Nov',
				'12' => 'Dec',
			);

			if ( 'all' === $period ) {
				$result = array(
					'Jan' => 0,
					'Feb' => 0,
					'Mar' => 0,
					'Apr' => 0,
					'May' => 0,
					'Jun' => 0,
					'Jul' => 0,
					'Aug' => 0,
					'Sep' => 0,
					'Oct' => 0,
					'Nov' => 0,
					'Dec' => 0,
				);

				// phpcs:disable
				$sql = sprintf(
					"SELECT DATE_FORMAT(created_at, '%%m') AS month, COUNT(*) AS total
					FROM %s
					%s
					GROUP BY month
					ORDER BY month",
					$table_name,
					$whereClause
				);
				// phpcs:enable

				$rows = $wpdb->get_results( $sql ); // phpcs:ignore

				foreach ( $rows as $row ) {
					$month_key            = $month_names[ $row->month ];
					$result[ $month_key ] = (int) $row->total;
				}

				return $result;
			} elseif ( 'last_30_days' === $period ) {
				$current_month = new \DateTime();
				$prev_month    = ( new \DateTime() )->modify( '-1 month' );

				$start_date = ( new \DateTime() )->modify( '-29 days' );
				$end_date   = new \DateTime();

				$prev_month_label    = $start_date->format( 'd M/Y' );
				$current_month_label = $end_date->format( 'd M/Y' );

				$months = array(
					$prev_month->format( 'Y-m' )    => $prev_month_label,
					$current_month->format( 'Y-m' ) => $current_month_label,
				);

				$result = array_fill_keys( array_values( $months ), 0 );

				$sql = sprintf(
					"SELECT DATE_FORMAT(created_at, '%%Y-%%m') AS \"year_month\", COUNT(*) AS total
					FROM %s
					%s
					GROUP BY DATE_FORMAT(created_at, '%%Y-%%m')
					ORDER BY DATE_FORMAT(created_at, '%%Y-%%m')",
					$table_name,
					$whereClause
				);

				$rows = $wpdb->get_results( $sql ); // phpcs:ignore
				foreach ( $rows as $row ) {
					if ( isset( $months[ $row->year_month ] ) ) {
						$result[ $months[ $row->year_month ] ] = (int) $row->total;
					}
				}

				return $result;
			} elseif ( 'last_90_days' === $period ) {
				$current_month = new \DateTime();
				$prev_month1   = ( new \DateTime() )->modify( '-1 month' );
				$prev_month2   = ( new \DateTime() )->modify( '-2 month' );

				$months = array(
					$prev_month2->format( 'Y-m' )   => $month_names[ $prev_month2->format( 'm' ) ] . '/' . $prev_month2->format( 'Y' ),
					$prev_month1->format( 'Y-m' )   => $month_names[ $prev_month1->format( 'm' ) ] . '/' . $prev_month1->format( 'Y' ),
					$current_month->format( 'Y-m' ) => $month_names[ $current_month->format( 'm' ) ] . '/' . $current_month->format( 'Y' ),
				);

				$result = array_fill_keys( array_values( $months ), 0 );

				$sql = sprintf(
					"SELECT DATE_FORMAT(created_at, '%%Y-%%m') AS \"year_month\", COUNT(*) AS total
					FROM %s
					%s
					GROUP BY DATE_FORMAT(created_at, '%%Y-%%m')
					ORDER BY DATE_FORMAT(created_at, '%%Y-%%m')",
					$table_name,
					$whereClause
				);

				$rows = $wpdb->get_results( $sql ); // phpcs:ignore
				foreach ( $rows as $row ) {
					if ( isset( $months[ $row->year_month ] ) ) {
						$result[ $months[ $row->year_month ] ] = (int) $row->total;
					}
				}

				return $result;
			} elseif ( 'last_year' === $period ) {
				$current_year = ( new \DateTime() )->format( 'Y' );
				$prev_year    = $current_year - 1;

				$years = array(
					$prev_year    => $prev_year,
					$current_year => $current_year,
				);

				$result = array_fill_keys( array_values( $years ), 0 );

				// phpcs:disable
				$sql = sprintf(
				"SELECT DATE_FORMAT(created_at, '%%Y') AS year, COUNT(*) AS total
					FROM %s
					%s
					GROUP BY year
					ORDER BY year",
					$table_name,
					$whereClause
				);
				// phpcs:enable

				$rows = $wpdb->get_results( $sql ); // phpcs:ignore
				foreach ( $rows as $row ) {
					$year = $row->year;
					if ( isset( $years[ $year ] ) ) {
						$result[ $years[ $year ] ] = (int) $row->total;
					}
				}

				return $result;
			}
		}

		if ( 'year' === $grouping ) {
			if ( 'all' === $period ) {
				$sql = sprintf(
					"SELECT DATE_FORMAT(created_at, '%%Y') AS year, COUNT(*) AS total
					FROM %s
					%s
					GROUP BY year
					ORDER BY year",
					$table_name,
					$whereClause
				);

				$rows   = $wpdb->get_results( $sql ); // phpcs:ignore
				$result = array();

				if ( 1 === count( $rows ) ) {
					$current_year            = $rows[0]->year;
					$prev_year               = $current_year - 1;
					$result[ $prev_year ]    = 0;
					$result[ $current_year ] = (int) $rows[0]->total;
				} else {
					foreach ( $rows as $row ) {
						$result[ $row->year ] = (int) $row->total;
					}
				}

				return $result;
			} else {
				$current_year = ( new \DateTime() )->format( 'Y' );
				$prev_year    = $current_year - 1;

				$years = array(
					$prev_year    => $prev_year,
					$current_year => $current_year,
				);

				$result = array_fill_keys( array_values( $years ), 0 );

				$sql = sprintf(
					"SELECT DATE_FORMAT(created_at, '%%Y') AS year, COUNT(*) AS total
					FROM %s
					%s
					GROUP BY year
					ORDER BY year",
					$table_name,
					$whereClause
				);

				$rows = $wpdb->get_results( $sql ); // phpcs:ignore
				foreach ( $rows as $row ) {
					if ( isset( $years[ $row->year ] ) ) {
						$result[ $years[ $row->year ] ] = (int) $row->total;
					}
				}

				return $result;
			}
		}

		$weekdays = array( 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun' );
		$result   = array_fill_keys( $weekdays, 0 );

		// phpcs:disable
		$sql = sprintf(
		"SELECT DATE_FORMAT(created_at, '%%a') AS label, COUNT(*) AS total
			FROM %s
			%s
			GROUP BY label
			ORDER BY FIELD(label, 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun')",
			$table_name,
			$whereClause
		);
		// phpcs:enable

		$rows = $wpdb->get_results( $sql ); // phpcs:ignore
		foreach ( $rows as $row ) {
			$weekday = $row->label;
			if ( isset( $result[ $weekday ] ) ) {
				$result[ $weekday ] = (int) $row->total;
			}
		}

		return $result;
	}

	public static function group_revenue_by_period( $data, $period, $grouping = 'week' ) {
		$results = array();

		foreach ( $data as $key => $item ) {
			$data[ $key ]['datetime'] = new \DateTime( $item['created_at'] );
		}

		if ( 'hour' === $grouping ) {
			$buckets = array();
			for ( $i = 0; $i < 8; $i++ ) {
				$hour          = str_pad( $i * 3, 2, '0', STR_PAD_LEFT );
				$buckets[ $i ] = array(
					'label' => $hour . ':00',
					'total' => 0,
				);
			}

			foreach ( $data as $item ) {
				$hour        = (int) $item['datetime']->format( 'H' );
				$bucketIndex = floor( $hour / 3 );
				if ( $bucketIndex >= 0 && $bucketIndex < 8 ) {
					$buckets[ $bucketIndex ]['total'] += $item['revenue'];
				}
			}

			$results = array_values( $buckets );

			$assoc_results = array();
			foreach ( $results as $total ) {
				$assoc_results[ $total['label'] ] = intval( $total['total'] );
			}

			$results = $assoc_results;
		} elseif ( 'day' === $grouping ) {
			$results = array();

			if ( ! empty( $data ) ) {
				$first_date = new \DateTime( min( array_column( $data, 'created_at' ) ) );
				$last_date  = new \DateTime( max( array_column( $data, 'created_at' ) ) );

				$current_date = $first_date;
				while ( $current_date <= $last_date ) {
					$day_label             = $current_date->format( 'M j' );
					$results[ $day_label ] = 0;
					$current_date          = $current_date->modify( '+1 day' );
				}

				foreach ( $data as $item ) {
					$day_label = $item['datetime']->format( 'M j' );
					if ( isset( $results[ $day_label ] ) ) {
						$results[ $day_label ] += $item['revenue'];
					}
				}
			}
		} elseif ( 'week' === $grouping && in_array( $period, array( 'all', 'last_7_days', 'last_30_days', 'last_90_days', 'last_year' ), true ) ) {
			$weekdays      = array( 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun' );
			$weekdayTotals = array_fill_keys( $weekdays, 0 );

			foreach ( $data as $item ) {
				$weekday = $item['datetime']->format( 'D' );
				if ( isset( $weekdayTotals[ $weekday ] ) ) {
					$weekdayTotals[ $weekday ] += $item['revenue'];
				}
			}

			foreach ( $weekdays as $day ) {
				$results[ $day ] = $weekdayTotals[ $day ];
			}
		} elseif ( 'month' === $grouping ) {
			$month_names = array(
				'01' => 'Jan',
				'02' => 'Feb',
				'03' => 'Mar',
				'04' => 'Apr',
				'05' => 'May',
				'06' => 'Jun',
				'07' => 'Jul',
				'08' => 'Aug',
				'09' => 'Sep',
				'10' => 'Oct',
				'11' => 'Nov',
				'12' => 'Dec',
			);

			if ( 'all' === $period ) {
				$results = array(
					'Jan' => 0,
					'Feb' => 0,
					'Mar' => 0,
					'Apr' => 0,
					'May' => 0,
					'Jun' => 0,
					'Jul' => 0,
					'Aug' => 0,
					'Sep' => 0,
					'Oct' => 0,
					'Nov' => 0,
					'Dec' => 0,
				);

				foreach ( $data as $item ) {
					$month                  = $item['datetime']->format( 'm' );
					$month_key              = $month_names[ $month ];
					$results[ $month_key ] += $item['revenue'];
				}
			} elseif ( 'last_30_days' === $period ) {
				$current_month = new \DateTime();
				$prev_month    = ( new \DateTime() )->modify( '-1 month' );

				$start_date = ( new \DateTime() )->modify( '-29 days' );
				$end_date   = new \DateTime();

				$prev_month_label    = $start_date->format( 'd M/Y' );
				$current_month_label = $end_date->format( 'd M/Y' );

				$results = array(
					$prev_month_label    => 0,
					$current_month_label => 0,
				);

				foreach ( $data as $item ) {
					$item_month = $item['datetime']->format( 'Y-m' );
					if ( $item_month === $prev_month->format( 'Y-m' ) ) {
						$results[ $prev_month_label ] += $item['revenue'];
					} elseif ( $item_month === $current_month->format( 'Y-m' ) ) {
						$results[ $current_month_label ] += $item['revenue'];
					}
				}
			} elseif ( 'last_90_days' === $period ) {
				$current_month = new \DateTime();
				$prev_month1   = ( new \DateTime() )->modify( '-1 month' );
				$prev_month2   = ( new \DateTime() )->modify( '-2 month' );

				$results = array(
					$month_names[ $prev_month2->format( 'm' ) ] . '/' . $prev_month2->format( 'Y' )     => 0,
					$month_names[ $prev_month1->format( 'm' ) ] . '/' . $prev_month1->format( 'Y' )     => 0,
					$month_names[ $current_month->format( 'm' ) ] . '/' . $current_month->format( 'Y' ) => 0,
				);

				foreach ( $data as $item ) {
					$item_month = $item['datetime']->format( 'Y-m' );
					if ( $item_month === $prev_month2->format( 'Y-m' ) ) {
						$results[ $month_names[ $prev_month2->format( 'm' ) ] . '/' . $prev_month2->format( 'Y' ) ] += $item['revenue'];
					} elseif ( $item_month === $prev_month1->format( 'Y-m' ) ) {
						$results[ $month_names[ $prev_month1->format( 'm' ) ] . '/' . $prev_month1->format( 'Y' ) ] += $item['revenue'];
					} elseif ( $item_month === $current_month->format( 'Y-m' ) ) {
						$results[ $month_names[ $current_month->format( 'm' ) ] . '/' . $current_month->format( 'Y' ) ] += $item['revenue'];
					}
				}
			} else {
				$weekdays      = array( 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun' );
				$weekdayTotals = array_fill_keys( $weekdays, 0 );

				foreach ( $data as $item ) {
					$weekday = $item['datetime']->format( 'D' );
					if ( isset( $weekdayTotals[ $weekday ] ) ) {
						$weekdayTotals[ $weekday ] += $item['revenue'];
					}
				}

				foreach ( $weekdays as $day ) {
					$results[ $day ] = $weekdayTotals[ $day ];
				}

				$assoc_results = array();
				foreach ( $results as $day => $total ) {
					$assoc_results[ $day ] = intval( $total );
				}

				$results = $assoc_results;
			}
		} elseif ( 'year' === $grouping ) {
			if ( 'all' === $period ) {
				$results = array();

				foreach ( $data as $item ) {
					$year = $item['datetime']->format( 'Y' );
					if ( ! isset( $results[ $year ] ) ) {
						$results[ $year ] = 0;
					}
					$results[ $year ] += $item['revenue'];
				}

				if ( 1 === count( $results ) ) {
					$current_year = array_keys( $results )[0];
					$prev_year    = $current_year - 1;
					$results      = array( $prev_year => 0 ) + $results;
				}
			} elseif ( 'last_year' === $period ) {
				$current_year = ( new \DateTime() )->format( 'Y' );
				$prev_year    = $current_year - 1;

				$results = array(
					$prev_year    => 0,
					$current_year => 0,
				);

				foreach ( $data as $item ) {
					$year = $item['datetime']->format( 'Y' );
					if ( isset( $results[ $year ] ) ) {
						$results[ $year ] += $item['revenue'];
					}
				}
			} else {
				$weekdays      = array( 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun' );
				$weekdayTotals = array_fill_keys( $weekdays, 0 );

				foreach ( $data as $item ) {
					$weekday = $item['datetime']->format( 'D' );
					if ( isset( $weekdayTotals[ $weekday ] ) ) {
						$weekdayTotals[ $weekday ] += $item['revenue'];
					}
				}

				foreach ( $weekdays as $day ) {
					$results[ $day ] = $weekdayTotals[ $day ];
				}

				$assoc_results = array();
				foreach ( $results as $day => $total ) {
					$assoc_results[ $day ] = intval( $total );
				}

				$results = $assoc_results;
			}
		} else {
			$weekdays      = array( 'Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun' );
			$weekdayTotals = array_fill_keys( $weekdays, 0 );

			foreach ( $data as $item ) {
				$weekday = $item['datetime']->format( 'D' );
				if ( isset( $weekdayTotals[ $weekday ] ) ) {
					$weekdayTotals[ $weekday ] += $item['revenue'];
				}
			}

			foreach ( $weekdays as $day ) {
				$results[ $day ] = $weekdayTotals[ $day ];
			}

			$assoc_results = array();
			foreach ( $results as $day => $total ) {
				$assoc_results[ $day ] = intval( $total );
			}

			$results = $assoc_results;
		}

		foreach ( $results as &$result ) {
			$result = ceil( $result );
		}

		return $results;
	}

	public static function get_data_by_period( $period, $table_name, $calc_id = null, $status = 'all' ) {
		global $wpdb;
		$custom_date = null;
		if ( 'custom' === $period && isset( $_GET['extra'] ) ) {
			$custom_date = sanitize_text_field( $_GET['extra'] );
		}

		$periods = ccb_get_date_range_for_period_pro( $period, $custom_date );
		$from    = $periods['from'];
		$to      = $periods['to'];

		$whereClause = '';
		if ( $from && $to ) {
			$whereClause = sprintf( 'WHERE created_at BETWEEN "%s" AND "%s" + INTERVAL 1 DAY', $from->format( 'Y-m-d H:i:s' ), $to->format( 'Y-m-d H:i:s' ) );
		}

		if ( ! empty( $calc_id ) ) {
			if ( empty( $whereClause ) ) {
				$whereClause = sprintf(
					'WHERE calc_id = %d',
					$calc_id
				);
			} else {
				$whereClause .= sprintf(
					' AND calc_id = %d',
					$calc_id
				);
			}
		}

		$orders_table_name = Orders::_table();
		if ( $table_name === $orders_table_name && 'all' !== $status ) {
			$status_value = 'completed' === $status ? 'complete' : $status;
			if ( empty( $whereClause ) ) {
				$whereClause = sprintf( 'WHERE status = "%s"', $status_value );
			} else {
				$whereClause .= sprintf( ' AND status = "%s"', $status_value );
			}
		}

		$sql = sprintf( 'SELECT * FROM %s %s', $table_name, $whereClause ); // phpcs:ignore

		return $wpdb->get_results( $sql ); // phpcs:ignore
	}

	public static function get_avarage_and_total( $table_name, $period, $calc_id = null ) {
		global $wpdb;

		$custom_date = null;
		if ( 'custom' === $period && isset( $_GET['extra'] ) ) {
			$custom_date = sanitize_text_field( $_GET['extra'] );
		}

		$periods = ccb_get_date_range_for_period_pro( $period, $custom_date );
		$from    = $periods['from'];
		$to      = $periods['to'];

		if ( ! $from || ! $to ) {
			$first_date_sql    = sprintf( 'SELECT MIN(created_at) as first_date FROM %s', $table_name );
			$first_date_result = $wpdb->get_row( $first_date_sql ); // phpcs:ignore

			if ( $first_date_result && $first_date_result->first_date ) {
				$from = new \DateTime( $first_date_result->first_date );
			} else {
				$from = new \DateTime();
			}

			$to = new \DateTime();
		}

		$whereClause = sprintf(
			'WHERE created_at BETWEEN "%s" AND "%s" + INTERVAL 1 DAY',
			$from->format( 'Y-m-d' ),
			$to->format( 'Y-m-d' )
		);

		if ( ! empty( $calc_id ) ) {
			$whereClause .= sprintf( ' AND calc_id = %d', $calc_id );
		}

		$sql = sprintf(
			'SELECT
				COUNT(*) AS total,
				COUNT(*) * 1.0 / (DATEDIFF("%s", "%s") + 1) AS average
			FROM %s %s',
			$to->format( 'Y-m-d' ),
			$from->format( 'Y-m-d' ),
			$table_name,
			$whereClause
		);

		$result = $wpdb->get_row( $sql ); // phpcs:ignore

		return array(
			'total'   => $result->total ?? 0,
			'average' => $result->average ?? 0,
		);
	}

	public static function get_single_calc_data() {
		$calc_id = $_GET['calc_id'];
		$period  = $_GET['period'];
		$extra   = $_GET['extra'];
		$status  = isset( $_GET['status'] ) ? sanitize_text_field( $_GET['status'] ) : 'all';
		$orders  = self::get_orders_by_calc( $calc_id, $period, $status );

		$all_calc_fields = self::get_all_calc_fields_for_analytics( $calc_id );

		$analytics_data = self::initialize_empty_analytics_for_fields( $all_calc_fields );

		$temp             = array();
		$processed_fields = array();

		foreach ( $orders as $order ) {
			if ( ! empty( $order->order_details ) ) {
				$order_details = json_decode( $order->order_details, true );

				foreach ( $order_details as $detail ) {
					if ( ! empty( $detail['alias'] ) ) {
						$field_name = preg_replace( '/_field_id.*/', '', $detail['alias'] );

						if ( in_array( $field_name, array( 'repeater', 'group' ), true ) ) {
							continue;
						}

						if ( ! empty( $field_name ) ) {
							$base_alias = $detail['alias'];

							if ( isset( $detail['repeaterAlias'] ) ) {
								if ( strpos( $detail['repeaterAlias'], 'repeater_field_id' ) !== false ) {
									if ( preg_match( '/^(.+)_(\d+)$/', $detail['alias'], $matches ) ) {
										$base_alias = $matches[1];
									}
								} elseif ( strpos( $detail['repeaterAlias'], 'group_field_id' ) !== false ) {
									$base_alias = $detail['alias'];
								}
							}

							if ( in_array( $field_name, array( 'dropDown', 'dropDown_with_img', 'radio', 'radio_with_img', 'checkbox', 'checkbox_with_img', 'toggle' ), true ) ) {
								$temp[ $base_alias ][] = $detail;
							} else {
								if ( isset( $detail['value'] ) ) {
									$temp[ $base_alias ][] = $detail['value'];
								}
							}
						}
					}
				}
			}
		}

		foreach ( $temp as $alias => $values ) {
			$field_name = preg_replace( '/_field_id.*/', '', $alias );

			if ( ! in_array( $alias, $processed_fields, true ) ) {
				$field_analytics = null;
				if ( in_array( $field_name, array( 'quantity', 'range', 'multi_range' ), true ) ) {
					$field_analytics = CCBFieldAnalytics::parse_numbers_field( $alias, $temp );
				} elseif ( in_array( $field_name, array( 'dropDown', 'dropDown_with_img', 'radio', 'radio_with_img' ), true ) ) {
					$field_analytics = CCBFieldAnalytics::parse_single_options_field( $alias, $temp );
				} elseif ( in_array( $field_name, array( 'checkbox', 'checkbox_with_img', 'toggle' ), true ) ) {
					$field_analytics = CCBFieldAnalytics::parse_multiple_options_field( $alias, $temp );
				}

				if ( $field_analytics ) {
					$analytics_data     = array_merge( $analytics_data, $field_analytics );
					$processed_fields[] = $alias;
				}
			}
		}

		$interactions_table_name = Orders::_table();
		$orders_table_name       = Orders::_table();
		$interactions_line_chart = 'day';

		if ( ! empty( $_GET['interactions_line_chart'] ) ) {
			$interactions_line_chart = in_array( $period, array( 'today', 'yesterday' ), true ) ? 'hour' : $_GET['interactions_line_chart'];
		}

		$interactions      = self::get_grouped_data_by_period( $period, $interactions_table_name, $interactions_line_chart, $calc_id );
		$interaction_extra = self::get_grouped_data_by_period( $period, $orders_table_name, $interactions_line_chart, $calc_id, $status );
		$interactions_cats = array_keys( $interactions ) >= array_keys( $interaction_extra ) ? array_keys( $interactions ) : array_keys( $interaction_extra );

		$base_analytics = array(
			'title'       => __( 'Orders vs Interactions', 'cost-calculator-builder-pro' ),
			'type'        => 'line',
			'series'      => array_values( $interactions ),
			'categories'  => $interactions_cats,
			'minValue'    => 0,
			'extraSeries' => array_values( $interaction_extra ),
			'key'         => 'interactions_line_chart',
		);

		$fields_settings  = array();
		$fields_analytics = array_merge( array( $base_analytics ), array_values( $analytics_data ) );

		$analytics_settings = get_option( 'ccb_analytics_data', array() );

		foreach ( $fields_analytics as $field_analytics ) {
			$fields_settings[ $field_analytics['key'] ] = true;
		}

		$page_settings    = ! empty( $analytics_settings[ $calc_id ]['page_settings'] ) ? $analytics_settings[ $calc_id ]['page_settings'] : array();
		$widgets          = ! empty( $analytics_settings[ $calc_id ]['widgets'] ) ? $analytics_settings[ $calc_id ]['widgets'] : array();
		$saved_charts     = ! empty( $page_settings['charts'] ) ? $page_settings['charts'] : array();
		$widgets_settings = array();

		$available_widgets = self::get_available_widgets( $calc_id, $period, $extra, $status );

		if ( ! empty( $widgets ) ) {
			$new_widgets_data = array();
			foreach ( $widgets as $widget ) {
				$current = array_filter(
					$available_widgets,
					function( $item ) use ( $widget ) {
						return $item['key'] === $widget['key'];
					}
				);

				if ( ! empty( $current ) ) {
					$current            = array_values( $current )[0];
					$current['size']    = $widget['size'];
					$new_widgets_data[] = $current;
				} elseif ( ! empty( $analytics_data[ $widget['key'] ] ) ) {
					$current         = $analytics_data[ $widget['key'] ];
					$current['size'] = $widget['size'];
					unset( $current['categories'] );

					$current['value']   = array_sum( $current['series'] );
					$current['series']  = array(
						array(
							'name' => $current['title'],
							'data' => $current['series'],
						),
					);
					$new_widgets_data[] = $current;
				} else {
					$new_widgets_data[] = $widget;
				}
			}

			if ( ! empty( $new_widgets_data ) ) {
				$widgets = $new_widgets_data;
			}
		}

		if ( ! empty( $analytics_data ) ) {
			foreach ( $analytics_data as $data ) {
				unset( $data['categories'] );
				$data['value']       = array_sum( $data['series'] );
				$data['series']      = array(
					array(
						'name' => $data['title'],
						'data' => $data['series'],
					),
				);
				$available_widgets[] = $data;
			}
		}

		if ( ! empty( $widgets ) ) {
			foreach ( $widgets as $widget ) {
				$widgets_settings[ $widget['key'] ] = true;
			}
		}

		$charts = array_merge( $fields_settings, $saved_charts );

		$saved_page_settings = ! empty( $page_settings['widgets'] ) ? $page_settings['widgets'] : array();

		if ( ! empty( $widgets ) ) {
			foreach ( $widgets as $widget ) {
				$widgets_settings[ $widget['key'] ] = true;
			}
		}

		$widgets_settings = array_merge( $widgets_settings, $saved_page_settings );

		$calc_settings   = CCBSettingsData::get_calc_single_settings( $calc_id );
		$global_settings = CCBSettingsData::get_calc_global_settings();

		$currency = ! empty( $calc_settings['currency']['currency'] ) ? $calc_settings['currency']['currency'] : '$';
		if ( ! empty( $global_settings['currency']['use_in_all'] ) ) {
			$currency = $global_settings['currency']['currency'];
		}

		return array(
			'carts'             => $widgets,
			'charts'            => $fields_analytics,
			'currency'          => $currency,
			'settings'          => array(
				'charts'  => $charts,
				'widgets' => $widgets_settings,
			),
			'available_widgets' => $available_widgets,
		);
	}

	public static function get_orders_by_calc( $calc_id, $period, $status = 'all' ) {
		global $wpdb;
		$custom_date = null;
		if ( 'custom' === $period && isset( $_GET['extra'] ) ) {
			$custom_date = sanitize_text_field( $_GET['extra'] );
		}

		$periods = ccb_get_date_range_for_period_pro( $period, $custom_date );
		$from    = $periods['from'];
		$to      = $periods['to'];

		$whereClause = '';
		if ( $from && $to ) {
			$whereClause = sprintf(
				'WHERE created_at BETWEEN "%s" AND "%s" + INTERVAL 1 DAY AND calc_id = %d',
				$from->format( 'Y-m-d H:i:s' ),
				$to->format( 'Y-m-d H:i:s' ),
				$calc_id,
			);
		} else {
			$whereClause = sprintf(
				'WHERE calc_id = %d',
				$calc_id
			);
		}

		if ( 'all' !== $status ) {
			$status_value = 'completed' === $status ? 'complete' : $status;
			$whereClause .= sprintf( ' AND status = "%s"', $status_value );
		}

		// phpcs:disable
		$sql = sprintf(
		'SELECT * FROM %s %s',
		Orders::_table(),
			$whereClause
		);
		// phpcs:enable

		return $wpdb->get_results( $sql ); // phpcs:ignore;
	}

	public static function get_calculators_list() {
		global $wpdb;

		$sql = sprintf(
			'SELECT DISTINCT calc_id FROM %s ORDER BY created_at DESC',
			Orders::_table()
		);

		$db_results   = $wpdb->get_results( $sql, ARRAY_A ); // phpcs:ignore
		$all_calc_ids = $wpdb->get_results( 'SELECT ID FROM ' . $wpdb->posts . ' WHERE post_type = "cost-calc" AND post_status = "publish"', ARRAY_A );

		// phpcs:disable
		$all_calc_ids = array_map( function( $item ) {
			return intval( $item['ID'] );
		}, $all_calc_ids );
		// phpcs:enable

		foreach ( $db_results as $result ) {
			$label         = get_post_meta( $result['calc_id'], 'stm-name', true );
			$calculators[] = array(
				'id'      => $result['calc_id'],
				'label'   => ! empty( $label ) ? $label : 'Untitled',
				'slug'    => '',
				'deleted' => ! in_array( intval( $result['calc_id'] ), $all_calc_ids, true ),
			);
		}

		return $calculators;
	}

	public static function get_available_widgets( $calc_id, $period, $extra = null, $status = 'all' ) {
		$available_widgets = array(
			'revenue'      => array(
				'title'   => __( 'Revenue', 'cost-calculator-builder-pro' ),
				'sub'     => '',
				'value'   => 0,
				'series'  => array(),
				'average' => 0,
				'key'     => 'revenue',
			),
			'orders'       => array(
				'title'   => __( 'Orders', 'cost-calculator-builder-pro' ),
				'sub'     => '',
				'value'   => 0,
				'series'  => array(),
				'average' => 0,
				'key'     => 'orders',
			),
			'views'        => array(
				'title'   => __( 'Views', 'cost-calculator-builder-pro' ),
				'sub'     => '',
				'value'   => 0,
				'series'  => array(),
				'average' => 0,
				'key'     => 'views',
			),
			'interactions' => array(
				'title'   => __( 'Interactions', 'cost-calculator-builder-pro' ),
				'sub'     => '',
				'value'   => 0,
				'series'  => array(),
				'average' => 0,
				'key'     => 'interactions',
			),
		);

		$orders_table_name       = Orders::_table();
		$views_table_name        = AnalyticsViews::_table();
		$interactions_table_name = AnalyticsInteractions::_table();

		$orders = array();

		if ( in_array( $period, array( 'today', 'yesterday' ), true ) ) {
			$cart_grouping = 'hour';
		} elseif ( 'custom' === $period ) {
			$custom_date   = isset( $_GET['extra'] ) ? sanitize_text_field( $_GET['extra'] ) : '';
			$periods       = ccb_get_date_range_for_period_pro( $period, $custom_date );
			$cart_grouping = ccb_get_optimal_grouping_for_custom_period( $periods['from'], $periods['to'] );
		} else {
			$cart_grouping = 'week';
		}

		$carts_data = array(
			array(
				'key'   => 'orders',
				'label' => __( 'Orders', 'cost-calculator-builder-pro' ),
				'table' => $orders_table_name,
			),
			array(
				'key'   => 'views',
				'label' => __( 'Views', 'cost-calculator-builder-pro' ),
				'table' => $views_table_name,
			),
			array(
				'key'   => 'interactions',
				'label' => __( 'Interactions', 'cost-calculator-builder-pro' ),
				'table' => $interactions_table_name,
			),
		);

		foreach ( $carts_data as $cart_data ) {
			if ( 'orders' === $cart_data['key'] ) {
				$_data        = self::get_data_by_period( $period, $cart_data['table'], $calc_id, $status );
				$_series_data = self::get_grouped_data_by_period( $period, $cart_data['table'], $cart_grouping, $calc_id, $status );
			} else {
				$_data        = self::get_data_by_period( $period, $cart_data['table'], $calc_id );
				$_series_data = self::get_grouped_data_by_period( $period, $cart_data['table'], $cart_grouping, $calc_id );
			}

			$_series = array_values( $_series_data );

			if ( 'orders' === $cart_data['key'] ) {
				$orders = $_data;
			}

			$cart_total_and_average = self::get_avarage_and_total( $cart_data['table'], $period, $calc_id );

			$available_widgets[ $cart_data['key'] ]['value']  = $cart_total_and_average['total'];
			$available_widgets[ $cart_data['key'] ]['series'] = array(
				array(
					'name' => $cart_data['label'],
					'data' => ccb_get_cart_series( $_series ),
				),
			);
		}

		$revenue = 0;
		$series  = array();

		foreach ( $orders as $order ) {
			$order_meta = get_option( 'calc_meta_data_order_' . $order->id );
			$totals     = json_decode( $order_meta['totals'], true );

			foreach ( $totals as $total ) {
				$revenue += $total['total'];
				$series[] = intval( $total['total'] );
			}
		}

		$available_widgets['revenue']['value']  = $revenue;
		$available_widgets['revenue']['series'] = array(
			array(
				'name' => __( 'Revenue', 'cost-calculator-builder-pro' ),
				'data' => ccb_get_cart_series( $series ),
			),
		);

		return array_values( $available_widgets );
	}

	public static function get_all_calc_fields_for_analytics( $calc_id ) {
		$calc_fields      = get_post_meta( $calc_id, 'stm-fields', true ) ?? array();
		$analytics_fields = array();

		$process_fields = function( $fields ) use ( &$process_fields, &$analytics_fields ) {
			foreach ( $fields as $field ) {
				if ( ! empty( $field['alias'] ) ) {
					$field_name = preg_replace( '/_field_id.*/', '', $field['alias'] );

					$supported_types = array( 'quantity', 'range', 'multi_range', 'dropDown', 'dropDown_with_img', 'radio', 'radio_with_img', 'checkbox', 'checkbox_with_img', 'toggle' );

					if ( in_array( $field_name, $supported_types, true ) ) {
						$analytics_fields[] = array(
							'alias'      => $field['alias'],
							'type'       => $field_name,
							'label'      => $field['label'] ?? $field['text'] ?? 'Untitled',
							'field_data' => $field,
						);
					}
				}

				if ( isset( $field['groupElements'] ) && is_array( $field['groupElements'] ) ) {
					$process_fields( $field['groupElements'] );
				}
			}
		};

		$process_fields( $calc_fields );

		return $analytics_fields;
	}

	public static function initialize_empty_analytics_for_fields( $fields ) {
		$analytics_data = array();

		foreach ( $fields as $field ) {
			$alias      = $field['alias'];
			$type       = $field['type'];
			$label      = $field['label'];
			$field_data = $field['field_data'];

			$empty_analytics = null;

			if ( in_array( $type, array( 'quantity', 'range', 'multi_range' ), true ) ) {
				$empty_analytics = self::create_empty_numbers_analytics( $alias, $label, $field_data );
			} elseif ( in_array( $type, array( 'dropDown', 'dropDown_with_img', 'radio', 'radio_with_img' ), true ) ) {
				$empty_analytics = self::create_empty_single_options_analytics( $alias, $label, $field_data );
			} elseif ( in_array( $type, array( 'checkbox', 'checkbox_with_img', 'toggle' ), true ) ) {
				$empty_analytics = self::create_empty_multiple_options_analytics( $alias, $label, $field_data );
			}

			if ( $empty_analytics ) {
				$analytics_data = array_merge( $analytics_data, $empty_analytics );
			}
		}

		return $analytics_data;
	}


	private static function create_empty_numbers_analytics( $alias, $label, $field_data ) {
		$min = 0;
		$max = 100;

		if ( isset( $field_data['min'] ) && isset( $field_data['max'] ) ) {
			$min = (int) $field_data['min'];
			$max = (int) $field_data['max'];
		}

		if ( isset( $field_data['minValue'] ) && isset( $field_data['maxValue'] ) ) {
			$min = (int) $field_data['minValue'];
			$max = (int) $field_data['maxValue'];
		}

		$range      = $max - $min + 1;
		$max_groups = 20;

		if ( $range <= 20 ) {
			$group_size = 1;
		} else {
			$base_group_size = max( 1, ceil( $range / $max_groups ) );

			if ( $base_group_size <= 5 ) {
				$group_size = $base_group_size;
			} else {
				$group_size = ceil( $base_group_size / 10 ) * 10;
			}
		}

		$num_groups = ceil( $range / $group_size );
		$categories = array();
		$series     = array();

		for ( $i = 0; $i < $num_groups; $i++ ) {
			$start = $min + ( $i * $group_size );
			$end   = min( $start + $group_size - 1, $max );

			if ( $start === $end ) {
				$categories[] = (string) $start;
			} else {
				$categories[] = $start . '-' . $end;
			}

			$series[] = 0;
		}

		return array(
			$alias => array(
				'title'       => $label,
				'type'        => 'bar',
				'series'      => $series,
				'categories'  => $categories,
				'minValue'    => 0,
				'extraSeries' => array(),
				'key'         => $alias,
			),
		);
	}

	private static function create_empty_single_options_analytics( $alias, $label, $field_data ) {
		$options    = $field_data['options'] ?? array();
		$categories = array();
		$series     = array();

		foreach ( $options as $option ) {
			$option_label = $option['label'] ?? 'No label';
			$categories[] = $option_label;
			$series[]     = 0;
		}

		return array(
			$alias => array(
				'title'       => $label,
				'type'        => 'bar',
				'series'      => $series,
				'categories'  => $categories,
				'minValue'    => 0,
				'extraSeries' => array(),
				'key'         => $alias,
			),
		);
	}

	private static function create_empty_multiple_options_analytics( $alias, $label, $field_data ) {
		$options    = $field_data['options'] ?? array();
		$categories = array();
		$series     = array();

		foreach ( $options as $option ) {
			$option_label = $option['label'] ?? 'No label';
			$categories[] = $option_label;
			$series[]     = 0;
		}

		return array(
			$alias => array(
				'title'       => $label,
				'type'        => 'bar',
				'series'      => $series,
				'categories'  => $categories,
				'minValue'    => 0,
				'extraSeries' => array(),
				'key'         => $alias,
			),
		);
	}

	private static function parse_grouping( $grouping, $period ) {
		if ( in_array( $period, array( 'today', 'yesterday' ), true ) ) {
			$grouping['weekdays_columns_chart']   = 'hour';
			$grouping['revenue_line_chart_chart'] = 'hour';
			$grouping['interactions_line_chart']  = 'hour';
		}

		return $grouping;
	}
}
