<?php
/**
 * Assets Controller Class
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Controllers;

use Elementor\Plugin;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SB\Helpers\BuilderFns;
use RadiusTheme\SBPRO\Helpers\BuilderFunPro;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SB\Controllers\AssetRegistry;
use RadiusTheme\SB\Controllers\AssetsController;

// Do not allow directly accessing this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit( 'This script cannot be accessed directly.' );
}
/**
 * Assets Controller Class
 */
class AssetsControllerPro {

	/**
	 * Singleton.
	 */
	use SingletonTrait;

	/**
	 * Plugin version
	 *
	 * @var string
	 */
	private $version;

	/**
	 * Styles.
	 *
	 * @var array
	 */
	private $styles = [];

	/**
	 * Ajax URL
	 *
	 * @var string
	 */
	private static $ajaxurl;

	/**
	 * Scripts.
	 *
	 * @var array
	 */
	private $scripts = [];

	/**
	 * Class Constructor
	 */
	public function __construct() {
		$this->version = ( defined( 'WP_DEBUG' ) && WP_DEBUG ) ? time() : RTSBPRO_VERSION;

		// phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
		if ( in_array( 'sitepress-multilingual-cms/sitepress.php', get_option( 'active_plugins' ) ) ) {
			self::$ajaxurl = admin_url( 'admin-ajax.php?lang=' . ICL_LANGUAGE_CODE );
		} else {
			self::$ajaxurl = admin_url( 'admin-ajax.php' );
		}

		/**
		 * Public scripts.
		 */
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_public_scripts' ], 20 );
		add_action( 'wp_enqueue_scripts', [ $this, 'pro_style_settings' ], 30 );
		/** Add_filter( 'rtsb/optimizer/scripts/shared', [ $this, 'extend_shared_scripts' ], 10, 2 ); */
		add_filter( 'rtsb/optimizer/styles/modules/shared', [ $this, 'extend_shared_module_styles' ] );
		add_filter( 'rtsb/optimizer/styles/elementor/shared', [ $this, 'extend_shared_elementor_styles' ] );
		add_filter( 'rtsb/optimizer/styles/shared/components', [ $this, 'extend_shared_component_styles' ] );
		add_filter( 'rtsb/optimizer/scripts/shared/components', [ $this, 'extend_shared_component_scripts' ] );
		add_filter( 'rtsb/optimizer/scripts/deps', [ $this, 'extend_shared_dependencies' ] );
		add_filter( 'rtsb/optimizer/module/remap', [ $this, 'extend_module_remap' ] );
		add_filter( 'rtsb/optimizer/modules/js_context', [ $this, 'extend_module_js_context' ] );
		add_filter( 'rtsb/optimizer/context_detect', [ $this, 'extend_context_detect' ] );

		/**
		 * Admin scripts.
		 */
		add_action( 'admin_enqueue_scripts', [ $this, 'register_backend_assets' ], 1 );
		add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_backend_scripts' ], 15 );
	}

	/**
	 * Registers Admin scripts.
	 *
	 * @return void
	 */
	public function register_backend_assets() {
		/**
		 * Styles.
		 */
		wp_register_style( 'rtsb-product-admin', rtsbpro()->get_assets_uri( 'css/backend/product-admin.css' ), '', $this->version );
		wp_register_style( 'gift-card-metabox', rtsbpro()->get_assets_uri( 'css/backend/gift-card-metabox.css' ), '', $this->version );

		/**
		 * Scripts.
		 */
		wp_register_script( 'rtsb-admin-license', rtsbpro()->get_assets_uri( 'js/backend/license-settings.js' ), [ 'jquery' ], $this->version, true );
		wp_register_script( 'rtsb-product-admin', rtsbpro()->get_assets_uri( 'js/backend/product-admin.js' ), [ 'jquery' ], $this->version, true );
	}

	/**
	 * Enqueues admin scripts.
	 *
	 * @return void
	 */
	public function enqueue_backend_scripts() {
		global $pagenow, $typenow;

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( 'admin.php' === $pagenow && ! empty( $_GET['page'] ) && 'rtsb-license' === $_GET['page'] ) {
			wp_enqueue_script( 'rtsb-admin-license' );
			wp_localize_script(
				'rtsb-admin-license',
				'rtsbLicenseParams',
				[
					'ajaxurl' => esc_url( self::$ajaxurl ),

				]
			);
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$is_orders_page = ( isset( $_REQUEST['post_type'] ) && 'shop_order' === sanitize_text_field( wp_unslash( $_REQUEST['post_type'] ) ) ) || ( isset( $_REQUEST['page'] ) && 'wc-orders' === sanitize_text_field( wp_unslash( $_REQUEST['page'] ) ) ) || 'shop_order' === $typenow;
		$screen         = get_current_screen();
		$is_coupon_page = ( 'post.php' === $pagenow || 'post-new.php' === $pagenow || 'edit.php' === $pagenow ) && 'shop_coupon' === $screen->post_type;

		if ( 'product' === $typenow || $is_orders_page || $is_coupon_page ) {
			wp_enqueue_style( 'rtsb-product-admin' );
		}

		if ( $this->is_product_edit_page() ) {
			global $post;

			$screen    = get_current_screen();
			$screen_id = $screen ? $screen->id : '';

			wp_enqueue_script( 'rtsb-product-admin' );

			wp_localize_script(
				'rtsb-product-admin',
				'rtsbAdminParamsPro',
				[
					'ajaxurl'       => admin_url( 'admin-ajax.php' ),
					rtsb()->nonceId => wp_create_nonce( rtsb()->nonceText ),
					'post_id'       => 'product' === $screen_id ? $post->ID : null,
					'addOnActive'   => FnsPro::is_module_active( 'product_add_ons' ) ? 'yes' : 'no',
				]
			);
		}
	}

	/**
	 * Localized Data.
	 *
	 * @static
	 * @return void
	 */
	public static function localizeData() {
		$mini_cart                 = Fns::get_options( 'modules', 'mini_cart' );
		$is_mini_cart_active       = ! empty( $mini_cart['active'] ) && 'on' === $mini_cart['active'];
		$mini_cart_custom_selector = $mini_cart['mini_cart_custom_selector'] ?? '';
		$time_text                 = apply_filters(
			'rtsb/localize/script/time/text',
			[
				'day'    => esc_html__( 'Day', 'shopbuilder-pro' ),
				'hour'   => esc_html__( 'Hr', 'shopbuilder-pro' ),
				'minute' => esc_html__( 'Min', 'shopbuilder-pro' ),
				'second' => esc_html__( 'Sec', 'shopbuilder-pro' ),
			]
		);

		$handle = Fns::optimized_handle( 'rtsb-public' );

		wp_localize_script(
			$handle,
			'rtsbPublicParamsPro',
			[
				'day'                             => $time_text['day'] ?? esc_html__( 'Day', 'shopbuilder-pro' ),
				'hour'                            => $time_text['hour'] ?? esc_html__( 'Hr', 'shopbuilder-pro' ),
				'minute'                          => $time_text['minute'] ?? esc_html__( 'Min', 'shopbuilder-pro' ),
				'second'                          => $time_text['second'] ?? esc_html__( 'Sec', 'shopbuilder-pro' ),
				'qucikCheckout'                   => esc_html__( 'Quick Checkout', 'shopbuilder-pro' ),
				'CouponSuccessText'               => esc_html__( 'Coupon applied!', 'shopbuilder-pro' ),
				'CouponUnSuccessText'             => esc_html__( 'Coupon could not be applied.', 'shopbuilder-pro' ),
				'CouponErrorOccurred'             => esc_html__( 'An error occurred.', 'shopbuilder-pro' ),
				'isMiniCartActive'                => $is_mini_cart_active,
				'miniCartCustomSelector'          => $mini_cart_custom_selector,
				'filterErrorText'                 => esc_html__( 'Filtering error occurred. Please ensure that the filter settings are correctly configured.', 'shopbuilder-pro' ),
				'filterSearchText'                => esc_html__( 'Search Terms', 'shopbuilder-pro' ),
				'urlQSExceptions'                 => apply_filters(
					'rtsb/elements/widget/url/query_exceptions',
					[
						'orderby',
						'displayview',
						'customize_changeset_uuid',
						'customize_theme',
						'customize_messenger_channel',
						'elementor-preview',
						'post_type',
						'shopview',
						'preview_id',
						'preview_nonce',
						'preview',
						'd',
						'fbclid',
					]
				),
				'resetFilterText'                 => apply_filters( 'rtsb/elements/elementor/reset_filter_text', esc_html__( 'Clear all filters', 'shopbuilder-pro' ) ),
				// Step checkout.
				'keyboard_nav'                    => true,
				'hide_last_prev'                  => true,
				'hide_last_back_to_cart'          => true,
				'skip_login_above_form'           => true,
				'skip_login_next_to_login_button' => true,
				'addonImageUploadAlert'           => esc_html__( 'You can only upload one image at a time. Please remove the current image before uploading a new one.', 'shopbuilder-pro' ),
			]
		);
	}

	/**
	 * Get all frontend scripts.
	 *
	 * @return void
	 */
	private function get_public_assets() {
		$this->get_public_styles()->get_public_scripts();
	}

	/**
	 * Get public styles.
	 *
	 * @return object
	 */
	private function get_public_styles() {
		$rtl_suffix = is_rtl() ? '-rtl' : '';
		$rtl_dir    = is_rtl() ? trailingslashit( 'rtl' ) : trailingslashit( 'css' );

		if ( Fns::is_optimization_enabled() ) {
			$this->styles[] = AssetRegistry::instance()->load_optimized_assets();
		} else {
			$this->styles[] = [
				'handle' => 'rtsb-general-addons',
				'src'    => rtsbpro()->get_assets_uri( $rtl_dir . 'frontend/general-addons' . $rtl_suffix . '.css' ),
			];
			$this->styles[] = [
				'handle' => 'rtsb-frontend',
				'src'    => rtsbpro()->get_assets_uri( $rtl_dir . 'frontend/frontend' . $rtl_suffix . '.css' ),
			];
		}

		return $this;
	}

	/**
	 * Get public scripts.
	 *
	 * @return object
	 */
	private function get_public_scripts() {
		$frontend_js             = 'frontend.js';
		$is_filter_widget_active = Fns::get_options( 'elements', 'ajax_product_filters' )['active'] ?? 'on';

		if ( ( BuilderFns::is_archive() || BuilderFns::is_shop() ) && $is_filter_widget_active ) {
			$frontend_js = 'frontend-archive.js';
		}
		$this->scripts[] = [
			'handle' => 'swiper-animation',
			'src'    => rtsbpro()->get_assets_uri( 'vendors/swiper/swiper.animation.min.js' ),
			'deps'   => [ 'jquery' ],
			'footer' => true,
		];

		if ( Fns::is_optimization_enabled() ) {
			$this->scripts[] = AssetRegistry::instance()->load_optimized_assets( 'js' );
		} else {
			$this->scripts[] = [
				'handle' => 'rtsb-public',
				'src'    => rtsbpro()->get_assets_uri( 'js/frontend/' . $frontend_js ),
				'deps'   => defined( 'ELEMENTOR_VERSION' ) && Plugin::$instance->preview->is_preview_mode() ? [ 'jquery', 'swiper-animation', 'imagesloaded', 'swiper', 'masonry' ] : [ 'jquery', 'swiper-animation', 'imagesloaded', 'swiper' ],
				'footer' => true,
			];
		}

		return $this;
	}

	/**
	 * Register public scripts.
	 *
	 * @return void
	 */
	public function register_public_scripts() {
		$this->get_public_assets();
		$this->styles  = array_filter( $this->styles );
		$this->scripts = array_filter( $this->scripts );

		wp_deregister_style( Fns::optimized_handle( 'rtsb-frontend' ) );
		wp_deregister_script( Fns::optimized_handle( 'rtsb-public' ) );

		/**
		 * No need re enqueue 'rtsb-frontend' and 'rtsb-public', The handle already enqueue from the free version
		 */
		// Register public styles.
		foreach ( $this->styles as $style ) {
			wp_register_style( $style['handle'], $style['src'], '', $this->version );
		}

		// Register public scripts.
		foreach ( $this->scripts as $script ) {
			wp_register_script( $script['handle'], $script['src'], $script['deps'], $this->version, $script['footer'] );
		}

		AssetsController::localizeData();
		self::localizeData();
	}

	/**
	 * Enqueues public scripts.
	 *
	 * @return void
	 */
	public function enqueue_public_scripts() {
		/**
		 * Register scripts.
		 */
		$this->register_public_scripts();

		/**
		 * Enqueue Script Below if need
		 */
		if ( defined( 'ELEMENTOR_VERSION' ) && Fns::should_load_elementor_scripts() ) {
			wp_enqueue_script( 'rtsb-sticky-sidebar' );
		}

		if ( Fns::is_shop_or_archive() ) {
			wp_enqueue_style( 'rtsb-noui-slider' );
		}
	}

	/**
	 * Enqueues general public scripts.
	 *
	 * @return void
	 */
	public function pro_style_settings() {
		$tooltip_font_size   = absint( Fns::get_option( 'general', 'tooltip', 'tooltip_font_size', 13 ) );
		$tooltip_font_weight = absint( Fns::get_option( 'general', 'tooltip', 'tooltip_font_weight', 500 ) );
		$tooltip_color       = Fns::get_option( 'general', 'tooltip', 'tooltip_color', '#fff' );
		$tooltip_color_bg    = Fns::get_option( 'general', 'tooltip', 'tooltip_bg', '#000' );

		$dynamic_css = '';

		if ( ! empty( $tooltip_font_size ) ) {
			$dynamic_css .= "body .tipsy .tipsy-inner { font-size: {$tooltip_font_size}px !important; }";
		}

		if ( ! empty( $tooltip_font_weight ) ) {
			$dynamic_css .= "body .tipsy .tipsy-inner { font-weight: {$tooltip_font_weight} !important; }";
		}

		if ( ! empty( $tooltip_color ) ) {
			$dynamic_css .= "body .tipsy .tipsy-inner { color: {$tooltip_color} !important; }";
		}

		if ( ! empty( $tooltip_color_bg ) ) {
			$dynamic_css .= "body .tipsy .tipsy-inner { background-color: {$tooltip_color_bg} !important; }";
			$dynamic_css .= "body .tipsy.tipsy-s .tipsy-arrow { border-top-color: {$tooltip_color_bg} !important; }";
			$dynamic_css .= "body .tipsy.tipsy-e .tipsy-arrow { border-left-color: {$tooltip_color_bg} !important; }";
			$dynamic_css .= "body .tipsy.tipsy-w .tipsy-arrow { border-right-color: {$tooltip_color_bg} !important; }";
			$dynamic_css .= "body .tipsy.tipsy-n .tipsy-arrow { border-bottom-color: {$tooltip_color_bg} !important; }";
		}

		if ( ! empty( $dynamic_css ) ) {
			wp_add_inline_style( Fns::optimized_handle( 'rtsb-frontend' ), $dynamic_css );
		}
	}

	/**
	 * Inject shared script dependencies for Pro modules like pre-order and back-order.
	 *
	 * @param array $shared_scripts Existing shared script paths.
	 * @param array $modules        Enabled core modules.
	 *
	 * @return array
	 */
	public function extend_shared_scripts( $shared_scripts, $modules ) {
		$contextual_loading = Fns::is_contextual_loading();

		if ( array_intersect( [ 'pre-order', 'flash-sale-countdown' ], $modules ) ) {
			if ( $contextual_loading ) {
				$contexts = Fns::is_module_active( 'quick_view' ) ? [ 'global' ] : [ 'shop', 'product' ];

				foreach ( $contexts as $ctx ) {
					$shared_scripts[ $ctx ][] = rtsbpro()->get_assets_path( 'js/modules/shared/countdown.js' );
				}
			} else {
				$shared_scripts[] = rtsbpro()->get_assets_path( 'js/modules/shared/countdown.js' );
			}
		}

		return $shared_scripts;
	}

	/**
	 * Inject shared style dependencies for Pro modules.
	 *
	 * @param array $shared_styles_map Existing shared style map.
	 *
	 * @return array
	 */
	public function extend_shared_module_styles( $shared_styles_map ) {
		$shared_styles_map[] = [
			'modules'   => [ 'pre-order', 'flash-sale-countdown' ],
			'file_name' => 'shared/countdown',
			'context'   => Fns::is_module_active( 'quick_view' ) ? [ 'global' ] : [ 'shop', 'product' ],
		];

		return $shared_styles_map;
	}

	/**
	 * Inject shared style dependencies for Pro widgets.
	 *
	 * @param array $shared_styles_map Existing shared style map.
	 *
	 * @return array
	 */
	public function extend_shared_elementor_styles( $shared_styles_map ) {
		$shared_styles_map[] = [
			'widgets'   => [
				'products-grid',
				'products-list',
				'products-slider',
			],
			'file_name' => 'shared/general-pro',
			'context'   => [ 'global' ],
		];

		$shared_styles_map[] = [
			'widgets'   => [
				'order-pay-form',
				'account-navigation-edit-shipping',
				'account-dashboard',
				'account-orders',
				'account-downloads',
				'account-billing-address',
				'account-shipping-address',
				'account-address-description',
				'account-details',
				'account-order-status',
				'account-order-details-download',
				'account-order-details-note',
				'account-order-details-table',
				'account-order-details-order-again',
				'account-order-details-order-shipping',
				'account-order-details-order-billing',
				'account-edit-billing-address',
				'account-edit-shipping-address',
			],
			'file_name' => 'shared/my-account',
			'context'   => [ 'account' ],
		];

		$shared_styles_map[] = [
			'widgets'   => [
				'products-grid',
				'products-list',
				'products-slider',
				'products-archive-catalog-custom',
				'upsells-product-custom',
				'upsells-products-slider-custom',
				'related-product-custom',
				'related-products-slider-custom',
				'cross-sells-custom',
				'cross-sells-custom-slider',
				'product-stock-counter',
			],
			'file_name' => 'shared/stock-count',
			'context'   => [ 'global' ],
		];

		$shared_styles_map[] = [
			'widgets'   => [
				'order-received-text',
				'order-details-summary',
				'order-details-table',
				'downloadable-products',
				'order-billing-address',
				'order-shipping-address',
				'order-custom-fields-data',
			],
			'file_name' => 'shared/order-received',
			'context'   => [ 'account' ],
		];

		$shared_styles_map[] = [
			'widgets'   => [
				'product-categories',
				'products-single-cateogory',
			],
			'file_name' => 'shared/product-categories',
			'context'   => [ 'global' ],
		];

		return $shared_styles_map;
	}

	/**
	 * Inject shared style dependencies for modules & widgets.
	 *
	 * @param array $styles Existing shared style map.
	 *
	 * @return array
	 */
	public function extend_shared_component_styles( $styles ) {
		return $styles + [
			'quantity'            => [
				'file_name' => 'shared/quantity',
				'modules'   => [ 'sticky-add-to-cart' ],
				'widgets'   => [ 'cart-table', 'product-add-to-cart' ],
				'context'   => Fns::is_module_active( 'quick_view' ) ? [ 'global' ] : [ 'product', 'cart' ],
			],
			'tippy'               => [
				'file_name' => 'shared/tippy',
				'modules'   => [ 'smart-coupons', 'bulk-discounts', 'variation-swatches', 'bogo-bxgy-free-gift-discounts' ],
				'widgets'   => [ 'product-lookbook' ],
				'context'   => [ 'global' ],
			],
			'coupons'             => [
				'file_name' => 'shared/coupons',
				'modules'   => [ 'smart-coupons', 'bogo-bxgy-free-gift-discounts' ],
				'widgets'   => [],
				'context'   => Fns::is_module_active( 'quick_checkout' ) ? [ 'global' ] : [ 'cart', 'checkout' ],
			],
			'product-add-to-cart' => [
				'file_name' => 'shared/product-add-to-cart',
				'modules'   => [],
				'widgets'   => [ 'highlighted-product' ],
				'context'   => Fns::is_elementor_widget_active( 'highlighted_product' ) ? [ 'global' ] : [ 'product' ],
			],
		];
	}

	/**
	 * Inject shared script dependencies for modules & widgets.
	 *
	 * @param array $scripts Shared scripts.
	 *
	 * @return array
	 */
	public function extend_shared_component_scripts( $scripts ) {
		$scripts['tippy'] = [
			'path'    => rtsbpro()->get_assets_path( 'js/modules/shared/tippy.js' ),
			'modules' => [ 'smart-coupons', 'bulk-discounts', 'variation-swatches', 'bogo-bxgy-free-gift-discounts' ],
			'widgets' => [],
		];

		$scripts['single-add-to-cart'] = [
			'path'    => rtsbpro()->get_assets_path( 'js/modules/shared/product-add-to-cart.js' ),
			'modules' => [],
			'widgets' => [ 'highlighted-product' ],
			'context' => Fns::is_elementor_widget_active( 'highlighted_product' ) ? [ 'global' ] : [ 'product' ],
		];

		$scripts['flash-sale-countdown'] = [
			'path'    => rtsbpro()->get_assets_path( 'js/modules/shared/countdown.js' ),
			'modules' => [ 'pre-order', 'flash-sale-countdown', 'quick-view' ],
			'widgets' => [ 'flash-sale-countdown' ],
		];

		return $scripts;
	}
	/**
	 * Shared dependencies.
	 *
	 * @param array $deps Dependencies.
	 *
	 * @return array
	 */
	public function extend_shared_dependencies( $deps ) {
		$deps[] = 'rtsb-sticky-sidebar';

		if ( defined( 'ELEMENTOR_VERSION' ) && Plugin::$instance->preview->is_preview_mode() ) {
			$deps[] = 'masonry';
		}

		if ( Fns::is_module_active( 'smart_coupons' ) ) {
			$deps[] = 'rtsb-tippy';
		}

		if ( Fns::is_elementor_widget_active( 'hero_slider' ) ) {
			$deps[] = 'swiper';
			$deps[] = 'swiper-animation';
		}

		if ( Fns::is_elementor_widget_active( 'ajax_product_filters' ) && Fns::is_shop_or_archive() ) {
			$deps[] = 'rtsb-noui-slider';
		}

		if ( Fns::is_module_active( 'product_add_ons' ) && ( is_product() || BuilderFns::is_product() ) ) {
			$deps[] = 'select2';
		}

		return $deps;
	}

	/**
	 * Module remap.
	 *
	 * @param string $module Module name.
	 *
	 * @return string
	 */
	public function extend_module_remap( $module ) {
		if ( in_array( $module, [ 'bulk-discounts', 'bogo-bxgy-free-gift-discounts' ], true ) ) {
			$module = 'bulk-discount-bogo';
		}

		return $module;
	}

	/**
	 * Module context.
	 *
	 * @param array $context Module context.
	 *
	 * @return array
	 */
	public function extend_module_js_context( $context ) {
		return $context + [
			'multi_step_checkout' => [ 'checkout' ],
			'sticky_add_to_cart'  => [ 'product' ],
			'pre_order'           => [ 'shop', 'product' ],
			'product_size_chart'  => [ 'shop', 'product' ],
			'product_add_ons'     => [ 'shop', 'product' ],
			'partial_pay'         => [ 'shop', 'product' ],
			'gift_card'           => [ 'cart', 'checkout' ],
		];
	}

	/**
	 * Context detection.
	 *
	 * @param string $context Context.
	 *
	 * @return string
	 */
	public function extend_context_detect( $context ) {
		if ( BuilderFunPro::is_account_page() ||
			BuilderFunPro::is_thank_you() ||
			BuilderFunPro::is_order_pay() ||
			is_order_received_page() ||
			is_checkout_pay_page() ) {
			$context = 'account';
		}

		return $context;
	}

	/**
	 * Checks if the current page is a WooCommerce product edit page in the WordPress admin.
	 *
	 * @return bool
	 */
	private function is_product_edit_page() {
		global $pagenow, $post;

		// Check if we are on the edit page for a product.
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( 'post.php' === $pagenow && isset( $_GET['action'] ) && 'edit' === $_GET['action'] ) {
			if ( isset( $post->ID ) && 'product' === get_post_type( $post->ID ) ) {
				return true;
			}
		}

		// Check if we are on the new post page for a product.
		if ( 'post-new.php' === $pagenow ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$post_type = isset( $_GET['post_type'] ) ? sanitize_text_field( wp_unslash( $_GET['post_type'] ) ) : 'post';

			if ( 'product' === $post_type ) {
				return true;
			}
		}

		return false;
	}
}
