<?php
/**
 * Main FilterHooks class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Controllers\Hooks;

use WP_Query;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SB\Helpers\BuilderFns;
use RadiusTheme\SB\Models\TemplateSettings;
use RadiusTheme\SBPRO\Helpers\BuilderFunPro;
use RadiusTheme\SBPRO\Elementor\Helpers\ControlSelectorPro;
use RadiusTheme\SBPRO\Controllers\Frontend\Shortcodes\CouponList;
use RadiusTheme\SBPRO\Controllers\Frontend\Shortcodes\GiftCardDisplay;
use RadiusTheme\SBPRO\Modules\CustomizeMyAccount\CustomizeMyAccountFns;
use RadiusTheme\SBPRO\Controllers\Frontend\Shortcodes\ProductStockCount;
use RadiusTheme\SBPRO\Elementor\Widgets\{
	Single,
	Archive,
	Cart,
	Thankyou,
	MyAccount,
	OrderPay,
	Checkout,
	General
};

defined( 'ABSPATH' ) || exit();

/**
 * Main FilterHooks class.
 */
class FilterHooksPro {
	/**
	 * Init Hooks.
	 *
	 * @return void
	 */
	public static function init_hooks() {
		add_filter( 'rtsb/columns/template/types/text', [ __CLASS__, 'product_columns_template_types_text' ], 15, 3 );
		add_filter( 'rtsb/columns/template/types/text', [ __CLASS__, 'archive_columns_template_types_text' ], 15, 3 );
		add_filter( 'rtsb/elements/elementor/widget/selectors', [ __CLASS__, 'widget_selectors' ], 15 );
		add_filter( 'rtsb/builder/content/parent_class', [ __CLASS__, 'class_added_template_before_content' ] );
		add_filter( 'rtsb/wcqv/product/content', [ __CLASS__, 'wcqv_product_content' ], 15 );
		add_filter( 'rtsb/product/page/builder/selected/products', [ __CLASS__, 'product_page_builder_selected_products' ], 11, 2 );
		add_filter( 'rtsb/template/builder/selected/categories', [ __CLASS__, 'page_template_builder_selected_categories' ], 11, 3 );
		add_filter( 'rtsb/template/builder/selected/tags', [ __CLASS__, 'page_template_builder_selected_tags' ], 11, 3 );

		add_filter( 'rtsb/core/elements/raw_list', [ __CLASS__, 'core_elements_raw_list' ], 15 );
		add_filter( 'rtsb/product/container/class', [ __CLASS__, 'product_container_class' ], 10, 2 );
		add_filter( 'rtsb/product/container/attr', [ __CLASS__, 'product_container_attr' ], 10, 2 );
		add_filter( 'rtsb/settings/tooltip/options', [ __CLASS__, 'tooltip_styles' ] );
		add_filter( 'rtsb/settings/guest_user/options', [ __CLASS__, 'guest_user_settings' ] );
		add_filter( 'rtsb/archive/custom/product/title', [ __CLASS__, 'title_highlight' ] );
		add_filter( 'rtsb/archive/custom/product/excerpt', [ __CLASS__, 'excerpt_highlight' ] );
		add_filter( 'rtsb/multi/step/checkout/widget', [ __CLASS__, 'multi_step_checkout_widget_builder_hooks' ] );

		// Disabling features for Guest Users.
		add_filter( 'woocommerce_get_price_html', [ __CLASS__, 'disable_price' ], 99, 2 );
		add_filter( 'woocommerce_cart_item_price', [ __CLASS__, 'disable_price' ], 99, 2 );
		add_filter( 'rtsb/render/cart/button', [ __CLASS__, 'disable_cart_button' ] );
		add_filter( 'woocommerce_checkout_registration_required', [ __CLASS__, 'disable_checkout' ] );

		// Shortcodes list.
		add_filter( 'rtsb/elements/shortcodes', [ __CLASS__, 'pro_shortcodes' ] );

		// Add SKU in the Search.
		add_filter( 'posts_search', [ __CLASS__, 'search_by_sku' ], 999, 2 );

		// Add my-account custom endpoint support.
		add_filter( 'rtsb/builder/register/page/types', [ __CLASS__, 'register_custom_template' ], 99 );
		add_filter( 'rtsb/builder/set/current/page/type/external', [ __CLASS__, 'set_custom_template' ], 11 );
		add_filter( 'rtsb/editor/enable/custom_endpoint', [ __CLASS__, 'enable_custom_endpoints' ] );
		add_filter( 'rtsb/core/elements/my_account/widget_list', [ __CLASS__, 'endpoint_dynamic_widget' ] );

		add_filter( 'rtsb/module/checkout_fields_editor/options', [ __CLASS__, 'checkout_fields_editor_options' ] );
	}

	/**
	 * @param array $controls Module.
	 * @return array
	 */
	public static function checkout_fields_editor_options( $controls ) {
		$extra_controls     = [
			'checkout_billing_title' => [
				'id'         => 'checkout_billing_title',
				'type'       => 'text',
				'label'      => esc_html__( 'Custom Billing Title', 'shopbuilder-pro' ),
				'help'       => esc_html__( 'Enter a custom title. This title will appear on the order edit page, in the order email, and on the thank you page.', 'shopbuilder-pro' ),
				'tab'        => 'billing_fields',
				'dependency' => [
					'rules' => [
						[
							'item'     => 'modules.checkout_fields_editor.modify_billing_form',
							'value'    => 'on',
							'operator' => '==',
						],
					],
				],
			],
		];
		$controls['fields'] = Fns::insert_controls( 'modify_billing_form', $controls['fields'], $extra_controls, true );
		$extra_controls     = [
			'checkout_shipping_title' => [
				'id'         => 'checkout_shipping_title',
				'type'       => 'text',
				'label'      => esc_html__( 'Custom Shipping Title', 'shopbuilder-pro' ),
				'help'       => esc_html__( 'Enter a custom title. This title will appear on the order edit page, in the order email, and on the thank you page.', 'shopbuilder-pro' ),
				'tab'        => 'shipping_fields',
				'dependency' => [
					'rules' => [
						[
							'item'     => 'modules.checkout_fields_editor.modify_shipping_form',
							'value'    => 'on',
							'operator' => '==',
						],
					],
				],
			],
		];
		$controls['fields'] = Fns::insert_controls( 'modify_shipping_form', $controls['fields'], $extra_controls, true );
		$extra_controls     = [
			'checkout_additional_title' => [
				'id'         => 'checkout_additional_title',
				'type'       => 'text',
				'label'      => esc_html__( 'Custom Information Title', 'shopbuilder-pro' ),
				'help'       => esc_html__( 'Enter a custom title. This title will appear on the order edit page, in the order email, and on the thank you page.', 'shopbuilder-pro' ),
				'tab'        => 'additional_fields',
				'dependency' => [
					'rules' => [
						[
							'item'     => 'modules.checkout_fields_editor.modify_additional_form',
							'value'    => 'on',
							'operator' => '==',
						],
					],
				],
			],
		];
		$controls['fields'] = Fns::insert_controls( 'modify_additional_form', $controls['fields'], $extra_controls, true );
		return $controls;
	}
	/**
	 * @param $list
	 *
	 * @return array
	 */
	public static function core_elements_raw_list( $list ) {

		// General Widget
		$type = BuilderFns::builder_type( get_the_ID() );

		if ( empty( $type ) ) {
			$list['highlighted_product']['base_class'] = General\HighLightedProduct::class;
			$list['highlighted_product']['active']     = 'on';
		}

		// Single Page Widget.
		$list['product_sales_count']['base_class'] = Single\ProductSalesCount::class;
		$list['product_sales_count']['active']     = 'on';

		$list['qv_product_sales_count']['base_class'] = Single\ProductSalesCount::class;
		$list['qv_product_sales_count']['active']     = 'on';

		$list['product_qr_code']['base_class'] = Single\ProductQRCode::class;
		$list['product_qr_code']['active']     = 'on';

		$list['product_size_chart']['base_class'] = Single\ProductSizeChart::class;
		$list['product_size_chart']['active']     = 'on';

		$list['product_quick_checkout']['base_class'] = Single\QuickCheckout::class;
		$list['product_quick_checkout']['active']     = 'on';

		$list['qv_product_quick_checkout']['base_class'] = Single\QuickCheckout::class;
		$list['qv_product_quick_checkout']['active']     = 'on';

		$list['qv_product_size_chart']['base_class'] = Single\ProductSizeChart::class;
		$list['qv_product_size_chart']['active']     = 'on';

		$list['flash_sale_countdown']['base_class'] = Single\FlashSaleCountdown::class;
		$list['flash_sale_countdown']['active']     = 'on';

		$list['advance_product_tabs']['base_class'] = Single\AdvancedProductTabs::class;
		$list['advance_product_tabs']['active']     = 'on';

		$list['product_stock_counter']['base_class'] = Single\ProductStockCount::class;
		$list['product_stock_counter']['active']     = 'on';

		// Archive Page Widget.
		$list['ajax_product_filters']['base_class'] = Archive\AjaxProductFilters::class;
		$list['ajax_product_filters']['active']     = 'on';

		// Cart Page Widget.
		$list['cart_coupon_form']['base_class'] = Cart\CouponForm::class;
		$list['cart_coupon_form']['active']     = 'on';

		// Checkout Page Widget.
		$list['multi_step_checkout_widget']['base_class'] = Checkout\MultiStepCheckout::class;
		$list['multi_step_checkout_widget']['active']     = 'on';

		// Order Thank you page.
		$list['order_received_text']['base_class'] = Thankyou\OrderReceivedText::class;
		$list['order_received_text']['active']     = 'on';

		$list['order_details_summary']['base_class'] = Thankyou\OrderDetailsSummary::class;
		$list['order_details_summary']['active']     = 'on';

		$list['order_details_table']['base_class'] = Thankyou\OrderDetailsTable::class;
		$list['order_details_table']['active']     = 'on';

		$list['downloadable_products']['base_class'] = Thankyou\DownloadableProducts::class;
		$list['downloadable_products']['active']     = 'on';

		$list['order_billing_address']['base_class'] = Thankyou\OrderBillingAddress::class;
		$list['order_billing_address']['active']     = 'on';

		$list['order_shipping_address']['base_class'] = Thankyou\OrderShippingAddress::class;
		$list['order_shipping_address']['active']     = 'on';

		$list['order_custom_fields_data']['base_class'] = Thankyou\CustomFieldsData::class;
		$list['order_custom_fields_data']['active']     = 'on';

		// Order Pay Page Widget.

		$list['order_pay_form']['base_class'] = OrderPay\OrderPayForm::class;
		$list['order_pay_form']['active']     = 'on';

		// My account Page widgets.
		$list['account_navigation_edit_shipping']['base_class'] = MyAccount\AccountNavigation::class;
		$list['account_navigation_edit_shipping']['active']     = 'on';

		$list['account_dashboard']['base_class'] = MyAccount\AccountDashboard::class;
		$list['account_dashboard']['active']     = 'on';

		$list['account_orders']['base_class'] = MyAccount\AccountOrders::class;
		$list['account_orders']['active']     = 'on';

		$list['account_downloads']['base_class'] = MyAccount\AccountDownloads::class;
		$list['account_downloads']['active']     = 'on';

		$list['account_billing_address']['base_class'] = MyAccount\AccountBillingAddress::class;
		$list['account_billing_address']['active']     = 'on';

		$list['account_shipping_address']['base_class'] = MyAccount\AccountShippingAddress::class;
		$list['account_billing_address']['active']      = 'on';

		$list['account_address_description']['base_class'] = MyAccount\AccountAddressDescription::class;
		$list['account_address_description']['active']     = 'on';

		$list['account_details']['base_class'] = MyAccount\AccountDetails::class;
		$list['account_details']['active']     = 'on';

		$list['account_order_status']['base_class'] = MyAccount\AccountOrderStatus::class;
		$list['account_order_status']['active']     = 'on';

		$list['account_order_details_download']['base_class'] = MyAccount\AccountOrderDownload::class;
		$list['account_order_details_download']['active']     = 'on';

		$list['account_order_details_note']['base_class'] = MyAccount\AccountOrderNote::class;
		$list['account_order_details_note']['active']     = 'on';

		$list['account_order_details_table']['base_class'] = MyAccount\AccountOrderTable::class;
		$list['account_order_details_table']['active']     = 'on';

		$list['account_order_details_order_again']['base_class'] = MyAccount\AccountOrderAgain::class;
		$list['account_order_details_order_again']['active']     = 'on';

		$list['account_order_details_order_shipping']['base_class'] = MyAccount\AccountOrderShipping::class;
		$list['account_order_details_order_shipping']['active']     = 'on';

		$list['account_order_details_order_billing']['base_class'] = MyAccount\AccountOrderBilling::class;
		$list['account_order_details_order_billing']['active']     = 'on';

		$list['account_edit_billing_address']['base_class'] = MyAccount\AccountEditBillingAddress::class;
		$list['account_edit_billing_address']['active']     = 'on';

		$list['account_edit_shipping_address']['base_class'] = MyAccount\AccountEditShippingAddress::class;
		$list['account_edit_shipping_address']['active']     = 'on';

		// My account Auth Page.
		$list['account_login']['base_class'] = MyAccount\AccountLogin::class;
		$list['account_login']['active']     = 'on';

		$list['account_login_form']['base_class'] = MyAccount\AccountLoginForm::class;
		$list['account_login_form']['active']     = 'on';

		$list['account_registration_form']['base_class'] = MyAccount\AccountRegistrationForm::class;
		$list['account_registration_form']['active']     = 'on';

		$list['account_lost_password']['base_class'] = MyAccount\AccountLostPassword::class;
		$list['account_lost_password']['active']     = 'on';

		$list['product_lookbook']['base_class'] = General\ProductLookBook\ProductLookBook::class;
		$list['product_lookbook']['active']     = 'on';

		$list['image_hotspots']['base_class'] = General\ImageHotspots\ImageHotspots::class;
		$list['image_hotspots']['active']     = 'on';

		$list['coupon_list']['base_class'] = General\CouponList\CouponList::class;
		$list['coupon_list']['active']     = 'on';
		$list['hero_slider']['base_class'] = General\HeroSlider\HeroSlider::class;
		$list['hero_slider']['active']     = 'on';

		// Quick View.
		// $list['account_lost_password']['base_class'] = MyAccount\AccountLostPassword::class;
		// $list['account_lost_password']['active']     = 'on';

		return $list;
	}

	/**
	 * Get and return the categories selected in an archive page.
	 *
	 * @param array  $categories The categories array.
	 * @param int    $post_id The post ID.
	 * @param string $template_type Template Type.
	 *
	 * @return array The categories selected in the archive page.
	 */
	public static function page_template_builder_selected_categories( $categories, $post_id, $template_type ) {
		$options_name = '';
		if ( empty( $post_id ) ) {
			return $categories;
		}
		if ( 'product' === $template_type ) {
			$options_name = BuilderFns::option_name_product_page_selected_cat( $post_id );
		} elseif ( 'archive' === $template_type ) {
			$options_name = BuilderFns::archive_option_name_by_template_id( $post_id );
		}
		if ( empty( $options_name ) ) {
			return $categories;
		}
		// return TemplateSettings::instance()->get_option( $options_name );.
		$categories = TemplateSettings::instance()->get_option( $options_name );
		return $categories;
	}
	/**
	 * Get and return the categories selected in an archive page.
	 *
	 * @param array  $tags The categories array.
	 * @param int    $post_id The post ID.
	 * @param string $template_type Template Type.
	 *
	 * @return array The categories selected in the archive page.
	 */
	public static function page_template_builder_selected_tags( $tags, $post_id, $template_type ) {
		if ( empty( $post_id ) ) {
			return $tags;
		}
		$tag_options_name = BuilderFns::option_name_product_page_selected_tag( $post_id );
		$tags             = TemplateSettings::instance()->get_option( $tag_options_name );
		return $tags;
	}

	/**
	 * Get selected products ids, saved in current page builder state.
	 *
	 * @param array $product_ids Array of selected product ids.
	 * @param int   $post_id Post id.
	 *
	 * @return array               Array of selected product ids
	 */
	public static function product_page_builder_selected_products( $product_ids, $post_id ) {
		$options_name = BuilderFns::option_name_by_template_id( $post_id );
		$product_ids  = TemplateSettings::instance()->get_option( $options_name );
		return $product_ids;
	}


	/**
	 * Filters the content of product form quick view.
	 *
	 * @param string $html content of product form quick view.
	 *
	 * @return string
	 */
	public static function wcqv_product_content( $html ) {
		$quick_view_id = BuilderFns::builder_page_id_by_type( 'quick-view' );
		if ( ! $quick_view_id ) {
			return $html;
		}
		global $product;
		if ( ! $product instanceof \WC_Product ) {
			return $html;
		}

		// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		return '<div id="rtsb-builder-content" class="rtsb-builder-content ">' . BuilderFns::get_builder_content( $quick_view_id ) . '</div>';
	}

	/**
	 * Filters archive page post types for displaying selected categories text.
	 *
	 * @param string $the_type The post type label name.
	 * @param string $type The post type.
	 * @param int    $post_id The post ID.
	 *
	 * @return string
	 */
	public static function archive_columns_template_types_text( $the_type, $template_type, $post_id ) {
		if ( 'archive' !== $template_type ) {
			return $the_type;
		}
		$categories = apply_filters( 'rtsb/template/builder/selected/categories', [], $post_id, $template_type );
		if ( ! empty( $categories ) && is_array( $categories ) ) {
			$the_type .= ' - Template For Selected Category';
		}

		return $the_type;
	}


	/**
	 * Filters the template types text for product
	 *
	 * @param string $the_type Template type.
	 * @param string $type Post type.
	 * @param int    $post_id Post ID.
	 *
	 * @return string
	 */
	public static function product_columns_template_types_text( $the_type, $type, $post_id ) {
		if ( 'product' !== $type ) {
			return $the_type;
		}
		$product_page_for = get_post_meta( $post_id, '_is_product_page_template_for', true );
		switch ( $product_page_for ) {
			case 'product_cats':
				$the_type .= ' - ' . esc_html__( 'Template For Selected Categories', 'shopbuilder-pro' );
				break;
			case 'product_tags':
				$the_type .= ' - ' . esc_html__( 'Template For Selected Tags', 'shopbuilder-pro' );
				break;
			case 'specific_products':
				$the_type .= ' - ' . esc_html__( 'Template For Selected Products', 'shopbuilder-pro' );
				break;
			default:
		}
		return $the_type;
	}

	/**
	 * Add new selectors to the widget selectors array.
	 *
	 * @param array $selectors The existing widget selectors.
	 *
	 * @return array
	 */
	public static function widget_selectors( $selectors ): array {
		return $selectors + ControlSelectorPro::selectors();
	}

	/**
	 * Append "woocommerce" class in selected pages.
	 *
	 * @param array $classes Array of classes.
	 *
	 * @return array
	 */
	public static function class_added_template_before_content( $classes ) {
		$type            = apply_filters( 'rtsb/builder/set/current/page/type', '' );
		$custom_endpoint = apply_filters( 'rtsb/editor/enable/custom_endpoint', false );

		if ( $custom_endpoint ) {
			$classes[] = 'woocommerce';
		}
		switch ( $type ) {
			case 'order-thank-you':
				$classes[] = 'woocommerce';
				if ( 'astra' === rtsb()->current_theme ) {
					$classes[] = 'woocommerce-order';
				}
				break;
			case 'myaccount-dashboard':
			case 'order-pay':
			case 'myaccount-edit-details':
			case 'myaccount-order-details':
			case 'myaccount-auth':
			case 'myaccount-lost-password':
			case 'myaccount-edit-billing-address':
			case 'myaccount-edit-shipping-address':
			case 'myaccount-address':
			case 'myaccount-downloads':
			case 'myaccount-order':
				$classes[] = 'woocommerce';
				break;
		}

		return $classes;
	}

	/**
	 * @param string $class Class of product container.
	 * @param array  $settings Array of all settings.
	 *
	 * @return mixed|string
	 */
	public static function product_container_class( $class, $settings ) {
		if ( ! empty( $settings['show_product_gallery'] ) && 'yes' === $settings['show_product_gallery'] ) {
			$class .= ' product-inner-gallery-container';
		}

		if ( ! empty( $settings['full_product_col'] ) ) {
			$position = $settings['full_product_col'];
			$class   .= ' full-product-column-' . esc_attr( $position );
		}

		if ( ! empty( $settings['products_per_col']['size'] ) ) {
			$per_column = $settings['products_per_col']['size'];
			$limit      = $settings['posts_limit'] ?? 0;
			$class     .= ' products-per-column-' . esc_attr( $per_column );
			$class     .= $limit ? ' products-limit-' . esc_attr( $limit ) : '';
			$class     .= ! empty( $settings['cols'] ) ? ' total-column-' . esc_attr( $settings['cols'] ) : '';
		}

		return $class;
	}

	/**
	 * @param string $attr Attributes of product container.
	 * @param array  $settings Array of all settings.
	 *
	 * @return string
	 */
	public static function product_container_attr( $attr, $settings ) {
		$per_column = $settings['products_per_col']['size'] ?? 3;
		$attr      .= '; --rtsb-products-per-column: ' . absint( $per_column );

		return $attr;
	}

	/**
	 * Tooltip Styles.
	 *
	 * @param array $settings Settings array.
	 *
	 * @return array
	 */
	public static function tooltip_styles( $settings ) {
		$settings['fields'] = [
			'tooltip_intro'       => [
				'id'   => 'tooltip_intro',
				'type' => 'description',
				'text' => esc_html__( 'Tooltip styles customization.', 'shopbuilder-pro' ),
			],
			'tooltip_font_size'   => [
				'id'    => 'tooltip_font_size',
				'label' => esc_html__( 'Font Size (px)', 'shopbuilder-pro' ),
				'type'  => 'number',
				'value' => 13,
			],
			'tooltip_font_weight' => [
				'id'      => 'tooltip_font_weight',
				'label'   => esc_html__( 'Font Weight', 'shopbuilder-pro' ),
				'type'    => 'select',
				'options' => [
					100 => esc_html__( '100 (Thin)', 'shopbuilder-pro' ),
					200 => esc_html__( '200 (Extra Light)', 'shopbuilder-pro' ),
					300 => esc_html__( '300 (Light)', 'shopbuilder-pro' ),
					400 => esc_html__( '400 (Normal)', 'shopbuilder-pro' ),
					500 => esc_html__( '500 (Medium)', 'shopbuilder-pro' ),
					600 => esc_html__( '600 (Semi Bold)', 'shopbuilder-pro' ),
					700 => esc_html__( '700 (Bold)', 'shopbuilder-pro' ),
					800 => esc_html__( '800 (Extra Bold)', 'shopbuilder-pro' ),
					900 => esc_html__( '900 (Black)', 'shopbuilder-pro' ),
				],
				'value'   => 500,
			],
			'tooltip_color'       => [
				'id'    => 'tooltip_color',
				'label' => esc_html__( 'Text Color', 'shopbuilder-pro' ),
				'type'  => 'color',
			],
			'tooltip_bg'          => [
				'id'    => 'tooltip_bg',
				'label' => esc_html__( 'Background Color', 'shopbuilder-pro' ),
				'type'  => 'color',
			],
		];

		return $settings;
	}

	/**
	 * Tooltip Styles.
	 *
	 * @param array $settings Settings array.
	 *
	 * @return array
	 */
	public static function guest_user_settings( $settings ) {
		$settings['fields'] = [
			'guest_user_intro'   => [
				'id'   => 'guest_user_intro',
				'type' => 'description',
				'text' => __( 'Customize visibility settings for various elements related to user interaction for guest users (non-logged-in users). <br />You can enable or disable specific features and elements to regulate the experience for guest users. <br />If certain features are disabled, guest users will need to log in to enable the feature.', 'shopbuilder-pro' ),
			],
			'hide_pricing'       => [
				'id'    => 'hide_pricing',
				'type'  => 'switch',
				'label' => esc_html__( 'Hide Product Pricing', 'shopbuilder-pro' ),
				'help'  => __( 'Enable this option if you don\'t want the product pricing to show for guest users. <br /><b>Note: </b> This setting will not work on Cart and Checkout page.', 'shopbuilder-pro' ),
			],
			'custom_price_text'  => [
				'type'       => 'text',
				'value'      => esc_html__( 'Login to see prices', 'shopbuilder-pro' ),
				'label'      => esc_html__( 'Custom Price Text', 'shopbuilder-pro' ),
				'help'       => esc_html__( 'Enter custom text that will show instead of product price. Leave blank to hide the price entirely.', 'shopbuilder-pro' ),
				'dependency' => [
					'rules' => [
						[
							'item'     => 'general.guest_user.hide_pricing',
							'value'    => 'on',
							'operator' => '==',
						],
					],
				],
			],
			'hide_add_to_cart'   => [
				'id'    => 'hide_add_to_cart',
				'type'  => 'switch',
				'label' => esc_html__( 'Hide Add to Cart Button', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option if you don\'t want the add to cart button to show for guest users.', 'shopbuilder-pro' ),
			],
			'hide_badges'        => [
				'id'    => 'hide_badges',
				'type'  => 'switch',
				'label' => esc_html__( 'Hide Product Badges', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option if you don\'t want the product badges to show for guest users.', 'shopbuilder-pro' ),
			],
			'hide_cart_page'     => [
				'id'    => 'hide_cart_page',
				'type'  => 'switch',
				'label' => esc_html__( 'Disable Cart Page', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option if you don\'t want the cart page to show for guest users.', 'shopbuilder-pro' ),
			],
			'hide_checkout_page' => [
				'id'    => 'hide_checkout_page',
				'type'  => 'switch',
				'label' => esc_html__( 'Disable Checkout Page', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option if you don\'t want the checkout page to show for guest users.', 'shopbuilder-pro' ),
			],
		];

		return $settings;
	}

	/**
	 * Highlight the search term in the given title.
	 *
	 * @param string $title The original title.
	 *
	 * @return string
	 */
	public static function title_highlight( $title ) {
		$search_term = FnsPro::get_search_term();

		return empty( $search_term ) ? $title : FnsPro::highlight_search_term( $title, $search_term );
	}

	/**
	 * Highlight the search term in the given excerpt.
	 *
	 * @param string $excerpt The original excerpt.
	 *
	 * @return string
	 */
	public static function excerpt_highlight( $excerpt ) {
		$search_term = FnsPro::get_search_term();

		return empty( $search_term ) ? $excerpt : FnsPro::highlight_search_term( $excerpt, $search_term );
	}

	/**
	 * Disable certain features for non-logged-in users.
	 *
	 * @param string $price The product price.
	 * @param object $product The WooCommerce product object.
	 *
	 * @return string
	 */
	public static function disable_price( $price, $product ) {
		if ( ! $product instanceof \WC_Product ) {
			return $price;
		}

		$price_disabled = Fns::is_guest_feature_disabled( 'hide_pricing', '' );
		$custom_price   = Fns::get_option( 'general', 'guest_user', 'custom_price_text', '' );

		if ( $price_disabled ) {
			$price = ! empty( $custom_price ) ? '<span class="rtsb-price-disabled"><a href="' . get_permalink( wc_get_page_id( 'myaccount' ) ) . '">' . esc_html( $custom_price ) . '</a></span>' : '';
		}

		return $price;
	}

	/**
	 * Disable the display of the cart button based on conditions.
	 *
	 * @param string $html The HTML markup for the cart button.
	 *
	 * @return string
	 */
	public static function disable_cart_button( $html ) {
		$add_to_cart_disabled = Fns::is_guest_feature_disabled( 'hide_add_to_cart', '' );

		if ( $add_to_cart_disabled ) {
			$html = '';
		}

		return $html;
	}

	/**
	 * Disable the checkout page.
	 *
	 * @param bool $required The current state of registration requirement.
	 *
	 * @return bool
	 */
	public static function disable_checkout( $required ) {
		$checkout_disabled = Fns::is_guest_feature_disabled( 'hide_checkout_page', '' );

		if ( $checkout_disabled || 'yes' !== get_option( 'woocommerce_enable_guest_checkout' ) ) {
			add_filter( 'woocommerce_checkout_registration_required', '__return_true' );
			add_filter( 'woocommerce_coupons_enabled', '__return_false' );

			$required = true;
		}

		return $required;
	}

	/**
	 * Pro shortcodes list.
	 *
	 * @param array $shortcodes Shortcodes List.
	 *
	 * @return array
	 */
	public static function pro_shortcodes( $shortcodes ) {
		return array_merge(
			$shortcodes,
			[
				CouponList::class,
				ProductStockCount::class,
                GiftCardDisplay::class
			]
		);
	}

	/*
	 * Check if builder has multistep checkout widget.
	 *
	 * @return bool
	 */
	public static function multi_step_checkout_widget_builder_hooks() {
		return ! FnsPro::checkout_page_has_multi_step_widget();
	}

	/**
	 * Search products and product variations by SKU.
	 *
	 * @param string   $search     The search SQL for the main query.
	 * @param WP_Query $query_vars The WP_Query instance (passed by reference).
	 *
	 * @return string
	 */
	public static function search_by_sku( $search, $query_vars ) {
		global $wpdb;

		if ( ! empty( $query_vars->query['s'] ) ) {
			$query = apply_filters(
				'rtsb/search_query/custom_field',
				[
					'relation' => 'OR',
					[
						'key'     => '_sku',
						'value'   => $query_vars->query['s'],
						'compare' => 'LIKE',
					],
				],
				$search,
				$query_vars
			);

			// simple products.
			$args = [
				'posts_per_page' => -1,
				'post_type'      => 'product',
				'meta_query'     => $query, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			];

			$posts    = get_posts( $args );
			$post_ids = [];

			foreach ( $posts as $post ) {
				$post_ids[] = $post->ID;
			}

			// Variable products.
			$args = [
				'posts_per_page' => -1,
				'post_type'      => 'product_variation',
				'meta_query'     => $query, // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			];

			$posts_variation = get_posts( $args );

			if ( empty( $posts_variation ) && empty( $posts ) ) {
				return $search;
			}

			foreach ( $posts_variation as $post ) {
				$post_ids[] = $post->post_parent;
			}

			if ( count( $post_ids ) > 0 ) {
				$search = str_replace( 'AND (((', "AND ((({$wpdb->posts}.ID IN (" . implode( ',', $post_ids ) . ')) OR (', $search );
			}
		}

		return $search;
	}

	/**
	 * Registers custom templates for custom endpoints on the My Account page.
	 *
	 * @param array $templates An array of existing templates.
	 * @return array
	 */
	public static function register_custom_template( $templates ) {
		if ( ! CustomizeMyAccountFns::is_custom_endpoint_enabled() ) {
			return $templates;
		}

		$endpoints = CustomizeMyAccountFns::custom_endpoints()['endpoints'] ?? [];

		foreach ( $endpoints as $endpoint => $endpoint_title ) {
			if ( false !== strpos( $endpoint, '_custom_link_' ) ) {
				continue;
			}

			$templates[ 'myaccount-' . esc_html( $endpoint ) ] = esc_html__( 'Endpoint: ', 'shopbuilder-pro' ) . esc_html( $endpoint_title );
		}

		return $templates;
	}

	/**
	 * Sets the custom template for the current My Account page if a custom endpoint is active.
	 *
	 * @param string $type The current template type.
	 * @return string
	 */
	public static function set_custom_template( $type ) {
		if ( ! FnsPro::is_my_account_page() || ! CustomizeMyAccountFns::is_custom_endpoint_enabled() ) {
			return $type;
		}

		$endpoints = array_keys( CustomizeMyAccountFns::custom_endpoints()['endpoints'] ?? [] );

		foreach ( $endpoints as $endpoint ) {
			if ( false !== strpos( $endpoint, '_custom_link_' ) ) {
				continue;
			}

			if ( BuilderFunPro::is_account_custom_endpoint( $endpoint ) ) {
				$type = 'myaccount-' . $endpoint;
			}
		}

		return $type;
	}

	/**
	 * Checks if any custom endpoints are active for the My Account page.
	 *
	 * @return bool
	 */
	public static function enable_custom_endpoints() {
		if ( ! CustomizeMyAccountFns::is_custom_endpoint_enabled() ) {
			return false;
		}

		$endpoints = array_keys( CustomizeMyAccountFns::custom_endpoints()['endpoints'] ?? [] );

		foreach ( $endpoints as $endpoint ) {
			if ( BuilderFunPro::is_account_custom_endpoint( $endpoint ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Endpoint Dynamic Widget.
	 *
	 * @param array $widgets Widget list.
	 *
	 * @return array
	 */
	public static function endpoint_dynamic_widget( $widgets ) {
		if ( ! CustomizeMyAccountFns::is_custom_endpoint_enabled() ) {
			return $widgets;
		}

		$widgets['account_custom_endpoint_content'] = [
			'id'            => 'account_custom_endpoint_content',
			'title'         => esc_html__( 'Endpoint Dynamic Content', 'shopbuilder-pro' ),
			'base_class'    => MyAccount\EndpointDynamicContent::class,
			'category'      => 'myaccount_dashboard',
			'is_front_page' => 'all_myaccount_dashboard_inner',
			'active'        => 'on',
			'fields'        => [],
		];

		return $widgets;
	}
}
