<?php
/**
 * Back-order Order Status Manager Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Controllers;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Modules\PreOrder\PreOrderFns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Order Status Manager Class.
 */
class OrderStatusController {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Options.
	 *
	 * @var array|mixed
	 */
	private array $options = [];

	const BO_STATUS = 'wc-rtsb-backordered';
	const PO_STATUS = 'wc-rtsb-preordered';
	const PP_STATUS = 'wc-rtsb-partial-pay';

	/**
	 * Class Constructor.
	 */
	private function __construct() {
		/**
		 * Get the options.
		 */
		$this->options = [
			'back_order'     => true,
			'pre_order'      => true,
			'partially_paid' => true,
		];

		/**
		 * Add new order statuses.
		 */
		add_filter( 'wc_order_statuses', [ $this, 'register_order_status' ] );
		add_filter( 'woocommerce_register_shop_order_post_statuses', [ $this, 'register_order_post_status' ] );

		// HPOS Support https://github.com/woocommerce/woocommerce/pull/35442.
		add_filter( 'bulk_actions-woocommerce_page_wc-orders', [ $this, 'define_bulk_actions' ] );
		add_filter( 'handle_bulk_actions-woocommerce_page_wc-orders', [ $this, 'handle_bulk_actions' ], 10, 3 );
		add_filter( 'admin_notices', [ $this, 'custom_notice' ] );

		// Legacy support.
		add_filter( 'bulk_actions-edit-shop_order', [ $this, 'define_bulk_actions_legacy' ], 99 );
		add_action( 'handle_bulk_actions-edit-shop_order', [ $this, 'handle_bulk_actions' ], 20, 3 );
	}

	/**
	 * Register custom order status.
	 *
	 * @param array $statuses Array of statuses.
	 *
	 * @return array
	 */
	public function register_order_status( $statuses ) {
		if ( $this->options['back_order'] ) {
			$statuses[ self::BO_STATUS ] = esc_html_x( 'Backordered', 'Order status', 'shopbuilder-pro' );
		}

		if ( $this->options['pre_order'] ) {
			$statuses[ self::PO_STATUS ] = esc_html_x( 'Pre-Ordered', 'Order status', 'shopbuilder-pro' );
		}

		if ( $this->options['partially_paid'] ) {
			$statuses[ self::PP_STATUS ] = esc_html_x( 'Partially Paid', 'Order status', 'shopbuilder-pro' );
		}

		return $statuses;
	}

	/**
	 * Register custom order post-status for orders.
	 *
	 * @param array $statuses Array of statuses.
	 *
	 * @return array
	 */
	public function register_order_post_status( $statuses ) {
		if ( $this->options['back_order'] ) {
			$statuses[ self::BO_STATUS ] = $this->get_backorder_post_status_props();
		}

		if ( $this->options['pre_order'] ) {
			$statuses[ self::PO_STATUS ] = $this->get_pre_order_post_status_props();
		}

		if ( $this->options['partially_paid'] ) {
			$statuses[ self::PP_STATUS ] = $this->get_partially_paid_post_status_props();
		}

		return $statuses;
	}

	/**
	 * Returns the properties of this post-status for registration.
	 *
	 * @return array
	 */
	public function get_backorder_post_status_props() {
		return [
			'label'                     => _x( 'Backordered', 'Order status', 'shopbuilder-pro' ),
			'public'                    => false,
			'exclude_from_search'       => false,
			'show_in_admin_all_list'    => true,
			'show_in_admin_status_list' => true,
			/* translators: %s: number of orders */
			'label_count'               => _n_noop( 'Backordered <span class="count">(%s)</span>', 'Backordered <span class="count">(%s)</span>', 'shopbuilder-pro' ),
		];
	}

	/**
	 * Returns the properties of this post-status for registration.
	 *
	 * @return array
	 */
	public function get_pre_order_post_status_props() {
		return [
			'label'                     => _x( 'Pre-Ordered', 'Order status', 'shopbuilder-pro' ),
			'public'                    => false,
			'exclude_from_search'       => false,
			'show_in_admin_all_list'    => true,
			'show_in_admin_status_list' => true,
			/* translators: %s: number of orders */
			'label_count'               => _n_noop( 'Pre-Ordered <span class="count">(%s)</span>', 'Pre-Ordered <span class="count">(%s)</span>', 'shopbuilder-pro' ),
		];
	}

	/**
	 * Returns the properties of this post-status for registration.
	 *
	 * @return array
	 */
	public function get_partially_paid_post_status_props() {
		return [
			'label'                     => _x( 'Partially Paid', 'Order status', 'shopbuilder-pro' ),
			'public'                    => false,
			'exclude_from_search'       => false,
			'show_in_admin_all_list'    => true,
			'show_in_admin_status_list' => true,
			/* translators: %s: number of orders */
			'label_count'               => _n_noop( 'Partially Paid <span class="count">(%s)</span>', 'Partially Paid <span class="count">(%s)</span>', 'shopbuilder-pro' ),
		];
	}

	/**
	 * Define bulk actions for WooCommerce orders.
	 *
	 * @param array $actions Array of bulk actions.
	 *
	 * @return array
	 */
	public function define_bulk_actions( $actions ) {
		$new_actions = [];

		foreach ( $actions as $key => $value ) {
			if ( 'trash' === $key ) {
				if ( FnsPro::is_module_active( 'back_order' ) && $this->options['back_order'] ) {
					$new_actions['mark_rtsb_backordered'] = esc_html__( 'Change status to backordered', 'shopbuilder-pro' );
				}

				if ( FnsPro::is_module_active( 'pre_order' ) && $this->options['pre_order'] ) {
					$new_actions['mark_rtsb_preordered'] = esc_html__( 'Change status to pre-ordered', 'shopbuilder-pro' );
				}

				if ( FnsPro::is_module_active( 'partial_pay' ) && $this->options['partially_paid'] ) {
					$new_actions['mark_rtsb_partially_paid'] = esc_html__( 'Change status to Partially Paid', 'shopbuilder-pro' );
				}
			}
			$new_actions[ $key ] = $value;
		}

		return $new_actions;
	}

	/**
	 * Handle custom bulk actions for WooCommerce orders.
	 *
	 * @param string $redirect URL to redirect to after handling the bulk action.
	 * @param string $action   The bulk action being triggered.
	 * @param array  $ids      Array of order IDs affected by the bulk action.
	 *
	 * @return string
	 */
	public function handle_bulk_actions( $redirect, $action, $ids ) {
		if ( FnsPro::is_module_active( 'back_order' ) && 'mark_rtsb_backordered' === $action ) {
			foreach ( $ids as $id ) {
				$order = wc_get_order( $id );
				$order->update_status( self::BO_STATUS );
			}

			$redirect = add_query_arg(
				[
					'bulk_action' => 'marked_rtsb_backordered',
					'changed'     => count( $ids ),
				],
				$redirect
			);
		}

		if ( FnsPro::is_module_active( 'pre_order' ) && 'mark_rtsb_preordered' === $action ) {
			foreach ( $ids as $id ) {
				$order = wc_get_order( $id );
				$order->update_status( self::PO_STATUS );
			}

			$redirect = add_query_arg(
				[
					'bulk_action' => 'marked_rtsb_preordered',
					'changed'     => count( $ids ),
				],
				$redirect
			);
		}

		if ( FnsPro::is_module_active( 'partial_pay' ) && 'mark_rtsb_partially_paid' === $action ) {
			foreach ( $ids as $id ) {
				$order = wc_get_order( $id );
				$order->update_status( self::PP_STATUS );
			}

			$redirect = add_query_arg(
				[
					'bulk_action' => 'marked_rtsb_partially_paid',
					'changed'     => count( $ids ),
				],
				$redirect
			);
		}

		return $redirect;
	}

	/**
	 * Display a custom notice after performing a bulk action for WooCommerce orders.
	 *
	 * @return void
	 */
	public function custom_notice() {
		$bulk_trigger = isset( $_REQUEST['bulk_action'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['bulk_action'] ) ) : null; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$changed      = isset( $_REQUEST['changed'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['changed'] ) ) : null; // phpcs:ignore WordPress.Security.NonceVerification.Recommended

		if ( 'marked_rtsb_backordered' === $bulk_trigger && $changed ) {
			/* translators: %d: number of orders */
			$message = '<div id="message" class="updated notice is-dismissible"><p>' . _n( '%d order status changed.', '%d order statuses changed.', $changed, 'shopbuilder-pro' ) . '</p></div>';

			Fns::print_html( sprintf( $message, $changed ) );
		}

		if ( 'marked_rtsb_preordered' === $bulk_trigger && $changed ) {
			/* translators: %d: number of orders */
			$message = '<div id="message" class="updated notice is-dismissible"><p>' . _n( '%d order status changed.', '%d order statuses changed.', $changed, 'shopbuilder-pro' ) . '</p></div>';

			Fns::print_html( sprintf( $message, $changed ) );
		}

		if ( 'marked_rtsb_partially_paid' === $bulk_trigger && $changed ) {
			/* translators: %d: number of orders */
			$message = '<div id="message" class="updated notice is-dismissible"><p>' . _n( '%d order status changed.', '%d order statuses changed.', $changed, 'shopbuilder-pro' ) . '</p></div>';

			Fns::print_html( sprintf( $message, $changed ) );
		}
	}

	/**
	 * Define legacy bulk actions for WooCommerce orders.
	 *
	 * @param array $actions Array of bulk actions.
	 *
	 * @return array
	 */
	public function define_bulk_actions_legacy( $actions ) {
		unset( $actions['trash'] );

		$actions['mark_rtsb_backordered']    = esc_html__( 'Change status to backordered', 'shopbuilder-pro' );
		$actions['mark_rtsb_preordered']     = esc_html__( 'Change status to pre-ordered', 'shopbuilder-pro' );
		$actions['mark_rtsb_partially_paid'] = esc_html__( 'Change status to partially paid', 'shopbuilder-pro' );
		$actions['trash']                    = esc_html__( 'Move to trash', 'shopbuilder-pro' );

		return $actions;
	}
}
