<?php
/**
 * Elementor Render Class for ShopBuilder Pro.
 *
 * This class contains render logics & output.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Elementor\Render;

// Do not allow directly accessing this file.
use RadiusTheme\SB\Helpers\Cache;
use WC_Product_Query;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

if ( ! defined( 'ABSPATH' ) ) {
	exit( 'This script cannot be accessed directly.' );
}

/**
 * Elementor Render Class for ShopBuilder Pro.
 */
class RenderPro {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Renders filter buttons.
	 *
	 * @param array  $metas Meta values.
	 * @param array  $settings Settings values.
	 * @param array  $args Args.
	 * @param object $widget Query object.
	 *
	 * @return void
	 */
	public static function render_tax_filter( $metas, $settings, $args, $widget ) { // phpcs:ignore Generic.Metrics.CyclomaticComplexity.MaxExceeded, Generic.Metrics.NestingLevel.TooHigh
		// Define a unique cache key based on relevant parameters.
		// phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.serialize_serialize
		$cache_key = 'tax_filter_' . md5( serialize( $metas ) . serialize( $settings ) . serialize( $args ) );
		// Check if the result is already in the cache.
		$cached_html = wp_cache_get( $cache_key, 'shopbuilder' );
		if ( false !== $cached_html ) {
			Fns::print_html( $cached_html );
			return;
		}
		$html               = '';
		$rand               = wp_rand();
		$is_term_selected   = false;
		$tax_filter_terms   = [];
		$attr_taxonomy_list = [];
		$all_select         = ' selected';
		$show_filter        = $metas['show_filter'];
		$tax_filter         = $metas['tax_filter'];
		$action_term        = $metas['action_term'];
		$filter_type        = $metas['filter_type'];
		$filter_class       = ' type-' . $metas['filter_type'];
		$cat_ids            = ! empty( $metas['categories'] ) ? array_map( 'absint', $metas['categories'] ) : [];
		$brand_ids          = ! empty( $metas['brands'] ) ? array_map( 'absint', $metas['brands'] ) : [];
		$tag_ids            = ! empty( $metas['tags'] ) ? array_map( 'absint', $metas['tags'] ) : [];
		$attr_ids           = ! empty( $metas['raw_settings']['filter_attributes'] ) ? array_map( 'absint', $metas['raw_settings']['filter_attributes'] ) : [];
		$show_count         = $metas['term_count'];
		$slider_nav         = $metas['nav_position'] . '-slider-nav';
		$filter_taxonomy    = ! empty( $settings['filter_taxonomy'] ) ? $settings['filter_taxonomy'] : 'product_cat';
		$ajax_filter_type   = $metas['ajax_filter_type'] ?? 'taxonomy';
		$custom_tabs        = $metas['custom_tabs'] ?? [];
		$is_special_layout  = 'special' === $ajax_filter_type && ! empty( $custom_tabs );
		if ( empty( $filter_type ) || ( ! in_array( '_taxonomy_filter', $metas['filters'], true ) && ! $show_filter ) ) {
			return;
		}
		if ( is_array( $cat_ids ) && ! empty( $cat_ids ) ) {
			$tax_filter_terms = array_merge( $tax_filter_terms, $cat_ids );
		}
		if ( is_array( $brand_ids ) && ! empty( $brand_ids ) ) {
			$tax_filter_terms = array_merge( $tax_filter_terms, $brand_ids );
		}
		if ( ! empty( $tag_ids ) && is_array( $tag_ids ) ) {
			$tax_filter_terms = array_merge( $tax_filter_terms, $tag_ids );
		}
		if ( ! empty( $attr_ids ) && is_array( $attr_ids ) ) {
			$tax_filter_terms = array_merge( $tax_filter_terms, $attr_ids );
		}
		if ( $is_special_layout ) {
			$tax_filter       = 'special_filter';
			$queries          = array_column( $custom_tabs, 'tab_query' );
			$titles           = array_column( $custom_tabs, 'tab_title' );
			$terms            = array_combine( $queries, $titles );
			$is_term_selected = empty( $settings['tax_show_all'] );
		} elseif ( 'product_attr' === $tax_filter ) {
			$attr_taxonomy_list = method_exists( Fns::class, 'tax_filter_attr_taxonomy' )
				? Fns::tax_filter_attr_taxonomy( $attr_ids )
				: [];
			if ( ! empty( $attr_taxonomy_list ) ) {
				foreach ( $attr_taxonomy_list as $taxonomy ) {
					$terms[] = Fns::get_all_terms( $taxonomy );
				}
				if ( ! empty( $terms ) && is_array( $terms ) ) {
					$terms = array_replace_recursive( ...$terms );
					$terms = array_unique( $terms );
				}
			}
		} else {
			$terms = Fns::get_all_terms( $tax_filter );
		}

		if ( $action_term && $tax_filter ) {
			$is_term_selected = true;
			$all_select       = null;
		}

		$html .= $show_filter ? '<div class="rtsb-tab-filters element-loading">' : '';
		if ( 'product_attr' === $tax_filter ) {
			$p_count = method_exists( Fns::class, 'count_products_by_attribute_terms' ) ? Fns::count_products_by_attribute_terms( $tax_filter_terms, $metas['raw_settings']['tax_relation'] ) : 0;
		} else {
			$p_count = $is_special_layout ? 0 : Fns::count_products_by_taxonomies( $tax_filter_terms, esc_html( $filter_taxonomy ) );
		}

		$p_count = $p_count > $args['limit'] ? $args['limit'] : $p_count;

		if ( 'button' === $filter_type || 'track' === $filter_type ) {
			// Adding filter container render attributes.
			$widget->add_attribute(
				'tab_filter_container_' . $rand,
				[
					'class'         => 'rtsb-filter-item-wrap rtsb-tax-filter ' . $slider_nav . $filter_class,
					'data-taxonomy' => $tax_filter,
				]
			);

			$html .= '<div ' . $widget->get_attribute_string( 'tab_filter_container_' . $rand ) . '>';
			$html .= 'track' === $filter_type ? '<div class="rtsb-tracks-wrapper">' : '';

			if ( ! empty( $settings['tax_show_all'] ) ) {
				$widget->add_attribute(
					'tab_filter_all_button_' . $rand,
					[
						'class'     => 'rtsb-filter-button-item' . $all_select,
						'data-term' => 'all',
					]
				);

				$html .= '<span ' . $widget->get_attribute_string( 'tab_filter_all_button_' . $rand ) . '>';
				$html .= ! empty( $settings['tax_show_all_text'] ) ? $settings['tax_show_all_text'] : '';
				$html .= $show_count ? ' <span class="rtsb-count">(' . absint( $p_count ) . ')</span>' : '';
				$html .= '</span>';
			}

			$show_tax_button    = is_array( $tax_filter_terms ) && ! empty( $tax_filter_terms );
			$show_custom_button = 'special' === $ajax_filter_type;

			if ( ! empty( $terms ) && ( $show_tax_button || $show_custom_button ) ) {
				if ( 'special' === $ajax_filter_type ) {
					foreach ( $terms as $id => $val ) {
						$termSelected = null;

						if ( $is_term_selected && reset( $terms ) === $val ) {
							$termSelected = ' selected';
						}

						// Adding filter button render attributes.
						$widget->add_attribute(
							'tab_filter_button_' . md5( $rand . $id ),
							[
								'class'     => 'rtsb-filter-button-item' . $termSelected,
								'data-term' => esc_html( $id ),
							]
						);

						$html .= '<span ' . $widget->get_attribute_string( 'tab_filter_button_' . md5( $rand . $id ) ) . '>';
						$html .= esc_html( $val );
						$html .= '</span>';
					}
				} elseif ( 'product_attr' === $tax_filter ) {
					$temp_args = $args;
					if ( $attr_taxonomy_list ) {
						foreach ( $attr_taxonomy_list as $taxonomy ) {
							foreach ( $tax_filter_terms as $id ) {
								$termSelected = null;
								$id           = Fns::wpml_object_id( $id, $taxonomy, 'current' );
								if ( empty( $terms[ $id ] ) ) {
									continue;
								}
								if ( $is_term_selected && $id === $action_term ) {
									$termSelected = ' selected';
								}

								if ( empty( $action_term ) && empty( $settings['tax_show_all'] ) && $id === $tax_filter_terms[0] ) {
									$termSelected = ' selected';
								}

								$filtered_term = get_term( $id, $taxonomy );
								if ( ! empty( $filtered_term ) && ! is_wp_error( $filtered_term ) ) {
									if ( $taxonomy === $filtered_term->taxonomy ) {
										$count             = 0;
										$args[ $taxonomy ] = $id;
										$args['tax_query'] = [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
											[
												'taxonomy' => $taxonomy,
												'field'    => 'term_id',
												'terms'    => $id,
												'operator' => 'IN',
											],
										];
										unset( $args['product_attribute_id'] );
										unset( $args['pa_color'] );
										$filtered_query = new WC_Product_Query( $args );

										$count = count( $filtered_query->get_products() );
										wp_reset_postdata();

										$args = $temp_args;

										// Adding filter button render attributes.
										$widget->add_attribute(
											'tab_filter_button_' . $rand . $id,
											[
												'class' => 'rtsb-filter-button-item' . $termSelected,
												'data-term' => absint( $id ),
											]
										);

										$html .= '<span ' . $widget->get_attribute_string( 'tab_filter_button_' . $rand . $id ) . '>';
										$html .= esc_html( $terms[ $id ] );
										$html .= $show_count ? ' <span class="rtsb-count">(' . absint( $count ) . ')</span>' : '';
										$html .= '</span>';
									}
								}
							}
						}
					}
				} else {
					$temp_args = $args;
					foreach ( $tax_filter_terms as $id ) {
						$termSelected = null;

						$id = Fns::wpml_object_id( $id, $tax_filter, 'current' );

						if ( empty( $terms[ $id ] ) ) {
							continue;
						}

						if ( $is_term_selected && $id === $action_term ) {
							$termSelected = ' selected';
						}

						if ( empty( $action_term ) && empty( $settings['tax_show_all'] ) && $id === $tax_filter_terms[0] ) {
							$termSelected = ' selected';
						}

						$filtered_term = get_term( $id, $tax_filter );
						$count         = 0;

						if ( 'product_tag' === $tax_filter ) {
							$action_taxonomy = 'product_tag_id';
						} elseif ( 'product_brand' === $tax_filter ) {
							$action_taxonomy = 'product_brand_id';
						} else {
							$action_taxonomy = 'product_category_id';
						}

						$args[ $action_taxonomy ] = $id;

						if ( ! empty( $filtered_term ) && ! is_wp_error( $filtered_term ) ) {
							// phpcs:ignore Squiz.PHP.CommentedOutCode.Found
							$filtered_query = new WC_Product_Query( $args );
							$count          = count( $filtered_query->get_products() );

							wp_reset_postdata();
						}

						$args = $temp_args;

						// Adding filter button render attributes.
						$widget->add_attribute(
							'tab_filter_button_' . $rand . $id,
							[
								'class'     => 'rtsb-filter-button-item' . $termSelected,
								'data-term' => absint( $id ),
							]
						);

						$html .= '<span ' . $widget->get_attribute_string( 'tab_filter_button_' . $rand . $id ) . '>';
						$html .= esc_html( $terms[ $id ] );
						$html .= $show_count ? ' <span class="rtsb-count">(' . absint( $count ) . ')</span>' : '';
						$html .= '</span>';
					}
				}
			}

			if ( 'track' === $filter_type ) {
				$html .= '<div class="rtsb-filter-item-track"></div>';
			}

			$html .= 'track' === $filter_type ? '</div>' : '';
			$html .= '</div>';
		} else {
			$html .= '';
		}

		$html .= $show_filter ? '</div>' : '';

		// Store the generated HTML in the cache.
		wp_cache_set( $cache_key, $html, 'shopbuilder', HOUR_IN_SECONDS );
		Cache::set_data_cache_key( $cache_key );
		Fns::print_html( apply_filters( 'rtsb/element/elementor/ajax_filter_output', $html, $settings ) );
	}

	/**
	 * Renders pagination.
	 *
	 * @param array  $meta Meta values.
	 * @param object $wp_query Query object.
	 * @param array  $args Args.
	 *
	 * @return void
	 */
	public static function render_pro_pagination( $meta, $wp_query, $args ) {
		$html = null;
		$text = $meta['load_more_text'];

		if ( 'pagination_ajax' === $meta['posts_loading_type'] ) {
			$html .= Fns::custom_pagination(
				$args['total_page'],
				$args['posts_per_page'],
				true
			);
		} elseif ( 'load_more' === $meta['posts_loading_type'] ) {
			if ( $args['total_page'] > 1 ) {
				$html .= "<div class='rtsb-load-more rtsb-pos-r'>
						<button data-paged='2' data-max-page='{$args['total_page']}'><span>" . esc_html( $text ) . '</span></button>
						<div class="rtsb-loadmore-loading rtsb-ball-clip-rotate"><div></div></div>
					</div>';
			}
		} elseif ( 'load_on_scroll' === $meta['posts_loading_type'] ) {
			if ( $args['total_page'] > 1 ) {
				$html .= "<div class='rtsb-infinite-scroll' data-trigger='1' data-paged='2' data-max-page='{$args['total_page']}'>
					<div class='rtsb-infinite-action'>
						<div class='rtsb-loadmore-loading rtsb-ball-clip-rotate'><div></div></div>
					</div>
				</div>";
			}
		}

		Fns::print_html( $html );
	}

	/**
	 * Render Filters View.
	 *
	 * @param array $templates Template name.
	 * @param array $settings Control settings.
	 *
	 * @return string
	 */
	public static function filters_view( $templates, $settings ) { // phpcs:ignore Generic.Metrics.CyclomaticComplexity.MaxExceeded
		$filters            = [];
		$html               = null;
		$reset              = ! empty( $settings['reset_btn'] );
		$ajax               = ! empty( $settings['ajax_mode'] );
		$ajax_class         = $ajax ? ' has-filter-ajax' : ' no-filter-ajax';
		$filter_target      = ! empty( $settings['filter_target'] ) ? $settings['filter_target'] : 'default';
		$default_selector   = ! empty( $settings['custom_class'] ) ? $settings['custom_class'] : 'ul.products';
		$reset_mode         = 'show' === $settings['reset_btn_behavior'] ? ' show-reset' : ' ondemand-reset';
		$reset_text         = ! empty( $settings['reset_btn_text'] ) ? $settings['reset_btn_text'] : esc_html__( 'Reset Filter', 'shopbuilder-pro' );
		$view_mode          = ! empty( $settings['collapse_mode'] ) ? ' has-collapse' : ' no-collapse';
		$collapse_mobile    = ! empty( $settings['collapse_mode_mobile'] ) && empty( $settings['collapse_mode'] ) ? ' has-collapse-mobile' : ' no-collapse-mobile';
		$floating_mode      = ! empty( $settings['floating_mode'] ) ? ' has-floating' : ' no-floating';
		$collapse_close     = ! empty( $settings['close_collapsed'] ) ? ' has-collapse-close' : ' no-collapse-close';
		$scroll_top         = ! empty( $settings['scroll_to_top'] ) ? ' has-scroll-mode' : ' no-scroll-mode';
		$scroll_offset      = ! empty( $settings['scroll_offset'] ) ? $settings['scroll_offset'] : 0;
		$toggle_btn         = ! empty( $settings['toggle_btn'] );
		$toggle_btn_txt     = ! empty( $settings['toggle_btn_text'] ) ? $settings['toggle_btn_text'] : esc_html__( 'Filter', 'shopbuilder-pro' );
		$toggle_btn_icon    = ! empty( $settings['toggle_btn_icon'] ) ? $settings['toggle_btn_icon'] : [];
		$toggle_class       = $toggle_btn ? ' has-toggle' : ' no-toggle';
		$collapse_type      = $view_mode . $collapse_close . ( ! empty( $settings['collapse_behavior'] ) ? ' collapse-' . $settings['collapse_behavior'] : '' );
		$scroll_mode        = ! empty( $settings['enable_scroll'] ) ? ' has-scroll' : ' no-scroll';
		$scroll_height      = ! empty( $settings['enable_scroll'] ) ? $settings['scroll_height']['size'] : 300;
		$count_alignment    = ! empty( $settings['count_alignment'] ) && 'space-between' === $settings['count_alignment'] ? ' count-align-right' : '';
		$additional_opt     = ! empty( $settings['more_less_btn'] ) ? ' has-additional-control' : ' no-additional-control';
		$mobile_toggle      = ! empty( $settings['toggle_btn_mobile'] );
		$mobile_toggle_text = ! empty( $settings['toggle_btn_mobile_text'] ) ? $settings['toggle_btn_mobile_text'] : esc_html__( 'Click to Filter', 'shopbuilder-pro' );
		$has_mobile_toggle  = $mobile_toggle ? ' has-mobile-toggle' : ' no-mobile-toggle';
		$additional_attr    = '';
		$scroll_attr        = '';

		if ( ! empty( $settings['enable_scroll'] ) ) {
			$scroll_attr = ' style="--rtsb-filter-scroll-height: ' . absint( $scroll_height ) . 'px;"';
		}

		foreach ( $settings['filter_types'] as $key => $item ) {
			if ( ! defined( 'RTWPVS_PLUGIN_FILE' ) && in_array( $item['input_type_all'], [ 'color', 'button', 'image' ], true ) ) {
				$item['input_type_all'] = 'checkbox';
			}
			$filters[] = [
				'filter_title' => $item['filter_title'],
				'show_icon'    => $item['show_title_icon'],
				'title_icon'   => $item['filter_title_icon'],
				'filter_items' => $item['filter_items'],
				'filter_attr'  => $item['filter_attr'],
				'input_type'   => 'product_attr' === $item['filter_items'] ? $item['input_type_all'] : $item['input_type'],
				'multiple'     => $item['multiple_selection'],
				'relation'     => ! empty( $item['multiple_relation'] ) ? $item['multiple_relation'] : 'AND',
			];

			if ( 'rating_filter' === $item['filter_items'] ) {
				$filters[ $key ]['template'] = $templates['rating'];
			} elseif ( 'price_filter' === $item['filter_items'] ) {
				wp_enqueue_script( 'rtsb-noui-slider' );

				$filters[ $key ]['input_type'] = 'slider';
				$filters[ $key ]['template']   = $templates['price'];
			} else {
				$filters[ $key ]['template'] = 'product_attr' === $item['filter_items'] ? $templates[ $item['input_type_all'] ] : $templates[ $item['input_type'] ];
			}
		}

		if ( ! empty( $settings['more_less_btn'] ) ) {
			$more            = ! empty( $settings['more_text'] ) ? $settings['more_text'] : esc_html__( 'More', 'shopbuilder-pro' );
			$less            = ! empty( $settings['less_text'] ) ? $settings['less_text'] : esc_html__( 'Less', 'shopbuilder-pro' );
			$additional_attr = ' data-more-text="' . esc_html( $more ) . '" data-less-text="' . esc_html( $less ) . '"';
		}

		$additional_attr .= ' data-filter-target="' . esc_attr( $filter_target ) . '-widget"';
		$additional_attr .= ' data-default-selector="' . esc_attr( $default_selector ) . '"';

		if ( ! empty( $settings['scroll_to_top'] ) ) {
			$additional_attr .= ' data-scroll-offset="' . absint( $scroll_offset ) . '"';
		}

		if ( ! empty( $settings['collapse_mode_mobile'] ) && empty( $settings['collapse_mode'] ) ) {
			$collapse_type = ' has-collapse has-collapse-close collapse-one';
		}

		if ( ! empty( $count_alignment ) ) {
			$collapse_type .= $count_alignment;
		}

		$args = [
			'filters'         => $filters,
			'collapse_type'   => $collapse_type,
			'collapse_mobile' => $collapse_mobile,
			'additional_opt'  => $additional_opt,
			'scroll_mode'     => $scroll_mode,
			'toggle_mode'     => $toggle_btn,
			'mobile_toggle'   => $mobile_toggle,
			'floating_mode'   => $floating_mode,
			'toggle_class'    => $toggle_class,
			'scroll_top'      => $scroll_top,
			'reset_mode'      => $reset_mode,
			'additional_attr' => $additional_attr,
			'scroll_attr'     => $scroll_attr,
			'reset'           => $reset,
			'ajax_class'      => $ajax_class,
			'reset_text'      => $reset_text,
			'settings'        => $settings,
		];

		$html .= '<div class="rtsb-archive-filters' . esc_attr( $has_mobile_toggle ) . '">';

		if ( $toggle_btn ) {
			$html .= '<div class="rtsb-filter-toggle">
						<button class="product-filter-toggle">
							<span class="icon">' . Fns::icons_manager( $toggle_btn_icon, 'toggle-icon' ) . '</span>
							<span class="text reset-text">' . esc_html( $toggle_btn_txt ) . '</span>
							<span></span>
						</button>
					</div>';
		} elseif ( $mobile_toggle ) {
			$html .= '<div class="rtsb-filter-mobile-toggle">
						<button class="product-filter-toggle">
							<span class="icon"><i aria-hidden="true" class="toggle-icon rtsb-icon rtsb-icon-filter"></i></span>
							<span class="text reset-text">' . esc_html( $mobile_toggle_text ) . '</span>
							<span></span>
						</button>
					</div>';
		}

		$html .= Fns::load_template( $templates['layout'], $args, true, '', rtsbpro()->get_plugin_template_path() );
		$html .= '</div>';

		return $html;
	}

	/**
	 * Generates an error message block.
	 *
	 * @param array  $title The title array of the error message block.
	 * @param string $wrapper Additional CSS class for the wrapper div.
	 *
	 * @return string The HTML representation of the error message block.
	 */
	public static function error_message( $title, $wrapper ) {
		$html  = '<div class="rtsb-product-filters ' . esc_attr( $wrapper ) . '">';
		$html .= FnsPro::get_filter_title( $title );
		$html .= '<div class="filter-content">';
		$html .= '<p class="no-filter">' . esc_html__( 'Nothing found.', 'shopbuilder-pro' ) . '</p>';
		$html .= '</div>';
		$html .= '</div>';

		return $html;
	}

	/**
	 * Render flash sale.
	 *
	 * @param array $settings Settings array.
	 * @return void
	 */
	public static function render_flash_sale( $settings ) {
		$html = null;

		if ( ! empty( $settings['show_flash_sale_countdown'] ) && shortcode_exists( 'rtsb_flash_sale_countdown' ) ) {
			$html .= do_shortcode( '[rtsb_flash_sale_countdown]' );
		}

		Fns::print_html( $html );
	}

	/**
	 * Render Social Share.
	 *
	 * @param array $settings Settings array.
	 * @return void
	 */
	public static function render_social_share_pro( $settings ) {
		$html = null;

		if ( ! empty( $settings['share_toggle'] ) ) {
			$share_icon = ! empty( $settings['share_toggle_icon'] ) ? $settings['share_toggle_icon'] : [];
			$icon_html  = Fns::icons_manager( $share_icon, 'icon-toggle-share' );

			$html .= '<div class="rtsb-share-toggle">';
			$html .= '<div class="toggle-icon">';
			$html .= $icon_html;
			$html .= '</div>';
			$html .= '</div>';
		}

		Fns::print_html( $html, true );
	}

	/**
	 * Render Additional Action Buttons.
	 *
	 * @param array $items Items array.
	 * @return void
	 */
	public static function render_action_buttons( $items ) {
		$html = null;

		$html .= Fns::get_action_button_by_type( $items, 'quick_checkout' );

		Fns::print_html( $html, true );
	}

	/**
	 * Display a filter header.
	 *
	 * @param array $settings The settings.
	 *
	 * @return void
	 */
	public static function filter_header( $settings ) {
		if ( empty( $settings['show_filter_header'] ) && empty( $settings['filter_header_text'] ) ) {
			return;
		}

		$layout         = ! empty( $settings['layout'] ) ? $settings['layout'] : 'vertical';
		$wrapper_class  = 'rtsb-filter-header ';
		$wrapper_class .= $layout . '-layout';
		?>

		<div class="<?php echo esc_attr( $wrapper_class ); ?>">
			<h3><?php echo esc_html( $settings['filter_header_text'] ); ?></h3>
		</div>

		<?php
	}

	/**
	 * Display a filter search form.
	 *
	 * @param array $settings The settings.
	 *
	 * @return void
	 */
	public static function filter_search( $settings ) {
		if ( empty( $settings['search_form'] ) ) {
			return;
		}

		$title         = [
			'title'         => ! empty( $settings['search_title'] ) ? $settings['search_title'] : '',
			'show_icon'     => ! empty( $settings['show_search_title_icon'] ),
			'icon'          => ! empty( $settings['filter_search_title_icon'] ) ? $settings['filter_search_title_icon'] : [],
			'icon_position' => '',
			'collapse'      => '',
			'preset'        => 'none',
		];
		$placeholder   = ! empty( $settings['search_placeholder'] ) ? $settings['search_placeholder'] : '';
		$btn_type      = ! empty( $settings['search_submit'] ) ? $settings['search_submit'] : 'icon';
		$btn_text      = ! empty( $settings['submit_button_title'] ) ? $settings['submit_button_title'] : esc_html__( 'Search', 'shopbuilder-pro' );
		$btn_icon      = ! empty( $settings['search_title_icon'] ) ? $settings['search_title_icon'] : [];
		$wrapper_class = 'rtsb-filter-search rtsb-product-search';
		$unique_id     = substr( uniqid(), -6 );
		$search_term   = isset( $_REQUEST['s'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['s'] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		?>

		<div class="<?php echo esc_attr( $wrapper_class ); ?>">
			<?php
			Fns::print_html( FnsPro::get_filter_title( $title ) );
			?>
			<div class="search-content">
				<div class="woocommerce-product-search<?php echo ! empty( $search_term ) ? ' active' : ''; ?>">
					<div class="search-input<?php echo 'text' === $btn_type ? ' has-submit-text' : ''; ?>">
						<label class="screen-reader-text" for="rtsb-product-search-form-<?php echo esc_attr( $unique_id ); ?>"><?php echo esc_html( $placeholder ); ?></label>
						<input type="search" id="rtsb-product-search-form-<?php echo esc_attr( $unique_id ); ?>" class="rtsb-search-field" placeholder="<?php echo esc_html( $placeholder ); ?>" value="<?php echo esc_attr( $search_term ); ?>" name="s" autocomplete="off">
						<button class="clear-search tipsy" aria-label="<?php echo esc_html__( 'Clear Search', 'shopbuilder-pro' ); ?>" title="<?php echo esc_html__( 'Clear Search', 'shopbuilder-pro' ); ?>">
							<i aria-hidden="true" class="delete-icon rtsb-icon rtsb-icon-delete"></i>
						</button>
					</div>
					<button class="rtsb-search-submit">
						<?php
						if ( 'icon' === $btn_type ) {
							Fns::print_html( Fns::icons_manager( $btn_icon, 'search-icon' ) );
						} else {
							echo '<span class="search-text">' . esc_html( $btn_text ) . '</span>';
						}
						?>
						<span></span>
					</button>
				</div>
			</div>
		</div>

		<?php
	}

	/**
	 * Modify arguments for the Category Argument Dataset Pro.
	 *
	 * @param array  $arg          The original arguments for the Cat Argument Dataset Pro.
	 * @param array  $meta         The metadata associated with the Cat Argument Dataset Pro.
	 * @param object $query        The WordPress query object.
	 * @param bool   $fullwidth    Whether the display is in full width.
	 * @param bool   $lazy_load    Whether lazy loading is enabled.
	 * @param array  $products     An array of products to analyze.
	 *
	 * @return array
	 */
	public static function cat_arg_dataset_pro( $arg, $meta, $query, $fullwidth, $lazy_load, $products ) {
		$_sale_count = 0;

		if ( $products ) {
			$_products   = array_filter(
				$products,
				function ( $product ) {
					return $product->is_on_sale();
				}
			);
			$_sale_count = count( $_products );
		}

		$arg['sale_count']    = $_sale_count;
		$arg['product_count'] = max( $query->count, 0 );
		$arg['count_text']    = $meta['after_count'];

		return $arg;
	}
}
