<?php
/**
 * Render class for Coupon List widget.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Elementor\Widgets\General\CouponList;

use WP_Query;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SB\Elementor\Render\GeneralAddons;
use RadiusTheme\SBPRO\Modules\SmartCoupons\RTSB_Coupon;

// Do not allow directly accessing this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit( 'This script cannot be accessed directly.' );
}

/**
 * Render class.
 *
 * @package RadiusTheme\SBPRO
 */
class Render extends GeneralAddons {
	/**
	 * Main render function for displaying content.
	 *
	 * @param array $data     Data to be passed to the template.
	 * @param array $settings Widget settings.
	 *
	 * @return string
	 */
	public function display_content( $data, $settings ) {
		$this->settings = $settings;

		$data = apply_filters( 'rtsb/general/coupon_list/args/' . $data['unique_name'], $data );
		$data = $this->get_data_content( $data );

		return $this->addon_view( $data );
	}

	/**
	 * Retrieves template args based on widget settings.
	 *
	 * @param int   $id Coupon ID.
	 * @param array $data Data to be passed to the template.
	 *
	 * @return array
	 */
	private function get_template_args( $id, $data ) {
		$coupon = new RTSB_Coupon( $id );

		$coupon_code   = $coupon->get_code();
		$expiry_date   = $coupon->get_date_expires();
		$product_ids   = $coupon->get_product_ids();
		$discount_type = get_post_meta( $id, 'discount_type', true );
		$amount        = $coupon->get_amount();

		$formatted_expiry_date = ! empty( $expiry_date ) ? gmdate( 'jS F', strtotime( $expiry_date ) ) : '';
		$days_left             = ! empty( $expiry_date ) ? ceil( ( $expiry_date->getTimestamp() - time() ) / DAY_IN_SECONDS ) : '';
		$data['grid_classes'] .= ! empty( $expiry_date ) && $days_left <= 0 ? ' coupon-expired' : '';
		$grid_classes          = explode( ' ', $data['grid_classes'] );

		if ( 'percent' === $discount_type ) {
			$discount_text        = '<span class="coupon-amount">' . $amount . '%</span>' . esc_html__( ' Discount!', 'shopbuilder-pro' );
			$data['grid_classes'] = implode( ' ', array_merge( $grid_classes, [ 'coupon-percent' ] ) );
		} elseif ( 'fixed_cart' === $discount_type || 'fixed_product' === $discount_type ) {
			$discount_text        = '<span class="coupon-amount">' . preg_replace( '/\.\d+/', '', wc_price( $amount ) ) . '</span>' . esc_html__( ' Discount!', 'shopbuilder-pro' );
			$data['grid_classes'] = implode( ' ', array_merge( $grid_classes, [ 'coupon-fixed' ] ) );
		} elseif ( 'rtsb_volume_discount' === $discount_type ) {
			$discount_text        = esc_html__( 'Volume Discount Available!', 'shopbuilder-pro' );
			$data['grid_classes'] = implode(
				' ',
				array_filter(
					array_merge(
						$grid_classes,
						[
							'coupon-volume',
							! empty( $product_ids ) ? 'has-active-product' : '',
						]
					)
				)
			);

			if ( ! empty( $product_ids ) ) {
				$product_names = array_map(
					function ( $product_id ) {
						$product = wc_get_product( $product_id );
						return $product ? $product->get_name() : '';
					},
					$product_ids
				);

				$product_names = array_filter( $product_names );
				$discount_text = '<span class="coupon-volume">' . esc_html__( 'Volume Discount for ', 'shopbuilder-pro' ) . '</span>' . implode( ', ', $product_names );
			}
		} else {
			$discount_text        = $amount;
			$data['grid_classes'] = implode( ' ', array_merge( $this->content_classes, [ 'coupon-amount' ] ) );
		}

		$grid_attr = [
			'class'            => $data['grid_classes'],
			'data-coupon-code' => $coupon_code,
		];

		$this->add_attribute( 'rtsb_coupon_list_attr_' . $id, $grid_attr );

		return [
			'coupon_code'           => $coupon_code,
			'coupon_description'    => ! empty( $this->settings['show_desc'] ) ? $coupon->get_description() : '',
			'expiry_date'           => $expiry_date,
			'discount_type'         => $discount_type,
			'amount'                => $amount,
			'formatted_expiry_date' => ! empty( $this->settings['show_expiration'] ) ? $formatted_expiry_date : '',
			'days_left'             => ! empty( $this->settings['show_validity'] ) ? $days_left : '',
			'discount_text'         => ! empty( $this->settings['show_discount'] ) ? $discount_text : '',
			'coupon'                => ! empty( $this->settings['show_coupon'] ) ? $coupon : '',
			'custom_text'           => ! empty( $this->settings['show_badge'] ) && ! empty( $this->settings['custom_text'] ) ? $this->settings['custom_text'] : '',
			'grid_attr'             => $this->get_attribute_string( 'rtsb_coupon_list_attr_' . $id ),
			'render'                => $this,
		];
	}

	/**
	 * Get data content.
	 *
	 * @param array $data Data.
	 *
	 * @return array
	 */
	private function get_data_content( $data ) {
		$cache_key   = 'rtsb_coupon_list_' . md5( wp_json_encode( $this->settings ) );
		$cached_data = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false !== $cached_data ) {
			return $cached_data;
		}

		$output               = '';
		$coupons_query        = new WP_Query( $this->coupon_query_args() );
		$data['grid_classes'] = implode( ' ', array_merge( $this->content_classes, [ $data['content_class'] ] ) );

		if ( $coupons_query->have_posts() ) {
			$coupons = $this->get_sorted_coupons( $coupons_query );

			if ( ! empty( $coupons ) ) {
				foreach ( $coupons as $coupon_id ) {
					$data    = wp_parse_args( $this->get_template_args( $coupon_id, $data ), $data );
					$output .= Fns::load_template( $data['template'], $data, true );
				}
			}

			wp_reset_postdata();
		} else {
			$output .= esc_html__( 'No coupons found.', 'shopbuilder-pro' );
		}

		$data['content'] = $output;

		wp_cache_set( $cache_key, $data, 'shopbuilder', 12 * HOUR_IN_SECONDS );
		Cache::set_data_cache_key( $cache_key );

		return $data;
	}

	/**
	 * Coupon query args.
	 *
	 * @return array
	 */
	private function coupon_query_args() {
		$args = [
			'post_type'      => 'shop_coupon',
			'posts_per_page' => ! empty( $this->settings['coupon_limit'] ) ? absint( $this->settings['coupon_limit'] ) : -1,
			'orderby'        => [
				'date_expires' => 'ASC',
			],
			'order'          => 'ASC',
		];

		// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
		$args['meta_query'] = [
			[
				'key'   => 'rtsb_show_on_frontend',
				'value' => 'on',
			],
		];

		if ( ! empty( $this->settings['include_coupon'] ) ) {
			$args['post__in'] = array_map( 'absint', (array) $this->settings['include_coupon'] );
		}

		if ( ! empty( $this->settings['exclude_coupon'] ) ) {
			$args['post__not_in'] = array_map( 'absint', (array) $this->settings['exclude_coupon'] );
		}

		if ( ! empty( $this->settings['coupon_type'] ) && 'all' !== $this->settings['coupon_type'] ) {
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			$args['meta_query'][] = [
				'key'   => 'discount_type',
				'value' => $this->settings['coupon_type'],
			];
		}

		if ( ! empty( $this->settings['filter_groups'] ) ) {
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
			$args['tax_query'] = [
				[
					'taxonomy' => 'rtsb_coupon_group',
					'field'    => 'term_id',
					'terms'    => (array) $this->settings['filter_groups'],
					'operator' => 'IN',
				],
			];
		}

		return $args;
	}

	/**
	 * Get sorted coupons.
	 *
	 * @param WP_Query $query Query.
	 *
	 * @return array
	 */
	private function get_sorted_coupons( $query ) {
		$valid_coupons     = [];
		$no_expiry_coupons = [];
		$expired_coupons   = [];

		while ( $query->have_posts() ) {
			$query->the_post();
			$coupon_id = get_the_ID();

			$expiry_date = get_post_meta( $coupon_id, 'date_expires', true );

			if ( empty( $expiry_date ) ) {
				$no_expiry_coupons[] = $coupon_id;
			} else {
				if ( $expiry_date >= time() ) {
					$valid_coupons[] = $coupon_id;
				} else {
					$expired_coupons[] = $coupon_id;
				}
			}
		}

		$visibility = ! empty( $this->settings['coupon_expiration_visibility'] ) ? $this->settings['coupon_expiration_visibility'] : [];
		$result     = [];

		if ( in_array( 'active', $visibility, true ) ) {
			$result = array_merge( $result, $valid_coupons );
		}

		if ( in_array( 'no_expiry', $visibility, true ) ) {
			$result = array_merge( $result, $no_expiry_coupons );
		}

		if ( in_array( 'expired', $visibility, true ) ) {
			$result = array_merge( $result, $expired_coupons );
		}

		if ( empty( $visibility ) ) {
			$result = array_merge( $valid_coupons, $no_expiry_coupons, $expired_coupons );
		}

		return $result;
	}
}
