<?php
/**
 * OrderReceived class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Elementor\Widgets\OrderPay;

use Automattic\WooCommerce\Internal\Utilities\Users;
use RadiusTheme\SB\Elementor\Helper\ControlHelper;
use RadiusTheme\SB\Elementor\Widgets\Controls\CheckoutPaymentSettings;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SB\Abstracts\ElementorWidgetBase;
use RadiusTheme\SBPRO\Elementor\Widgets\Controls\OrderDetailsTableSettings;
use RadiusTheme\SBPRO\Helpers\BuilderFunPro;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use WC_Order;

// Do not allow directly accessing this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit( 'This script cannot be accessed directly.' );
}

class OrderPayForm extends ElementorWidgetBase {

	/**
	 * Construct function
	 *
	 * @param array $data default array.
	 * @param mixed $args default arg.
	 */
	public function __construct( $data = [], $args = null ) {
		$this->rtsb_name = esc_html__( 'Order Pay Form', 'shopbuilder-pro' );
		$this->rtsb_base = 'rtsb-order-pay-form';
		parent::__construct( $data, $args );
	}

	/**
	 * Widget Field
	 *
	 * @return void|array
	 */
	public function widget_fields() {
		return array_merge(
			$this->table_settings(),
			$this->payment_settings()
		);
	}

	public function table_settings() {
		return OrderDetailsTableSettings::table_settings( $this );
	}
	public function payment_settings() {

		$fields       = CheckoutPaymentSettings::widget_fields( $this );
		$insert_array = [
			'wrapper_border_radius' => [
				'label'      => esc_html__( 'Wrapper Border Radius (px)', 'shopbuilder-pro' ),
				'type'       => 'dimensions',
				'size_units' => [ 'px' ],
				'selectors'  => [
					$this->selectors['wrapper_border_radius'] => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}}!important;overflow:hidden;',
				],
			],
		];
		return Fns::insert_controls( 'payment_method_wrap_padding', $fields, $insert_array, true );
	}

	/**
	 * Set Widget Keyword.
	 *
	 * @return array
	 */
	public function get_keywords() {
		return [ 'Order Pay' ] + parent::get_keywords();
	}

	public function guest_should_verify_email( WC_Order $order, string $context ): bool {
		// If we cannot match the order with the current user, ask that they verify their email address.
		$nonce_is_valid = wp_verify_nonce( filter_input( INPUT_POST, 'check_submission' ), 'wc_verify_email' );
		$supplied_email = null;
		$order_id       = $order->get_id();

		if ( $nonce_is_valid ) {
			$supplied_email = sanitize_email( wp_unslash( filter_input( INPUT_POST, 'email' ) ) );
		}

		return Users::should_user_verify_order_email( $order_id, $supplied_email, $context );
	}

	/**
	 * Render Function
	 *
	 * @return void
	 */
	protected function render() {
		$controllers = $this->get_settings_for_display();

		$this->theme_support();
		$order = BuilderFunPro::get_order( ! $this->is_builder_mode() );
		if ( ! $order ) {
			FnsPro::order_message();
			return;
		}
		$data                      = [
			'template'    => 'elementor/orderpay/order-pay-form',
			'controllers' => $controllers,
			'order'       => $order,
		];
		$data['guest_user_verify'] = ! $this->is_builder_mode() && $this->guest_should_verify_email( $order, 'order-pay' );
		$data['needs_payment']     = ! $this->is_builder_mode() && ! $order->needs_payment();

		Fns::load_template( $data['template'], $data, false, '', rtsbpro()->get_plugin_template_path() );

		$this->theme_support( 'render_reset' );
	}
}
