<?php
/**
 * Customize My Account Module Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Models;

use DateTime;
use RadiusTheme\SB\Helpers\Fns;
use WC_Product;

defined( 'ABSPATH' ) || exit();

/**
 * Discount Calculator Base Class.
 */
class DiscountCalculatorBase {
	/**
	 * Cache storage.
	 *
	 * @var array
	 */
	public $cache = [];

	/**
	 * Discount rule configuration.
	 *
	 * @var array
	 */
	protected $discountRule;

	/**
	 * WooCommerce product instance.
	 *
	 * @var WC_Product
	 */
	protected $product;
	/**
	 * Discount rule configuration.
	 *
	 * @var array
	 */
	protected $cart_item;
	/**
	 * Constructor.
	 *
	 * @param WC_Product $product WooCommerce product instance.
	 * @param array      $rule    Discount rule configuration.
	 */
	public function __construct( $product, $rule = [], $cart_item = [] ) {
		$this->product      = $product;
		$this->cart_item    = $cart_item;
		$defaults           = [
			'enable_date_range'  => '',
			'start_date'         => '',
			'end_date'           => '',
			'allowed_roles'      => [],
			'applicable_product' => [],
		];
		$this->discountRule = wp_parse_args( $rule, $defaults );
	}

	/**
	 * Checks if the discount is applicable.
	 *
	 * @return bool True if the discount is valid, false otherwise.
	 */
	public function is_discount_applicable() {
		return $this->is_discount_start()
			&& $this->is_discount_not_end()
			&& $this->is_user_permitted()
			&& $this->applicable_product()
			&& $this->quentity_validation();
	}

	/**
	 * Retrieves the product instance.
	 *
	 * @return WC_Product|null Returns the WC_Product instance or null if invalid.
	 */
	public function get_product() {
		if ( $this->product instanceof WC_Product ) {
			return $this->product;
		}
		return null;
	}

	/**
	 * Gets the current timestamp in WordPress timezone.
	 *
	 * @return int Current timestamp.
	 */
	protected function currentTimestamp() {
		$wp_timezone = wp_timezone();
		$now         = new DateTime( 'now', $wp_timezone ); // Current time in WP timezone.
		return $now->getTimestamp();
	}

	/**
	 * Checks if the discount start date has been reached.
	 *
	 * @param string $format Date format, default 'Y-m-d\TH:i'.
	 * @return bool True if discount can start, false otherwise.
	 */
	protected function is_discount_start( $format = false ) {
		if ( empty( $this->discountRule['enable_date_range'] ) ) {
			return true;
		}
		$wp_timezone = wp_timezone();
		if ( empty( $this->discountRule['start_date'] ) ) {
			return true;
		}

		$datetime = $this->discountRule['start_date'];
		if ( ! $format ) {
			$format   = 'Y-m-d H:i';
			$datetime = str_replace( 'T', ' ', $datetime );
		}
		$start_date_obj = date_create_from_format( $format, $datetime, $wp_timezone );
		$start_date     = $start_date_obj ? $start_date_obj->getTimestamp() : 0;
		return $start_date <= $this->currentTimestamp();
	}

	/**
	 * Checks if the discount end date has passed.
	 *
	 * @param string $format Date format, default 'Y-m-d\TH:i'.
	 * @return bool True if discount has ended, false otherwise.
	 */
	protected function is_discount_not_end( $format = false ) {
		if ( empty( $this->discountRule['enable_date_range'] ) ) {
			return true;
		}
		$wp_timezone = wp_timezone();
		if ( empty( $this->discountRule['end_date'] ) ) {
			return true;
		}
		$datetime = $this->discountRule['end_date'];
		if ( ! $format ) {
			$format   = 'Y-m-d H:i';
			$datetime = str_replace( 'T', ' ', $datetime );
		}

		$end_date_obj = date_create_from_format( $format, $datetime, $wp_timezone );
		$end_date     = $end_date_obj ? $end_date_obj->getTimestamp() : 0;
		return $end_date > $this->currentTimestamp();
	}

	/**
	 * Checks if the current user is permitted for the discount.
	 *
	 * @return bool True if user has permission, false otherwise.
	 */
	protected function is_user_permitted() {
		if ( ! empty( $this->discountRule['allowed_roles'] ) && ! array_intersect( $this->discountRule['allowed_roles'], Fns::get_current_user_roles() ) ) {
			return false;
		}
		return true;
	}


	/**
	 * Checks if the current user is permitted for the discount.
	 *
	 * @return bool True if user has permission, false otherwise.
	 */
	protected function quentity_validation() {
		if ( ! empty( $this->cart_item['quantity'] ) && $this->cart_item['quantity'] < $this->discountRule['required_quantity'] ) {
			return false;
		}
		return true;
	}


	/**
	 * Checks if the product is eligible for the discount.
	 *
	 * @return bool True if the product is applicable, false otherwise.
	 */
	protected function applicable_product() {
		if ( ! empty( $this->discountRule['applicable_product'] ) && ! in_array( $this->product->get_id(), $this->discountRule['applicable_product'], true ) ) {
			return false;
		}
		return true;
	}
}
