<?php
/**
 * Product Add-Ons Functions Class.
 *
 * @package Rse\SB
 */

namespace RadiusTheme\SBPRO\Modules\AddOns;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Helpers\AdminInputsFns;
use RadiusTheme\SB\Elementor\Helper\RenderHelpers;
use RadiusTheme\SBPRO\Modules\AddOns\Frontend\Product;
use RadiusTheme\SBPRO\Modules\FlashSaleCountdown\CountdownFns;
use RadiusTheme\SBPRO\Modules\PreOrder\Frontend\Product as PreOrderProduct;

defined( 'ABSPATH' ) || exit();

/**
 * Product Add-Ons Functions Class.
 */
class AddOnsFns {
	/**
	 * Returns an array of settings fields.
	 *
	 * @return array
	 */
	public static function settings_field() {
		return [
			'add_ons_settings_intro'            => [
				'id'    => 'add_ons_settings_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Add-On Settings', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'add_ons_position'                  => [
				'id'      => 'add_ons_position',
				'type'    => 'select',
				'label'   => __( 'Product Add-Ons Position', 'shopbuilder-pro' ),
				'help'    => esc_html__( 'Select the product add-ons position in the product details page.', 'shopbuilder-pro' ),
				'value'   => 'before',
				'options' => [
					'before' => esc_html__( 'Before Add-to-Cart Button', 'shopbuilder-pro' ),
					'after'  => esc_html__( 'After Add-to-Cart Button', 'shopbuilder-pro' ),
				],
				'tab'     => 'general',
			],
			'add_ons_cart_button_label'         => [
				'id'          => 'add_ons_cart_button_label',
				'label'       => __( 'Loop Add to Cart Button Label', 'shopbuilder-pro' ),
				'type'        => 'text',
				'value'       => esc_html__( 'Select Options', 'shopbuilder-pro' ),
				'placeholder' => esc_html__( 'Ex. Select Options', 'shopbuilder-pro' ),
				'help'        => esc_html__( 'Enter the text to override the default \'Add to Cart\' button label for the product which has addons on shop/archive pages. Leave empty for default label.', 'shopbuilder-pro' ),
				'tab'         => 'general',
			],
			'add_ons_update_base_price'         => [
				'id'    => 'add_ons_update_base_price',
				'type'  => 'switch',
				'label' => __( 'Update Product Base Price', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Allow the product\'s base price (displayed beneath the title) to dynamically update based on the total price of selected options or add-ons. Please note that it will not work with variable products.', 'shopbuilder-pro' ),
				'value' => 'on',
				'tab'   => 'general',
			],
			'add_ons_hook_priority'             => [
				'id'          => 'add_ons_hook_priority',
				'label'       => __( 'Product Add-Ons Hook Priority', 'shopbuilder-pro' ),
				'type'        => 'text',
				'placeholder' => esc_html__( 'Ex. 15', 'shopbuilder-pro' ),
				'help'        => esc_html__( 'Enter the hook priority for displaying the product add-ons.', 'shopbuilder-pro' ),
				'tab'         => 'general',
			],
			'allowed_image_extensions'          => [
				'id'       => 'allowed_image_extensions',
				'label'    => __( 'Allowed Image Extensions for Image Upload', 'shopbuilder-pro' ),
				'type'     => 'select',
				'multiple' => true,
				'options'  => apply_filters(
					'rtsb/modules/add_ons/allowed_image_extensions',
					[
						'jpg'  => esc_html__( 'jpg', 'shopbuilder-pro' ),
						'jpeg' => esc_html__( 'jpeg', 'shopbuilder-pro' ),
						'png'  => esc_html__( 'png', 'shopbuilder-pro' ),
						'gif'  => esc_html__( 'gif', 'shopbuilder-pro' ),
						'webp' => esc_html__( 'webp', 'shopbuilder-pro' ),
						'pdf'  => esc_html__( 'pdf', 'shopbuilder-pro' ),
					]
				),
				'value'    => [ 'jpg', 'png', 'webp' ],
				'help'     => esc_html__( 'Select allowed image extensions for image upload field.', 'shopbuilder-pro' ),
				'tab'      => 'general',
			],
			'price_box_settings_intro'          => [
				'id'    => 'price_box_settings_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Price Box Settings', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'add_ons_price_total_box'           => [
				'id'      => 'add_ons_price_total_box',
				'type'    => 'select',
				'label'   => __( 'Total Price Box Display', 'shopbuilder-pro' ),
				'help'    => esc_html__( 'Select which information to show in the total price box.', 'shopbuilder-pro' ),
				'value'   => 'all',
				'options' => [
					'all'   => esc_html__( 'Show product price & add-ons', 'shopbuilder-pro' ),
					'total' => esc_html__( 'Show only the final total', 'shopbuilder-pro' ),
					'hide'  => esc_html__( 'Hide total price box', 'shopbuilder-pro' ),
				],
				'tab'     => 'general',
			],
			'add_ons_price_box_text'            => [
				'id'          => 'add_ons_price_box_text',
				'label'       => __( 'Total Price Box Text', 'shopbuilder-pro' ),
				'type'        => 'text',
				'value'       => esc_html__( 'Grand Total', 'shopbuilder-pro' ),
				'placeholder' => esc_html__( 'Ex. Grand Total', 'shopbuilder-pro' ),
				'help'        => esc_html__( 'Enter the text for the total price box.', 'shopbuilder-pro' ),
				'tab'         => 'general',
			],
			'add_ons_visibility_settings_intro' => [
				'id'    => 'add_ons_visibility_settings_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Add-Ons Visibility', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'add_ons_show_add_to_cart'          => [
				'id'    => 'add_ons_show_add_to_cart',
				'type'  => 'switch',
				'label' => __( 'Show Add to Cart After Selection', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option to show add to cart button after selection of the product add-ons.', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'add_ons_show_in_cart'              => [
				'id'    => 'add_ons_show_in_cart',
				'type'  => 'switch',
				'label' => __( 'Show Add-Ons in Cart Page', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option to show the product add-ons in the cart or checkout page.', 'shopbuilder-pro' ),
				'value' => 'on',
				'tab'   => 'general',
			],
			'add_ons_show_in_order_email'       => [
				'id'    => 'add_ons_show_in_order_email',
				'type'  => 'switch',
				'label' => __( 'Show Add-Ons in Order Email', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option to show the product add-ons in the order email.', 'shopbuilder-pro' ),
				'value' => 'on',
				'tab'   => 'general',
			],
			'add_ons_user_settings_intro'       => [
				'id'    => 'add_ons_user_settings_intro',
				'type'  => 'title',
				'label' => esc_html__( 'User Access and Permissions', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'add_ons_user_role_management'      => [
				'id'      => 'add_ons_user_role_management',
				'label'   => __( 'Specify Users Roles', 'shopbuilder-pro' ),
				'help'    => esc_html__( 'Determine which user groups can see product add-ons.', 'shopbuilder-pro' ),
				'type'    => 'select',
				'value'   => 'all',
				'options' => [
					'all'      => esc_html__( 'All users', 'shopbuilder-pro' ),
					'specific' => esc_html__( 'Only users with a specific role', 'shopbuilder-pro' ),
				],
				'tab'     => 'general',
			],
			'add_ons_roles'                     => [
				'id'         => 'add_ons_roles',
				'label'      => esc_html__( 'Select User Roles', 'shopbuilder-pro' ),
				'help'       => esc_html__( 'Please choose user roles.', 'shopbuilder-pro' ),
				'type'       => 'select',
				'value'      => [ 'administrator', 'customer' ],
				'options'    => FnsPro::get_all_user_roles(),
				'multiple'   => true,
				'tab'        => 'general',
				'dependency' => [
					'rules' => [
						[
							'item'     => 'modules.product_add_ons.add_ons_user_role_management',
							'value'    => 'specific',
							'operator' => '==',
						],
					],
				],
			],
			'add_on_groups_intro'               => [
				'id'   => 'add_on_groups_intro',
				'type' => 'description',
				'text' => esc_html__( 'To add new product add-ons, simply click on the \'Add New Group\' button below.', 'shopbuilder-pro' ),
				'tab'  => 'add_ons',
			],
			'add_on_groups'                     => [
				'id'           => 'add_on_groups',
				'type'         => 'product_addons_special_settings',
				'tab'          => 'add_ons',
				'cat_list'     => Fns::products_category_query(),
				'product_list' => Fns::get_post_types( null, [ 'post_type' => 'product' ] ),
			],
			'addons_style_intro'                => [
				'id'    => 'addons_style_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Group Header', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'group_header_tag'                  => [
				'id'      => 'group_header_tag',
				'label'   => __( 'Group Heading Tag', 'shopbuilder-pro' ),
				'help'    => esc_html__( 'Select the HTML tag to use for the group heading.', 'shopbuilder-pro' ),
				'type'    => 'select',
				'value'   => 'h2',
				'options' => [
					'h1' => esc_html__( 'H1', 'shopbuilder-pro' ),
					'h2' => esc_html__( 'H2', 'shopbuilder-pro' ),
					'h3' => esc_html__( 'H3', 'shopbuilder-pro' ),
					'h4' => esc_html__( 'H4', 'shopbuilder-pro' ),
					'h5' => esc_html__( 'H5', 'shopbuilder-pro' ),
					'h6' => esc_html__( 'H6', 'shopbuilder-pro' ),
				],
				'tab'     => 'styles',
			],
			'group_header_font_size'            => [
				'id'    => 'group_header_font_size',
				'type'  => 'text',
				'label' => esc_html__( 'Group Heading Font Size', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Font Size. Ex. 16px', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'group_desc_font_size'              => [
				'id'    => 'group_desc_font_size',
				'type'  => 'text',
				'label' => esc_html__( 'Group Description Font Size', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Font Size. Ex. 16px', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'group_header_background_color'     => [
				'id'    => 'group_header_background_color',
				'label' => esc_html__( 'Group Header Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the background color for the group header.', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'group_header_text_color'           => [
				'id'    => 'group_header_text_color',
				'label' => esc_html__( 'Group Header Text Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the text color for the group header.', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'group_header_padding'              => [
				'id'    => 'group_header_padding',
				'type'  => 'text',
				'label' => esc_html__( 'Group Header Padding', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Padding. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'group_header_margin'               => [
				'id'    => 'group_header_margin',
				'type'  => 'text',
				'label' => esc_html__( 'Group Header Margin', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Margin. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'groups_style_intro'                => [
				'id'    => 'groups_style_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Field Groups', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'field_label_tag'                   => [
				'id'      => 'field_label_tag',
				'label'   => __( 'Field Label Tag', 'shopbuilder-pro' ),
				'help'    => esc_html__( 'Select the HTML tag to use for the field label.', 'shopbuilder-pro' ),
				'type'    => 'select',
				'value'   => 'h3',
				'options' => [
					'h1' => esc_html__( 'H1', 'shopbuilder-pro' ),
					'h2' => esc_html__( 'H2', 'shopbuilder-pro' ),
					'h3' => esc_html__( 'H3', 'shopbuilder-pro' ),
					'h4' => esc_html__( 'H4', 'shopbuilder-pro' ),
					'h5' => esc_html__( 'H5', 'shopbuilder-pro' ),
					'h6' => esc_html__( 'H6', 'shopbuilder-pro' ),
				],
				'tab'     => 'styles',
			],
			'field_label_font_size'             => [
				'id'    => 'field_label_font_size',
				'type'  => 'text',
				'label' => esc_html__( 'Field Label Font Size', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Font Size. Ex. 16px', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'field_desc_font_size'              => [
				'id'    => 'field_desc_font_size',
				'type'  => 'text',
				'label' => esc_html__( 'Field Description Font Size', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Font Size. Ex. 16px', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'field_label_color'                 => [
				'id'    => 'field_label_color',
				'label' => esc_html__( 'Field Title Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the text color for the field titles.', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'group_background_color'            => [
				'id'    => 'group_background_color',
				'label' => esc_html__( 'Group Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the background color for the field group.', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'group_border_color'                => [
				'id'    => 'group_border_color',
				'label' => esc_html__( 'Group Border Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the text color for the field header.', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'group_radius'                      => [
				'id'    => 'group_radius',
				'type'  => 'text',
				'label' => esc_html__( 'Group Border Radius', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Border Radius. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'group_tooltip_position'            => [
				'id'      => 'group_tooltip_position',
				'label'   => __( 'Tooltip Position', 'shopbuilder-pro' ),
				'help'    => esc_html__( 'Select the position of the tooltip.', 'shopbuilder-pro' ),
				'type'    => 'select',
				'value'   => 'right',
				'options' => [
					'top'    => esc_html__( 'Top', 'shopbuilder-pro' ),
					'bottom' => esc_html__( 'Bottom', 'shopbuilder-pro' ),
					'left'   => esc_html__( 'Left', 'shopbuilder-pro' ),
					'right'  => esc_html__( 'Right', 'shopbuilder-pro' ),
				],
				'tab'     => 'styles',
			],
			'group_padding'                     => [
				'id'    => 'group_padding',
				'type'  => 'text',
				'label' => esc_html__( 'Group Padding', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Padding. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'group_margin'                      => [
				'id'    => 'group_margin',
				'type'  => 'text',
				'label' => esc_html__( 'Group Margin', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Margin. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'form_style_intro'                  => [
				'id'    => 'form_style_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Form Styles', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'form_primary_color'                => [
				'id'    => 'form_primary_color',
				'type'  => 'color',
				'label' => esc_html__( 'Field Primary Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the primary color for the fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'form_background_color'             => [
				'id'    => 'form_background_color',
				'type'  => 'color',
				'label' => esc_html__( 'Field Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the background color for the fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'field_text_color'                  => [
				'id'    => 'field_text_color',
				'label' => esc_html__( 'Field Text Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the text color for the fields.', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'form_border_color'                 => [
				'id'    => 'form_border_color',
				'type'  => 'color',
				'label' => esc_html__( 'Field Border Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the border color for the fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'checkbox_background_color'         => [
				'id'    => 'checkbox_background_color',
				'type'  => 'color',
				'label' => esc_html__( 'Checkbox Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the background color for the checkbox fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'checkbox_label_color'              => [
				'id'    => 'checkbox_label_color',
				'type'  => 'color',
				'label' => esc_html__( 'Checkbox Label Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the label color for the checkbox fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'checkbox_border_color'             => [
				'id'    => 'checkbox_border_color',
				'type'  => 'color',
				'label' => esc_html__( 'Checkbox Border Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the border color for the checkbox fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'radio_background_color'            => [
				'id'    => 'radio_background_color',
				'type'  => 'color',
				'label' => esc_html__( 'Radio Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the background color for the radio fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'radio_label_color'                 => [
				'id'    => 'radio_label_color',
				'type'  => 'color',
				'label' => esc_html__( 'Radio Label Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the label color for the radio fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'radio_border_color'                => [
				'id'    => 'radio_border_color',
				'type'  => 'color',
				'label' => esc_html__( 'Radio Border Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the border color for the radio fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'text_select_background_color'      => [
				'id'    => 'text_select_background_color',
				'type'  => 'color',
				'label' => esc_html__( 'Text-Select Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the background color for the text-select fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'text_select_label_color'           => [
				'id'    => 'text_select_label_color',
				'type'  => 'color',
				'label' => esc_html__( 'Text-Select Label Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the label color for the text-select fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'switch_background_color'           => [
				'id'    => 'switch_background_color',
				'type'  => 'color',
				'label' => esc_html__( 'Switch Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the background color for the switch fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'switch_label_color'                => [
				'id'    => 'switch_label_color',
				'type'  => 'color',
				'label' => esc_html__( 'Switch Label Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the label color for the switch fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'image_button_color'                => [
				'id'    => 'image_button_color',
				'type'  => 'color',
				'label' => esc_html__( 'Image Upload Button Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the button color for the image upload fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'image_button_bg_color'             => [
				'id'    => 'image_button_bg_color',
				'type'  => 'color',
				'label' => esc_html__( 'Image Upload Button Background', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the button background color for the image upload fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'image_button_hover_color'          => [
				'id'    => 'image_button_hover_color',
				'type'  => 'color',
				'label' => esc_html__( 'Image Upload Button Hover Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the button hover color for the image upload fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'image_button_hover_bg_color'       => [
				'id'    => 'image_button_hover_bg_color',
				'type'  => 'color',
				'label' => esc_html__( 'Image Upload Button Hover Background', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the button hover background color for the image upload fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'price_box_style_intro'             => [
				'id'    => 'price_box_style_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Price Box', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'price_box_text_color'              => [
				'id'    => 'price_box_text_color',
				'label' => esc_html__( 'Price Box Text Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the text color for the price box.', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'price_box_background_color'        => [
				'id'    => 'price_box_background_color',
				'label' => esc_html__( 'Price Box Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the background color for the price box.', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'price_box_border_color'            => [
				'id'    => 'price_box_border_color',
				'label' => esc_html__( 'Price Box Border Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the border color for the price box.', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'price_box_radius'                  => [
				'id'    => 'price_box_radius',
				'type'  => 'text',
				'label' => esc_html__( 'Price Box Border Radius', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter price box Border Radius. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'price_box_padding'                 => [
				'id'    => 'price_box_padding',
				'type'  => 'text',
				'label' => esc_html__( 'Price Box Padding', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Padding. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'price_box_margin'                  => [
				'id'    => 'price_box_margin',
				'type'  => 'text',
				'label' => esc_html__( 'Price Box Margin', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Margin. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
		];
	}

	/**
	 * Renders the HTML template for a product add-on group in the admin area.
	 *
	 * @param array|null $group       An associative array containing group details.
	 * @param int        $group_index The index of the group.
	 *
	 * @return string The HTML output of the group template.
	 */
	public static function render_admin_group_template( $group = null, $group_index = 0 ) {
		ob_start();
		?>
		<div class="rtsb-addon-group" data-index="<?php echo esc_attr( $group_index ); ?>">
			<?php
			Fns::print_html(
				AdminInputsFns::render_title_with_actions(
					[
						'type'         => 'h3',
						'title_class'  => 'addons-group-title',
						'title'        => $group['name'] ?? esc_html__( 'Untitled Group', 'shopbuilder-pro' ),
						'drag_tip'     => esc_html__( 'Drag and drop to rearrange this group.', 'shopbuilder-pro' ),
						'remove_class' => 'remove-group-button',
					]
				),
				true
			);
			?>
			<div class="rtsb-addon-group-content hidden">
				<div class="data">
					<?php
					$group_id_part = 'rtsb_addon_groups[' . esc_attr( $group_index ) . ']';
					$fields        = AdminInputsFns::generate_text_input(
						[
							'id'            => $group_id_part . '[name]',
							'label'         => esc_html__( 'Group Name', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Product Addons group name', 'shopbuilder-pro' ),
							'description'   => __( 'Enter the group name. <br />This is a required field.', 'shopbuilder-pro' ),
							'placeholder'   => esc_html__( 'Enter group name', 'shopbuilder-pro' ),
							'value'         => $group['name'] ?? '',
							'input_class'   => 'rtsb-addon-group-name',
							'wrapper_class' => 'form-field group-name',
						]
					);

					$fields .= AdminInputsFns::generate_textarea_input(
						[
							'id'            => $group_id_part . '[description]',
							'label'         => esc_html__( 'Group Description', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Product Addons group description', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Enter the group description.', 'shopbuilder-pro' ),
							'placeholder'   => esc_html__( 'Enter group description', 'shopbuilder-pro' ),
							'value'         => $group['description'] ?? '',
							'input_class'   => 'short rtsb-addon-group-description',
							'wrapper_class' => 'form-field group-description',
						]
					);

					$fields .= AdminInputsFns::generate_switch_input(
						[
							'id'            => $group_id_part . '[show_name]',
							'label'         => esc_html__( 'Display Group Name?', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Show Product Addons group name', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Switch on to show group name in the frontend.', 'shopbuilder-pro' ),
							'checked'       => ! empty( $group['show_name'] ) && 'on' === $group['show_name'] ? 'on' : 'off',
							'value'         => 'yes',
							'input_class'   => 'switch-input',
							'wrapper_class' => 'form-field show-name',
						]
					);

					$fields .= AdminInputsFns::generate_switch_input(
						[
							'id'            => $group_id_part . '[show_desc]',
							'label'         => esc_html__( 'Display Group Description?', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Show Product Addons group description', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Switch on to show group description in the frontend.', 'shopbuilder-pro' ),
							'checked'       => ! empty( $group['show_desc'] ) && 'on' === $group['show_desc'] ? 'on' : 'off',
							'value'         => 'yes',
							'input_class'   => 'switch-input',
							'wrapper_class' => 'form-field show-description',
						]
					);

					Fns::print_html( $fields, true );
					?>
					<div class="rtsb-addon-inputs<?php echo empty( $group['inputs'] ) ? ' hidden' : ''; ?>">
						<div class="addon-inputs-wrapper">
							<?php
							if ( ! empty( $group['inputs'] ) ) {
								foreach ( $group['inputs'] as $input_index => $input ) {
									Fns::print_html( self::render_admin_input_template( $input, $group_index, $input_index ), true );
								}
							}
							?>
						</div>
					</div>

					<button type="button" class="button add-input-button"><?php echo esc_html__( 'Add New Field', 'shopbuilder-pro' ); ?></button>
				</div>
			</div>
		</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Renders the HTML template for a product add-on input field in the admin area.
	 *
	 * @param array|null $input       An associative array containing input details.
	 * @param int        $group_index The index of the group the input belongs to.
	 * @param int        $input_index The index of the input field.
	 *
	 * @return string The HTML output of the input field template.
	 */
	public static function render_admin_input_template( $input = null, $group_index = 0, $input_index = 0 ) {
		ob_start();
		?>
		<div class="form-row rtsb-addon-input" data-index="<?php echo esc_attr( $input_index ); ?>">
			<?php
			Fns::print_html(
				AdminInputsFns::render_title_with_actions(
					[
						'type'         => 'h4',
						'title_class'  => 'addons-field-title',
						'title'        => $input['label'] ?? esc_html__( 'Untitled Field', 'shopbuilder-pro' ),
						'drag_tip'     => esc_html__( 'Drag and drop to rearrange this field.', 'shopbuilder-pro' ),
						'remove_class' => 'remove-input-button',
					]
				),
				true
			);
			?>
			<div class="rtsb-addon-field-content">
				<div class="flex-row">
					<?php
					$field_id_part = 'rtsb_addon_groups[' . esc_attr( $group_index ) . '][inputs][' . esc_attr( $input_index ) . ']';
					$fields        = AdminInputsFns::generate_text_input(
						[
							'id'            => $field_id_part . '[label]',
							'label'         => esc_html__( 'Field Label', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Product Addons field label', 'shopbuilder-pro' ),
							'description'   => __( 'Enter the field label. <br />This is a required field.', 'shopbuilder-pro' ),
							'placeholder'   => esc_html__( 'Enter field label', 'shopbuilder-pro' ),
							'value'         => $input['label'] ?? '',
							'input_class'   => 'rtsb-addon-label-name',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-row input-label',
						]
					);

					$fields .= AdminInputsFns::generate_select_input(
						[
							'id'            => $field_id_part . '[type]',
							'label'         => esc_html__( 'Field Type', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Product Addons field type', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Select your desired field type.', 'shopbuilder-pro' ),
							'value'         => $input['type'] ?? '',
							'input_class'   => 'rtsb-addon-input-type',
							'label_class'   => 'field-label',
							'options'       => self::field_types(),
							'wrapper_class' => 'form-row input-type',
						]
					);

					$fields .= AdminInputsFns::generate_number_input(
						[
							'id'            => $field_id_part . '[price]',
							'label'         => esc_html__( 'Field Price', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Product Addons field price', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Enter the field price. Field price will not work if option is added.', 'shopbuilder-pro' ),
							'value'         => $input['price'] ?? '',
							'min'           => '0',
							'step'          => '0.01',
							'input_class'   => 'rtsb-addon-input-price',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-row input-price',
						]
					);

					$fields .= AdminInputsFns::generate_textarea_input(
						[
							'id'            => $field_id_part . '[description]',
							'label'         => esc_html__( 'Field Description', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Product Addons field description', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Enter the field description.', 'shopbuilder-pro' ),
							'placeholder'   => esc_html__( 'Enter field description', 'shopbuilder-pro' ),
							'value'         => $input['description'] ?? '',
							'input_class'   => 'rtsb-addon-input-description',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-row input-description',
						]
					);

					$fields .= AdminInputsFns::generate_text_input(
						[
							'id'            => $field_id_part . '[placeholder]',
							'label'         => esc_html__( 'Field Placeholder', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Product Addons field placeholder', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Enter the field placeholder. This can also be used if the input type is select.', 'shopbuilder-pro' ),
							'placeholder'   => esc_html__( 'Enter field placeholder', 'shopbuilder-pro' ),
							'value'         => $input['placeholder'] ?? '',
							'input_class'   => 'rtsb-addon-input-placeholder',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-row input-placeholder',
						]
					);

					Fns::print_html( $fields, true );
					?>
				</div>

				<div class="form-row">
					<?php
					$fields = AdminInputsFns::generate_switch_input(
						[
							'id'            => $field_id_part . '[required]',
							'label'         => esc_html__( 'Is this field Required?', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Show Product Addons required field', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Switch on to add this field as required.', 'shopbuilder-pro' ),
							'checked'       => ! empty( $input['required'] ) && 'on' === $input['required'] ? 'on' : 'off',
							'value'         => 'yes',
							'input_class'   => 'switch-input',
							'wrapper_class' => 'form-field input-required',
						]
					);

					$fields .= AdminInputsFns::generate_switch_input(
						[
							'id'            => $field_id_part . '[tooltip]',
							'label'         => esc_html__( 'Show description as tooltip?', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Show Product Addons field description as tooltip', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Switch on to show the field description as a tooltip.', 'shopbuilder-pro' ),
							'checked'       => ! empty( $input['tooltip'] ) && 'on' === $input['tooltip'] ? 'on' : 'off',
							'value'         => 'yes',
							'input_class'   => 'switch-input',
							'wrapper_class' => 'form-field input-tooltip',
						]
					);

					$fields .= AdminInputsFns::generate_switch_input(
						[
							'id'            => $field_id_part . '[multiple]',
							'label'         => esc_html__( 'Allow multiple selection?', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Select Product Addons field values as multiple', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Switch on to allow multiple selection.', 'shopbuilder-pro' ),
							'checked'       => ! empty( $input['multiple'] ) && 'on' === $input['multiple'] ? 'on' : 'off',
							'value'         => 'yes',
							'input_class'   => 'switch-input',
							'wrapper_class' => 'form-field input-multiple hidden',
						]
					);

					Fns::print_html( $fields, true );
					?>
				</div>

				<div class="form-row form-row-full rtsb-addon-options<?php echo empty( $input['options'] ) ? ' hidden' : ''; ?>">
					<div class="addon-options-wrapper">
						<?php
						if ( isset( $input['options'] ) ) {
							foreach ( $input['options'] as $option_index => $option ) {
								Fns::print_html( self::render_admin_option_template( $option, $group_index, $input_index, $option_index ), true );
							}
						}
						?>
					</div>
				</div>

				<div class="field-button hidden">
					<button type="button" class="button add-option-button"><?php esc_html_e( 'Add New Option', 'shopbuilder-pro' ); ?></button>
				</div>
			</div>
		</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Renders the HTML template for a product add-on option field in the admin area.
	 *
	 * @param string|null $option       The value of the option.
	 * @param int         $group_index  The index of the group the input belongs to.
	 * @param int         $input_index  The index of the input field.
	 * @param int         $option_index The index of the option.
	 *
	 * @return string The HTML output of the option field template.
	 */
	public static function render_admin_option_template( $option = null, $group_index = 0, $input_index = 0, $option_index = 0 ) {
		ob_start();
		?>
		<div class="rtsb-addon-option" data-index="<?php echo esc_attr( $option_index ); ?>">
			<div class="rtsb-addon-option-content flex-row">
				<?php
				$option_id_part = 'rtsb_addon_groups[' . esc_attr( $group_index ) . '][inputs][' . esc_attr( $input_index ) . '][options][' . esc_attr( $option_index ) . ']';
				$fields         = AdminInputsFns::generate_text_input(
					[
						'id'            => $option_id_part . '[name]',
						'label'         => esc_html__( 'Option Name', 'shopbuilder-pro' ),
						'desc_tip'      => false,
						'placeholder'   => esc_html__( 'Enter option name', 'shopbuilder-pro' ),
						'value'         => $option['name'] ?? '',
						'input_class'   => 'rtsb-addon-option-name',
						'label_class'   => 'field-label',
						'wrapper_class' => 'form-row option-name',
					]
				);

				$fields .= AdminInputsFns::generate_number_input(
					[
						'id'            => $option_id_part . '[price]',
						'label'         => esc_html__( 'Option Price', 'shopbuilder-pro' ),
						'desc_tip'      => false,
						'value'         => $option['price'] ?? '',
						'min'           => '0',
						'step'          => '0.01',
						'input_class'   => 'rtsb-addon-option-price',
						'label_class'   => 'field-label',
						'wrapper_class' => 'form-row option-price',
					]
				);

				$fields .= AdminInputsFns::generate_switch_input(
					[
						'id'            => $option_id_part . '[default]',
						'label'         => esc_html__( 'Default Selected?', 'shopbuilder-pro' ),
						'desc_tip'      => false,
						'checked'       => ! empty( $option['default'] ) && 'on' === $option['default'] ? 'on' : 'off',
						'value'         => 'yes',
						'input_class'   => 'switch-input',
						'wrapper_class' => 'form-field option-default',
					]
				);

				Fns::print_html( $fields, true );
				?>
			</div>
			<?php
			Fns::print_html(
				AdminInputsFns::render_title_with_actions(
					[
						'type'         => 'div',
						'title_class'  => 'addons-option-title',
						'title'        => '',
						'drag_tip'     => esc_html__( 'Drag and drop to rearrange this option', 'shopbuilder-pro' ),
						'remove_class' => 'remove-option-button',
						'has_edit'     => false,
					]
				),
				true
			);
			?>
		</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Saves the product add-ons data from the admin form.
	 *
	 * @param int $post_id The ID of the product post.
	 *
	 * @return void
	 */
	public static function save_product_add_ons( $post_id ) {
		// Check if post_id is valid.
		if ( ! self::is_valid( absint( $post_id ) ) ) {
			return;
		}

		// Verify nonce.
		check_admin_referer( rtsb()->nonceText, rtsb()->nonceId );

		$disable_addons = isset( $_REQUEST['_rtsb_disable_addons'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_disable_addons'] ) ) : 'off';
		$enable_addons  = isset( $_REQUEST['_rtsb_enable_addons'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_enable_addons'] ) ) : 'off';
		$disable_addons = 'yes' === $disable_addons ? 'on' : $disable_addons;
		$enable_addons  = 'yes' === $enable_addons ? 'on' : $enable_addons;
		$addon_groups   = isset( $_REQUEST['rtsb_addon_groups'] ) ? map_deep( wp_unslash( $_REQUEST['rtsb_addon_groups'] ), 'sanitize_text_field' ) : [];
		$add_ons        = self::process_addon_groups( $addon_groups );

		self::update_post_meta( $post_id, $disable_addons, $enable_addons, $add_ons );
		wp_cache_delete( 'rtsb_product_addons_for_' . $post_id, 'shopbuilder' );
		wp_cache_delete( 'rtsb_product_addons_' . $post_id, 'shopbuilder' );
	}

	/**
	 * Checks if the provided post-ID is valid.
	 *
	 * @param int|string $post_id The post ID to validate.
	 *
	 * @return bool
	 */
	private static function is_valid( $post_id ) {
		return ! empty( $post_id ) && is_numeric( $post_id );
	}

	/**
	 * Processes an array of addon groups to filter out empty groups and sanitize data.
	 *
	 * @param array $addon_groups An array of addon groups to process.
	 *
	 * @return array
	 */
	private static function process_addon_groups( $addon_groups ) {
		$add_ons = [];

		// Filter out empty groups.
		$addon_groups = array_filter(
			$addon_groups,
			function ( $group ) {
				return ! empty( $group['name'] );
			}
		);

		foreach ( $addon_groups as $group_index => $group ) {
			if ( ! empty( $group['name'] ) ) {
				$group_data = [
					'id'          => absint( $group_index ),
					'name'        => sanitize_text_field( $group['name'] ),
					'description' => ! empty( $group['description'] ) ? wp_kses_post( $group['description'] ) : '',
					'show_name'   => ! empty( $group['show_name'] ) && ( 'on' === $group['show_name'] || 'yes' === $group['show_name'] ) ? 'on' : 'off',
					'show_desc'   => ! empty( $group['show_desc'] ) && ( 'on' === $group['show_desc'] || 'yes' === $group['show_desc'] ) ? 'on' : 'off',
					'inputs'      => self::process_inputs( $group['inputs'] ?? [] ),
				];

				if ( ! empty( $group_data['inputs'] ) || empty( $group['inputs'] ) ) {
					$add_ons[] = $group_data;
				}
			}
		}

		return $add_ons;
	}

	/**
	 * Processes an array of input data within addon groups.
	 *
	 * @param array $inputs An array of inputs to process.
	 *
	 * @return array
	 */
	private static function process_inputs( $inputs ) {
		$processed_inputs = [];

		if ( ! empty( $inputs ) && is_array( $inputs ) ) {
			foreach ( $inputs as $input_index => $input ) {
				if ( ! empty( $input['label'] ) ) {
					$input_data = [
						'id'          => absint( $input_index ),
						'type'        => sanitize_text_field( $input['type'] ),
						'label'       => sanitize_text_field( $input['label'] ),
						'price'       => isset( $input['price'] ) ? sanitize_text_field( $input['price'] ) : '',
						'description' => isset( $input['description'] ) ? wp_kses_post( $input['description'] ) : '',
						'placeholder' => isset( $input['placeholder'] ) ? sanitize_text_field( $input['placeholder'] ) : '',
						'required'    => ! empty( $input['required'] ) && ( 'on' === $input['required'] || 'yes' === $input['required'] ) ? 'on' : 'off',
						'tooltip'     => ! empty( $input['tooltip'] ) && ( 'on' === $input['tooltip'] || 'yes' === $input['tooltip'] ) ? 'on' : 'off',
						'multiple'    => ! empty( $input['multiple'] ) && ( 'on' === $input['multiple'] || 'yes' === $input['multiple'] ) ? 'on' : 'off',
						'options'     => self::process_options( $input['options'] ?? [] ),
					];

					$processed_inputs[] = $input_data;
				}
			}
		}

		return $processed_inputs;
	}

	/**
	 * Processes an array of options for a given input.
	 *
	 * @param array $options An array of options to process.
	 *
	 * @return array
	 */
	private static function process_options( $options ) {
		$processed_options = [];

		if ( isset( $options ) && is_array( $options ) ) {
			foreach ( $options as $option_index => $option ) {
				$option_data = [
					'id'      => absint( $option_index ),
					'name'    => sanitize_text_field( $option['name'] ),
					'price'   => isset( $option['price'] ) ? sanitize_text_field( $option['price'] ) : '',
					'default' => ! empty( $option['default'] ) && ( 'on' === $option['default'] || 'yes' === $option['default'] ) ? 'on' : 'off',
				];

				$processed_options[] = $option_data;
			}
		}

		return $processed_options;
	}

	/**
	 * Updates or deletes post meta based on the provided data.
	 *
	 * @param int    $post_id The ID of the post to update or delete meta data for.
	 * @param string $disable_addons The value indicating whether addons are disabled.
	 * @param string $enable_addons The value indicating whether addons are enabled.
	 * @param array  $add_ons An array of processed addon groups to save as post-meta.
	 *
	 * @return void
	 */
	private static function update_post_meta( $post_id, $disable_addons, $enable_addons, $add_ons ) {
		if ( ! empty( $disable_addons ) && 'on' === $disable_addons ) {
			update_post_meta( $post_id, '_rtsb_disable_addons', $disable_addons );
		} else {
			delete_post_meta( $post_id, '_rtsb_disable_addons' );
		}

		if ( ! empty( $enable_addons ) && 'on' === $enable_addons ) {
			update_post_meta( $post_id, '_rtsb_enable_addons', $enable_addons );
		} else {
			delete_post_meta( $post_id, '_rtsb_enable_addons' );
		}

		if ( ! empty( $add_ons ) ) {
			update_post_meta( $post_id, '_rtsb_product_addons', $add_ons );
		} else {
			delete_post_meta( $post_id, '_rtsb_product_addons' );
		}
	}

	/**
	 * Get pre-order settings data.
	 *
	 * @return array
	 */
	public static function get_add_ons_settings_data() {
		$settings    = Fns::get_options( 'modules', 'product_add_ons' ) ?? [];
		$cache_key   = 'rtsb_addons_settings_data_' . md5( wp_json_encode( $settings ) );
		$cached_data = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false !== $cached_data ) {
			return $cached_data;
		}

		$groups = RenderHelpers::get_data( $settings, 'add_on_groups', [] );

		if ( ! empty( $groups ) && is_string( $groups ) ) {
			$groups = json_decode( $groups, true );
		}

		$settings_data = [
			'position'             => RenderHelpers::get_data( $settings, 'add_ons_position', 'before' ),
			'cart_button_label'    => RenderHelpers::get_data( $settings, 'add_ons_cart_button_label', esc_html__( 'Select Options', 'shopbuilder-pro' ) ),
			'update_base_price'    => RenderHelpers::get_data( $settings, 'add_ons_update_base_price', '' ),
			'allowed_extensions'   => RenderHelpers::get_data( $settings, 'allowed_image_extensions', [ 'jpg', 'png' ] ),
			'hook_priority'        => RenderHelpers::get_data( $settings, 'add_ons_hook_priority', '' ),
			'price_total_box'      => RenderHelpers::get_data( $settings, 'add_ons_price_total_box', 'all' ),
			'price_box_text'       => RenderHelpers::get_data( $settings, 'add_ons_price_box_text', esc_html__( 'Grand Total', 'shopbuilder-pro' ) ),
			'show_add_to_cart'     => RenderHelpers::get_data( $settings, 'add_ons_show_add_to_cart', '' ),
			'show_in_cart'         => RenderHelpers::get_data( $settings, 'add_ons_show_in_cart', '' ),
			'show_in_order_email'  => RenderHelpers::get_data( $settings, 'add_ons_show_in_order_email', '' ),
			'user_role_management' => RenderHelpers::get_data( $settings, 'add_ons_user_role_management', 'all' ),
			'roles'                => RenderHelpers::get_data( $settings, 'add_ons_roles', [ 'administrator', 'customer' ] ),
			'groups'               => $groups,
		];

		wp_cache_set( $cache_key, $settings_data, 'shopbuilder', 12 * HOUR_IN_SECONDS );
		Cache::set_data_cache_key( $cache_key );

		return $settings_data;
	}

	/**
	 * Get Product addon data.
	 *
	 * @param number $product_id .
	 * @return array
	 */
	public static function get_addon_meta_data( $product_id ) {
		if ( ! $product_id ) {
			return [];
		}

		$cache_key   = 'rtsb_addons_meta_data_' . $product_id;
		$cached_data = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false !== $cached_data ) {
			return $cached_data;
		}

		$product = wc_get_product( $product_id );

		if ( ! $product ) {
			return [];
		}

		$meta_mapping = self::product_meta_data_mapping();
		$product_meta = [];

		foreach ( $meta_mapping as $meta_key => $custom_field_id ) {
			$product_meta[ $meta_key ] = $product->get_meta( $custom_field_id );
		}

		wp_cache_set( $cache_key, $product_meta, 'shopbuilder', 12 * HOUR_IN_SECONDS );
		Cache::set_data_cache_key( $cache_key );

		return $product_meta;
	}

	/**
	 * Maps product meta-keys to their corresponding custom field IDs.
	 *
	 * @return array
	 */
	public static function product_meta_data_mapping() {
		return [
			'disable'  => '_rtsb_disable_addons',
			'override' => '_rtsb_enable_addons',
			'groups'   => '_rtsb_product_addons',
		];
	}

	/**
	 * Check if the global add-ons are disabled for a specific product.
	 *
	 * @param int $product_id The ID of the product to check.
	 *
	 * @return bool
	 */
	public static function is_global_addons_disabled( $product_id ) {
		return 'on' === ( self::get_addon_meta_data( $product_id )['disable'] ?? 'off' );
	}

	/**
	 * Check if the global add-ons settings are overridden for a specific product.
	 *
	 * @param int $product_id The ID of the product to check.
	 *
	 * @return bool
	 */
	public static function is_global_addons_overridden( $product_id ) {
		return 'on' === ( self::get_addon_meta_data( $product_id )['override'] ?? 'off' );
	}

	/**
	 *
	 * @return mixed
	 */
	public static function get_global_addon_groups() {
		return self::get_add_ons_settings_data()['groups'];
	}

	/**
	 * Get global and product addon lists.
	 *
	 * @param number $product_id .
	 * @return array;
	 */
	public static function get_addon_groups( $product_id ) {
		$cache_key   = 'rtsb_product_addon_groups_' . $product_id;
		$cached_data = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false !== $cached_data ) {
			return $cached_data;
		}

		if ( self::is_global_addons_disabled( $product_id ) ) {
			return [];
		}

		if ( self::is_global_addons_overridden( $product_id ) ) {
			$addon_groups = self::get_addon_meta_data( $product_id )['groups'];
		} else {
			$global_addons = self::get_global_addon_groups();
			$addon_groups  = self::get_relevant_addons( $global_addons, $product_id );
		}

		wp_cache_set( $cache_key, $addon_groups, 'shopbuilder', 12 * HOUR_IN_SECONDS );
		Cache::set_data_cache_key( $cache_key );

		return $addon_groups;
	}

	/**
	 * Checks if a product has any addons.
	 *
	 * @param object $product The product object to check.
	 * @param bool   $variation Variation check.
	 *
	 * @return bool
	 */
	public static function product_has_addons( $product, $variation = false ) {
		$product_id    = $variation ? $product->get_parent_id() : $product->get_id();
		$cache_key     = 'rtsb_product_addons_for_' . $product_id;
		$cached_result = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false !== $cached_result ) {
			return $cached_result;
		}

		$has_addons = ! empty( self::get_addon_groups( $product_id ) );

		wp_cache_set( $cache_key, $has_addons, 'shopbuilder', 12 * HOUR_IN_SECONDS );
		Cache::set_data_cache_key( $cache_key );

		return $has_addons;
	}

	/**
	 * Filters and retrieves relevant add-ons based on product ID and grouping settings.
	 *
	 * @param array $settings An array of settings.
	 * @param int   $product_id The ID of the product.
	 *
	 * @return array
	 */
	public static function get_relevant_addons( $settings, $product_id ) {
		$relevant_addons = [];

		foreach ( $settings as $setting ) {
			$is_relevant_addon = self::is_relevant_addon( $setting, $product_id );

			if ( $is_relevant_addon ) {
				$relevant_addons[] = $setting;
			}
		}

		return array_values( $relevant_addons );
	}

	/**
	 * Determines if a setting is relevant based on the product ID and group settings.
	 *
	 * @param array $setting The setting array.
	 * @param int   $product_id The ID of the product.
	 *
	 * @return bool
	 */
	private static function is_relevant_addon( $setting, $product_id ) {
		$group_product = $setting['group_product'] ?? [];
		$group_cat     = $setting['group_cat'] ?? [];
		$group_by      = $setting['group_by'] ?? 'category';

		if ( 'product' === $group_by ) {
			$group_product_ids = self::get_group_ids( $group_product );

			return self::is_relevant_by_product( $group_product_ids, $product_id );
		} elseif ( 'category' === $group_by ) {
			$group_cat_ids = self::get_group_ids( $group_cat );

			return self::is_relevant_by_category( $group_cat_ids, $product_id );
		}

		return false;
	}

	/**
	 * Get group product IDs.
	 *
	 * @param array $ids The group id array.
	 *
	 * @return array
	 */
	public static function get_group_ids( array $ids ) {
		$group_ids = [];

		foreach ( $ids as $id ) {
			$group_ids[] = $id['value'] ?? null;
		}

		return $group_ids;
	}

	/**
	 * Checks if the setting is relevant based on product ID for 'product' grouping.
	 *
	 * @param array $group_product The product group array.
	 * @param int   $product_id The ID of the product.
	 *
	 * @return bool
	 */
	private static function is_relevant_by_product( $group_product, $product_id ) {
		return empty( $group_product )
			|| ( in_array( $product_id, $group_product, true ) && ( wp_doing_ajax() || ! is_admin() ) );
	}

	/**
	 * Checks if the setting is relevant based on product ID for 'category' grouping.
	 *
	 * @param array $group_cat_ids The category group ids array.
	 * @param int   $product_id The ID of the product.
	 *
	 * @return bool
	 */
	private static function is_relevant_by_category( $group_cat_ids, $product_id ) {
		if ( empty( $group_cat_ids ) ) {
			return true;
		}

		$product_terms = get_the_terms( $product_id, 'product_cat' );

		if ( is_array( $product_terms ) ) {
			$product_term_ids = wp_list_pluck( $product_terms, 'term_id' );

			return ! empty( array_intersect( $product_term_ids, $group_cat_ids ) );
		}

		return false;
	}

	/**
	 * Add-Ons group heading tag.
	 *
	 * @return string
	 */
	public static function group_heading_tag() {
		$tag = Fns::get_options( 'modules', 'product_add_ons' )['group_header_tag'] ?? 'h2';

		return Fns::get_validated_html_tag( $tag );
	}

	/**
	 * Add-Ons field label tag.
	 *
	 * @return string
	 */
	public static function field_heading_tag() {
		$tag = Fns::get_options( 'modules', 'product_add_ons' )['field_label_tag'] ?? 'h3';

		return method_exists( Fns::class, 'get_validated_html_tag' ) ? Fns::get_validated_html_tag( $tag ) : esc_attr( $tag );
	}

	/**
	 * Input field types.
	 *
	 * @return array
	 */
	public static function field_types() {
		return [
			'text'         => esc_html__( 'Text', 'shopbuilder-pro' ),
			'textarea'     => esc_html__( 'Textarea', 'shopbuilder-pro' ),
			'number'       => esc_html__( 'Number', 'shopbuilder-pro' ),
			'radio'        => esc_html__( 'Radio', 'shopbuilder-pro' ),
			'select'       => esc_html__( 'Select', 'shopbuilder-pro' ),
			'checkbox'     => esc_html__( 'Checkbox', 'shopbuilder-pro' ),
			'text_select'  => esc_html__( 'Text Select', 'shopbuilder-pro' ),
			'image_upload' => esc_html__( 'Image Upload', 'shopbuilder-pro' ),
			'switch'       => esc_html__( 'Switch', 'shopbuilder-pro' ),
		];
	}

	/**
	 * Conditionally wraps content in a label tag if required.
	 *
	 * @param string $content The content to wrap.
	 * @param bool   $label_required Whether to wrap the content in a label tag.
	 * @param string $field_id The ID of the field to use for the 'for' attribute of the label.
	 *
	 * @return string
	 */
	public static function wrap_with_label( $content, $label_required = false, $field_id = '' ) {
		$html = '';

		if ( $label_required ) {
			$html .= '<label class="rtsb-addon-label" for="' . esc_attr( $field_id ) . '">';
			$html .= $content;
			$html .= '</label>';
		} else {
			$html .= $content;
		}

		return $html;
	}

	/**
	 * Initialize option data with defaults.
	 *
	 * @param array        $option The option data.
	 * @param string|array $value The field value.
	 *
	 * @return array
	 */
	public static function field_options_data( $option, $value ) {
		$option_name     = RenderHelpers::get_data( $option, 'name', '' );
		$default_checked = ! empty( $option['default'] ) && 'on' === $option['default'];
		$price           = Product::convert_currency( RenderHelpers::get_data( $option, 'price', '' ) );

		return [
			RenderHelpers::get_data( $option, 'id', 0 ),
			RenderHelpers::get_data( $option, 'name', '' ),
			$price,
			$default_checked,
			is_array( $value ) ? in_array( $option_name, $value, true ) : $option_name === $value,
		];
	}

	/**
	 * Checks if any key in the given array matches a specified regular expression pattern.
	 *
	 * @param string $pattern The regular expression pattern to match against the array keys.
	 * @param array  $array   The array to search through for matching keys.
	 *
	 * @return bool
	 */
	public static function key_exists( $pattern, $array ) {
		return count( preg_grep( $pattern, array_keys( $array ) ) ) > 0;
	}

	/**
	 * Display formatted price with currency symbol.
	 *
	 * @param string $class Class.
	 * @param string $price Product price.
	 * @param string $position Currency symbol position.
	 * @param string $symbol_html Symbol html.
	 *
	 * @return string
	 */
	public static function display_price( $class, $price, $position, $symbol_html ) {
		$html = '<span class="amount">';

		if ( 'left' === $position || 'left_space' === $position ) {
			$html .= $symbol_html;
			$html .= 'left_space' === $position ? ' ' : '';
		}

		$html .= '<span class="total ' . esc_attr( $class ?? '' ) . '">';
		$html .= wc_format_decimal( $price, wc_get_price_decimals() );
		$html .= '</span>';

		if ( 'right' === $position || 'right_space' === $position ) {
			$html .= 'right_space' === $position ? ' ' : '';
			$html .= $symbol_html;
		}

		$html .= '</span>';

		return $html;
	}

	/**
	 * Verifies the validity of a request to add a product to the cart.
	 *
	 * @param int $product_id The ID of the product being added to the cart.
	 *
	 * @return bool
	 */
	public static function verify_add_to_cart_request( $product_id ) {
		// Verify nonce.
		if ( ! check_ajax_referer( rtsb()->nonceText, rtsb()->nonceId, false ) ) {
			return false;
		}

		// Check if request and product_id are valid.
		if ( empty( $_REQUEST ) || empty( $product_id ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Apply campaign discount to the prices if a campaign is available.
	 *
	 * @param array $prices The original prices (price, regular_price, sale_price).
	 * @param array $campaign The campaign data.
	 *
	 * @return array
	 */
	public static function apply_campaign_discount( $prices, $campaign ) {
		$regular_price = floatval( $prices['regular_price'] );
		$price         = floatval( $prices['price'] );
		$sale_price    = floatval( $prices['sale_price'] );

		if ( ! empty( $campaign ) ) {
			$discount_amount = $campaign['discount_amount'] ?? 0;

			if ( ! empty( $discount_amount ) ) {
				$discount_price = floatval( CountdownFns::campaign_discount_price( $regular_price, $campaign ) );
				$sale_price     = $regular_price - $discount_price;
				$price          = $sale_price;
			}
		} else {
			$price = ! $sale_price ? $regular_price : $sale_price;
		}

		return compact( 'price', 'regular_price', 'sale_price' );
	}

	/**
	 * Apply the pre-order discount to the prices if a product is in pre-order.
	 *
	 * @param array  $prices The original prices (price, regular_price, sale_price).
	 * @param object $product The product data.
	 *
	 * @return array
	 */
	public static function apply_preorder_price( $prices, $product ) {
		$regular_price = floatval( $prices['regular_price'] );
		$sale_price    = floatval( $prices['sale_price'] );
		$product_id    = $product->get_id();

		if ( ! empty( $regular_price ) ) {
			$sale_price = PreOrderProduct::calculate_preorder_price( $product_id, $regular_price );
		}

		$price = $sale_price;

		if ( ! empty( $prices['rtsb_partial_pay_amount'] ) ) {
			$price = floatval( $prices['price'] );
		}

		return compact( 'price', 'regular_price', 'sale_price' );
	}

	/**
	 * Apply the pre-order discount to the prices if a product is in pre-order.
	 *
	 * @param array  $prices The original prices (price, regular_price, sale_price).
	 * @param object $product The product data.
	 * @param array  $variation_data The variation data.
	 *
	 * @return array
	 */
	public static function apply_preorder_price_with_variations( $prices, $product, $variation_data = [] ) {
		$regular_price     = floatval( $prices['regular_price'] );
		$sale_price        = floatval( $prices['sale_price'] );
		$product_id        = $product->get_id();
		$preorder_price    = null;
		$lowest_sale_price = null;

		if ( $product->is_type( 'variable' ) ) {
			foreach ( $product->get_available_variations() as $variation ) {
				$variation_id = $variation['variation_id'];

				$variation_regular_price = floatval( get_post_meta( $variation_id, '_regular_price', true ) );
				$variation_sale_price    = PreOrderProduct::calculate_preorder_price( $variation_id, $variation_regular_price );

				if ( is_null( $lowest_sale_price ) || $variation_sale_price < $lowest_sale_price ) {
					$lowest_sale_price = $variation_sale_price;
				}

				if ( ! empty( $variation_data ) ) {
					$stock_status = get_post_meta( $variation_id, '_stock_status', true );

					if ( 'preorder' === $stock_status ) {
						$variation_data[ $variation_id ]['sale_price'] = floatval( $variation_sale_price );
					}
				}
			}

			$preorder_price = $lowest_sale_price;
		} else {
			if ( ! empty( $regular_price ) ) {
				$sale_price     = PreOrderProduct::calculate_preorder_price( $product_id, $regular_price );
				$preorder_price = $sale_price;
			}
		}

		$price = ! empty( $sale_price ) ? $sale_price : $regular_price;

		return empty( $variation_data ) ? compact( 'price', 'regular_price', 'sale_price', 'preorder_price' ) : compact( 'price', 'regular_price', 'sale_price', 'preorder_price', 'variation_data' );
	}
}
