<?php
/**
 * AddonProcessor Model Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\AddOns\Frontend;

use RadiusTheme\SBPRO\Modules\AddOns\AddOnsFns;

defined( 'ABSPATH' ) || exit();

/**
 * AddonProcessor Model Class.
 */
class AddonProcessor {
	/**
	 * Product ID for which add-ons are being processed.
	 *
	 * @var int
	 */
	private $product_id;

	/**
	 * Array of add-ons associated with the product.
	 *
	 * @var array
	 */
	private $product_addons;

	/**
	 * Cart item data to which add-ons will be added.
	 *
	 * @var array
	 */
	private $cart_item_data;

	/**
	 * Class Constructor.
	 *
	 * @param int   $product_id The ID of the product.
	 * @param array $cart_item_data The cart item data.
	 *
	 * @return void
	 */
	public function __construct( $product_id, $cart_item_data ) {
		$this->product_id     = $product_id;
		$this->cart_item_data = $cart_item_data;
		$this->product_addons = AddOnsFns::get_addon_groups( $product_id );

		if ( empty( $this->cart_item_data['rtsb_product_addons'] ) ) {
			$this->cart_item_data['rtsb_product_addons'] = [];
		}
	}

	/**
	 * Processes the add-ons for the given product and cart item data.
	 *
	 * @return array
	 */
	public function process() {
		foreach ( $this->product_addons as $group_index => $addon_group ) {
			$addon_data = $this->process_addon_group( $addon_group, $group_index );

			if ( ! empty( $addon_data['fields'] ) ) {
				$this->cart_item_data['rtsb_product_addons'][] = $addon_data;
			}
		}

		return $this->cart_item_data;
	}

	/**
	 * Processes a group of add-ons.
	 *
	 * @param array $addon_group The add-on group data.
	 * @param int   $group_index The index of the add-on group.
	 *
	 * @return array
	 */
	private function process_addon_group( $addon_group, $group_index ) {
		$addon_data = [
			'group_name' => $addon_group['name'],
			'fields'     => [],
		];

		foreach ( $addon_group['inputs'] as $input_index => $input ) {
			$field_name = "rtsb_addon_field_{$this->product_id}_{$group_index}_{$input_index}";
			$field_data = $this->process_input_field( $input, $field_name );

			if ( ! empty( $field_data ) ) {
				$addon_data['fields'][] = $field_data;
			}
		}

		return $addon_data;
	}

	/**
	 * Processes a single input field.
	 *
	 * @param array  $input The input field data.
	 * @param string $field_name The name of the field.
	 *
	 * @return array
	 */
	private function process_input_field( $input, $field_name ) {
		$field_data = [
			'field_id' => $field_name,
			'type'     => $input['type'],
			'label'    => $input['label'],
			'value'    => [],
			'price'    => [],
		];

		return $this->set_field_data( $field_data, $input, $field_name );
	}

	/**
	 * Sets the field data based on the input and field name.
	 *
	 * @param array  $field_data The field data to be set.
	 * @param array  $input The input data.
	 * @param string $field_name The name of the field.
	 *
	 * @return array
	 */
	private function set_field_data( $field_data, $input, $field_name ) {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( ! empty( $_REQUEST[ $field_name ] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$field_data['value'] = sanitize_text_field( wp_unslash( $_REQUEST[ $field_name ] ) );
			$field_data['price'] = $input['price'];
		}

		$has_options = ! ( 'text' === $input['type'] || 'textarea' === $input['type'] || 'number' === $input['type'] || 'image_upload' === $input['type'] );

		if ( $has_options && ! empty( $input['options'] ) ) {
			$field_data = $this->process_input_options( $field_data, $input, $field_name );
		}

		return $field_data;
	}

	/**
	 * Processes input options for a field.
	 *
	 * @param array  $field_data The field data to be updated.
	 * @param array  $input The input data containing options.
	 * @param string $field_name The name of the field.
	 *
	 * @return array
	 */
	private function process_input_options( $field_data, $input, $field_name ) {
		foreach ( $input['options'] as $option ) {
			$option_field_name = $this->get_option_field_name( $input, $field_name, $option );

			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( ! empty( $_REQUEST[ $option_field_name ] ) ) {
				// phpcs:ignore WordPress.Security.NonceVerification.Recommended
				if ( is_array( $_REQUEST[ $option_field_name ] ) ) {
					// phpcs:ignore WordPress.Security.NonceVerification.Recommended
					$option_values = array_map( 'sanitize_text_field', wp_unslash( $_REQUEST[ $option_field_name ] ) );
					$field_data    = $this->process_multiple_options( $field_data, $option_values, $input, $option );
				} else {
					// phpcs:ignore WordPress.Security.NonceVerification.Recommended
					$option_value = sanitize_text_field( wp_unslash( $_REQUEST[ $option_field_name ] ) );
					$field_data   = $this->process_single_option( $field_data, $option_value, $option );
				}
			}
		}

		return $field_data;
	}

	/**
	 * Gets the name of the option field.
	 *
	 * @param array  $input The input data.
	 * @param string $field_name The base field name.
	 * @param array  $option The option data.
	 *
	 * @return string
	 */
	private function get_option_field_name( $input, $field_name, $option ) {
		$multiple = 'on' === ( $input['multiple'] ?? 'off' );
		return $multiple && 'select' !== $input['type'] ? "{$field_name}_{$option['id']}" : $field_name;
	}

	/**
	 * Processes multiple options for a field.
	 *
	 * @param array $field_data The field data to be updated.
	 * @param array $option_values The array of selected option values.
	 * @param array $input The input data containing options.
	 * @param array $option The option data.
	 *
	 * @return array
	 */
	private function process_multiple_options( $field_data, $option_values, $input, $option ) {
		if ( count( $option_values ) === 1 ) {
			$single_value = current( $option_values );
			if ( $single_value === $option['name'] ) {
				if ( ! is_array( $field_data['value'] ) ) {
					$field_data['value'] = [];
				}

				if ( ! is_array( $field_data['price'] ) ) {
					$field_data['price'] = [];
				}

				$field_data['value'][] = $single_value;
				$field_data['price'][] = floatval( $option['price'] ?? 0 );
			}
		} elseif ( 'select' === $input['type'] ) {
			foreach ( $option_values as $value ) {
				if ( $value === $option['name'] ) {
					if ( ! is_array( $field_data['price'] ) ) {
						$field_data['price'] = [];
					}

					$field_data['value']   = $option_values;
					$field_data['price'][] = floatval( $option['price'] ?? 0 );
				}
			}
		}

		return $field_data;
	}

	/**
	 * Processes a single option for a field.
	 *
	 * @param array  $field_data The field data to be updated.
	 * @param string $option_value The selected option value.
	 * @param array  $option The option data.
	 *
	 * @return array
	 */
	private function process_single_option( $field_data, $option_value, $option ) {
		if ( $option_value === $option['name'] ) {
			$field_data['value'] = $option_value;
			$field_data['price'] = floatval( $option['price'] ?? 0 );
		}

		return $field_data;
	}
}
