<?php
/**
 * Back-order Module Admin Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\BackOrder;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Helpers\AdminInputsFns;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Module Admin Class.
 */
class BackOrderAdmin {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Global settings.
	 *
	 * @var array
	 */
	private $data = [];

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		$this->get_data();
		$this->product_meta_boxes();
	}

	/**
	 * Get global backorder settings.
	 *
	 * @return void
	 */
	private function get_data() {
		$limit = absint( Fns::get_option( 'modules', 'back_order', 'back_order_max_allowed_order' ) );
		$date  = Fns::get_option( 'modules', 'back_order', 'back_order_avail_date' );

		$this->data = [
			// Global Settings.
			'global_limit'      => $limit,
			'global_avail_date' => $date,

			// Placeholders.
			'limit_placeholder' => $limit ? sprintf(
				'%s (%s)',
				esc_html__( 'Override global limit', 'shopbuilder-pro' ),
				'Set as ' . esc_html( $limit )
			) : esc_html__( 'Override global limit (no limit is set)', 'shopbuilder-pro' ),
			'availability_date' => ! empty( $date ) ? sprintf(
				'%s (%s)',
				esc_html__( 'Entering date here will override global availability date', 'shopbuilder-pro' ),
				'Set as ' . esc_html( date_i18n( get_option( 'date_format' ), strtotime( $date ) ) )
			) : '',
		];
	}

	/**
	 * Adds meta-boxes for back order settings in WooCommerce.
	 *
	 * @return void
	 */
	public function product_meta_boxes() {
		$this
			->simple_products_support()
			->variable_products_support();
	}

	/**
	 * Sets up support for single products in the back order settings.
	 *
	 * @return BackOrderAdmin
	 */
	public function simple_products_support() {
		add_action( 'woocommerce_product_options_stock_status', [ $this, 'add_simple_meta_box' ] );
		add_action( 'woocommerce_process_product_meta', [ $this, 'save_simple_meta_box' ] );

		return $this;
	}

	/**
	 * Sets up support for variable products in the back order settings.
	 *
	 * @return void
	 */
	public function variable_products_support() {
		add_action( 'woocommerce_product_after_variable_attributes', [ $this, 'add_variation_meta_box' ], 10, 3 );
		add_action( 'woocommerce_save_product_variation', [ $this, 'save_variation_meta_box' ], 10, 2 );
	}

	/**
	 * Adds back order meta-fields to the product settings.
	 *
	 * @return void
	 */
	public function add_simple_meta_box() {
		$product_id = get_the_id();
		$product    = wc_get_product( $product_id );
		$status     = 'onbackorder' !== $product->get_stock_status() ? 'hidden' : '';

		if ( 'variable' === $product->get_type() ) {
			$status = 'hidden';
		}

		$backorder_limit        = get_post_meta( $product_id, '_rtsb_max_backorder_allowed', true );
		$backorder_availability = get_post_meta( $product_id, '_rtsb_back_order_avail_date', true );

		echo '<div id="rtsb_back_order_options_group" class="options_group ' . esc_attr( $status ) . '">';
		echo '<div class="rtsb-group-heading"><h2>' . esc_html__( 'Back-Order Settings', 'shopbuilder-pro' ) . '</h2></div>';

		// Text input.
		woocommerce_wp_text_input(
			[
				'id'                => '_rtsb_max_backorder_allowed',
				'label'             => esc_html__( 'Backorder Limit', 'shopbuilder-pro' ),
				'desc_tip'          => true,
				'description'       => esc_html__( 'Set the maximum number of items a customer can place on back-order. Leave empty to apply global limit. Enter -1 for no limit.', 'shopbuilder-pro' ),
				'placeholder'       => $this->data['limit_placeholder'],
				'type'              => 'number',
				'value'             => $backorder_limit,
				'custom_attributes' => [
					'step' => 'any',
				],
			]
		);

		// Date input.
		AdminInputsFns::generate_date_picker_input(
			$product_id,
			[
				'id'          => '_rtsb_back_order_avail_date',
				'label'       => esc_html__( 'Availability Date', 'shopbuilder-pro' ),
				'desc_tip'    => true,
				'value'       => $backorder_availability,
				'description' => esc_html__( 'Enter the estimated date when back-ordered items will be available for shipment. ', 'shopbuilder-pro' ) . $this->data['availability_date'],
			]
		);

		echo '</div>';
	}

	/**
	 * Saves meta-box data for the specified post-ID.
	 *
	 * @param int $post_id The ID of the post being saved.
	 *
	 * @return void
	 */
	public function save_simple_meta_box( $post_id ) {
		// Check if the nonce is set and valid.
		if (
			! isset( $_REQUEST['_wpnonce'] ) &&
			! isset( $_REQUEST['post_ID'] ) &&
			! wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['_wpnonce'] ) ), 'update-post_' . sanitize_text_field( wp_unslash( $_REQUEST['post_ID'] ) ) ) ) {
			return;
		}

		// Get and sanitize backorder limit and availability date from request.
		$backorder_limit = ! empty( $_REQUEST['_rtsb_max_backorder_allowed'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_max_backorder_allowed'] ) ) : null;

		if ( ! empty( $backorder_limit ) ) {
			$backorder_limit = strlen( trim( $backorder_limit ) ) <= 0 ? '' : intval( $backorder_limit );
		}

		$backorder_availability = ! empty( $_REQUEST['_rtsb_back_order_avail_date'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_back_order_avail_date'] ) ) : null;

		// Update post-meta with backorder limit and availability date.
		update_post_meta( $post_id, '_rtsb_max_backorder_allowed', $backorder_limit );
		update_post_meta( $post_id, '_rtsb_back_order_avail_date', $backorder_availability );
	}

	/**
	 * Add variation meta-box with backorder limit and availability date inputs.
	 *
	 * @param int    $loop           The loop index of the variation.
	 * @param array  $variation_data The variation data.
	 * @param object $variation      The variation object.
	 *
	 * @return void
	 */
	public function add_variation_meta_box( $loop, $variation_data, $variation ) {
		$backorders             = get_post_meta( $variation->ID, '_stock_status', true );
		$backorder_limit        = get_post_meta( $variation->ID, '_rtsb_max_backorder_allowed', true );
		$backorder_availability = get_post_meta( $variation->ID, '_rtsb_back_order_avail_date', true );
		$status                 = 'onbackorder' !== $backorders ? 'hidden' : '';

		echo sprintf( '<div id="rtsb_back_order_variation_options_group_%s" class="%s">', esc_attr( $loop ), esc_attr( $status ) );
		echo '<div class="rtsb-group-heading"><h2>' . esc_html__( 'Back-Order Settings', 'shopbuilder-pro' ) . '</h2></div>';

		// Text input.
		woocommerce_wp_text_input(
			[
				'id'                => '_rtsb_max_backorder_allowed' . $loop,
				'name'              => '_rtsb_max_backorder_allowed' . "[$loop]",
				'label'             => esc_html__( 'Backorder Limit', 'shopbuilder-pro' ),
				'desc_tip'          => true,
				'description'       => esc_html__( 'Set the maximum number of items a customer can place on back-order. Leave empty to apply global limit. Enter -1 for no limit.', 'shopbuilder-pro' ),
				'placeholder'       => $this->data['limit_placeholder'],
				'type'              => 'number',
				'value'             => $backorder_limit,
				'wrapper_class'     => 'form-row form-row-first',
				'custom_attributes' => [
					'step' => 'any',
				],
			]
		);

		// Date input.
		AdminInputsFns::generate_date_picker_input(
			$variation->ID,
			[
				'id'            => '_rtsb_back_order_avail_date' . $loop,
				'name'          => '_rtsb_back_order_avail_date' . "[$loop]",
				'label'         => esc_html__( 'Availability Date', 'shopbuilder-pro' ),
				'desc_tip'      => true,
				'value'         => $backorder_availability,
				'description'   => esc_html__( 'Enter the estimated date when back-ordered items will be available for shipment. ', 'shopbuilder-pro' ) . $this->data['availability_date'],
				'wrapper_class' => 'form-row form-row-last',
			]
		);

		echo '</div>';
	}

	/**
	 * Save variation meta-box data.
	 *
	 * @param int $variation_id The ID of the variation being saved.
	 * @param int $loop The loop index of the variation.
	 *
	 * @return void
	 */
	public function save_variation_meta_box( $variation_id, $loop ) {
		// Check security.
		check_ajax_referer( 'save-variations', 'security' );

		// Get and sanitize backorder limit and availability date from request.
		$backorder_limit = ! empty( $_REQUEST['_rtsb_max_backorder_allowed'][ $loop ] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_max_backorder_allowed'][ $loop ] ) ) : null;

		if ( ! empty( $backorder_limit ) ) {
			$backorder_limit = strlen( trim( $backorder_limit ) ) <= 0 ? '' : intval( $backorder_limit );
		}

		$backorder_availability = ! empty( $_REQUEST['_rtsb_back_order_avail_date'][ $loop ] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_back_order_avail_date'][ $loop ] ) ) : null;

		// Update post-meta with backorder limit and availability date.
		update_post_meta( $variation_id, '_rtsb_max_backorder_allowed', $backorder_limit );
		update_post_meta( $variation_id, '_rtsb_back_order_avail_date', $backorder_availability );
	}
}
