<?php
/**
 * Back-order Functions Class.
 *
 * @package Rse\SB
 */

namespace RadiusTheme\SBPRO\Modules\BackOrder;

use RadiusTheme\SB\Helpers\Fns;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Functions Class.
 */
class BackOrderFns {
	/**
	 * Returns an array of settings fields.
	 *
	 * @return array
	 */
	public static function settings_field() {
		return [
			'back_order_max_allowed_order' => [
				'id'    => 'back_order_max_allowed_order',
				'type'  => 'number',
				'value' => 3,
				'size'  => 'small',
				'min'   => 0,
				'max'   => 999,
				'label' => __( 'Backorder Limit', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Set the maximum number of items a customer can place on back-order. Leave empty for no limit.', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'back_order_avail_date'        => [
				'id'           => 'back_order_avail_date',
				'label'        => esc_html__( 'Expected Availability Date', 'shopbuilder-pro' ),
				'type'         => 'datetime',
				'format'       => 'YYYY-MM-DD',
				'showtime'     => false,
				'disableDates' => true,
				'help'         => esc_html__( 'Enter the estimated date when back-ordered items will be available.', 'shopbuilder-pro' ),
				'tab'          => 'general',
			],
			'back_order_avail_msg'         => [
				'id'    => 'back_order_avail_msg',
				'label' => __( 'General Availability Message <br />for Back-Order', 'shopbuilder-pro' ),
				'type'  => 'textarea',
				'value' => esc_html__( 'On Backorder: Expected availability by {availability_date}.', 'shopbuilder-pro' ),
				'help'  => __( 'Customize the message displayed for back-ordered items. Use the <code>{availability_date}</code> placeholder to dynamically show the specified availability date.', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'back_order_text_style_intro'  => [
				'id'    => 'back_order_text_style_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Back-Order Text', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'back_order_text_color'        => [
				'id'    => 'back_order_text_color',
				'label' => esc_html__( 'Back-Order Text Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose color for Back-Order Text', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'back_order_bg_color'          => [
				'id'    => 'back_order_bg_color',
				'label' => esc_html__( 'Back-Order Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose background color for Back-Order Text', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'back_order_font_size'         => [
				'id'    => 'back_order_font_size',
				'type'  => 'text',
				'label' => esc_html__( 'Font Size', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Font Size. Ex. 16px', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'back_order_font_weight'       => [
				'id'      => 'back_order_font_weight',
				'type'    => 'select',
				'options' => [
					100 => esc_html__( '100 (Thin)', 'shopbuilder-pro' ),
					200 => esc_html__( '200 (Extra Light)', 'shopbuilder-pro' ),
					300 => esc_html__( '300 (Light)', 'shopbuilder-pro' ),
					400 => esc_html__( '400 (Normal)', 'shopbuilder-pro' ),
					500 => esc_html__( '500 (Medium)', 'shopbuilder-pro' ),
					600 => esc_html__( '600 (Semi Bold)', 'shopbuilder-pro' ),
					700 => esc_html__( '700 (Bold)', 'shopbuilder-pro' ),
					800 => esc_html__( '800 (Extra Bold)', 'shopbuilder-pro' ),
					900 => esc_html__( '900 (Black)', 'shopbuilder-pro' ),
				],
				'value'   => 400,
				'label'   => esc_html__( 'Font Weight', 'shopbuilder-pro' ),
				'help'    => esc_html__( 'Choose Font Weight.', 'shopbuilder-pro' ),
				'tab'     => 'styles',
			],
			'back_order_radius'            => [
				'id'    => 'back_order_radius',
				'type'  => 'text',
				'label' => esc_html__( 'Border Radius', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Border Radius. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'back_order_padding'           => [
				'id'    => 'back_order_padding',
				'type'  => 'text',
				'label' => esc_html__( 'Padding', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Padding. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'back_order_margin'            => [
				'id'    => 'back_order_margin',
				'type'  => 'text',
				'label' => esc_html__( 'Margin', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Margin. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'back_order_badge_style_intro' => [
				'id'    => 'back_order_badge_style_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Back-Order Badge', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'back_order_badge_text_color'  => [
				'id'    => 'back_order_badge_text_color',
				'label' => esc_html__( 'Back-Order Badge Text Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose color for Back-Order badge text', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'back_order_badge_bg_color'    => [
				'id'    => 'back_order_badge_bg_color',
				'label' => esc_html__( 'Back-Order Badge Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose background color for Back-Order badge text', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
		];
	}

	/**
	 * Get back order data.
	 *
	 * @return array
	 */
	public static function get_backorder_data() {
		$limit   = absint( Fns::get_option( 'modules', 'back_order', 'back_order_max_allowed_order' ) );
		$date    = Fns::get_option( 'modules', 'back_order', 'back_order_avail_date' );
		$message = Fns::get_option( 'modules', 'back_order', 'back_order_avail_msg' );

		return [
			'limit'   => ! empty( $limit ) ? $limit : 0,
			'date'    => $date,
			'message' => $message,
		];
	}

	/**
	 * Get back order availability date for a specific product.
	 *
	 * @param int $product_id Product ID.
	 *
	 * @return string
	 */
	public static function get_backorder_date( $product_id ) {
		$global_avail_date = self::get_backorder_data()['date'];
		$local_avail_date  = get_post_meta( $product_id, '_rtsb_back_order_avail_date', true );

		return ! empty( $local_avail_date ) ? $local_avail_date : $global_avail_date;
	}

	/**
	 * Get formatted date based on WordPress settings.
	 *
	 * @param string $date Date to format.
	 *
	 * @return string
	 */
	public static function get_formatted_date( $date ) {
		if ( empty( $date ) ) {
			return '';
		}

		return date_i18n( get_option( 'date_format' ), strtotime( $date ) );
	}

	/**
	 * Get back order limit for a specific product.
	 *
	 * @param int $product_id Product ID.
	 *
	 * @return int
	 */
	public static function get_backorder_limit( $product_id ) {
		$global_limit = self::get_backorder_data()['limit'];
		$local_limit  = get_post_meta( $product_id, '_rtsb_max_backorder_allowed', true );

		return ! empty( $local_limit ) ? $local_limit : $global_limit;
	}

	/**
	 * Get a back order message with availability date.
	 *
	 * @param string $date Availability date.
	 *
	 * @return string
	 */
	public static function get_backorder_message( $date ) {
		$message = self::get_backorder_data()['message'];

		if ( empty( $date ) ) {
			$date = esc_html__( 'shortly', 'shopbuilder-pro' );
		}

		return str_replace( '{availability_date}', $date, $message );
	}

	/**
	 * Check if back order is allowed for a specific product.
	 *
	 * @param int $id Product ID.
	 * @param int $stock Product stock.
	 * @param int $variation_id Product variation ID.
	 *
	 * @return bool
	 */
	public static function is_backorder_allowed( $id, $stock, $variation_id = '' ) {
		if ( ! empty( $variation_id ) ) {
			$id = $variation_id;
		}

		$limit = self::get_backorder_limit( $id );

		if ( empty( $limit ) || $limit < 0 ) {
			return true;
		} else {
			return ( $stock + $limit ) > 0;
		}
	}

	/**
	 * Check if back order stock for a specific product.
	 *
	 * @param int $id Product ID.
	 * @param int $stock Product stock.
	 * @param int $variation_id Product variation ID.
	 *
	 * @return int|null
	 */
	public static function check_backorder_stock( $id, $stock, $variation_id = '' ) {
		if ( ! empty( $variation_id ) ) {
			$id = $variation_id;
		}

		$limit = (int) self::get_backorder_limit( $id );
		$stock = (int) $stock;

		if ( ! empty( $limit ) || $limit > 0 ) {
			return $stock + $limit;
		}

		return null;
	}

	/**
	 * Display a notice related to the product's back-order status.
	 *
	 * @param object $product The WooCommerce product object.
	 * @param int    $amount  The maximum amount available for back-order.
	 *
	 * @return void
	 */
	public static function backorder_product_notice( $product, $amount ) {
		$product_title = $product->get_title();

		if ( $amount <= 0 ) {
			/* translators: %s: product title */
			$notice  = esc_html__( 'Sorry, "%s" has reached its back-order limit.', 'shopbuilder-pro' );
			$message = sprintf( $notice, $product_title );
		} else {
			/* translators: %1$s: product title, %2$s: product amount */
			$notice  = esc_html__( '"%1$s" is available for back-order. You can add a maximum of %2$d unit to your cart.', 'shopbuilder-pro' );
			$message = sprintf( $notice, $product_title, intval( $amount ) );
		}

		wc_add_notice( '<strong>' . $message . '</strong>', 'error' );
	}
}
