<?php
/**
 * Back-order Frontend Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\BackOrder;

use WC_Order;
use WC_Product;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\PreOrder\PreOrderFns;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Frontend Class.
 */
class BackOrderFrontEnd {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Options.
	 *
	 * @var array|mixed
	 */
	private array $options;

	/**
	 * Asset Handle
	 *
	 * @var string
	 */
	private $handle = 'rtsb-backorder';

	/**
	 * Class Constructor.
	 */
	private function __construct() {
		/**
		 * Get the options.
		 */
		$this->options = Fns::get_options( 'modules', 'back_order' );

		/**
		 * Product page modifications.
		 */
		add_filter( 'woocommerce_get_availability_text', [ $this, 'backorder_text' ], 10, 2 );
		add_filter( 'woocommerce_get_availability_class', [ $this, 'backorder_class' ], 10, 2 );
		add_filter( 'rtsb/elementor/product_stock_icons', [ $this, 'backorder_icon' ], 10, 3 );
		add_filter( 'woocommerce_check_cart_items', [ $this, 'backorder_limit_notice' ], 99, 4 );

		/**
		 * Cart page modifications.
		 */
		add_filter( 'woocommerce_add_to_cart_validation', [ $this, 'cart_validation' ], 10, 5 );
		add_filter( 'woocommerce_cart_item_backorder_notification', '__return_false' );
		add_action( 'woocommerce_after_cart_item_name', [ $this, 'backorder_cart_message' ] );

		/**
		 * Checkout page modifications.
		add_filter( 'woocommerce_cart_item_name', [ $this, 'backorder_checkout_message' ], 10, 2 );
		 */

		/**
		 * Thank you page modifications.
		 */
		add_action( 'woocommerce_thankyou', [ $this, 'change_order_status' ] );
		add_action( 'rtsb_woocommerce_thankyou', [ $this, 'change_order_status' ] );

		/**
		 * Order meta.
		 */
		add_action( 'woocommerce_checkout_create_order_line_item', [ $this, 'add_order_meta' ], 99, 4 );
		add_filter( 'woocommerce_order_item_get_formatted_meta_data', [ $this, 'get_order_meta' ], 10, 4 );

		/**
		 * Dynamic styles.
		 */
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_public_scripts' ], 99 );
	}

	/**
	 * Modify the availability text based on backorder status.
	 *
	 * @param string $text    The original text.
	 * @param object $product Product object.
	 *
	 * @return string
	 */
	public function backorder_text( $text, $product ) {
		$id           = $this->get_product_id( $product );
		$stock        = $product->get_stock_quantity();
		$availability = null;

		if ( true === $product->managing_stock() || 'parent' === $product->managing_stock() ) {
			if ( $product->is_on_backorder( 1 ) ) {
				if ( BackOrderFns::is_backorder_allowed( $id, $stock ) ) {
					if ( $product->backorders_require_notification() ) {
						$availability = $this->get_backorder_availability( $id );
					}
				} else {
					$text = esc_html__( 'Out of stock.', 'shopbuilder-pro' );
				}
			}
		} else {
			if ( $product->is_on_backorder( 1 ) ) {
				if ( BackOrderFns::is_backorder_allowed( $id, $stock ) ) {
					$availability = $this->get_backorder_availability( $id );
				}
			}
		}

		if ( ! empty( $availability ) ) {
			$text = esc_html( BackOrderFns::get_backorder_message( $availability ) );
		}

		return ! empty( $text ) ? '<span class="rtsb-stock-text">' . $text . '</span>' : '';
	}

	/**
	 * Adjust the product class based on backorder status.
	 *
	 * @param string $class   The original class.
	 * @param object $product Product object.
	 *
	 * @return string
	 */
	public function backorder_class( $class, $product ) {
		$new_class = $this->adjust_backorder_display(
			$product,
			[
				'backorder'    => 'available-on-backorder',
				'out_of_stock' => 'out-of-stock',
			]
		);

		return ! empty( $new_class ) ? $new_class : $class;
	}

	/**
	 * Adjust the product icon based on backorder status.
	 *
	 * @param string $icon        The original icon.
	 * @param array  $controllers The controllers for managing icons.
	 * @param object $product     Product object.
	 *
	 * @return string
	 */
	public function backorder_icon( $icon, $controllers, $product ) {
		$new_icon = $this->adjust_backorder_display(
			$product,
			[
				'backorder'    => Fns::icons_manager( $controllers['stock_backorder_icon'] ),
				'out_of_stock' => Fns::icons_manager( $controllers['out_of_stock_icon'] ),
			]
		);

		return ! empty( $new_icon ) ? $new_icon : $icon;
	}

	/**
	 * Display back-order limit notice and adjust cart item quantity if necessary.
	 *
	 * @return void
	 */
	public function backorder_limit_notice() {
		// Get cart contents.
		$cart = WC()->cart->get_cart();

		foreach ( $cart as $cart_item_key => $cart_item ) {
			// Get product data.
			$product_id   = $cart_item['product_id'];
			$variation_id = $cart_item['variation_id'];
			$id           = $variation_id ?: $product_id;
			$quantity     = $cart_item['quantity'];
			$product      = $cart_item['data'];

			if ( ! $product ) {
				continue;
			}

			$stock      = $product->get_stock_quantity();
			$amount     = 'parent' === $product->managing_stock() ? BackOrderFns::check_backorder_stock( $product_id, $stock ) : BackOrderFns::check_backorder_stock( $product_id, $stock, $variation_id );
			$validation = BackOrderFns::is_backorder_allowed( $id, $stock );

			if ( empty( $amount ) ) {
				continue;
			}

			// Check if the cart quantity exceeds the available back-order amount.
			if ( $quantity > $amount ) {
				$validation = $this->validate_backorder( $validation, $product, $product_id, $stock, $quantity, $amount );
			}

			if ( ! $validation ) {
				WC()->cart->set_quantity( $cart_item_key, $amount );

				// Display error notice.
				BackOrderFns::backorder_product_notice( $product, $amount );
			}
		}
	}

	/**
	 * Perform cart validation for back-ordered products.
	 *
	 * @param bool  $passed_validation Whether the validation has passed.
	 * @param int   $product_id        The ID of the product being added to the cart.
	 * @param int   $quantity          The quantity of the product being added to the cart.
	 * @param int   $variation_id      The ID of the product variation being added to the cart.
	 * @param array $variation         The variation attributes of the product being added to the cart.
	 *
	 * @return bool
	 */
	public function cart_validation( $passed_validation, $product_id, $quantity, $variation_id = '', $variation = [] ) {
		$id      = $variation_id ?: $product_id;
		$product = wc_get_product( $id );

		if ( ! $product ) {
			return $passed_validation;
		}

		if ( ! $product->is_on_backorder( 1 ) ) {
			return $passed_validation;
		}

		$stock  = $product->get_stock_quantity();
		$amount = 'parent' === $product->managing_stock() ? BackOrderFns::check_backorder_stock( $product_id, $stock ) : BackOrderFns::check_backorder_stock( $product_id, $stock, $variation_id );

		if ( true === $product->managing_stock() || 'parent' === $product->managing_stock() ) {
			if ( empty( $amount ) ) {
				BackOrderFns::backorder_product_notice( $product, $amount );

				return false;
			}
		} else {
			if ( empty( $amount ) || $amount < 0 ) {
				return true;
			}
		}

		$cart_item_key     = WC()->cart->find_product_in_cart( WC()->cart->generate_cart_id( $product_id, $variation_id, $variation ) );
		$passed_validation = BackOrderFns::is_backorder_allowed( $id, $stock );

		if ( $quantity > $amount ) {
			$passed_validation = false;
		}

		if ( $cart_item_key ) {
			$cart_item = WC()->cart->get_cart_item( $cart_item_key );

			$quantity = $cart_item['quantity'] + $quantity;

			// Check if the cart quantity exceeds the available back-order amount.
			if ( $quantity >= $amount ) {
				$passed_validation = $this->validate_backorder( $passed_validation, $product, $product_id, $stock, $quantity, $amount );
			}
		}

		if ( ! $passed_validation ) {
			BackOrderFns::backorder_product_notice( $product, $amount );

			$passed_validation = false;
		}

		return $passed_validation;
	}

	/**
	 * Back-order cart message.
	 *
	 * @param array $cart_item Cart Item.
	 *
	 * @return void
	 */
	public function backorder_cart_message( $cart_item ) {
		if ( ! is_cart() ) {
			return;
		}
		$id      = $cart_item['variation_id'] ?: $cart_item['product_id'];
		$product = $cart_item['data'];
		$stock   = $product->get_stock_quantity();
		$message = '';

		if ( true === $product->managing_stock() || 'parent' === $product->managing_stock() ) {
			if ( $cart_item['quantity'] > $stock ) {
				if ( $product->backorders_require_notification() ) {
					$message = $this->get_backorder_availability( $id );
				}
			}
		} else {
			if ( $product->is_on_backorder( 1 ) ) {
				if ( $cart_item['quantity'] > $stock ) {
					$message = $this->get_backorder_availability( $id );
				}
			}
		}

		if ( ! empty( $message ) ) {
			echo '<span class="rtsb-back-order-text-badge">' . esc_html__( 'Back-Order', 'shopbuilder-pro' ) . '</span>';
			echo '<div class="stock available-on-backorder rtsb-backorder-message backorder_notification">
				<span class="wc-item-meta-label rtsb-stock-text">' . esc_html( BackOrderFns::get_backorder_message( $message ) ) . '
				</span>
			</div>';
		}
	}

	/**
	 * Back-order checkout message.
	 *
	 * @param string $name      Name.
	 * @param array  $cart_item Cart Item.
	 *
	 * @return string
	 */
	public function backorder_checkout_message( $name, $cart_item ) {
		if ( ! is_checkout() ) {
			return $name;
		}
		$product = $cart_item['data'];

		if ( ! $product->is_on_backorder( 1 ) ) {
			return $name;
		}

		$text = '<span class="rtsb-checkout-text"><span class="rtsb-back-order-text-badge">' . esc_html__( 'Back-Order', 'shopbuilder-pro' ) . '</span></span>';

		return $name . $text;
	}

	/**
	 * Change the order status to backordered if one or more items are on backorder.
	 *
	 * @param int $order_id The ID of the order to check and update.
	 *
	 * @return void
	 */
	public function change_order_status( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( ! $order || $order->has_status( 'wc-rtsb-backordered' ) ) {
			return;
		}

		if ( $this->has_pre_order_items( $order ) ) {
			return;
		}

		foreach ( $order->get_items() as $item ) {
			$product_id = $item->get_variation_id() ?: $item->get_product_id();
			$product    = wc_get_product( $product_id );

			if ( ! $product instanceof WC_Product ) {
				continue;
			}

			if ( $product->managing_stock() && $product->backorders_allowed() && $product->get_stock_quantity() >= 0 ) {
				continue;
			}

			if ( $product->is_on_backorder( 1 ) ) {
				$order->update_status( 'wc-rtsb-backordered', esc_html__( 'Order status changed to backordered because one or more items are on backorder.', 'shopbuilder-pro' ) );

				break;
			}
		}
	}

	/**
	 * Display backorder notification in the order item meta.
	 *
	 * @param int    $item_id   The ID of the item in the order.
	 * @param object $item      The order item object.
	 * @param object $order     The order object.
	 * @param bool   $plain_tex Whether the text should be plain or not.
	 *
	 * @return void
	 */
	public function order_item_meta_end( $item_id, $item, $order, $plain_tex ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed
		$product_data = $item->get_product();

		if ( $product_data->is_type( 'variation' ) ) {
			$product_id = $product_data->get_parent_id();
		} else {
			$product_id = $product_data->get_id();
		}

		if ( $product_data->managing_stock() && $product_data->backorders_allowed() && $product_data->get_stock_quantity() >= 0 ) {
			return;
		}

		echo '<div class="rtsb-wc-item-meta" style="display: inline-block;">';

		if ( $product_data->is_on_backorder() ) {
			echo '<div class="stock available-on-backorder rtsb-backorder-message backorder_notification">';
			echo '<p class="wc-item-meta-label rtsb-stock-text">' . esc_html( BackOrderFns::get_backorder_message( $this->get_backorder_availability( $product_id ) ) ) . '</p>';
			echo '</li>';
		}

		echo '</ul>';
	}

	/**
	 * Add order meta-data for a product if it is on backorder.
	 *
	 * @param mixed  $item The item being added to the order.
	 * @param string $cart_item_key The key of the item in the cart.
	 * @param array  $values The values of the item.
	 * @param object $order The order object.
	 *
	 * @return void
	 */
	public function add_order_meta( $item, $cart_item_key, $values, $order ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed
		$product = $values['data'];

		if ( $product->is_type( 'variation' ) ) {
			$product_id = $product->get_parent_id();
		} else {
			$product_id = $product->get_id();
		}

		if ( $product->managing_stock() && $product->backorders_allowed() && $product->get_stock_quantity() >= 0 ) {
			return;
		}

		if ( $product->is_on_backorder( $values['quantity'] ) ) {
			$backorder_availability = '<span class="rtsb-backorder-availability-text">' . BackOrderFns::get_formatted_date( BackOrderFns::get_backorder_date( $product_id ) ) . '</span>';
			$text                   = '<span class="rtsb-back-order-text-badge">' . esc_html__( 'Back-Order', 'shopbuilder-pro' ) . '</span>';

			$item->add_meta_data( 'rtsb_backorder_text', $text, true );
			$item->add_meta_data( 'rtsb_backorder_availability', $backorder_availability, true );
		}
	}

	/**
	 * Retrieves the order metadata and formats it.
	 *
	 * @param array $formatted_meta The formatted meta data.
	 * @param mixed $item The item to retrieve the meta-data for.
	 *
	 * @return array
	 */
	public function get_order_meta( $formatted_meta, $item ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.FoundAfterLastUsed
		foreach ( $formatted_meta as $key => $meta ) {
			if ( 'rtsb_backorder_availability' === $meta->key ) {
				$meta->display_key = '<span class="rtsb-order-meta order-status status-rtsb-backordered"><span class="wc-item-meta-label rtsb-stock-text">' . esc_html__( 'Availability Date', 'shopbuilder-pro' ) . '</span></span>';
			}

			if ( 'rtsb_backorder_text' === $meta->key ) {
				$meta->display_key = '<span class="rtsb-order-meta order-status status-rtsb-backordered"><span class="wc-item-meta-label rtsb-stock-text">' . esc_html__( 'Order Type', 'shopbuilder-pro' ) . '</span></span>';
			}
		}

		return $formatted_meta;
	}

	/**
	 * Dynamic styles.
	 *
	 * @return void
	 */
	public function enqueue_public_scripts() {
		$cache_key       = 'rtsb_backorder_dynamic_styles_' . md5( wp_json_encode( $this->options ) );
		$backorder       = '.available-on-backorder:not(.backorder_notification)';
		$backorder_text  = $backorder . ' .rtsb-stock-text';
		$backorder_text .= ', ' . $backorder . ' .rtsb-icon, ' . $backorder . ' svg';
		$backorder_text .= ', ' . $backorder . ' .rtsb-icon, ' . $backorder . ' svg';
		$backorder_badge = '.rtsb-back-order-text-badge';

		// phpcs:disable
		// Disabled phpcs for one-liner display.
		$css_properties = [
			'back_order_text_color'       => [ 'selector' => $backorder_text, 'property' => 'color' ],
			'back_order_bg_color'         => [ 'selector' => $backorder, 'property' => 'background-color' ],
			'back_order_font_size'        => [ 'selector' => $backorder_text, 'property' => 'font-size' ],
			'back_order_font_weight'      => [ 'selector' => $backorder_text, 'property' => 'font-weight' ],
			'back_order_radius'           => [ 'selector' => $backorder, 'property' => 'border-radius' ],
			'back_order_padding'          => [ 'selector' => $backorder, 'property' => 'padding' ],
			'back_order_margin'           => [ 'selector' => $backorder, 'property' => 'margin' ],
			'back_order_badge_text_color' => [ 'selector' => $backorder_badge, 'property' => 'color' ],
			'back_order_badge_bg_color'   => [ 'selector' => $backorder_badge, 'property' => 'background-color' ],
		];
		// phpcs:enable

		// Generate dynamic CSS.
		FnsPro::dynamic_styles( $this->options, $cache_key, $css_properties, Fns::optimized_handle( 'rtsb-frontend' ) );
	}

	/**
	 * Adjust the backorder display property based on product and context.
	 *
	 * @param object $product Product object.
	 * @param array  $context The context for adjusting the display property.
	 *
	 * @return string
	 */
	private function adjust_backorder_display( $product, $context = [] ) {
		$id    = $product->get_id();
		$stock = $product->get_stock_quantity();
		$prop  = '';

		if ( true === $product->managing_stock() || 'parent' === $product->managing_stock() ) {
			if ( $product->is_on_backorder( 1 ) ) {
				if ( BackOrderFns::is_backorder_allowed( $id, $stock ) ) {
					if ( $product->backorders_require_notification() ) {
						$prop = $context['backorder'];
					}
				} else {
					$prop = $context['out_of_stock'];
				}
			}
		} else {
			if ( $product->is_on_backorder( 1 ) ) {
				if ( BackOrderFns::is_backorder_allowed( $id, $stock ) ) {
					$prop = $context['backorder'];
				}
			}
		}

		return $prop;
	}

	/**
	 * Validates the backorder for a given product.
	 *
	 * @param bool   $validation The current validation status.
	 * @param object $product The product object.
	 * @param int    $product_id The ID of the product.
	 * @param int    $stock The stock quantity.
	 * @param int    $quantity The quantity to be added.
	 * @param int    $amount The maximum allowed quantity.
	 *
	 * @return bool
	 */
	private function validate_backorder( $validation, $product, $product_id, $stock, $quantity, $amount ) {
		$is_backorder_allowed = BackOrderFns::is_backorder_allowed( $product_id, $stock ) && ( $quantity > $amount );

		if ( $product->is_on_backorder( $quantity ) && $is_backorder_allowed ) {
			$validation = false;
		}

		return $validation;
	}

	/**
	 * Get the formatted backorder availability date for a given product.
	 *
	 * @param int $product_id The product ID.
	 *
	 * @return string
	 */
	private function get_backorder_availability( $product_id ) {
		if ( empty( $product_id ) ) {
			return '';
		}

		return BackOrderFns::get_formatted_date( BackOrderFns::get_backorder_date( $product_id ) );
	}

	/**
	 * Get the product ID.
	 *
	 * @param object $product Product object.
	 *
	 * @return int|null
	 */
	private function get_product_id( $product ) {
		if ( ! $product ) {
			return null;
		}

		return $product->is_type( 'variation' ) && ( $product->managing_stock() === 'parent' ) ? $product->get_parent_id() : $product->get_id();
	}

	/**
	 * Check if an order has any pre-order items.
	 *
	 * @param WC_Order $order The order to check.
	 *
	 * @return bool
	 */
	private function has_pre_order_items( $order ) {
		foreach ( $order->get_items() as $item ) {
			$product_id = $item->get_variation_id() ?: $item->get_product_id();
			$product    = wc_get_product( $product_id );

			if ( FnsPro::is_module_active( 'pre_order' ) && PreOrderFns::is_on_pre_order( $product ) ) {
				return true;
			}
		}

		return false;
	}
}
