<?php
/**
 * Customize My Account Module Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\BogoBxGy;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\AdminInputsFns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Module Class.
 */
class BogoGiftBxGyAdmin {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		add_filter( 'woocommerce_product_data_tabs', [ $this, 'add_product_add_ons_tab' ] );
		add_action( 'woocommerce_product_data_panels', [ $this, 'render_meta_boxes' ] );
		add_action( 'woocommerce_process_product_meta', [ $this, 'save_bogo_bxgy_discount_meta' ], 10, 2 );
	}
	/**
	 * Sanitize an array of strings.
	 *
	 * @param string $meta_key Meta key.
	 *
	 * @return array
	 */
	public function sanitize_array( $meta_key ) {
		return isset( $_REQUEST[ $meta_key ] ) && is_array( $_REQUEST[ $meta_key ] ) ? array_map( 'sanitize_text_field', wp_unslash( $_REQUEST[ $meta_key ] ) ) : []; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	}
	/**
	 * Saves discount meta fields for a product.
	 *
	 * This function handles saving discount enable/disable options
	 * when a product is updated in the admin panel.
	 *
	 * @param int      $post_id The ID of the product being saved.
	 * @param \WP_Post $post    The post object for the product.
	 * @return void
	 */
	public function save_bogo_bxgy_discount_meta( $post_id, $post ) {
		// Verify nonce.
		check_admin_referer( rtsb()->nonceText, rtsb()->nonceId );

		$disable_discount  = isset( $_REQUEST['_rtsb_disable_bogo_bxgy'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_disable_bogo_bxgy'] ) ) : null;
		$enable_discount   = isset( $_REQUEST['_rtsb_enable_bogo_bxgy'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_enable_bogo_bxgy'] ) ) : null;
		$bogo_multiapply   = isset( $_REQUEST['_rtsb_disable_bogo_multiapply'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_disable_bogo_multiapply'] ) ) : null;
		$bxgy_multiapply   = isset( $_REQUEST['_rtsb_disable_bxgy_multiapply'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_disable_bxgy_multiapply'] ) ) : null;
		$enable_date_range = isset( $_REQUEST['_rtsb_enable_date_range'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_enable_date_range'] ) ) : null;

		$discount_type       = isset( $_REQUEST['_rtsb_bogo_bxgy_type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_bogo_bxgy_type'] ) ) : null;
		$discount_value_type = isset( $_REQUEST['_rtsb_bogo_bxgy_value_type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_bogo_bxgy_value_type'] ) ) : null;
		$minimum_quantity    = isset( $_REQUEST['_rtsb_bxgy_minimum_required_quantity'] ) ? absint( wp_unslash( $_REQUEST['_rtsb_bxgy_minimum_required_quantity'] ) ) : null;
		$discounted_quantity = isset( $_REQUEST['_rtsb_bxgy_discounted_products_quantity'] ) ? absint( wp_unslash( $_REQUEST['_rtsb_bxgy_discounted_products_quantity'] ) ) : null;
		$amount              = isset( $_REQUEST['_rtsb_bxgy_discounted_amount'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_bxgy_discounted_amount'] ) ) : null;

		$roles               = $this->sanitize_array( '_rtsb_bogo_bxgy_user_roles' );
		$discounted_products = [];
		if ( isset( $_REQUEST['_rtsb_bogo_bxgy_discounted_products'] ) ) {
			if ( is_array( $_REQUEST['_rtsb_bogo_bxgy_discounted_products'] ) ) {
				$discounted_products = $this->sanitize_array( '_rtsb_bogo_bxgy_discounted_products' );
			} else {
				$discounted_products = sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_bogo_bxgy_discounted_products'] ) );
			}
		}

		$start_date = isset( $_REQUEST['_rtsb_bogo_bxgy_start_date_and_time'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_bogo_bxgy_start_date_and_time'] ) ) : null;
		$end_date   = isset( $_REQUEST['_rtsb_bogo_bxgy_end_date_and_time'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['_rtsb_bogo_bxgy_end_date_and_time'] ) ) : null;

		// Disable Balk Discount.
		if ( ! empty( $disable_discount ) && 'on' === $disable_discount ) {
			update_post_meta( $post_id, '_rtsb_disable_bogo_bxgy', $disable_discount );
		} else {
			delete_post_meta( $post_id, '_rtsb_disable_bogo_bxgy' );
		}
		// Enable Balk Discount.
		if ( ! empty( $enable_discount ) && 'on' === $enable_discount ) {
			update_post_meta( $post_id, '_rtsb_enable_bogo_bxgy', $enable_discount );
		} else {
			delete_post_meta( $post_id, '_rtsb_enable_bogo_bxgy' );
		}
		// Discount Type.

		update_post_meta( $post_id, '_rtsb_bxgy_minimum_required_quantity', $minimum_quantity );
		update_post_meta( $post_id, '_rtsb_bxgy_discounted_products_quantity', $discounted_quantity );
		update_post_meta( $post_id, '_rtsb_bxgy_discounted_amount', $amount );

		update_post_meta( $post_id, '_rtsb_disable_bogo_multiapply', $bogo_multiapply );
		update_post_meta( $post_id, '_rtsb_disable_bxgy_multiapply', $bxgy_multiapply );
		update_post_meta( $post_id, '_rtsb_enable_date_range', $enable_date_range );

		update_post_meta( $post_id, '_rtsb_bogo_bxgy_type', $discount_type );
		update_post_meta( $post_id, '_rtsb_bogo_bxgy_discounted_products', $discounted_products );
		update_post_meta( $post_id, '_rtsb_bogo_bxgy_value_type', $discount_value_type );
		update_post_meta( $post_id, '_rtsb_bogo_bxgy_user_roles', $roles );
		update_post_meta( $post_id, '_rtsb_bogo_bxgy_start_date_and_time', $start_date );
		update_post_meta( $post_id, '_rtsb_bogo_bxgy_end_date_and_time', $end_date );
	}

	/**
	 * Adds the Product Add-Ons tab to the product data tabs.
	 *
	 * @param array $tabs Existing tabs.
	 *
	 * @return array Modified tabs.
	 */
	public function add_product_add_ons_tab( $tabs ) {
		$tabs['rtsb_product_bogo_bxgy'] = [
			'label'    => esc_html__( 'Buy-X Get-Y | BOGO', 'shopbuilder-pro' ),
			'target'   => 'rtsb_product_bogo_bxgy',
			'class'    => [ 'show_if_simple', 'show_if_variable' ],
			'priority' => 65,
		];

		return $tabs;
	}

	/**
	 * Renders meta-boxes to the WooCommerce product data panels.
	 *
	 * @return void
	 */
	public function render_meta_boxes() {
		global $post;
		$disable_bogo_bxgy   = get_post_meta( $post->ID, '_rtsb_disable_bogo_bxgy', true );
		$enable_bogo_bxgy    = get_post_meta( $post->ID, '_rtsb_enable_bogo_bxgy', true );
		$bogo_multiapply     = get_post_meta( $post->ID, '_rtsb_disable_bogo_multiapply', true );
		$bxgy_multiapply     = get_post_meta( $post->ID, '_rtsb_disable_bxgy_multiapply', true );
		$enable_date_range   = get_post_meta( $post->ID, '_rtsb_enable_date_range', true );
		$discount_value_type = get_post_meta( $post->ID, '_rtsb_bogo_bxgy_value_type', true );
		$discount_type       = get_post_meta( $post->ID, '_rtsb_bogo_bxgy_type', true );
		$minimum_quantity    = get_post_meta( $post->ID, '_rtsb_bxgy_minimum_required_quantity', true );
		$discounted_quantity = get_post_meta( $post->ID, '_rtsb_bxgy_discounted_products_quantity', true );
		$amount              = get_post_meta( $post->ID, '_rtsb_bxgy_discounted_amount', true );

		$user_roles = get_post_meta( $post->ID, '_rtsb_bogo_bxgy_user_roles', true );
		$start_date = get_post_meta( $post->ID, '_rtsb_bogo_bxgy_start_date_and_time', true );
		$end_date   = get_post_meta( $post->ID, '_rtsb_bogo_bxgy_end_date_and_time', true );

		$discounted_products = get_post_meta( $post->ID, '_rtsb_bogo_bxgy_discounted_products', true );
		$disable_checked     = 'on' === $disable_bogo_bxgy ? 'on' : 'off';
		$enable_checked      = 'on' === $enable_bogo_bxgy ? 'on' : 'off';
		$enable_visibility   = 'on' === $enable_bogo_bxgy ? ' hidden' : '';

		$discounted_product_titles = [];
		if ( ! empty( $discounted_products ) ) {
			if ( is_array( $discounted_products ) ) {
				foreach ( $discounted_products as $product_id ) {
					$discounted_product_titles[ $product_id ] = get_the_title( $product_id );
				}
			} else {
				$discounted_product_titles[ $discounted_products ] = get_the_title( $discounted_products );
			}
		}

		?>
		<div id='rtsb_product_bogo_bxgy' class='panel woocommerce_options_panel rtsb_panel hidden'>
			<div class="rtsb-group-heading">
				<h2><?php esc_html_e( 'Discount Settings', 'shopbuilder-pro' ); ?></h2>
			</div>
			<div id="rtsb_product_bogo_bxgy_options_group" class="options_group">
				<?php wp_nonce_field( rtsb()->nonceText, rtsb()->nonceId ); ?>
				<div class="bogo-bxgy-discount-group-control">
					<div id="rtsb_disable_bogo_bxgy_group">
						<?php
						Fns::print_html(
							AdminInputsFns::generate_switch_input(
								[
									'id'            => '_rtsb_disable_bogo_bxgy',
									'label'         => esc_html__( 'Disable Global Rules?', 'shopbuilder-pro' ),
									'desc_tip'      => true,
									'aria_label'    => esc_html__( 'Disable Global Product Discount', 'shopbuilder-pro' ),
									'description'   => esc_html__( 'Switch on to disable the ShopBuilder global Discount rules for this product.', 'shopbuilder-pro' ),
									'checked'       => esc_attr( $disable_checked ),
									'value'         => 'on',
									'input_class'   => 'switch-input',
									'wrapper_class' => 'form-field disable-rules',
								]
							),
							true
						);
						?>
					</div>
					<div id="rtsb_enable_bogo_bxgy_group">
						<?php
						Fns::print_html(
							AdminInputsFns::generate_switch_input(
								[
									'id'            => '_rtsb_enable_bogo_bxgy',
									'label'         => esc_html__( 'Override Global Rules?', 'shopbuilder-pro' ),
									'desc_tip'      => true,
									'aria_label'    => esc_html__( 'Override Global Product Discount', 'shopbuilder-pro' ),
									'description'   => esc_html__( 'Switch on to override the ShopBuilder global Discount rules.', 'shopbuilder-pro' ),
									'checked'       => esc_attr( $enable_checked ),
									'value'         => 'on',
									'input_class'   => 'switch-input',
									'wrapper_class' => 'form-field ',
								]
							),
							true
						);
						?>
					</div>
				</div>
				<div id="rtsb_bogo_bxgy_groups" class="rtsb-bogo-bxgy-discount-fields<?php echo esc_attr( $enable_visibility ); ?>">
					<?php
					$fields  = AdminInputsFns::generate_select_input(
						[
							'id'            => '_rtsb_bogo_bxgy_type',
							'label'         => esc_html__( 'Discount Type', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Discount type', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Select your desired discount type.', 'shopbuilder-pro' ),
							'value'         => $discount_type ?? 'bogo',
							'input_class'   => 'short',
							'label_class'   => 'field-label',
							'options'       => [
								'bogo'        => esc_html__( 'BOGO', 'shopbuilder-pro' ),
								'buy-x-get-y' => esc_html__( 'Buy X Get Y', 'shopbuilder-pro' ),
							],
							'wrapper_class' => 'form-field input-type rtsb-bogo-bxgy-type',
						]
					);
					$fields .= AdminInputsFns::generate_number_input(
						[
							'id'            => '_rtsb_bxgy_minimum_required_quantity',
							'label'         => esc_html__( 'Required Quantity', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Minimum Quantity Needed to Claim This Offer', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Minimum Quantity Needed to Claim This Offer.', 'shopbuilder-pro' ),
							'value'         => $minimum_quantity ?? '',
							'placeholder'   => esc_html__( 'Quantity', 'shopbuilder-pro' ),
							'min'           => '0',
							'step'          => '1',
							'input_class'   => 'rtsb-max-product-qty',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-field minimum-product-qty',
						]
					);
					$fields .= AdminInputsFns::generate_product_search_multiselect_input(
						[
							'id'            => '_rtsb_bogo_bxgy_discounted_products',
							'label'         => esc_html__( 'Discounted Products', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Discounted Products', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Select your desired discounted products.', 'shopbuilder-pro' ),
							'value'         => ! empty( $discounted_products ) ? $discounted_products : '',
							'options'       => $discounted_product_titles,
							'multiple'      => true,
							'input_class'   => 'short',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-field input-type ',
						]
					);
					$fields .= AdminInputsFns::generate_number_input(
						[
							'id'            => '_rtsb_bxgy_discounted_products_quantity',
							'label'         => esc_html__( 'Discounted Products Quantity', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Discounted Products Quantity', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'When the discount is available, the specified quantity of free or offered products will be automatically added to the cart.', 'shopbuilder-pro' ),
							'value'         => $discounted_quantity,
							'placeholder'   => esc_html__( 'Quantity', 'shopbuilder-pro' ),
							'min'           => '0',
							'step'          => '1',
							'input_class'   => 'rtsb-max-product-qty',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-field discounted-product-qty',

						]
					);
					// Only one gift product is allowed per order?
					$fields .= AdminInputsFns::generate_switch_input(
						[
							'id'            => '_rtsb_disable_bogo_multiapply',
							'label'         => esc_html__( 'One Gift Product Allowed?', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Only one gift product is allowed per order', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Only one gift product can be added to the cart per order, regardless of the quantity of the main product. Even if multiple eligible products are added, only a single gift item will be included..', 'shopbuilder-pro' ),
							'checked'       => esc_attr( $bogo_multiapply ),
							'value'         => 'on',
							'input_class'   => 'switch-input',
							'wrapper_class' => 'form-field bogo-multiapply',
						]
					);
					$fields .= AdminInputsFns::generate_switch_input(
						[
							'id'            => '_rtsb_disable_bxgy_multiapply',
							'label'         => esc_html__( 'Allow MultiApply?', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Multiple Time Apply Base on Product Quantity.', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Multiple Time Apply Base on Product Quantity.', 'shopbuilder-pro' ),
							'checked'       => esc_attr( $bxgy_multiapply ),
							'value'         => 'on',
							'input_class'   => 'switch-input',
							'wrapper_class' => 'form-field bxgy-multiapply',
						]
					);
					$fields .= AdminInputsFns::generate_select_input(
						[
							'id'            => '_rtsb_bogo_bxgy_value_type',
							'label'         => esc_html__( 'Discount Value Type', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Discount Value type', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Choose how the discount is applied to the Get (Y) product.', 'shopbuilder-pro' ),
							'value'         => $discount_value_type,
							'input_class'   => 'short',
							'label_class'   => 'field-label',
							'options'       => [
								'full'       => esc_html__( '100% Discount (Free)', 'shopbuilder-pro' ),
								'percentage' => esc_html__( 'Percentage Discount (%)', 'shopbuilder-pro' ),
								'fixed'      => esc_html__( 'Fixed Amount Discount (Per Item)', 'shopbuilder-pro' ),
							],
							'wrapper_class' => 'form-field input-type discount-value-field-type',
						]
					);
					$fields .= AdminInputsFns::generate_number_input(
						[
							'id'            => '_rtsb_bxgy_discounted_amount',
							'label'         => esc_html__( 'Discount Value', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'Discount Value', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Enter the discount value based on the selected discount type (fixed amount or percentage).', 'shopbuilder-pro' ),
							'value'         => $amount,
							'placeholder'   => esc_html__( 'Enter Discount Value', 'shopbuilder-pro' ),
							'min'           => '0',
							'step'          => '0.01',
							'input_class'   => 'rtsb-discount-value',
							'label_class'   => 'field-label',
							'wrapper_class' => 'form-field discount-amount-field',
						]
					);
					$fields .= AdminInputsFns::generate_select_input(
						[
							'id'            => '_rtsb_bogo_bxgy_user_roles',
							'label'         => esc_html__( 'User Roles', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'aria_label'    => esc_html__( 'User Roles', 'shopbuilder-pro' ),
							'description'   => esc_html__( 'Select desired user roles type.', 'shopbuilder-pro' ),
							'value'         => ! empty( $user_roles ) ? $user_roles : [],
							'input_class'   => 'rtsb-select2-input wc-enhanced-select',
							'label_class'   => 'field-label',
							'options'       => Fns::get_all_user_roles(),
							'wrapper_class' => 'form-field input-type ',
							'multiple'      => true,
						]
					);
					$fields .= AdminInputsFns::generate_switch_input(
						[
							'id'            => '_rtsb_enable_date_range',
							'label'         => esc_html__( 'Enable Date range?', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'checked'       => esc_attr( $enable_date_range ),
							'value'         => 'on',
							'input_class'   => 'switch-input',
							'wrapper_class' => 'form-field bxgy-date-range',
						]
					);
					$fields .= AdminInputsFns::generate_date_picker_input(
						$post->ID,
						[
							'id'            => '_rtsb_bogo_bxgy_start_date_and_time',
							'label'         => esc_html__( 'Discount Start Date', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'value'         => $start_date ?? '',
							'class'         => 'short',
							'description'   => esc_html__( 'Enter the estimated date when Discount will be available. ', 'shopbuilder-pro' ),
							'wrapper_class' => 'rtsb-bxgy-date',
						],
						true,
						false
					);
					$fields .= AdminInputsFns::generate_date_picker_input(
						$post->ID,
						[
							'id'            => '_rtsb_bogo_bxgy_end_date_and_time',
							'label'         => esc_html__( 'Discount End Date', 'shopbuilder-pro' ),
							'desc_tip'      => true,
							'value'         => $end_date ?? '',
							'class'         => 'short',
							'description'   => esc_html__( 'Enter the estimated date when Discount will end. ', 'shopbuilder-pro' ),
							'wrapper_class' => 'rtsb-bxgy-date',
						],
						true,
						false
					);
					Fns::print_html( $fields, true )
					?>
				</div>

			</div>
		</div>
		<?php
	}
}
