<?php
/**
 * Customize My Account Module Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\BogoBxGy;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Modules\BulkDiscount\BulkDiscountFns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use WC_Product;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Module Class.
 */
class BogoGiftBxGyApply {
	/**
	 * @var array
	 */
	public $cache = [];

	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * CLass Construct
	 */
	private function __construct() {
		add_filter( 'woocommerce_loop_add_to_cart_link', [ $this, 'bulk_discount_note' ], 11, 2 );
		add_action( 'woocommerce_before_add_to_cart_form', [ $this, 'single_bulk_discount_note' ], 11 );
		// Custom Widget Support.
		add_action( 'rtsb/after/shop/product/title', [ $this, 'custom_layout_bulk_discount_note' ], 11 );

		add_action( 'woocommerce_before_calculate_totals', [ $this, 'custom_discounted_cart_item_price' ], 20, 1 );
		add_filter( 'woocommerce_cart_item_quantity', [ $this, 'disable_quantity_selector_for_free_gifts' ], 10, 3 );
		add_filter( 'woocommerce_cart_contents_changed', [ $this, 'adjust_cart_subtotal' ], 25, 1 );
		add_filter( 'woocommerce_cart_item_removed', [ $this, 'cart_item_removed' ], 10, 2 );
		add_filter( 'woocommerce_cart_item_name', [ $this, 'cart_item_discount_description' ], 20, 2 );
		add_action( 'woocommerce_before_cart', [ $this, 'add_cart_notice' ] );
		add_action( 'woocommerce_cart_calculate_fees', [ $this, 'remove_coupons' ] );
		// Bogo BxGy Order Page.
		add_action( 'woocommerce_checkout_create_order_line_item', [ $this, 'add_volume_discount_to_order_items' ], 10, 3 );
		add_filter( 'woocommerce_order_item_get_formatted_meta_data', [ $this, 'format_volume_discount_meta' ], 10, 2 );
		add_filter( 'woocommerce_hidden_order_itemmeta', [ $this, 'hide_admin_meta' ], 10, 1 );
	}
	/**
	 * Hides the volume discount metadata from the admin order details view.
	 *
	 * This prevents the `_rtsb_bogo_bxgy_volume` meta key from being displayed
	 * in the order details section for better UI clarity.
	 *
	 * @param array $hidden_meta_data List of meta keys to be hidden.
	 * @return array Updated list of hidden meta keys.
	 */
	public function hide_admin_meta( $hidden_meta_data ) {
		$hidden_meta_data[] = '_rtsb_bogo_bxgy_volume';
		return $hidden_meta_data;
	}
	/**
	 * Formats and adds the volume discount metadata to order line items.
	 *
	 * Retrieves the discount applied per item and adds it to the order metadata
	 * in a formatted structure for display.
	 *
	 * @param array  $meta_data Existing meta data for the line item.
	 * @param object $item      The order line item object.
	 * @return array Updated meta data with volume discount information.
	 */
	public function format_volume_discount_meta( $meta_data, $item ) {
		$volume_discount = $item->get_meta( '_rtsb_bogo_bxgy_volume' );
		if ( $volume_discount ) {
			$meta_data[] = (object) [
				'key'           => esc_html__( 'Offer Applied', 'shopbuilder-pro' ),
				'value'         => '',
				'display_key'   => esc_html__( 'Offer', 'shopbuilder-pro' ),
				'display_value' => wp_strip_all_tags( $volume_discount ),
			];
		}
		return $meta_data;
	}

	/**
	 * Adds the volume discount metadata to order line items.
	 *
	 * Calculates the discount for the cart item and stores it as meta data
	 * in the order line item to track applied bulk discounts.
	 *
	 * @param object $item          The order line item object.
	 * @param string $cart_item_key The unique cart item key.
	 * @param array  $values        The cart item data containing pricing and discount details.
	 * @return void
	 */
	public function add_volume_discount_to_order_items( $item, $cart_item_key, $values ) {
		if ( ! isset( $values[ BogoGiftBxGyFns::DISCOUNTED_CART_KEY ] ) ) {
			return;
		}
		$message = '';
		if ( 'bogo' === $values['rule'] ) {
			$message = esc_html__( 'Free Product (Buy One, Get One)', 'shopbuilder-pro' );
		} elseif ( 'bxgy' === $values['rule'] ) {
			$message = esc_html__( 'Buy X, Get Y', 'shopbuilder-pro' );
			$price   = wc_price( ! empty( $values['data']->get_sale_price() ) ? $values['data']->get_sale_price() : $values['data']->get_regular_price() );
			if ( 'percentage' === $values['discount_type'] ) {
				$message  = sprintf(
					'Original Price: %s | ',
					$price
				);
				$message .= sprintf(
					'Discount: %s ',
					esc_html( $values['amount'] ) . '%'
				);
			} elseif ( 'fixed' === $values['discount_type'] ) {
				$message  = sprintf(
					'Original Price: %s | ',
					$price
				);
				$message .= sprintf(
					'Discount: %s (Buy X, Get Y)',
					wc_price( $values['amount'] )
				);
			} elseif ( 'full' === $values['discount_type'] ) {
				$message = esc_html__( 'Buy X, Get Y Discount: 100% ', 'shopbuilder-pro' );
			}
		}
		if ( empty( $message ) ) {
			return;
		}
		$item->add_meta_data( '_rtsb_bogo_bxgy_volume', wp_strip_all_tags( $message ), true );
	}
	/**
	 * @param object $cart cart item.
	 * @return void
	 */
	public function remove_coupons( $cart ) {
		$discounts        = BogoGiftBxGyFns::get_options();
		$isDisabledCoupon = 'on' === ( $discounts['bxgybogo_disable_coupons'] ?? '' );
		$discount_applied = WC()->session->get( 'rtsb_discount_applied_session' );
		if ( $isDisabledCoupon && $discount_applied ) {
			$cart->remove_coupons();
		}
	}
	/**
	 * @return void
	 */
	public function add_cart_notice() {
		$discounts        = BogoGiftBxGyFns::get_options();
		$discount_applied = WC()->session->get( 'rtsb_discount_applied_session' );
		if ( empty( $discount_applied ) || ! is_array( $discount_applied ) ) {
			return;
		}
		$discount_applied = array_unique( $discount_applied );
		if ( ! empty( $discounts['bxgybogo_cart_message'] ) ) {
			$product_list = [];
			foreach ( $discount_applied as $id ) {
				$product_list[] = get_the_title( $id );
			}
			$product_list = implode( ', ', $product_list );
			// Replace the placeholder.
			$message = str_replace( '{{bxgy_products}}', $product_list, $discounts['bxgybogo_cart_message'] );
			wc_print_notice(
				sprintf(
					'<span class="discount-notice">%s</span>',
					$message
				),
				'success'
			);
			WC()->session->__unset( 'rtsb_discount_applied_session' );
		}
	}
	/**
	 * @param $product_name
	 * @param $cart_item
	 * @return string
	 */
	public function cart_item_discount_description( $product_name, $cart_item ) {
		if ( $this->is_discounted( $cart_item ) && 'bogo' === $cart_item['rule'] ) {
			$product_name = '<div class="rtsb-dicount-summery"><span class="rtsb-free-gift-wrapper"><span class="rtsb-free-gift-label"><svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M12 7v13m0-13H8.464c-.52 0-1.02-.21-1.389-.586A2.019 2.019 0 0 1 6.5 5c0-.53.207-1.04.575-1.414A1.947 1.947 0 0 1 8.465 3C11.214 3 12 7 12 7Zm0 0h3.536c.52 0 1.02-.21 1.389-.586.368-.375.575-.884.575-1.414 0-.53-.207-1.04-.575-1.414A1.947 1.947 0 0 0 15.535 3C12.786 3 12 7 12 7Zm-7 5h14v5.8c0 1.12 0 1.68-.218 2.108a2 2 0 0 1-.874.874C17.48 21 16.92 21 15.8 21H8.2c-1.12 0-1.68 0-2.108-.218a2 2 0 0 1-.874-.874C5 19.48 5 18.92 5 17.8V12Zm-.4 0h14.8c.56 0 .84 0 1.054-.109a1 1 0 0 0 .437-.437C21 11.24 21 10.96 21 10.4V8.6c0-.56 0-.84-.109-1.054a1 1 0 0 0-.437-.437C20.24 7 19.96 7 19.4 7H4.6c-.56 0-.84 0-1.054.109a1 1 0 0 0-.437.437C3 7.76 3 8.04 3 8.6v1.8c0 .56 0 .84.109 1.054a1 1 0 0 0 .437.437C3.76 12 4.04 12 4.6 12Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg</span>' . $product_name . '</span></div>';
		} elseif ( $this->is_discounted( $cart_item ) && 'bxgy' === $cart_item['rule'] ) {
			$amount         = $cart_item['amount'];
			$original_price = ! empty( $cart_item['data']->get_sale_price() ) ? $cart_item['data']->get_sale_price() : $cart_item['data']->get_regular_price();
			$message        = '';
			if ( 'fixed' === $cart_item['discount_type'] ) {
				$message = sprintf(
					'<div class="rtsb-dicount-summery"><div class="rtsb-original-price">%s %s </div> <div class="rtsb-discount"> %s %s</div> </div>',
					esc_html__( 'Original Price:', 'shopbuilder-pro' ),
					wc_price( $original_price ),
					esc_html__( 'Discount:', 'shopbuilder-pro' ),
					wc_price( $amount )
				);
			} elseif ( 'percentage' === $cart_item['discount_type'] ) {
				$message = sprintf(
					'<div class="rtsb-dicount-summery"><div class="rtsb-original-price">%s %s </div> <div class="rtsb-discount"> %s %s%%</div> </div>',
					esc_html__( 'Original Price:', 'shopbuilder-pro' ),
					wc_price( $original_price ),
					esc_html__( 'Discount:', 'shopbuilder-pro' ),
					esc_html( $amount )
				);
			} elseif ( 'full' === $cart_item['discount_type'] ) {
				$message = sprintf(
					'<div class="rtsb-dicount-summery"> <div class="rtsb-original-price">%s %s </div> <div class="rtsb-discount"> %s </div> </div>',
					esc_html__( 'Original Price:', 'shopbuilder-pro' ),
					wc_price( $original_price ),
					esc_html__( 'Discount: 100%', 'shopbuilder-pro' )
				);
			}
			$product_name .= apply_filters( 'rtsb/discount/note/html', $message, $original_price, $amount, $cart_item );
		}
		return $product_name;
	}

	/**
	 * Removes free gift products when the main product is removed from the cart.
	 *
	 * @param string  $cart_item_key The key of the removed cart item.
	 * @param WC_Cart $cart          The WooCommerce cart object.
	 *
	 * @return void
	 */
	public function cart_item_removed( $cart_item_key, $cart ) {
		// Check if the removed item was a main product with a discount rule applied.
		$removed_item = $cart->removed_cart_contents[ $cart_item_key ] ?? null;
		if ( ! $removed_item || isset( $removed_item[ BogoGiftBxGyFns::DISCOUNTED_CART_KEY ] ) ) {
			return;
		}
		// Loop through cart and remove free gifts.
		foreach ( $cart->get_cart() as $key => $item ) {
			if ( isset( $item[ BogoGiftBxGyFns::DISCOUNTED_CART_KEY ] ) && $removed_item['product_id'] === $item[ BogoGiftBxGyFns::DISCOUNTED_CART_KEY ] ) {
				$cart->remove_cart_item( $key );
			}
		}
	}
	/**
	 * Disables the quantity selector for gift items in the cart.
	 *
	 * @param string $product_quantity Original quantity HTML.
	 * @param string $cart_item_key    Cart item key.
	 * @param array  $cart_item        Cart item data.
	 *
	 * @return string
	 */
	public function disable_quantity_selector_for_free_gifts( $product_quantity, $cart_item_key, $cart_item ) {
		if ( $this->is_discounted( $cart_item ) ) {
			return '<span class="rtsb-no-quantity">' . esc_html( $cart_item['quantity'] ) . '</span>';
		}
		return $product_quantity;
	}

	/**
	 * Checks if a cart item is a free gift.
	 *
	 * @param array $cart_item Cart item data.
	 *
	 * @return bool
	 */
	private function is_discounted( $cart_item ) {
		return isset( $cart_item[ BogoGiftBxGyFns::DISCOUNTED_CART_KEY ] ) && $cart_item[ BogoGiftBxGyFns::DISCOUNTED_CART_KEY ];
	}

	/**
	 * @param object $cart cart item.
	 * @return void
	 */
	public function custom_discounted_cart_item_price( $cart ) {
		if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
			return;
		}
		if ( did_action( 'woocommerce_before_calculate_totals' ) >= 2 ) {
			return;
		}
		foreach ( $cart->get_cart() as $cart_item ) {
			if ( ! isset( $cart_item[ BogoGiftBxGyFns::DISCOUNTED_CART_KEY ] ) ) {
				$product = $cart_item['data'];
				if ( $product->is_type( 'variation' ) ) {
					$parent_id = $product->get_parent_id();
					$product   = wc_get_product( $parent_id );
				}
				new DiscountHandler( $product, $cart_item );
			}
		}
	}

	/**
	 * @param $cart_contents
	 * @return array
	 */
	public function adjust_cart_subtotal( $cart_contents ) {
		foreach ( $cart_contents as $cart_item ) {
			if ( $this->is_discounted( $cart_item ) && 'bogo' === $cart_item['rule'] ) {
				$cart_item['data']->set_price( 0 );
			}
			$price = ! empty( $cart_item['data']->get_sale_price() ) ? $cart_item['data']->get_sale_price() : $cart_item['data']->get_regular_price();
			if ( $this->is_discounted( $cart_item ) && 'bxgy' === $cart_item['rule'] ) {
				if ( 'fixed' === $cart_item['discount_type'] ) {
					$price = $price > $cart_item['amount'] ? $price - $cart_item['amount'] : $price;
				} elseif ( 'percentage' === $cart_item['discount_type'] ) {
					$price = $price - ( $price * $cart_item['amount'] / 100 );
				} elseif ( 'full' === $cart_item['discount_type'] ) {
					$price = 0;
				}
				$cart_item['data']->set_price( $price );
			}
			$regular_price = $cart_item['data']->get_regular_price();
			$cart_item['data']->set_regular_price( $regular_price );
		}

		return $cart_contents;
	}
	/**
	 * Generates an availability notice for a product if applicable.
	 *
	 * @param WC_Product $product The WooCommerce product object.
	 * @return string The formatted availability notice HTML or an empty string if no notice is available.
	 */
	private function available_notice( $product, $loop = true ) {
		$DiscountHandler = new DiscountHandler( $product );
		$price_html      = $DiscountHandler->get_availability_notice();
		$parent_class    = $loop ? 'rtsb-product-showcase' : 'rtsb-product-details';
		if ( $price_html ) {
			return '<div class="rtsb-bogo-bxgy-availability-notice ' . $parent_class . '">' . $price_html . '</div>';
		}
		return '';
	}
	/**
	 * Adds a bulk discount notice to the product button in the loop.
	 *
	 * This function checks if the bulk discount message should be displayed
	 * in the product loop. If enabled, it calculates the discount and appends
	 * the discount notice before returning the button.
	 *
	 * @param string     $button  The original product button HTML.
	 * @param WC_Product $product The WooCommerce product object.
	 * @return string  The modified button HTML with the bulk discount notice (if applicable).
	 */
	public function bulk_discount_note( $button, $product ) {
		$discounts = BogoGiftBxGyFns::get_options();
		if ( 'on' !== ( $discounts['discount_note_in_product_loop'] ?? '' ) ) {
			return $button;
		}
		return $this->available_notice( $product ) . $button;
	}

	/**
	 * Adds a bulk discount notice to the product button in the loop.
	 *
	 * This function checks if the bulk discount message should be displayed
	 * in the product loop. If enabled, it calculates the discount and appends
	 * the discount notice before returning the button.
	 *
	 * @param string     $button  The original product button HTML.
	 * @param WC_Product $product The WooCommerce product object.
	 * @return string  The modified button HTML with the bulk discount notice (if applicable).
	 */
	public function custom_layout_bulk_discount_note() {
		$discounts = BogoGiftBxGyFns::get_options();
		if ( 'on' !== ( $discounts['discount_note_in_product_loop'] ?? '' ) ) {
			return;
		}
		global $product;
		$note = $this->available_notice( $product );
		Fns::print_html( $note, true );
	}

	/**
	 * Displays the bulk discount notice on the single product page.
	 *
	 * This function retrieves the global WooCommerce product and calls
	 *
	 * @return void
	 */
	public function single_bulk_discount_note() {
		$discounts = BogoGiftBxGyFns::get_options();
		if ( 'on' !== ( $discounts['discount_note_in_product_details'] ?? '' ) ) {
			return;
		}
		global $product;
		Fns::print_html( $this->available_notice( $product, false ), true );
	}
}
