<?php
/**
 * Customize My Account Module Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\BogoBxGy;

use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SBPRO\Modules\BogoBxGy\Discount\BogoDiscount;
use RadiusTheme\SBPRO\Modules\BogoBxGy\Discount\BxGyDiscount;
use WC_Product;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Module Class.
 */
class DiscountHandler {
	/**
	 * @var array
	 */
	public $cache = [];

	/**
	 * @var
	 */
	protected $product;
	/**
	 * @var
	 */
	protected $cart_item;
	/**
	 * @var
	 */
	protected $discount = [];

	/**
	 * Module Class Constructor.
	 */
	public function __construct( WC_Product $product, $cart_item = [] ) {
		$this->cart_item = $cart_item;
		$this->product   = $product;
		$this->discount_type_handler();
	}

	/**
	 * @return void
	 */
	public function discount_type_handler() {
		if ( apply_filters( 'rtsb/module/bxgy_bogo/global/disable', false, $this->product ) ) {
			return;
		}

		$rules = $this->get_discount_rules();

		if ( empty( $rules ) ) {
			$this->remove_discounted_item();
		}

		$disable_bogo_bxgy = get_post_meta( $this->product->get_id(), '_rtsb_disable_bogo_bxgy', true );
		$enable_bogo_bxgy  = get_post_meta( $this->product->get_id(), '_rtsb_enable_bogo_bxgy', true );
		$disable_checked   = 'on' === $disable_bogo_bxgy;
		$enable_bogo_bxgy  = 'on' === $enable_bogo_bxgy;
		if ( $disable_checked ) {
			return;
		}
		$generate_option = [];
		if ( $enable_bogo_bxgy ) {
			$discount_type            = get_post_meta( $this->product->get_id(), '_rtsb_bogo_bxgy_type', true );
			$minimum_quantity         = get_post_meta( $this->product->get_id(), '_rtsb_bxgy_minimum_required_quantity', true );
			$discounted_quantity      = get_post_meta( $this->product->get_id(), '_rtsb_bxgy_discounted_products_quantity', true );
			$user_roles               = get_post_meta( $this->product->get_id(), '_rtsb_bogo_bxgy_user_roles', true );
			$enable_date_range        = get_post_meta( $this->product->get_id(), '_rtsb_enable_date_range', true );
			$start_date               = get_post_meta( $this->product->get_id(), '_rtsb_bogo_bxgy_start_date_and_time', true );
			$end_date                 = get_post_meta( $this->product->get_id(), '_rtsb_bogo_bxgy_end_date_and_time', true );
			$discount_products        = get_post_meta( $this->product->get_id(), '_rtsb_bogo_bxgy_discounted_products', true );
			$discount_value_type      = get_post_meta( $this->product->get_id(), '_rtsb_bogo_bxgy_value_type', true );
			$apply_only_one_gift_item = get_post_meta( $this->product->get_id(), '_rtsb_disable_bogo_multiapply', true );
			$bxgy_multiple_apply      = get_post_meta( $this->product->get_id(), '_rtsb_disable_bxgy_multiapply', true );
			$amount                   = get_post_meta( $this->product->get_id(), '_rtsb_bxgy_discounted_amount', true );

			// Option Date Setup.
			if ( ! empty( $start_date ) ) {
				$st_date = new \DateTime( $start_date );
				$generate_option['discount_start_date_and_time'] = $st_date->format( 'Y-m-d H:i' );
			} else {
				$generate_option['discount_start_date_and_time'] = '';
			}
			if ( ! empty( $end_date ) ) {
				$end_date                                      = new \DateTime( $end_date );
				$generate_option['discount_end_date_and_time'] = $end_date->format( 'Y-m-d H:i' );
			} else {
				$generate_option['discount_end_date_and_time'] = '';
			}
			$generate_option['enable_date_range']         = $enable_date_range ?? '';
			$generate_option['discount_type']             = $discount_type ?? '';
			$generate_option['user_roles']                = $user_roles;
			$generate_option['applicable_product']        = [ $this->product->get_id() ];
			$generate_option['discount_product']          = ! empty( $discount_products ) ? $discount_products : [];
			$generate_option['required_quantity']         = $minimum_quantity;
			$generate_option['discount_product_quantity'] = $discounted_quantity;
			$generate_option['discount_calculation']      = $discount_value_type ?? '';
			$generate_option['apply_only_one_gift_item']  = $apply_only_one_gift_item;
			$generate_option['bxgy_multiple_apply']       = $bxgy_multiple_apply;
			$generate_option['bxgy_amount']               = $amount;

		}

		foreach ( $rules as $option ) {
			if ( $enable_bogo_bxgy ) {
				$option = $generate_option;
			}
			$rule = [
				'enable_date_range'         => 'on' === ( $option['enable_date_range'] ?? '' ) ? 'on' : '',
				'start_date'                => $option['discount_start_date_and_time'] ?? '',
				'end_date'                  => $option['discount_end_date_and_time'] ?? '',
				'allowed_roles'             => $option['user_roles'] ?? [],
				'apply_only_one_gift_item'  => $option['apply_only_one_gift_item'] ?? '',
				'bxgy_multiple_apply'       => $option['bxgy_multiple_apply'] ?? '',
				'bxgy_amount'               => $option['bxgy_amount'] ?? '',
				'discount_product_quantity' => $option['discount_product_quantity'] ?? '',
				'discount_calculation'      => $option['discount_calculation'] ?? '',
				'required_quantity'         => $option['required_quantity'] ?? 1,
				'applicable_product'        => ! empty( $generate_option['applicable_product'] ) ? $generate_option['applicable_product'] : array_column( ( $option['applicable_product'] ?? [] ), 'value' ),
				'discount_product'          => ! empty( $generate_option['discount_product'] ) ? $generate_option['discount_product'] : array_column( ( $option['discount_product'] ?? [] ), 'value' ),
			];
			if ( empty( $rule['discount_product'] ) ) {
				return;
			}
			unset(
				$option['enable_date_range'],
				$option['discount_start_date_and_time'],
				$option['discount_end_date_and_time'],
				$option['required_quantity'],
				$option['user_roles'],
				$option['bxgy_amount'],
				$option['applicable_product'],
				$option['bxgy_multiple_apply'],
				$option['apply_only_one_gift_item'],
				$option['discount_product'],
				$option['discount_calculation'],
				$option['discount_product_quantity']
			);

			$rule['option'] = $option;
			switch ( $option['discount_type'] ) {
				case 'bogo':
					$rule['required_quantity'] = 1;
					$discount                  = new BogoDiscount( $this->product, $rule, $this->cart_item );
					if ( $discount->is_discount_applicable() ) {
						$this->discount[ $this->product->get_id() ] = $discount;
					} else {
						if ( ! empty( $this->cart_item ) ) {
							$this->remove_discounted_item();
						}
					}
					break;
				case 'buy-x-get-y':
					$discount = new BxGyDiscount( $this->product, $rule, $this->cart_item );
					if ( $discount->is_discount_applicable() ) {
						$this->discount[ $this->product->get_id() ] = $discount;
					} else {
						if ( ! empty( $this->cart_item ) ) {
							$this->remove_discounted_item();
						}
					}
					break;
			}
			// Cart/Checkout Page Will Apply The discount.
			if ( ! empty( $this->cart_item ) && ! empty( $this->discount[ $this->product->get_id() ] ) ) {
				$this->discount[ $this->product->get_id() ]->apply_discount();
			}
		}
	}
	/**
	 * Remove BxGy free products from the cart if requirements are not met.
	 *
	 * @param number $required_quantity Quantity.
	 *
	 * @return void
	 */
	public function remove_discounted_item() {
		$cart = WC()->cart;
		if ( empty( $cart ) ) {
			return;
		}
		foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {
			$parent = $cart_item[ BogoGiftBxGyFns::DISCOUNTED_CART_KEY ] ?? '';
			if ( empty( $parent ) ) {
				continue;
			}

			if ( $this->product->get_id() !== $parent ) {
				continue;
			}

			// if ( $this->cart_item['quantity'] < $required_quantity ) {
				 $cart->remove_cart_item( $cart_item_key );
			// }
		}
	}
	/**
	 * @return string
	 */
	public function get_availability_notice() {
		if ( ! empty( $this->discount[ $this->product->get_id() ] ) ) {
			$discount         = $this->discount[ $this->product->get_id() ]->discount_message();
			$note             = ( $discount['note'] ?? esc_html__( 'Offer Available', 'shopbuilder-pro' ) );
			$type             = $discount['discount_type'] ?? 'fixed';
			$discount_message = $discount['discount_message'] ?? '';
			$info_icon        = '<svg width="16" height="16" viewBox="0 0 24 24" xmlns="http://www.w3.org/2000/svg" fill="currentColor"><g fill="none" stroke="currentColor" stroke-linecap="round" stroke-linejoin="round" stroke-width="2"><circle cx="12" cy="12" data-name="--Circle" r="10"/><path d="M12 12v4M12 8h0"/></g></svg>';
			$info_icon        = apply_filters( 'rtsb/bogo/bxgy/discount/notice/icon', $info_icon, $type );
			// data-trigger="mouseenter".
			$sale_icon  = '<svg xmlns="http://www.w3.org/2000/svg" width="11" height="11" viewBox="0 0 11 11" fill="none"><path d="M6 1L1 6.4H5.5L5 10L10 4.6H5.5L6 1Z" stroke="currentColor" stroke-width="0.8" stroke-linecap="round" stroke-linejoin="round"/></svg>';
			$price_html = '<span class="rtsb-bxgy-bogo-discount-note " data-trigger="mouseenter" data-theme="rtsb-bxgy-bogo-discount"> 
            <span class="rtsb-bxgy-bogo-discount-notice">' . $note . '</span> 
            <span class="rtsb-bxgy-bogo-discount-info-icon" >' . $info_icon . '<span class="rtsb-tippy-data"> <span class="rtsb-tippy-content"> <span class="rtsb-bxgy-bogo-discount-badge">' . esc_html__( 'Sale', 'shopbuilder-pro' ) . $sale_icon . '</span>  ' . $discount_message . '</span></span></span></span>';
			return apply_filters( 'rtsb/bogo/bxg/discount/notice/html', $price_html, $type );
		}
		return '';
	}

	/**
	 * Get product min/max price.
	 *
	 * @return array|false
	 */
	private function get_discount_rules() {
		// Check if the cached result exists.
		$cache_key     = 'get_bogo_bxgy_free_gift_rules';
		$cached_result = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false !== $cached_result ) {
			// Return the cached result.
			return $cached_result;
		}
		$discounts_rules = BogoGiftBxGyFns::get_options();
		$campaign_list   = json_decode( $discounts_rules['bxgybogo_rules'] ?? '', true );
		if ( empty( $campaign_list ) ) {
			return [];
		}
		// Store the result in the cache.
		wp_cache_set( $cache_key, $campaign_list, 'shopbuilder', 3600 ); // Cache for 1 hour.
		Cache::set_data_cache_key( $cache_key );
		return $campaign_list;
	}
}
