<?php
/**
 * Customize My Account Module Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\BulkDiscount;

use DateTime;
use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SB\Helpers\Fns;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Module Class.
 */
abstract class DiscountCalculator {
	/**
	 * @var array
	 */
	public $cache = [];

	/**
	 * Base Settings
	 *
	 * @var array
	 */
	protected $baseSettings;

	/**
	 * Main Product Id
	 *
	 * @var int
	 */
	protected $mainProductId;

	/**
	 * Has Discount
	 *
	 * @var bool
	 */
	protected $hasDiscount;

	/**
	 * Discount Rule
	 *
	 * @var array
	 */
	protected $discountRule;

	/**
	 * Product
	 *
	 * @var object
	 */
	protected $product;

	/**
	 * Module Class Constructor.
	 */
	public function __construct() {
		$this->product       = $this->set_product();
		$this->mainProductId = $this->set_main_product_id();
		$this->baseSettings  = $this->get_base_settings();
		$this->hasDiscount   = $this->is_eligible_for_discount();
	}
	/**
	 * Set the product object.
	 *
	 * @return \WC_Product
	 */
	abstract public function set_product();
	/**
	 * Get bool
	 */
	public function has_product_discount() {
		return $this->hasDiscount;
	}
	/**
	 * Get Product Id
	 */
	public function get_main_product_id() {
		return $this->mainProductId;
	}

	/**
	 * Get Product Id
	 *
	 * @param string $stringData String Data.
	 *
	 * @return array
	 */
	public function condition_persing( $stringData ) {
		// Split by comma to get each condition.
		$conditions = explode( '|', $stringData );
		if ( empty( $conditions ) ) {
			return [];
		}
		$discounts = [];
		foreach ( $conditions as $condition ) {
			if ( empty( $condition ) ) {
				continue;
			}
			list( $quantity, $value ) = array_pad( explode( ':', $condition, 2 ), 2, 0 );
			if ( ! $value ) {
				continue;
			}
			$discounts[ intval( $quantity ) ] = floatval( $value ); // Convert to numbers.
		}
		return $discounts;
	}
	/**
	 * Get Product Id
	 */
	public function get_discount_rules() {
		return $this->discountRule;
	}

	/**
	 * Get Product Id
	 */
	protected function is_golbal_disable() {
		if ( apply_filters( 'rtsb/module/bulk_discounts/global/disable', false, $this->product, $this->get_main_product_id() ) ) {
			return true;
		}
		return 'on' === get_post_meta( $this->get_main_product_id(), '_rtsb_disable_bulk_discount', true );
	}
	/**
	 * Get Product Id
	 */
	protected function is_override() {
		if ( $this->is_golbal_disable() ) {
			return false;
		}
		return 'on' === get_post_meta( $this->get_main_product_id(), '_rtsb_enable_bulk_discount', true );
	}
	/**
	 * Get Product Id
	 */
	public function set_main_product_id() {
		if ( 'variation' === $this->product->get_type() ) {
			return $this->product->get_parent_id();
		}
		return $this->product->get_id();
	}
	/**
	 * Get Product Id
	 */
	protected function is_override_condition() {
		if ( ! $this->is_override() ) {
			return false;
		}
		$id            = $this->get_main_product_id();
		$type          = get_post_meta( $id, '_rtsb_enable_bulk_discount_type', true );
		$condition     = get_post_meta( $id, '_rtsb_bulk_discount_condition', true );
		$allowed_roles = get_post_meta( $id, '_rtsb_user_roles', true );
		$start_date    = get_post_meta( $id, '_rtsb_discount_start_date_and_time', true );
		$end_date      = get_post_meta( $id, '_rtsb_discount_end_date_and_time', true );
		$apply_on      = get_post_meta( $id, '_discount_apply_on', true );
		if ( empty( $condition ) ) {
			return false;
		}
		$cond = $this->condition_persing( $condition );
		if ( empty( $cond ) ) {
			return false;
		}
		// Get WordPress timezone.
		$wp_timezone = wp_timezone();
		$now         = new DateTime( 'now', $wp_timezone ); // Current time in WP timezone.
		$now_strtime = $now->getTimestamp();
		if ( ! empty( $start_date ) ) {
			$start_date_obj = date_create_from_format( 'Y-m-d\TH:i', $start_date, $wp_timezone );
			$start_date     = $start_date_obj ? $start_date_obj->getTimestamp() : 0;
			if ( $start_date > $now_strtime ) {
				return false;
			}
		}
		if ( ! empty( $end_date ) ) {
			$end_date_obj = date_create_from_format( 'Y-m-d\TH:i', $end_date, $wp_timezone );
			$end_date     = $end_date_obj ? $end_date_obj->getTimestamp() : 0;
			if ( $end_date < $now_strtime ) {
				return false;
			}
		}

		if ( ! empty( $allowed_roles ) && ! array_intersect( $allowed_roles, Fns::get_current_user_roles() ) ) {
			return false;
		}
		$this->discountRule = [
			'discount_type'           => $type ?: 'percentage',
			'bulk_discount_condition' => $cond,
			'discount_apply_on'       => $apply_on ?: 'regular-price',
		];
		return true;
	}

	/**
	 * Check if a product is eligible for bulk discount.
	 *
	 * @return bool True if eligible, false otherwise.
	 */
	protected function is_eligible_for_discount() { // phpcs:ignore Generic.Metrics.CyclomaticComplexity.MaxExceeded
		if ( $this->is_golbal_disable() ) {
			$this->discountRule = [];
			return false;
		}
		if ( $this->is_override() ) {
			if ( $this->is_override_condition() && ! empty( $this->get_discount_rules() ) ) {
				return true;
			}
			return false;
		}
		$has_discount = false;
		// Get WordPress timezone.
		$wp_timezone = wp_timezone();
		$now         = new DateTime( 'now', $wp_timezone ); // Current time in WP timezone.
		$now_strtime = $now->getTimestamp();
		foreach ( $this->baseSettings as $key => $option ) {
			if ( empty( $option['bulk_discount_condition'] ) ) {
				continue;
			}
			$cond = $this->condition_persing( $option['bulk_discount_condition'] );
			if ( empty( $cond ) ) {
				continue;
			}
			$apply_for     = $option['apply_for'] ?? 'product';
			$allowed_roles = $option['user_roles'] ?? [];
			$start_date    = $option['discount_start_date_and_time'] ?? false;
			$end_date      = $option['discount_end_date_and_time'] ?? false;
			$date_range    = 'on' === ( $option['enable_date_range'] ?? '' );
			// Check if user role is allowed (if specified).
			if ( ! empty( $allowed_roles ) && ! array_intersect( $allowed_roles, Fns::get_current_user_roles() ) ) {
				continue;
			}

			if ( $date_range && $start_date ) {
				$start_date_obj = date_create_from_format( 'Y-m-d H:i', $start_date, $wp_timezone );
				$start_date     = $start_date_obj ? $start_date_obj->getTimestamp() : 0;
				if ( $start_date > $now_strtime ) {
					continue; // Exit if start date is in the future.
				}
			}

			if ( $date_range && $end_date ) {
				$end_date_obj = date_create_from_format( 'Y-m-d H:i', $end_date, $wp_timezone );
				$end_date     = $end_date_obj ? $end_date_obj->getTimestamp() : 0;
				if ( $end_date < $now_strtime ) {
					continue; // Exit if start date is in the future.
				}
			}

			$applicable_products   = ! empty( $option['applicable_products'] ) ? array_column( $option['applicable_products'], 'value' ) : [];
			$excluded_products     = ! empty( $option['exclude_product'] ) ? array_column( $option['exclude_product'] ?? [], 'value' ) : [];
			$applicable_categories = ! empty( $option['applicable_categories'] ) ? array_column( $option['applicable_categories'], 'value' ) : [];
			$applicable_tags       = ! empty( $option['applicable_tags'] ) ? array_column( $option['applicable_tags'], 'value' ) : [];

			if ( 'product' === $apply_for && ! empty( $applicable_products ) && ! in_array( $this->get_main_product_id(), $applicable_products, true ) ) {
				continue;
			} elseif ( 'product_cat' === $apply_for || 'product_tag' === $apply_for ) {
				if ( ! empty( $excluded_products ) && in_array( $this->get_main_product_id(), $excluded_products, true ) ) {
					continue;
				}
			}
			if ( 'product_cat' === $apply_for && ! empty( $applicable_categories ) && ! has_term( $applicable_categories, 'product_cat', $this->get_main_product_id() ) ) {
				continue;
			}

			// Skip if product is excluded.
			if ( 'product_tag' === $apply_for && ! empty( $applicable_tags ) && ! has_term( $applicable_tags, 'product_tag', $this->get_main_product_id() ) ) {
				continue;
			}
			$this->discountRule = [
				'discount_type'           => $option['discount_type'] ?? 'percentage',
				'bulk_discount_condition' => $cond,
				'discount_apply_on'       => $option['discount_apply_on'] ?? 'regular-price',
			];
			$has_discount       = true;
			break;
		}
		return $has_discount; // No matching discount found.
	}
	/**
	 * @return array
	 */
	protected function get_base_settings() {
		// This will functional Latter.
		$cache_key = 'set_base_settings';
		if ( ! empty( $this->cache[ $cache_key ] ) ) {
			return $this->cache[ $cache_key ];
		}
		$this->cache[ $cache_key ] = $this->get_bulk_discounts_list();
		return $this->cache[ $cache_key ];
	}

	/**
	 * Get product min/max price.
	 *
	 * @return array|false
	 */
	protected function get_bulk_discounts_list(): array {
		// Check if the cached result exists.
		$cache_key     = 'get_bulk_discounts_rules';
		$cached_result = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false !== $cached_result ) {
			// Return the cached result.
			return $cached_result;
		}
		$bulkDiscount  = BulkDiscountFns::get_options();
		$campaign_list = json_decode( $bulkDiscount['rules'] ?? '', true );
		if ( empty( $campaign_list ) ) {
			return [];
		}
		// Store the result in the cache.
		wp_cache_set( $cache_key, $campaign_list, 'shopbuilder', 3600 ); // Cache for 1 hour.
		Cache::set_data_cache_key( $cache_key );
		return $campaign_list;
	}
}
