<?php
/**
 * Main FilterHooks class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\CurrencySwitcher;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Main FilterHooks class.
 */
class CurrencyFrontend extends CurrencyBase {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Currency Settings
	 *
	 * @var array
	 */
	private $settings;

	/**
	 * Asset Handle
	 *
	 * @var string
	 */
	private $handle = 'rtsb-currency-switcher';

	/**
	 * Notifications hooks.
	 */
	private function __construct() {
		parent::__construct();
		$this->settings = $this->get_currency_settings();
		// ShortCode.
		add_shortcode( 'currency_switcher', [ $this, 'currency_switcher_shortcode' ] );

		/*
		 * Footer Script.
		 */
		add_action( 'wp_footer', [ $this, 'currency_switcher_footer_script' ], 99 );

		/**
		 * Public scripts.
		 */
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_public_scripts' ], 99 );

		/**
		 * Public scripts.
		 */
		add_filter( 'woocommerce_add_to_cart_fragments', [ $this, 'recalculate_cart_after_currency_change' ], 1 );

		/**
		 * Format price
		 */
		 add_filter( 'wc_get_price_decimals', [ $this, 'set_decimals' ], );

		 /**
		  * Symbol position
		  * Ref "pre_option_{$option}"
		 */
		add_filter( 'pre_option_woocommerce_currency_pos', [ $this, 'currency_position' ], 30 );

		/**
		 * Symbol position.
		 */
		add_filter( 'woocommerce_currency', [ $this, 'set_woocommerce_currency' ], 15 );

		/**
		 * Set Currency Symbol.
		 */
		add_filter( 'woocommerce_currency_symbol', [ $this, 'set_currency_symbol' ], 15, 2 );

		/**
		 * Price Separator.
		 */
		add_filter( 'wc_get_price_decimal_separator', [ $this, 'price_decimal_separator' ], 15 );
		add_filter( 'wc_get_price_thousand_separator', [ $this, 'price_thousand_separator' ], 15 );

		/*
		 * Available Payment Getaways
		 */
		add_filter( 'woocommerce_available_payment_gateways', [ $this, 'available_payment_gateways' ], 15 );
	}

	/**
	 * Add Nav Menu Link Atts.
	 *
	 * @param array $atts Nav Menu Link Atts.
	 * @param array $item Nav Menu Link Item.
	 * @param array $args Nav Menu Link Args.
	 * @param int   $depth Nav Menu Link Depth.
	 *
	 * @return array
	 */
	public function add_nav_menu_link_atts( $atts, $item, $args, $depth ) {
		global $currency_switcher_options; // Define global variable.
		$is_geo = 'on' === ( $currency_switcher_options['enable_geolocation'] ?? 'off' );

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( empty( $_GET['currency'] ) || ! $is_geo ) {
			return $atts;
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$currency = sanitize_text_field( wp_unslash( $_GET['currency'] ) );
		if ( Fns::getSession( 'rtsb_currency' ) !== $currency ) {
			$atts['href'] = remove_query_arg( 'currency', $atts['href'] );
			return $atts;
		}
		$atts['href'] =
			add_query_arg(
				[
					'currency' => $currency,
				],
				$atts['href']
			);
		return $atts;
	}
	/**
	 * @param array $available_gateways Gateways List.
	 * @return array
	 */
	public function available_payment_gateways( $available_gateways ) {
		$gateway = $this->get_available_gateway();
		if ( count( $gateway ) ) {
			$available_gateways = array_intersect_key( $available_gateways, array_flip( $gateway ) );
		}
		return $available_gateways;
	}

	/**
	 * Return the decimal separator for prices.
	 *
	 * @param string $separator separator.
	 * @since  2.3
	 * @return string
	 */
	public function price_decimal_separator( $separator ) {
		if ( '' !== ( $this->settings['decimal_separator'] ?? '' ) ) {
			$separator = stripslashes( $this->settings['decimal_separator'] );
		}
		return $separator;
	}

	/**
	 * Return the thousand separator for prices.
	 *
	 * @param string $separator separator.
	 * @return string
	 * @since  2.3
	 */
	public function price_thousand_separator( $separator ) {
		if ( '' !== ( $this->settings['thousand_separator'] ?? '' ) ) {
			$separator = stripslashes( $this->settings['thousand_separator'] );
		}
		return $separator;
	}
	/**
	 * Number Of Decimals.
	 *
	 * @param int $decimal number value.
	 * @return int
	 */
	public function set_decimals( $decimal ) {
		if ( '' !== ( $this->settings['number_of_decimals'] ?? '' ) ) {
			$decimal = absint( $this->settings['number_of_decimals'] );
		}
		return (int) $decimal;
	}

	/**
	 * Currency Position.
	 *
	 * @param string $pre position.
	 * @return string|bool
	 */
	public function currency_position( $pre ) {
		if ( ! empty( $this->settings['currency_position'] ) ) {
			$pre = $this->settings['currency_position'];
		}
		return $pre;
	}
	/**
	 * Re-calculate After Switch Currency.
	 *
	 * @param array $fragments fragments.
	 *
	 * @return array
	 */
	public function recalculate_cart_after_currency_change( $fragments ) {
		if ( Fns::getSession( 'rtsb_currency_changed' ) ) {
			WC()->cart->calculate_totals();
			Fns::removeSession( 'rtsb_currency_changed' );
		}
		return $fragments;
	}
	/**
	 * @return void
	 */
	public function enqueue_public_scripts() {
		global $currency_switcher_options;

		// Enqueue assets.
		$this->handle = Fns::enqueue_module_assets(
			$this->handle,
			'currency-switcher',
			[
				'context' => rtsbpro(),
				'version' => RTSBPRO_VERSION,
				'type'    => 'css',
			]
		);

		$dynamic_css = '';

		if ( ! empty( $currency_switcher_options['currency_text_color'] ) ) {
			$dynamic_css .= '.switcher-list a{';
			$dynamic_css .= 'color:' . $currency_switcher_options['currency_text_color'] . ';';
			$dynamic_css .= '}';
		}
		if ( ! empty( $currency_switcher_options['currency_text_font_size'] ) ) {
			$dynamic_css .= '.switcher-list a{';
			$dynamic_css .= 'font-size:' . $currency_switcher_options['currency_text_font_size'] . ';';
			$dynamic_css .= '}';
		}

		if ( ! empty( $currency_switcher_options['currency_active_text_color'] ) ) {
			$dynamic_css .= '.switcher-list a:hover, .switcher-list a.currency-selected{';
			$dynamic_css .= 'color:' . $currency_switcher_options['currency_active_text_color'] . ';';
			$dynamic_css .= '}';
		}

		if ( ! empty( $currency_switcher_options['switcher_bg_color'] ) ) {
			$dynamic_css .= '.switcher-list .switcher-btn-wrapper{';
			$dynamic_css .= 'background:' . $currency_switcher_options['switcher_bg_color'] . ';';
			$dynamic_css .= '}';
		}

		if ( ! empty( $currency_switcher_options['currency_label_font_size'] ) ) {
			$dynamic_css .= '.switcher-list .placeholder-label-text{';
			$dynamic_css .= 'font-size:' . $currency_switcher_options['currency_label_font_size'] . ';';
			$dynamic_css .= '}';
		}

		if ( ! empty( $currency_switcher_options['currency_icon_color'] ) ) {
			$dynamic_css .= '.switcher-list .switcher-placeholder-icon{';
			$dynamic_css .= 'color:' . $currency_switcher_options['currency_icon_color'] . ';';
			$dynamic_css .= '}';
		}
		if ( ! empty( $currency_switcher_options['currency_icon_font_size'] ) ) {
			$dynamic_css .= '.switcher-list .switcher-placeholder-icon{';
			$dynamic_css .= 'font-size:' . $currency_switcher_options['currency_icon_font_size'] . ';';
			$dynamic_css .= '}';
		}

		if ( ! empty( $currency_switcher_options['currency_icon_bg_color'] ) ) {
			$dynamic_css .= '.switcher-list .switcher-toggle-switch.active,.switcher-list .switcher-placeholder-icon{';
			$dynamic_css .= 'background:' . $currency_switcher_options['currency_icon_bg_color'] . ';';
			$dynamic_css .= '}';
		}

		if ( ! empty( $currency_switcher_options['currency_label_bg_color'] ) ) {
			$dynamic_css .= '.switcher-list .switcher-placeholder{';
			$dynamic_css .= 'background:' . $currency_switcher_options['currency_label_bg_color'] . ';';
			$dynamic_css .= '}';
		}
		if ( ! empty( $currency_switcher_options['currency_label_text_color'] ) ) {
			$dynamic_css .= '.switcher-list .placeholder-label-text{';
			$dynamic_css .= 'color:' . $currency_switcher_options['currency_label_text_color'] . ';';
			$dynamic_css .= '}';
		}

		if ( ! empty( $dynamic_css ) ) {
			wp_add_inline_style( $this->handle, $dynamic_css );
		}
	}

	/**
	 * Get the pages.
	 *
	 * @param string $key Key.
	 *
	 * @return array
	 */
	private function get_the_value( $key ) {
		$pages = [];
		global $currency_switcher_options;
		if ( ! empty( $currency_switcher_options[ $key ] ) && is_array( $currency_switcher_options[ $key ] ) ) {
			foreach ( $currency_switcher_options[ $key ] as $val ) {
				$item    = json_decode( stripslashes( $val ), true );
				$pages[] = $item['value'];
			}
		}
		return $pages;
	}


	/**
	 * @return bool
	 */
	private function is_show_sticky_currency_swatch() {
		global $currency_switcher_options;
		if ( 'on' !== ( $currency_switcher_options['show_sticky_currency_widget'] ?? '' ) ) {
			return false;
		}
		return true;

		/* // phpcs:ignore Squiz.PHP.CommentedOutCode.Found
		$object_id = get_queried_object_id();
		if ( is_shop() ) {
			// Get the shop page ID.
			$object_id = wc_get_page_id( 'shop' );
			// Use $shop_page_id as needed.
		}
		$include_page = $this->get_the_value( 'switcher_btn_include_page' );
		if ( ! empty( $include_page ) && ! in_array( $object_id, $include_page, true ) ) {
			return false;
		}
		$exclude_page = $this->get_the_value( 'switcher_btn_exclude_page' );
		if ( ! empty( $exclude_page ) && in_array( $object_id, $exclude_page, true ) ) {
			return false;
		}
		if ( is_archive() ) {
			$name            = get_queried_object()->name;
			$include_archive = $this->get_the_value( 'switcher_btn_include_archive' );

			if ( ! empty( $include_archive ) && ! in_array( $name, $include_archive, true ) ) {
				return false;
			}
		}
		if ( is_single() ) {
			$name          = get_post_type( $object_id );
			$details_pages = $this->get_the_value( 'switcher_btn_include_details_page' );
			if ( ! empty( $details_pages ) && ! in_array( $name, $details_pages, true ) ) {
				return false;
			}
		}
		return true;
		*/
	}

	/**
	 * Currency Switcher Shortcode.
	 *
	 * @param array $atts Shortcode attributes.
	 *
	 * @return string
	 */
	public function currency_switcher_shortcode( $atts ) {
		if ( empty( $this->get_available_currency() ) ) {
			return;
		}
		// Parse shortcode attributes and set defaults.
		$atts = shortcode_atts(
			[
				'position'           => 'default', // default position if no attribute is provided.
				'switcher_preset'    => 'preset1',
				'show_flag'          => 'show',
				'show_currency_text' => 'show',
			],
			$atts,
			'currency_switcher'
		);

		if ( 'show' !== ( $atts['show_flag'] ?? '' ) && 'show' !== ( $atts['show_currency_text'] ) ) {
			return;
		}

		$switcher_position = $atts['position'] ?? 'default';
		$switcher_preset   = $atts['switcher_preset'] ?? 'preset1';

		global $wp;
		$root_url = home_url( add_query_arg( [], $wp->request ) );

		ob_start();

		?>
		<div class="switcher-list switcher-position-<?php echo esc_attr( $switcher_position . ' layout-' . $switcher_preset ); ?>">
			<?php if ( 'preset2' === $switcher_preset && 'default' !== $switcher_position ) { ?>
			   <div class="switcher-placeholder">
					<span class="placeholder-label-text"> <?php echo esc_html__( 'Currency', 'shopbuilder-pro' ); ?> </span>
					<div class="switcher-placeholder-icon"> <?php Fns::print_html( $this->get_currency_symbol( $this->get_current_currency() ) ); ?> </div>
			   </div>
			<?php } ?>

			<?php if ( 'default' == $switcher_position ) { ?>
				<div class="switcher-using-shortcode">
					<?php
					wp_enqueue_style( 'dashicons' );

					$key      = $this->get_current_currency();
					$country  = CsFns::get_country_or_currency( $key, 'country' );
					$path     = RTSBPRO_PATH . '/assets/images/flags/' . strtolower( $country ) . '.svg';
					$currency = $this->get_available_currency();
					$flag_url = rtsbpro()->get_assets_uri( 'images/flags/' . strtolower( $country ) . '.svg' );
					$val      = $currency[ $key ] ?? [];
					$data     = [
						'template'           => 'currency-switcher/' . $switcher_preset,
						'class'              => 'currency-selected',
						'path'               => $path,
						'url'                => '#',
						'currency'           => $key,
						'currency_info'      => $val,
						'flag_url'           => $flag_url,
						'is_active'          => $key === $this->get_current_currency(),
						'show_flag'          => ! empty( $atts['show_flag'] ) && 'show' === $atts['show_flag'],
						'show_currency_text' => ! empty( $atts['show_currency_text'] ) && 'show' === $atts['show_currency_text'],
					];
					Fns::load_template( $data['template'], $data, false, '', rtsbpro()->get_plugin_template_path() );
					?>
					<span class="dashicons dashicons-arrow-down-alt2"></span>
				</div>
			<?php } ?>
			<div class="switcher-btn-wrapper">
				<?php
				$nonce = wp_create_nonce( rtsb()->nonceText );
				foreach ( $this->get_available_currency() as $key => $val ) {
					$class   = $key === $this->get_current_currency() ? 'currency-selected' : 'currency-no-selected';
					$country = CsFns::get_country_or_currency( $key, 'country' );
					$url     =
						add_query_arg(
							[
								'currency'      => $key,
								rtsb()->nonceId => $nonce,
							],
							$root_url
						);

					$path     = RTSBPRO_PATH . '/assets/images/flags/' . strtolower( $country ) . '.svg';
					$flag_url = rtsbpro()->get_assets_uri( 'images/flags/' . strtolower( $country ) . '.svg' );
					$data     = [
						'template'           => 'currency-switcher/' . $switcher_preset,
						'class'              => $class,
						'path'               => $path,
						'url'                => $url,
						'currency'           => $key,
						'currency_info'      => $val,
						'flag_url'           => $flag_url,
						'is_active'          => $key === $this->get_current_currency(),
						'show_flag'          => ! empty( $atts['show_flag'] ) && 'show' === $atts['show_flag'],
						'show_currency_text' => ! empty( $atts['show_currency_text'] ) && 'show' === $atts['show_currency_text'],
					];
					Fns::load_template( $data['template'], $data, false, '', rtsbpro()->get_plugin_template_path() );
				}
				?>
			</div>
		</div>
		<?php
		return ob_get_clean();
	}

	/**
	 * Set Currency
	 *
	 * @return void
	 */
	public function currency_switcher_footer_script() {
		if ( ! $this->is_show_sticky_currency_swatch() ) {
			return;
		}
		global $currency_switcher_options;
		$switcher_position  = $currency_switcher_options['currency_switcher_position'] ?? 'default';
		$switcher_preset    = $currency_switcher_options['switcher_preset'] ?? 'preset1';
		$show_flag          = ! empty( $currency_switcher_options['enable_currency_flag'] ) && 'on' === $currency_switcher_options['enable_currency_flag'] ? 'show' : '';
		$show_currency_text = ! empty( $currency_switcher_options['enable_currency_text'] ) && 'on' === $currency_switcher_options['enable_currency_text'] ? 'show' : '';
		echo do_shortcode( '[currency_switcher position="' . $switcher_position . '" switcher_preset="' . $switcher_preset . '" show_flag="' . $show_flag . '" show_currency_text="' . $show_currency_text . '" /]' );
	}
	/**
	 * Set Currency
	 *
	 * @param array $currency currency settings.
	 * @return bool
	 */
	public function set_woocommerce_currency( $currency ) {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$curr = sanitize_text_field( wp_unslash( $_GET['currency'] ?? '' ) );
		if ( 'reset' === $curr ) {
			 Fns::removeSession( 'rtsb_currency' );
		}

		if ( $this->is_available_currency() ) {
			$currency = $this->get_current_currency();
		}
		return $currency;
	}

	/**
	 * @param string $currency_symbol icon.
	 * @param string $currency Currency code.
	 * @return string
	 */
	public function set_currency_symbol( $currency_symbol, $currency ) {
		return $this->get_currency_symbol( $currency ) ?? $currency_symbol;
	}
}
