<?php
/**
 * Customize My Account Functions Class.
 *
 * @package Rse\SB
 */

namespace RadiusTheme\SBPRO\Modules\CustomizeMyAccount;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Functions Class.
 */
class CustomizeMyAccountFns {
	/**
	 * Returns an array of settings fields.
	 *
	 * @return array
	 */
	public static function settings_field() {
		return [
			'endpoint_field_intro'         => [
				'id'   => 'endpoint_field_intro',
				'type' => 'description',
				'text' => esc_html__( 'Below is a list of all My Account endpoints available for customization. You can customize these endpoints, including their slugs, icons, and content. Additionally, you can rearrange the order in which they are displayed on the My Account page. You can create custom endpoints or links by clicking the \'Add New Endpoint / Link\' button at the bottom.', 'shopbuilder-pro' ),
				'tab'  => 'endpoints',
			],
			'endpoint_field'               => [
				'id'          => 'endpoint_field',
				'type'        => 'repeaters',
				'buttonLabel' => esc_html__( 'Add New Endpoint / Link', 'shopbuilder-pro' ),
				'tab'         => 'endpoints',
				'defaults'    => self::get_default_options(),
				'repeat'      => [
					'menu_visibility'  => [
						'id'    => 'menu_visibility',
						'type'  => 'switch',
						'label' => esc_html__( 'Show in Navigation Menu', 'shopbuilder-pro' ),
						'help'  => esc_html__( 'Enable to show this endpoint in My Account Navigation Menu', 'shopbuilder-pro' ),
						'value' => 'on',
					],
					'repeater_type'    => [
						'id'      => 'repeater_type',
						'label'   => esc_html__( 'Endpoint Type', 'shopbuilder-pro' ),
						'help'    => esc_html__( 'Select endpoint type.', 'shopbuilder-pro' ),
						'type'    => 'select',
						'value'   => 'endpoint',
						'options' => [
							'endpoint' => esc_html__( 'Endpoint', 'shopbuilder-pro' ),
							'url'      => esc_html__( 'Custom Link', 'shopbuilder-pro' ),
						],
					],
					'title'            => [
						'id'          => 'title',
						'label'       => esc_html__( 'Endpoint Label', 'shopbuilder-pro' ),
						'help'        => esc_html__( 'Enter endpoint label.', 'shopbuilder-pro' ),
						'type'        => 'text',
						'placeholder' => esc_html__( 'My Custom Endpoint', 'shopbuilder-pro' ),
						'value'       => esc_html__( 'My Custom Endpoint', 'shopbuilder-pro' ),
					],
					'slug'             => [
						'id'          => 'slug',
						'label'       => esc_html__( 'Endpoint Slug', 'shopbuilder-pro' ),
						'help'        => esc_html__( 'Enter a unique endpoint slug.', 'shopbuilder-pro' ),
						'type'        => 'text',
						'placeholder' => esc_html__( 'my-custom-endpoint', 'shopbuilder-pro' ),
						'value'       => esc_html__( 'my-custom-endpoint', 'shopbuilder-pro' ),
						'dependency'  => [
							'rules' => [
								[
									'item'     => 'modules.customize_my_account.endpoint_field.repeater_type',
									'value'    => 'endpoint',
									'operator' => '==',
								],
							],
						],
					],
					'link'             => [
						'id'          => 'link',
						'label'       => esc_html__( 'Endpoint Custom URL', 'shopbuilder-pro' ),
						'help'        => esc_html__( 'Enter endpoint custom link.', 'shopbuilder-pro' ),
						'type'        => 'text',
						'placeholder' => esc_html__( 'https://example.com', 'shopbuilder-pro' ),
						'dependency'  => [
							'rules' => [
								[
									'item'     => 'modules.customize_my_account.endpoint_field.repeater_type',
									'value'    => 'url',
									'operator' => '==',
								],
							],
						],
					],
					'new_tab'          => [
						'id'         => 'new_tab',
						'type'       => 'switch',
						'label'      => esc_html__( 'Open in New Tab?', 'shopbuilder-pro' ),
						'help'       => esc_html__( 'Enable to open the custom link in a new tab.', 'shopbuilder-pro' ),
						'dependency' => [
							'rules' => [
								[
									'item'     => 'modules.customize_my_account.endpoint_field.repeater_type',
									'value'    => 'url',
									'operator' => '==',
								],
							],
						],
					],
					'icon'             => [
						'id'      => 'icon',
						'label'   => esc_html__( 'Endpoint Icon', 'shopbuilder-pro' ),
						'help'    => esc_html__( 'Please select the endpoint icon.', 'shopbuilder-pro' ),
						'type'    => 'select',
						'value'   => 'none',
						'options' => [
							'none'   => esc_html__( 'None', 'shopbuilder-pro' ),
							'custom' => esc_html__( 'Select Icon', 'shopbuilder-pro' ),
							'upload' => esc_html__( 'Upload Icon', 'shopbuilder-pro' ),
						],
					],
					'custom_icon'      => [
						'id'         => 'custom_icon',
						'label'      => esc_html__( 'Select Endpoint Icon', 'shopbuilder-pro' ),
						'help'       => esc_html__( 'Please choose the endpoint custom icon.', 'shopbuilder-pro' ),
						'type'       => 'icon_select',
						'searchable' => true,
						'options'    => self::get_icons(),
						'dependency' => [
							'rules' => [
								[
									'item'     => 'modules.customize_my_account.endpoint_field.icon',
									'value'    => 'custom',
									'operator' => '==',
								],
							],
						],
					],
					'image_icon'       => [
						'id'         => 'image_icon',
						'type'       => 'fileupload',
						'label'      => esc_html__( 'Upload Custom Icon', 'shopbuilder-pro' ),
						'help'       => esc_html__( 'Please upload custom icon.', 'shopbuilder-pro' ),
						'dependency' => [
							'rules' => [
								[
									'item'     => 'modules.customize_my_account.endpoint_field.icon',
									'value'    => 'upload',
									'operator' => '==',
								],
							],
						],
					],
					'visibility'       => [
						'id'      => 'visibility',
						'label'   => esc_html__( 'Endpoint Visibility', 'shopbuilder-pro' ),
						'help'    => esc_html__( 'Please choose whether to show this endpoint to all users or only to specific users.', 'shopbuilder-pro' ),
						'type'    => 'select',
						'value'   => 'all',
						'options' => [
							'all'      => esc_html__( 'All users', 'shopbuilder-pro' ),
							'specific' => esc_html__( 'Only users with a specific role', 'shopbuilder-pro' ),
						],
					],
					'roles'            => [
						'id'         => 'roles',
						'label'      => esc_html__( 'Select User Roles', 'shopbuilder-pro' ),
						'help'       => esc_html__( 'Please choose user roles to show this endpoint.', 'shopbuilder-pro' ),
						'type'       => 'select',
						'value'      => [ 'administrator', 'customer' ],
						'options'    => FnsPro::get_all_user_roles(),
						'multiple'   => true,
						'dependency' => [
							'rules' => [
								[
									'item'     => 'modules.customize_my_account.endpoint_field.visibility',
									'value'    => 'specific',
									'operator' => '==',
								],
							],
						],
					],
					'content'          => [
						'id'          => 'content',
						'label'       => esc_html__( 'Endpoint Content', 'shopbuilder-pro' ),
						'help'        => esc_html__( 'Enter endpoint content.', 'shopbuilder-pro' ),
						'type'        => 'text_editor',
						'placeholder' => esc_html__( 'Endpoint Content', 'shopbuilder-pro' ),
						'sanitize_fn' => 'pass_all',
						'dependency'  => [
							'rules' => [
								[
									'item'     => 'modules.customize_my_account.endpoint_field.repeater_type',
									'value'    => 'endpoint',
									'operator' => '==',
								],
							],
						],
					],
					'content_position' => [
						'id'         => 'content_position',
						'label'      => esc_html__( 'Choose Content Position', 'shopbuilder-pro' ),
						'help'       => esc_html__( 'Please choose where you want to display the endpoint content.', 'shopbuilder-pro' ),
						'type'       => 'select',
						'value'      => 'replace',
						'options'    => [
							'prepend' => esc_html__( 'Before Default Content', 'shopbuilder-pro' ),
							'append'  => esc_html__( 'After Default Content', 'shopbuilder-pro' ),
							'replace' => esc_html__( 'Replace Default Content', 'shopbuilder-pro' ),
						],
						'dependency' => [
							'rules' => [
								[
									'item'     => 'modules.customize_my_account.endpoint_field.uniqueid',
									'value'    => [
										'dashboard',
										'orders',
										'downloads',
										'edit-address',
										'payment-methods',
										'edit-account',
									],
									'operator' => 'in',
								],
							],
						],
					],
				],
			],
			'endpoint_font_intro'          => [
				'id'    => 'endpoint_styles_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Menu Items Font', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'my_acc_nav_font_size'         => [
				'id'    => 'my_acc_nav_font_size',
				'type'  => 'text',
				'label' => esc_html__( 'Font Size', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter font size. ex. 16px', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'my_acc_nav_font_weight'       => [
				'id'      => 'my_acc_nav_font_weight',
				'type'    => 'select',
				'options' => [
					100 => esc_html__( '100 (Thin)', 'shopbuilder-pro' ),
					200 => esc_html__( '200 (Extra Light)', 'shopbuilder-pro' ),
					300 => esc_html__( '300 (Light)', 'shopbuilder-pro' ),
					400 => esc_html__( '400 (Normal)', 'shopbuilder-pro' ),
					500 => esc_html__( '500 (Medium)', 'shopbuilder-pro' ),
					600 => esc_html__( '600 (Semi Bold)', 'shopbuilder-pro' ),
					700 => esc_html__( '700 (Bold)', 'shopbuilder-pro' ),
					800 => esc_html__( '800 (Extra Bold)', 'shopbuilder-pro' ),
					900 => esc_html__( '900 (Black)', 'shopbuilder-pro' ),
				],
				'value'   => 400,
				'label'   => esc_html__( 'Font Weight', 'shopbuilder-pro' ),
				'help'    => esc_html__( 'Choose font weight.', 'shopbuilder-pro' ),
				'tab'     => 'styles',
			],
			'my_acc_nav_icon_size'         => [
				'id'    => 'my_acc_nav_icon_size',
				'type'  => 'text',
				'label' => esc_html__( 'Icon Size', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter icon size. ex. 16px', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'endpoint_color_intro'         => [
				'id'    => 'endpoint_styles_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Menu Item Colors', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'my_acc_nav_text_color'        => [
				'id'    => 'my_acc_nav_text_color',
				'label' => esc_html__( 'Item Text Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose color for nav menu text', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'my_acc_nav_bg_color'          => [
				'id'    => 'my_acc_nav_bg_color',
				'label' => esc_html__( 'Item Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose background color for nav menu text', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'my_acc_nav_icon_color'        => [
				'id'    => 'my_acc_nav_icon_color',
				'label' => esc_html__( 'Item Icon Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose icon color for nav menu', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'my_acc_nav_hover_color'       => [
				'id'    => 'my_acc_nav_hover_color',
				'label' => esc_html__( 'Item Hover Text Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose hover color for nav menu text', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'my_acc_nav_hover_bg_color'    => [
				'id'    => 'my_acc_nav_hover_bg_color',
				'label' => esc_html__( 'Item Hover Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose hover background color for nav menu', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'my_acc_nav_hover_icon_color'  => [
				'id'    => 'my_acc_nav_hover_icon_color',
				'label' => esc_html__( 'Item Hover Icon Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose hover icon color for nav menu', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'my_acc_nav_active_color'      => [
				'id'    => 'my_acc_nav_active_color',
				'label' => esc_html__( 'Active Item Text Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose active color for nav menu text', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'my_acc_nav_active_bg_color'   => [
				'id'    => 'my_acc_nav_active_bg_color',
				'label' => esc_html__( 'Active Item Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose active background color for nav menu', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'my_acc_nav_active_icon_color' => [
				'id'    => 'my_acc_nav_active_icon_color',
				'label' => esc_html__( 'Active Item Icon Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose active icon color for nav menu', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'endpoint_spacing_intro'       => [
				'id'    => 'endpoint_spacing_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Menu Items Border and Spacings', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'my_acc_nav_border'            => [
				'id'    => 'my_acc_nav_border',
				'type'  => 'text',
				'label' => esc_html__( 'Item Border Width', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter border width. Ex. 10px', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'my_acc_nav_border_color'      => [
				'id'    => 'my_acc_nav_border_color',
				'label' => esc_html__( 'Item Border Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose nav menu border color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'my_acc_nav_padding'           => [
				'id'    => 'my_acc_nav_padding',
				'type'  => 'text',
				'label' => esc_html__( 'Item Padding', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Padding. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'my_acc_nav_icon_spacing'      => [
				'id'    => 'my_acc_nav_icon_spacing',
				'type'  => 'text',
				'label' => esc_html__( 'Left Spacing for Icon', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Spacing. Ex. 10px', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'my_acc_nav_icon_padding'      => [
				'id'    => 'my_acc_nav_icon_padding',
				'type'  => 'text',
				'label' => esc_html__( 'Padding for Item with Icon', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Padding. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'my_acc_nav_margin'            => [
				'id'    => 'my_acc_nav_margin',
				'type'  => 'text',
				'label' => esc_html__( 'Item Margin', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Margin. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
		];
	}

	/**
	 * Retrieve an array of custom icons with their HTML representation.
	 *
	 * @return array
	 */
	public static function get_icons() {

		if ( method_exists( Fns::class, 'get_icons' ) ) {
			return Fns::get_icons();
		}
		// TODO:: Below Code Will Remove In the future.
		$icon        = Fns::get_custom_icon_names();
		$icons_array = [];
		foreach ( $icon as $value ) {
			$icons_array[ $value ] = '<i class="rtsb-icon rtsb-icon-' . $value . '"></i> <span class="icon-name">' . ucfirst( str_replace( '-', ' ', $value ) ) . '</span>';
		}
		return $icons_array;
	}

	/**
	 * Retrieve the default endpoints and their titles.
	 *
	 * @return array
	 */
	public static function get_default_endpoints() {
		$endpoints = [
			'orders'          => get_option( 'woocommerce_myaccount_orders_endpoint', 'orders' ),
			'downloads'       => get_option( 'woocommerce_myaccount_downloads_endpoint', 'downloads' ),
			'edit-address'    => get_option( 'woocommerce_myaccount_edit_address_endpoint', 'edit-address' ),
			'payment-methods' => get_option( 'woocommerce_myaccount_payment_methods_endpoint', 'payment-methods' ),
			'edit-account'    => get_option( 'woocommerce_myaccount_edit_account_endpoint', 'edit-account' ),
			'customer-logout' => get_option( 'woocommerce_logout_endpoint', 'customer-logout' ),
		];

		$items = [
			'dashboard'       => esc_html__( 'Dashboard', 'shopbuilder-pro' ),
			'orders'          => esc_html__( 'Orders', 'shopbuilder-pro' ),
			'downloads'       => esc_html__( 'Downloads', 'shopbuilder-pro' ),
			'edit-address'    => _n( 'Address', 'Addresses', ( 1 + (int) wc_shipping_enabled() ), 'shopbuilder-pro' ),
			'payment-methods' => esc_html__( 'Payment methods', 'shopbuilder-pro' ),
			'edit-account'    => esc_html__( 'Account details', 'shopbuilder-pro' ),
			'customer-logout' => esc_html__( 'Log out', 'shopbuilder-pro' ),
		];

		foreach ( $endpoints as $endpoint_id => $endpoint ) {
			if ( empty( $endpoint ) ) {
				unset( $items[ $endpoint_id ] );
			}
		}

		if ( class_exists( 'WC_Subscriptions' ) ) {
			unset( $items['subscriptions'] );
		}

		$menu_items     = apply_filters( 'woocommerce_account_menu_items', $items, $endpoints );
		$formatted_data = [];

		foreach ( $menu_items as $endpoint_id => $title ) {
			$formatted_data[ $endpoint_id ] = [
				'title' => $title,
				'slug'  => $endpoints[ $endpoint_id ] ?? '',
			];
		}

		return $formatted_data;
	}

	/**
	 * Retrieve the title for a specific endpoint.
	 *
	 * @param string $endpoint The endpoint identifier.
	 * @return string
	 */
	public static function get_endpoint_title( $endpoint ) {
		$items = self::get_default_endpoints();

		return $items[ $endpoint ]['title'] ?? '';
	}

	/**
	 * Retrieve the default options for endpoints.
	 *
	 * @return array
	 */
	public static function get_default_options() {
		$common_fields = [
			'menu_visibility'  => 'on',
			'deletable'        => 'no',
			'icon'             => 'none',
			'custom_icon'      => '',
			'image_icon'       => '',
			'visibility'       => 'all',
			'roles'            => [],
			'content'          => '',
			'content_position' => 'replace',
		];

		$default_options = [
			'dashboard' => [
				'uniqueid' => 'dashboard',
				'title'    => self::get_endpoint_title( 'dashboard' ) ?: esc_html__( 'Dashboard', 'shopbuilder-pro' ),
			] + $common_fields,
		];

		$default_endpoints = self::get_default_endpoints();

		foreach ( $default_endpoints as $endpoint => $data ) {
			if ( ! empty( $data['slug'] ) ) {
				$endpoint_options = [ 'uniqueid' => $endpoint ] + $data + $common_fields;

				if ( 'customer-logout' === $endpoint ) {
					unset( $endpoint_options['content'], $endpoint_options['content_position'] );
				}

				$default_options[ $endpoint ] = $endpoint_options;
			}
		}

		return $default_options;
	}

	/**
	 * Generates HTML markup for displaying an endpoint icon.
	 *
	 * @param array   $data     The endpoint data containing icon information.
	 * @param string  $endpoint The endpoint identifier.
	 * @param boolean $wrapper  Whether to wrap the icon in a span element.
	 *
	 * @return string
	 */
	public static function get_icon_html( $data, $endpoint, $wrapper = true ) {
		if ( empty( $data ) || 'none' === $data['icon'] ) {
			return '';
		}

		$html = $wrapper ? '<span class="icon ' . esc_attr( $endpoint ) . '_icon">' : '';

		if ( 'custom' === $data['icon'] ) {
			$html .= ! empty( $data['custom_icon'] ) ? '<i class="rtsb-icon rtsb-icon-' . esc_attr( $data['custom_icon'] ) . '"></i>' : '';
		} else {
			$icon_html = '';
			$icon_url  = $data['image_icon']['source'] ?? '';
			$icon_id   = $data['image_icon']['id'] ?? 0;
			$extension = ! empty( $icon_url ) ? strtolower( substr( strrchr( $data['image_icon']['source'], '.' ), 1 ) ) : '';

			if ( 'svg' === $extension ) {
				$content = file_get_contents( $icon_url ); // phpcs:ignore WordPress.WP.AlternativeFunctions.file_get_contents_file_get_contents
				$is_svg  = strpos( $content, '<svg' );

				if ( false !== $is_svg ) {
					$icon_html = substr( $content, $is_svg );
				}
			} else {
				$attr = [
					'class' => 'rtsb-icon-img',
					'alt'   => esc_html( ucfirst( $endpoint ) ) . esc_html__( ' Icon', 'shopbuilder-pro' ),
				];

				$icon_html = wp_get_attachment_image( absint( $icon_id ), 'full', true, $attr );
			}

			$html .= $icon_html;
		}

		$html .= $wrapper ? '</span>' : '';

		return $html;
	}

	/**
	 * Retrieves custom endpoints for the My Account page.
	 *
	 * @return array
	 */
	public static function custom_endpoints() {
		return apply_filters( 'rtsb/global/my_account_custom_endpoints', [] );
	}

	/**
	 * Checks if custom endpoints are enabled for the My Account page.
	 *
	 * @return bool
	 */
	public static function is_custom_endpoint_enabled() {
		if ( ! self::custom_endpoints() ) {
			return false;
		}

		$endpoint_enabled = ! empty( self::custom_endpoints()['enabled'] ) && self::custom_endpoints()['enabled'];

		if ( $endpoint_enabled ) {
			return true;
		}

		return false;
	}

	/**
	 * Retrieve custom endpoints options.
	 *
	 * @return array
	 */
	public static function get_endpoints_options() {
		$raw_options      = Fns::get_options( 'modules', 'customize_my_account' );
		$repeater_options = [];
		$endpoint_fields  = [];
		$i                = 1;

		if ( 'on' === ( $raw_options['active'] ?? '' ) ) {
			$repeater_options = json_decode( $raw_options['endpoint_field'] ?? '', true );
		}

		foreach ( $repeater_options as $endpoint ) {
			$unique_id = $endpoint['uniqueid'] ?? null;

			if ( $unique_id ) {
				unset( $endpoint['uniqueid'] );
				$endpoint_fields[ $unique_id ] = $endpoint;

				if ( strpos( $unique_id, 'rtsb_repeater_item_' ) !== false ) {
					if ( 'url' === ( $endpoint['repeater_type'] ?? 'endpoint' ) ) {
						$slug = 'rtsb_custom_link_' . $i;
						$i++;
					} else {
						$slug = $endpoint['slug'] ?? 'rtsb-custom-ep';
					}

					$endpoint_fields[ $slug ] = $endpoint;

					unset( $endpoint_fields[ $unique_id ] );
				}
			}
		}

		return $endpoint_fields;
	}

	/**
	 * Get link attributes for a given endpoint.
	 *
	 * @param array  $data      The array containing endpoint data.
	 * @param string $endpoint The specific endpoint.
	 *
	 * @return array
	 */
	public static function get_link_attributes( $data, $endpoint ) {
		$endpoint_type = $data[ $endpoint ]['repeater_type'] ?? 'endpoint';
		$custom_link   = $data[ $endpoint ]['link'] ?? '';
		$open_new_tab  = $data[ $endpoint ]['new_tab'] ?? '';
		$link_target   = ! empty( $open_new_tab ) && 'on' === $open_new_tab ? ' target="_blank"' : '';
		$href          = 'url' === $endpoint_type ? $custom_link : wc_get_account_endpoint_url( $endpoint );

		return [
			'href'   => $href,
			'target' => $link_target,
		];
	}
}
