<?php
/**
 * Customize My Account Module Front End Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\CustomizeMyAccount;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Module Class.
 */
class CustomizeMyAccountFrontEnd {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Options.
	 *
	 * @var array|mixed
	 */
	private array $options;

	/**
	 * Cache.
	 *
	 * @var array
	 */
	private $cache = [];

	/**
	 * Endpoint title.
	 *
	 * @var array|mixed
	 */
	private array $endpoint_title = [];

	/**
	 * Endpoint content.
	 *
	 * @var string
	 */
	private string $endpoint_content = '';

	/**
	 * All Endpoints.
	 *
	 * @var array|mixed
	 */
	private array $all_endpoints;

	/**
	 * Custom Endpoints.
	 *
	 * @var array|mixed
	 */
	private array $custom_endpoints = [];

	/**
	 * Asset Handle
	 *
	 * @var string
	 */
	private $handle = 'rtsb-customize-my-account';

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		/**
		 * Get the options.
		 */
		$this->options       = Fns::get_options( 'modules', 'customize_my_account' );
		$this->all_endpoints = $this->get_all_endpoints_data();

		// Custom endpoints.
		$this->add_custom_endpoints();

		// Change endpoint title.
		$this->change_endpoint_title();

		// Change endpoint slug.
		$this->change_endpoint_slug();
		$this->detect_wc_options_change();

		// Change navigation template.
		$this->change_navigation();

		// Endpoint Content.
		$this->endpoint_content();

		// Styles.
		$this->dynamic_styles();
	}

	/**
	 * Change the titles of My Account page endpoints based on settings.
	 *
	 * @return void
	 */
	public function change_endpoint_title() {
		add_filter( 'woocommerce_account_menu_items', [ $this, 'change_menu_title' ], 999 );
		add_action( 'template_redirect', [ $this, 'change_page_title' ] );
	}

	/**
	 * Change endpoint slug.
	 *
	 * @return void
	 */
	public function change_endpoint_slug() {
		$endpoint_slugs = $this->detect_endpoint_changes( 'slug' );

		if ( ! empty( $endpoint_slugs ) ) {
			foreach ( $endpoint_slugs as $endpoint_id => $slug ) {
				$this->update_endpoint_slug( $endpoint_id, $slug );
			}

			$this->need_rewrite_flush();
		}

		Fns::maybe_flush_rewrite_rules();
	}

	/**
	 * Sets up action hook to check WooCommerce options change.
	 *
	 * @return void
	 */
	public function detect_wc_options_change() {
		add_action( 'woocommerce_update_options_advanced', [ $this, 'woocommerce_endpoint_settings' ], 10, 2 );
	}

	/**
	 * Updates WooCommerce endpoint settings based on user input.
	 *
	 * @return void
	 */
	public function woocommerce_endpoint_settings() {
		$default_endpoints = array_merge(
			array_keys( wc_get_account_menu_items() ),
			[
				'payment-methods',
			]
		);

		foreach ( $default_endpoints as $endpoint ) {
			$option_name = 'woocommerce_myaccount_' . str_replace( '-', '_', $endpoint ) . '_endpoint';

			if ( 'customer-logout' === $endpoint ) {
				$option_name = 'woocommerce_logout_endpoint';
			}

			// phpcs:ignore WordPress.Security.NonceVerification.Missing
			if ( ! empty( $_POST[ $option_name ] ) ) {
				// phpcs:ignore WordPress.Security.NonceVerification.Missing
				$slug = sanitize_title( wp_unslash( $_POST[ $option_name ] ) );
				update_option( $option_name, $slug );

				FnsPro::update_repeater_options(
					'modules',
					'customize_my_account',
					'endpoint_field',
					$endpoint,
					'slug',
					$slug,
				);
			}
		}
	}

	/**
	 * Changes the navigation for the WooCommerce account page.
	 *
	 * @return void
	 */
	public function change_navigation() {
		if ( FnsPro::is_my_account_page() ) {
			return;
		}

		// Remove WooCommerce default account navigation.
		$priority = has_action( 'woocommerce_account_navigation', 'woocommerce_account_navigation' );
		remove_action( 'woocommerce_account_navigation', 'woocommerce_account_navigation', $priority );

		// Add custom navigation template.
		add_action( 'woocommerce_account_navigation', [ $this, 'load_navigation_template' ] );
	}

	/**
	 * Endpoint content.
	 *
	 * @return void
	 */
	public function endpoint_content() {
		$displayed = $this->get_displayed_endpoint();

		if ( empty( $displayed ) ) {
			return;
		}

		$this->endpoint_content = $this->all_endpoints[ $displayed ]['content'] ?? '';
		$content_position       = $this->all_endpoints[ $displayed ]['content_position'] ?? 'replace';

		if ( empty( $this->endpoint_content ) ) {
			return;
		}

		if ( 'prepend' === $content_position ) {
			add_action( 'woocommerce_account_content', [ $this, 'display_endpoint_content' ], 6 );
		} elseif ( 'append' === $content_position ) {
			add_action( 'woocommerce_account_content', [ $this, 'display_endpoint_content' ], 18 );
		} else {
			remove_action( 'woocommerce_account_content', 'woocommerce_account_content' );
			add_action( 'woocommerce_account_content', [ $this, 'display_endpoint_content' ] );
		}
	}

	/**
	 * Dynamic Styles.
	 *
	 * @return void
	 */
	public function dynamic_styles() {
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_public_scripts' ], 99 );
	}

	/**
	 * Adds custom endpoints for the My Account page.
	 *
	 * @return void
	 */
	public function add_custom_endpoints() {
		add_action( 'init', [ $this, 'add_rewrite_endpoint' ], 99 );
		add_filter( 'rtsb/global/my_account_custom_endpoints', [ $this, 'custom_endpoints' ] );
		add_filter( 'woocommerce_get_query_vars', [ $this, 'add_query_vars' ], 0 );
	}

	/**
	 * Add custom query vars for custom endpoints.
	 *
	 * @param array $vars Default woocommerce query vars.
	 *
	 * @return array
	 */
	public function add_query_vars( $vars ) {
		if ( empty( $this->custom_endpoints ) ) {
			return $vars;
		}

		foreach ( $this->custom_endpoints as $custom_ep => $custom_ep_data ) {
			$vars[ $custom_ep ] = $custom_ep_data['slug'];
		}

		return $vars;
	}

	/**
	 * Add custom rewrite endpoints.
	 *
	 * @return void
	 */
	public function add_rewrite_endpoint() {
		if ( empty( $this->custom_endpoints ) ) {
			return;
		}

		$wc_endpoint_mask = WC()->query->get_endpoints_mask();
		$rewrite          = false;

		foreach ( $this->custom_endpoints as $custom_ep => $custom_ep_data ) {
			if ( 'url' === ( $custom_ep_data['repeater_type'] ?? 'endpoint' ) ) {
				continue;
			}

			$slug    = $custom_ep_data['slug'] ?? $custom_ep;
			$rewrite = true;

			add_rewrite_endpoint( $slug, $wc_endpoint_mask );
		}

		if ( $rewrite ) {
			$this->need_rewrite_flush();
		}
	}

	/**
	 * Retrieve custom endpoints data.
	 *
	 * @return array
	 */
	public function custom_endpoints() {
		if ( empty( $this->custom_endpoints ) ) {
			return [];
		}

		$endpoints = [];

		foreach ( $this->custom_endpoints as $custom_ep => $custom_ep_data ) {
			$endpoints['endpoints'][ $custom_ep ]      = $custom_ep_data['title'];
			$endpoints['endpoints_data'][ $custom_ep ] = $custom_ep_data;
		}

		if ( ! empty( $endpoints ) ) {
			$endpoints['enabled'] = true;
		}

		return $endpoints;
	}

	/**
	 * Loads the custom navigation template for the WooCommerce account page.
	 *
	 * @return void
	 */
	public function load_navigation_template() {
		$args = [
			'template' => 'myaccount/account-navigation',
			'data'     => $this->all_endpoints,
		];

		// Load custom template.
		Fns::load_template( $args['template'], $args, false, '', rtsbpro()->get_plugin_template_path() );
	}

	/**
	 * Change the titles of menu items.
	 *
	 * @param array $items The original menu items.
	 *
	 * @return array
	 */
	public function change_menu_title( $items ) {
		$sorted_items = [];

		if ( empty( $this->all_endpoints ) && empty( $this->custom_endpoints ) ) {
			return $items;
		}

		foreach ( $this->custom_endpoints as $custom_endpoint_id => $custom_ep_data ) {
			$items[ $custom_endpoint_id ] = esc_html( $custom_ep_data['title'] );
		}

		foreach ( $this->all_endpoints as $endpoint_id => $data ) {
			$this->endpoint_title[ $endpoint_id ] = $data['title'];

			if ( isset( $items[ $endpoint_id ] ) ) {
				$sorted_items[ $endpoint_id ] = $items[ $endpoint_id ];
			}
		}

		if ( empty( $this->endpoint_title ) ) {
			return $sorted_items;
		}

		foreach ( $this->endpoint_title as $endpoint_id => $title ) {
			if ( isset( $sorted_items[ $endpoint_id ] ) ) {
				$sorted_items[ $endpoint_id ] = esc_html( $title );
			}
		}

		return $sorted_items;
	}

	/**
	 * Change the title of the displayed page.
	 *
	 * @return void
	 */
	public function change_page_title() {
		global $post;

		$displayed = $this->get_displayed_endpoint();

		if ( empty( $displayed ) || empty( $post ) || empty( $this->endpoint_title ) ) {
			return;
		}

		foreach ( $this->endpoint_title as $endpoint_id => $title ) {
			if ( $endpoint_id === $displayed ) {
				$post->post_title = esc_html( $title );
			} else {
				$default_title    = CustomizeMyAccountFns::get_endpoint_title( $displayed );
				$post->post_title = $default_title ?: $post->post_title;
			}
		}
	}

	/**
	 * Display endpoint content.
	 *
	 * @return void
	 */
	public function display_endpoint_content() {
		global $wp_embed;

		echo '<div class="rtsb-endpoint-dynamic-content">';
		echo do_shortcode( $wp_embed->autoembed( apply_filters( 'the_content', $this->endpoint_content ) ) );
		echo '</div>';
	}

	/**
	 * Dynamic styles.
	 *
	 * @return void
	 */
	public function enqueue_public_scripts() {
		// Enqueue assets.
		$this->handle = Fns::enqueue_module_assets(
			$this->handle,
			'customize-my-account',
			[
				'context' => rtsbpro(),
				'version' => RTSBPRO_VERSION,
				'type'    => 'css',
			]
		);

		$parent        = '.woocommerce-account .rtsb-custom-wc-nav .woocommerce-MyAccount-navigation ul ';
		$selector_li   = $parent . 'li.woocommerce-MyAccount-navigation-link';
		$selector_a    = $parent . 'li.woocommerce-MyAccount-navigation-link a';
		$selector_icon = "$parent li.woocommerce-MyAccount-navigation-link.has-custom-icon .icon i, $parent li.woocommerce-MyAccount-navigation-link.has-custom-icon .icon img, $parent li.woocommerce-MyAccount-navigation-link.has-custom-icon .icon svg";
		$css_rules     = [];

		$css_properties = [
			'my_acc_nav_font_size'         => 'font-size',
			'my_acc_nav_font_weight'       => 'font-weight',
			'my_acc_nav_icon_size'         => [ 'font-size', 'width' ],
			'my_acc_nav_text_color'        => 'color',
			'my_acc_nav_bg_color'          => 'background-color',
			'my_acc_nav_icon_color'        => 'color',
			'my_acc_nav_hover_color'       => 'color',
			'my_acc_nav_hover_bg_color'    => 'background-color',
			'my_acc_nav_hover_icon_color'  => 'color',
			'my_acc_nav_active_color'      => 'color',
			'my_acc_nav_active_bg_color'   => 'background-color',
			'my_acc_nav_active_icon_color' => 'color',
			'my_acc_nav_border'            => 'border-width',
			'my_acc_nav_border_color'      => 'border-color',
			'my_acc_nav_padding'           => 'padding',
			'my_acc_nav_icon_spacing'      => 'left',
			'my_acc_nav_icon_padding'      => 'padding',
			'my_acc_nav_margin'            => 'margin',
		];

		foreach ( $css_properties as $option => $property ) {
			if ( ! empty( $this->options[ $option ] ) ) {
				$value    = esc_html( $this->options[ $option ] );
				$selector = $selector_a;

				switch ( $option ) {
					case ( strpos( $option, 'nav_icon_color' ) !== false ):
						$selector = $selector_li . '.has-custom-icon span.icon i';
						break;
					case ( strpos( $option, 'hover_icon_color' ) !== false ):
						$selector = $selector_li . '.has-custom-icon:hover span.icon i';
						break;
					case ( strpos( $option, 'active_icon_color' ) !== false ):
						$selector = $selector_li . '.is-active.has-custom-icon span.icon i';
						break;
					case ( strpos( $option, 'icon_padding' ) !== false ):
						$selector = $selector_li . '.has-custom-icon > a';
						break;
					case ( strpos( $option, 'icon_spacing' ) !== false ):
						$selector = $selector_li . '.has-custom-icon span.icon';
						break;
					case ( strpos( $option, 'icon' ) !== false ):
						$selector = $selector_icon;
						break;
					case ( strpos( $option, 'font' ) !== false ):
						$selector = $selector_li;
						break;
					case ( strpos( $option, 'hover' ) !== false ):
						$selector .= ':hover';
						break;
					case ( strpos( $option, 'active' ) !== false ):
						$selector = $selector_li . '.is-active a';
						break;
					case ( strpos( $option, 'hover' ) !== false && strpos( $option, 'active' ) !== false ):
						$selector = $selector_li . '.is-active a:hover';
						break;
					case ( strpos( $option, 'border' ) !== false ):
						$selector = $selector_li . ',' . $selector_a;
						break;
				}

				if ( strpos( $option, 'icon_size' ) !== false ) {
					foreach ( $property as $icon ) {
						$css_rules[] = $selector . ' {' . $icon . ': ' . $value . ';}';
					}
				} else {
					$css_rules[] = $selector . ' {' . $property . ': ' . $value . ';}';
				}
			}
		}

		$dynamic_css = implode( '', $css_rules );

		if ( ! empty( $dynamic_css ) ) {
			wp_add_inline_style( $this->handle, $dynamic_css );
		}
	}

	/**
	 * Retrieves all endpoint data for the "My Account" page.
	 *
	 * @return array
	 */
	public function get_all_endpoints_data() {
		$cache_key = 'get_my_account_endpoints';

		if ( isset( $this->cache[ $cache_key ] ) ) {
			return $this->cache[ $cache_key ];
		}

		$repeater_fields = json_decode( $this->options['endpoint_field'] ?? '[]', true );
		$endpoint_fields = $this->build_endpoint_fields_array( $repeater_fields );

		$this->cache[ $cache_key ] = $endpoint_fields;

		return $endpoint_fields;
	}

	/**
	 * Builds the array of endpoint fields based on the provided repeater fields.
	 *
	 * @param array|string $repeater_fields The repeater fields containing endpoint data.
	 *
	 * @return array
	 */
	private function build_endpoint_fields_array( $repeater_fields ) {
		$endpoint_fields = [];
		$i               = 1;

		if ( ! empty( $repeater_fields ) && is_string( $repeater_fields ) ) {
			$repeater_fields = json_decode( $repeater_fields, true );
		}

		foreach ( $repeater_fields as $endpoint ) {
			$has_user_restriction = 'specific' === ( $endpoint['visibility'] ?? null );

			if ( $has_user_restriction && ( ! FnsPro::is_visible_for_user( $endpoint['roles'] ) ) ) {
				continue;
			}

			$unique_id = $endpoint['uniqueid'] ?? null;

			if ( $unique_id ) {
				unset( $endpoint['uniqueid'] );
				$endpoint_fields[ $unique_id ] = $endpoint;

				if ( strpos( $unique_id, 'rtsb_repeater_item_' ) !== false ) {
					if ( 'url' === ( $endpoint['repeater_type'] ?? 'endpoint' ) ) {
						$slug = 'rtsb_custom_link_' . $i;
						$i++;
					} else {
						$slug = $endpoint['slug'] ?? 'rtsb-custom-ep';
					}

					$endpoint_fields[ $slug ]        = $endpoint;
					$this->custom_endpoints[ $slug ] = $endpoint;

					unset( $endpoint_fields[ $unique_id ] );
				}
			}
		}

		return $endpoint_fields;
	}

	/**
	 * Detect changes in endpoint titles or slugs between default and customized endpoints.
	 *
	 * @param string $field The field to check for changes ('title' or 'slug').
	 *
	 * @return array
	 */
	private function detect_endpoint_changes( $field = 'title' ) {
		$cache_key = 'detect_endpoint_changes_' . $field;

		if ( isset( $this->cache[ $cache_key ] ) ) {
			return $this->cache[ $cache_key ];
		}

		$default_endpoints    = CustomizeMyAccountFns::get_default_endpoints();
		$changes              = [];
		$customized_endpoints = [];

		foreach ( $this->all_endpoints as $key => $value ) {
			$customized_endpoints[ $key ] = [
				'title' => $value['title'],
				'slug'  => 'url' === ( $value['repeater_type'] ?? 'endpoint' ) ? ( $value['link'] ?? '' ) : ( $value['slug'] ?? '' ),
				'type'  => $value['repeater_type'] ?? 'endpoint',
			];
		}

		foreach ( $customized_endpoints as $endpoint_id => $customized_data ) {
			if ( ! empty( $default_endpoints[ $endpoint_id ] ) ) {
				$default_data = $default_endpoints[ $endpoint_id ];

				if ( $default_data[ $field ] !== $customized_data[ $field ] ) {
					// phpcs:ignore Generic.CodeAnalysis.RequireExplicitBooleanOperatorPrecedence.MissingParentheses
					if ( ( 'slug' === $field && 'dashboard' === $endpoint_id ) || 'slug' === $field && 'url' === $customized_data['type'] ) {
						continue;
					}

					$changes[ $endpoint_id ] = $customized_data[ $field ];
				}
			} else {
				$changes[ $customized_data['slug'] ] = $customized_data[ $field ];

				$this->update_endpoint_slug( $endpoint_id, $customized_data['slug'] );
				$this->need_rewrite_flush();
			}
		}

		$this->cache[ $cache_key ] = $changes;

		return $changes;
	}

	/**
	 * Updates the slug for a specific endpoint option.
	 *
	 * @param string $endpoint The endpoint identifier.
	 * @param string $slug The new slug value.
	 *
	 * @return void
	 */
	private function update_endpoint_slug( $endpoint, $slug ) {
		$option_name = 'woocommerce_myaccount_' . sanitize_text_field( str_replace( '-', '_', $endpoint ) ) . '_endpoint';

		if ( 'customer-logout' === $endpoint ) {
			$option_name = 'woocommerce_logout_endpoint';
		}

		update_option( $option_name, sanitize_text_field( $slug ) );
	}

	/**
	 * Retrieves the currently displayed endpoint.
	 *
	 * @return string
	 */
	private function get_displayed_endpoint() {
		$cache_key = 'get_displayed_endpoint';

		if ( isset( $this->cache[ $cache_key ] ) ) {
			return $this->cache[ $cache_key ];
		}

		$displayed     = '';
		$current_url   = ! empty( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
		$dashboard_url = esc_url_raw( get_permalink( wc_get_page_id( 'myaccount' ) ) );
		$current_url   = rtrim( $current_url, '/' );
		$dashboard_url = rtrim( wp_parse_url( $dashboard_url, PHP_URL_PATH ), '/' );

		if ( $current_url === $dashboard_url ) {
			$displayed = 'dashboard';
		} else {
			// Check for other endpoints.
			foreach ( WC()->query->get_query_vars() as $key => $value ) {
				if ( strpos( $current_url, $value ) !== false ) {
					$displayed = $key;
				}
			}
		}

		$this->cache[ $cache_key ] = $displayed;

		return $displayed;
	}

	/**
	 * Marks the need for flushing rewrite rules.
	 *
	 * @return void
	 */
	private function need_rewrite_flush() {
		update_option( 'rtsb_permalinks_need_flush', 'yes' );
	}
}
