<?php
/**
 * Main FilterHooks class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\FlashSaleCountdown;

use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Modules\PreOrder\PreOrderFns;

defined( 'ABSPATH' ) || exit();

/**
 * Main FilterHooks class.
 */
class CountdownFns {

	/**
	 * @return array
	 */
	public static function settings_field() {
		return apply_filters(
			'rtsb/module/flash_sale_countdown/fields',
			[
				'counter_shop_wrapper_heading'            => [
					'id'    => 'counter_shop_wrapper_heading',
					'type'  => 'title',
					'label' => esc_html__( 'Shop Page Settings', 'shopbuilder-pro' ),
					'tab'   => 'general',
				],
				'show_counter_on_loop'                    => [
					'id'    => 'show_counter_on_loop',
					'value' => 'on',
					'type'  => 'switch',
					'label' => esc_html__( 'Show In Shop Page', 'shopbuilder-pro' ),
					'tab'   => 'general',
				],
				'loop_counter_position'                   => [
					'id'         => 'loop_counter_position',
					'type'       => 'select',
					'value'      => 'custom',
					'label'      => esc_html__( 'Shop Page Counter Position', 'shopbuilder-pro' ),
					'help'       => esc_html__( 'You can manage counter position in shop page.', 'shopbuilder-pro' ),
					'options'    => [
						'before_add_to_cart' => esc_html__( 'Before Add To Cart', 'shopbuilder-pro' ),
						'after_add_to_cart'  => esc_html__( 'After Add To Cart', 'shopbuilder-pro' ),
						'shortcode'          => esc_html__( 'Use Shortcode', 'shopbuilder-pro' ),
						'custom'             => esc_html__( 'Custom Position', 'shopbuilder-pro' ),
					],
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.flash_sale_countdown.show_counter_on_loop',
								'value'    => 'on',
								'operator' => '==',
							],
						],
					],
					'tab'        => 'general',
				],
				'loop_counter_position_shortcode'         => [
					'type'       => 'raw',
					'label'      => ' ',
					'html'       => sprintf(
					/* translators: 1: The shortcode.*/
						esc_html__( 'Choose where to show counter in WooCommerce product\'s loop. Copy this shortcode %1$s and paste it where you want to show.', 'shopbuilder-pro' ),
						'<code>[rtsb_flash_sale_countdown]</code>'
					),

					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.flash_sale_countdown.show_counter_on_loop',
								'value'    => 'on',
								'operator' => '==',
							],
							[
								'item'     => 'modules.flash_sale_countdown.loop_counter_position',
								'value'    => 'shortcode',
								'operator' => '==',
							],
						],
					],
					'tab'        => 'general',
				],
				'loop_custom_hook_name'                   => [
					'id'         => 'loop_custom_hook_name',
					'type'       => 'text',
					'label'      => esc_html__( 'Enter Hook Name', 'shopbuilder-pro' ),
					'help'       => sprintf(
					/* translators: 1: The shortcode.*/
						esc_html__( 'Or Copy the php code %1$s and paste it in your product query where you want to show the button.', 'shopbuilder-pro' ),
						"<code>&lt;?php do_action( 'rtsb/modules/flash_sale_countdown/frontend/display' ); ?&gt;</code>"
					),
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.flash_sale_countdown.show_counter_on_loop',
								'value'    => 'on',
								'operator' => '==',
							],
							[
								'item'     => 'modules.flash_sale_countdown.loop_counter_position',
								'value'    => 'custom',
								'operator' => '==',
							],
						],
					],
					'tab'        => 'general',
				],
				'loop_hook_priority'                      => [
					'id'         => 'loop_hook_priority',
					'type'       => 'number',
					'size'       => 'small',
					'min'        => 0,
					'max'        => 999,
					'label'      => esc_html__( 'Hook Priority', 'shopbuilder-pro' ),
					'help'       => esc_html__( 'Counter position hook priority.', 'shopbuilder-pro' ),
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.flash_sale_countdown.show_counter_on_loop',
								'value'    => 'on',
								'operator' => '==',
							],
						],
					],
					'tab'        => 'general',
				],
				'loop_counter_layout'                     => [
					'id'         => 'loop_counter_layout',
					'type'       => 'select',
					'value'      => 'horizontal',
					'label'      => esc_html__( 'Shop Page Counter layout ', 'shopbuilder-pro' ),
					'options'    => [
						'horizontal' => esc_html__( 'Horizontal', 'shopbuilder-pro' ),
						'vertical'   => esc_html__( 'Vertical', 'shopbuilder-pro' ),
					],
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.flash_sale_countdown.show_counter_on_loop',
								'value'    => 'on',
								'operator' => '==',
							],
						],
					],
					'tab'        => 'general',
				],

				'loop_counter_vertical_position'          => [
					'id'         => 'loop_counter_vertical_position',
					'type'       => 'select',
					'value'      => 'top-left',
					'label'      => esc_html__( 'Vertical position', 'shopbuilder-pro' ),
					'options'    => [
						'top-left'  => esc_html__( 'Top Left', 'shopbuilder-pro' ),
						'top-right' => esc_html__( 'Top Right', 'shopbuilder-pro' ),
					],
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.flash_sale_countdown.loop_counter_layout',
								'value'    => 'vertical',
								'operator' => '==',
							],
						],
					],
					'tab'        => 'general',
				],
				'counter_single_wrapper_heading'          => [
					'id'    => 'counter_single_wrapper_heading',
					'type'  => 'title',
					'label' => esc_html__( 'Product Page Settings', 'shopbuilder-pro' ),
					'tab'   => 'general',
				],
				'show_counter_product_page'               => [
					'id'    => 'show_counter_product_page',
					'value' => 'on',
					'type'  => 'switch',
					'label' => esc_html__( 'Show Counter in Product Page', 'shopbuilder-pro' ),
					'tab'   => 'general',
				],
				'product_counter_position'                => [
					'id'         => 'product_counter_position',
					'label'      => esc_html__( 'Product Page Counter Position', 'shopbuilder-pro' ),
					'help'       => esc_html__( 'You can manage compare Counter position in single product page.', 'shopbuilder-pro' ),
					'type'       => 'select',
					'value'      => 'custom',
					'options'    => [
						'before_add_to_cart' => esc_html__( 'Before Add To Cart', 'shopbuilder-pro' ),
						'after_add_to_cart'  => esc_html__( 'After Add To Cart', 'shopbuilder-pro' ),
						'after_thumbnail'    => esc_html__( 'After Image', 'shopbuilder-pro' ),
						'after_summary'      => esc_html__( 'After Summary', 'shopbuilder-pro' ),
						'after_short_desc'   => esc_html__( 'After Short Description', 'shopbuilder-pro' ),
						'shortcode'          => esc_html__( 'Use Shortcode', 'shopbuilder-pro' ),
						'custom'             => esc_html__( 'Custom Position', 'shopbuilder-pro' ),
					],
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.flash_sale_countdown.show_counter_product_page',
								'value'    => 'on',
								'operator' => '==',
							],
						],
					],
					'tab'        => 'general',
				],
				'product_counter_position_shortcode'      => [
					'type'       => 'raw',
					'label'      => ' ',
					'html'       => sprintf(
					/* translators: 1: The shortcode.*/
						esc_html__( 'Choose where to show button on the product page. Copy this shortcode %1$s and paste it where you want to show the button.', 'shopbuilder-pro' ),
						'<code>[rtsb_wishlist_button]</code>'
					),
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.flash_sale_countdown.show_counter_product_page',
								'value'    => 'on',
								'operator' => '==',
							],
							[
								'item'     => 'modules.flash_sale_countdown.product_counter_position',
								'value'    => 'shortcode',
								'operator' => '==',
							],
						],
					],
					'tab'        => 'general',
				],
				'product_custom_hook_name'                => [
					'id'         => 'product_custom_hook_name',
					'type'       => 'text',
					'label'      => esc_html__( 'Product Hook Name', 'shopbuilder-pro' ),
					'tab'        => 'general',
					'help'       => sprintf(
					/* translators: 1: The shortcode.*/
						esc_html__( 'Or Copy the php code %1$s and paste it in your product query where you want to show the button.', 'shopbuilder-pro' ),
						"<code>&lt;?php do_action( 'rtsb/modules/flash_sale_countdown/frontend/display' ); ?&gt;</code>"
					),
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.flash_sale_countdown.show_counter_product_page',
								'value'    => 'on',
								'operator' => '==',
							],
							[
								'item'     => 'modules.flash_sale_countdown.product_counter_position',
								'value'    => 'custom',
								'operator' => '==',
							],
						],
					],
				],
				'product_hook_priority'                   => [
					'id'         => 'product_hook_priority',
					'type'       => 'number',
					'size'       => 'small',
					'min'        => 0,
					'max'        => 999,
					'label'      => esc_html__( 'Hook Priority', 'shopbuilder-pro' ),
					'tab'        => 'general',
					'dependency' => [
						'rules'    => [
							[
								'item'     => 'modules.flash_sale_countdown.show_counter_product_page',
								'value'    => 'on',
								'operator' => '==',
							],
						],
						'relation' => 'and',
					],
				],
				// Style Tab
				'loop_counter_vertical_top_distance'      => [
					'id'         => 'loop_counter_vertical_top_distance',
					'type'       => 'text',
					'value'      => '20px',
					'label'      => esc_html__( 'Top Distance', 'shopbuilder-pro' ),
					'tab'        => 'style',
					'help'       => esc_html__( 'Distance Example: 30px Or 30%.', 'shopbuilder-pro' ),
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.flash_sale_countdown.loop_counter_layout',
								'value'    => 'vertical',
								'operator' => '==',
							],
						],
					],
				],
				'loop_counter_vertical_vertical_distance' => [
					'id'         => 'loop_counter_vertical_vertical_distance',
					'type'       => 'text',
					'value'      => '20px',
					'label'      => esc_html__( 'Vertical Distance', 'shopbuilder-pro' ),
					'help'       => esc_html__( 'Distance Example: 20px Or 10%.', 'shopbuilder-pro' ),
					'tab'        => 'style',
					'dependency' => [
						'rules' => [
							[
								'item'     => 'modules.flash_sale_countdown.loop_counter_layout',
								'value'    => 'vertical',
								'operator' => '==',
							],
						],
					],
				],

				'counter_background_color'                => [
					'id'    => 'counter_background_color',
					'label' => esc_html__( 'Counter Background Color', 'shopbuilder-pro' ),
					'help'  => esc_html__( 'Change Notification background color', 'shopbuilder-pro' ),
					'type'  => 'color',
					'tab'   => 'style',
				],

				'counter_text_color'                      => [
					'id'    => 'counter_text_color',
					'label' => esc_html__( 'Counter Text Color', 'shopbuilder-pro' ),
					'help'  => esc_html__( 'Text Color', 'shopbuilder-pro' ),
					'type'  => 'color',
					'tab'   => 'style',
				],

				'counter_button_margin'                   => [
					'id'    => 'counter_button_margin',
					'type'  => 'text',
					'size'  => 'small',
					'min'   => 0,
					'max'   => 999,
					'label' => esc_html__( 'Wrapper Margin', 'shopbuilder-pro' ),
					'help'  => esc_html__( 'Enter countdown wrapper margin. Example: 0 15px 15px 15px', 'shopbuilder-pro' ),
					'tab'   => 'style',
				],

				// End Single Product Page.
				'campaign_field_intro'                    => [
					'id'   => 'campaign_field_intro',
					'type' => 'description',
					'text' => esc_html__( 'To add new campaigns, simply click on the \'Add New\' button below.', 'shopbuilder-pro' ),
					'tab'  => 'campaign',
				],
				'campaign_field'                          => [
					'id'     => 'campaign_field',
					'type'   => 'repeaters',
					'label'  => '',
					'tab'    => 'campaign',
					'repeat' => [
						'enable_campaign'       => [
							'id'    => 'enable_campaign',
							'type'  => 'switch',
							'label' => esc_html__( 'Enable Campaign', 'shopbuilder-pro' ),
							'help'  => esc_html__( 'Switch On for Run this Campaign.', 'shopbuilder-pro' ),
						],
						'title'                 => [
							'id'          => 'title',
							'label'       => esc_html__( 'Campaign Title', 'shopbuilder-pro' ),
							'type'        => 'text',
							'placeholder' => 'Black Friday',
							'value'       => 'Black Friday ',
						],
						'custom_date'           => [
							'id'    => 'custom_date',
							'type'  => 'switch',
							'value' => 'on',
							'label' => esc_html__( 'Custom date', 'shopbuilder-pro' ),
							'help'  => esc_html__( 'Switch On for set Custom date. Product "Sale price dates" Will overwrite form here.', 'shopbuilder-pro' ),
						],
						'start_date'            => [
							'id'         => 'start_date',
							'label'      => esc_html__( 'Campaign Start', 'shopbuilder-pro' ),
							'type'       => 'datetime',
							'format'     => 'YYYY-MM-DD HH:mm',
							'showtime'   => true,
							'help'       => esc_html__( 'Format: "YYYY-MM-DD HH:mm"', 'shopbuilder-pro' ),
							'dependency' => [
								'rules' => [
									[
										'item'     => 'modules.flash_sale_countdown.campaign_field.custom_date',
										'value'    => 'on',
										'operator' => '==',
									],
								],
							],
						],
						'end_date'              => [
							'id'         => 'end_date',
							'label'      => esc_html__( 'Campaign End', 'shopbuilder-pro' ),
							'type'       => 'datetime',
							'format'     => 'YYYY-MM-DD HH:mm',
							'showtime'   => true,
							'help'       => esc_html__( 'End Date must be greater than now. Format: "YYYY-MM-DD HH:mm"', 'shopbuilder-pro' ),
							'dependency' => [
								'rules' => [
									[
										'item'     => 'modules.flash_sale_countdown.campaign_field.custom_date',
										'value'    => 'on',
										'operator' => '==',
									],
								],
							],
						],
						'apply_for'             => [
							'id'      => 'apply_for',
							'type'    => 'select',
							'value'   => 'product',
							'label'   => esc_html__( 'Applicable for', 'shopbuilder-pro' ),
							'options' => [
								'product'     => esc_html__( 'Product', 'shopbuilder-pro' ),
								'product_cat' => esc_html__( 'Product Categories', 'shopbuilder-pro' ),
							],
						],
						/*
						'apply_on_all_product'  => [
							'id'         => 'apply_on_all_product',
							'type'       => 'switch',
							'label'      => esc_html__( 'Apply On All Products', 'shopbuilder-pro' ),
							'help'       => esc_html__( 'Enable this option for Apply On all product.', 'shopbuilder-pro' ),
							'dependency' => [
								'rules' => [
									[
										'item'     => 'modules.flash_sale_countdown.campaign_field.apply_for',
										'value'    => 'product',
										'operator' => '==',
									],
								],
							],
						],
						*/
						'applicable_products'   => [
							'id'              => 'applicable_products',
							'type'            => 'search_and_multi_select',
							'label'           => esc_html__( 'Applicable Products', 'shopbuilder-pro' ),
							'help'            => esc_html__( 'Choose products to include. Leave blank to apply in all product.', 'shopbuilder-pro' ),
							'placeholder'     => esc_html__( 'Search Products', 'shopbuilder-pro' ),
							'func_with_param' => [ Fns::class, 'get_post_types', [ 'post_type' => [ 'product' ] ] ],
							'options'         => Fns::get_post_types( null, [ 'post_type' => [ 'product' ] ] ),
							'dependency'      => [
								'rules' => [
									[
										'item'     => 'modules.flash_sale_countdown.campaign_field.apply_for',
										'value'    => 'product',
										'operator' => '==',
									],
								],
							],
						],

						'applicable_categories' => [
							'id'              => 'applicable_categories',
							'type'            => 'search_and_multi_select',
							'label'           => esc_html__( 'Applicable Categories', 'shopbuilder-pro' ),
							'help'            => esc_html__( 'Choose Categories to include. Leave blank to apply in all categories.', 'shopbuilder-pro' ),
							'placeholder'     => esc_html__( 'Search Category', 'shopbuilder-pro' ),
							// Function will receive one params for search string.
							'func_with_param' => [
								Fns::class,
								'products_category_query',
							],
							'options'         => Fns::products_category_query(),
							'dependency'      => [
								'rules' => [
									[
										'item'     => 'modules.flash_sale_countdown.campaign_field.apply_for',
										'value'    => 'product_cat',
										'operator' => '==',
									],
								],
							],
						],

						'discount_type'         => [
							'id'      => 'discount_type',
							'type'    => 'select',
							'value'   => 'fixed',
							'label'   => esc_html__( 'Discount Type', 'shopbuilder-pro' ),
							'options' => [
								'fixed'   => esc_html__( 'Fixed', 'shopbuilder-pro' ),
								'percent' => esc_html__( 'Percent', 'shopbuilder-pro' ),
							],
						],

						'discount_amount'       => [
							'id'          => 'discount_amount',
							'type'        => 'number',
							'label'       => esc_html__( 'Discount Amount', 'shopbuilder-pro' ),
							'placeholder' => '15',
						],

						'minimum_amount'        => [
							'id'          => 'discount_amount',
							'type'        => 'number',
							'label'       => esc_html__( 'Product Minimum Amount after discount', 'shopbuilder-pro' ),
							'placeholder' => '15',
						],
					],
				],

			]
		);
	}

	/**
	 * Get product min/max price.
	 *
	 * @return array|false
	 */
	public static function get_flash_sale_campain_list() {
		// Check if the cached result exists.
		$cache_key     = 'flash_sale_campaign_list';
		$cached_result = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false !== $cached_result ) {
			// Return the cached result.
			return $cached_result;
		}
		// If the cached result doesn't exist, fetch it from the database.
		$options       = Fns::get_options( 'modules', 'flash_sale_countdown' );
		$campaign_list = json_decode( $options['campaign_field'] ?? '', true );

		// Store the result in the cache.
		wp_cache_set( $cache_key, $campaign_list, 'shopbuilder', 3600 ); // Cache for 1 hour.
		Cache::set_data_cache_key( $cache_key );
		return $campaign_list;
	}

	/**
	 * @param $apply_for
	 *
	 * @return array
	 */
	public static function get_active_campain_list() {
		$options = self::get_flash_sale_campain_list();
		if ( ! is_array( $options ) ) {
			return [];
		}

		return array_filter(
			$options,
			function ( $object ) {
				return 'on' === ( $object['enable_campaign'] ?? null );
			}
		);
	}

	/**
	 * @param $product
	 *
	 * @return array
	 */
	public static function get_campaign_for_current_product( $product = null ) {

		$available_campaign = [];

		if ( ! $product instanceof \WC_Product ) {
			global $product;
		}

		if ( ! $product instanceof \WC_Product ) {
			return $available_campaign;
		}

		$product_id = 'variation' === $product->get_type() ? $product->get_parent_id() : $product->get_id();
		if ( ! $product_id ) {
			return $available_campaign;
		}

		$active_campaign = self::get_active_campain_list();
		if ( ! is_array( $active_campaign ) || ! count( $active_campaign ) ) {
			return $available_campaign;
		}

		foreach ( $active_campaign as $index => $campaign ) {

			$campaign_apply_for = $campaign['apply_for'] ?? '';
			if ( 'product' === $campaign_apply_for ) {
				$_products           = $campaign['applicable_products'] ?? '';
				$applicable_products = Fns::multiselect_settings_field_value( $_products );

				$is_all_products = empty( $applicable_products );
				//phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
				if ( ! $is_all_products && ! in_array( $product_id, $applicable_products ) ) {
					continue;
				}
			} elseif ( 'product_cat' === $campaign_apply_for ) {
				$term_ids = [];
				$terms    = get_the_terms( $product_id, 'product_cat' );
				if ( $terms && ! is_wp_error( $terms ) ) {
					// Using array_map to extract term IDs directly.
					$term_ids = array_map(
						function ( $term ) {
							return $term->term_id;
						},
						$terms
					);
					// Filter out null values to handle cases where get_the_terms returns false or an empty array.
					$term_ids = array_filter( $term_ids );
				}

				$applicable_cat = $campaign['applicable_categories'] ?? '';
				$applicable_cat = Fns::multiselect_settings_field_value( $applicable_cat );

				$applicable_cat  = array_map( 'absint', $applicable_cat );
				$is_all_products = empty( $applicable_cat );

				$has_cat = array_intersect( $term_ids, $applicable_cat );

				if ( ! $is_all_products && ! count( $has_cat ) ) {
					continue;
				}
			}

			$custom_date = $campaign['custom_date'] ?? false;
			$now         = current_time( 'timestamp' );

			if ( 'off' === $custom_date ) {
				$sale_price_dates_from_timestamp = $product->get_date_on_sale_from() ? $product->get_date_on_sale_from()->getOffsetTimestamp() : false;
				$sale_price_dates_to_timestamp   = $product->get_date_on_sale_to() ? $product->get_date_on_sale_to()->getOffsetTimestamp() : false;
				$campaign['start_date']          = $sale_price_dates_from_timestamp ? wp_date( 'Y-m-d H:i:s', $sale_price_dates_from_timestamp ) : '';
				$campaign['end_date']            = $sale_price_dates_to_timestamp ? wp_date( 'Y-m-d H:i:s', $sale_price_dates_to_timestamp ) : '';
			}

			$start_date = strtotime( $campaign['start_date'] ?? '' );
			$end_date   = strtotime( $campaign['end_date'] ?? '' );

			if ( $start_date && $start_date > $now ) {
				continue;
			}

			if ( $end_date && $end_date < $now ) {
				continue;
			}

			$available_campaign = $campaign;
		}

		return apply_filters( 'rtsb/module/flash_sale_countdown/campaign', $available_campaign, $product );
	}

	/**
	 * @param $product
	 *
	 * @return float|int|mixed
	 */
	public static function campaign_discount_price( $product_price, $campaign ) {
		$discount_price = 0;
		// Get product prices.
		if ( empty( $campaign ) ) {
			return $discount_price;
		}

		$campaign_amount = floatval( $campaign['discount_amount'] ?? 0 );
		$minimum_amount  = floatval( $campaign['minimum_amount'] ?? 0 );

		if ( $product_price <= 0 || $campaign_amount < 1 ) {
			return $discount_price;
		}
		$price_type = $campaign['discount_type'] ?? 'fixed';
		if ( 'fixed' === $price_type ) {
			$discount_price = min( $product_price, $campaign_amount );
		} elseif ( 'percent' === $price_type ) {
			$discount_price = $product_price * $campaign_amount / 100;
		}

		$available_max_discount = $minimum_amount > 0 && $product_price > $minimum_amount ? $product_price - $minimum_amount : $discount_price;
		if ( $discount_price >= $available_max_discount ) {
			$discount_price = $available_max_discount;
		}

		return $discount_price;
	}
}
