<?php
/**
 * Customize My Account Module Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\GiftCard;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use WC_Tax;

defined( 'ABSPATH' ) || exit();

/**
 * Gift Card Admin Module Class.
 */
class GiftCardCartCheckout {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Module Class Constructor.
	 *
	 * Initializes the Gift Card product type by adding necessary filters and actions.
	 */
	private function __construct() {
		/**
		 *  Add_filter( 'woocommerce_get_shop_coupon_data', [ $this, 'generate_virtual_coupon_data' ], 10, 2 );
		 */
		add_action( 'woocommerce_after_calculate_totals', [ $this, 'apply_gift_cards_discount' ], 20 );
		// Register.
		add_action( 'woocommerce_new_order', [ $this, 'register_gift_cards_usage' ] );
		add_filter( 'woocommerce_get_order_item_totals', [ $this, 'show_gift_cards_total_applied_to_order' ], 10, 2 );
		add_filter( 'woocommerce_product_get_tax_class', [ $this, 'exclude_tax_for_gift_card' ], 10, 2 );
		add_filter( 'woocommerce_get_cart_item_from_session', [ $this, 'set_price_in_cart_from_session' ], 20, 2 );
	}
	/**
	 * Sets the custom price for gift cards when cart is restored from session (fixes mini cart display).
	 *
	 * @param array $cart_item cart item.
	 * @param array $values values.
	 * @return array
	 */
	public function set_price_in_cart_from_session( $cart_item, $values ) {
		if ( $cart_item['data'] instanceof GiftCardWCProduct && isset( $cart_item['rtsb_gift_card_amount'] ) ) {
			$amount = floatval( $values['rtsb_gift_card_amount'] );
			$cart_item['data']->set_price( $amount );
		}
		return $cart_item;
	}

	/**
	 * Exclude tax for gift card products.
	 *
	 * This function ensures that tax is not applied to products of type "gift card" or those
	 * in the "gift-card" product category. The tax status is set to 'none' for such products.
	 *
	 * @param string     $tax_status The tax class for the product.
	 * @param WC_Product $product The WooCommerce product object.
	 * @return string The modified tax class, set to 'none' for gift cards.
	 */
	public function exclude_tax_for_gift_card( $tax_status, $product ) {
		// Check if the product is a "gift card" (adjust this based on your product type or category).
		if ( GiftCardFns::GIFT_CARD_KEY === $product->get_type() ) {
			// Set tax status to "none" for gift cards.
			$tax_status = 'none';
		}
		return $tax_status;
	}
	/**
	 * Update the balance for all gift cards applied to an order
	 *
	 * @param int $order_id order id.
	 */
	public function register_gift_cards_usage( $order_id ) {
		/**
		 * Adding two race condition fields to the order
		 */
		$order                     = wc_get_order( $order_id );
		$applied_discount          = 0.00;
		$applied_gift_cards_amount = isset( WC()->cart ) ? WC()->cart->applied_gift_cards_amounts : [];
		if ( isset( $applied_gift_cards_amount ) && is_array( $applied_gift_cards_amount ) ) {
			WC()->mailer();
			foreach ( $applied_gift_cards_amount as $code => $amount ) {
				$giftCard = GiftCardFns::get_card_by_code( $code );
				if ( $giftCard->exists() ) {
					$applied_discount += $amount;
					$new_balance       = max( 0.00, $giftCard->card_balance() - $amount );
					$giftCard->update_balance( $new_balance );
					$giftCard->add_card_activity( $order_id, $amount, $new_balance );
					do_action( 'rtsb/gift/card/activity/email', $giftCard, $order );
				}
			}
		}
		$order_total = $order->get_total();

		$order->update_meta_data( 'rtsb_applied_gift_cards', $applied_gift_cards_amount );
		$order->update_meta_data( 'rtsb_applied_gift_cards_totals', $applied_discount );
		$order->update_meta_data( 'rtsb_applied_gift_cards_order_total', $order_total );
		/* translators: %s = order total amount */
		$order->add_order_note( sprintf( esc_html__( 'Order paid with gift cards for a total amount of %s.', 'shopbuilder-pro' ), wc_price( $applied_discount ) ) );
		$order->save();

		GiftCardFns::empty_gift_cards_session();
	}

	/**
	 * Generates virtual coupon data for gift cards.
	 *
	 * @param array  $coupon_data Coupon data.
	 * @param string $card_number Coupon code.
	 * @return array Modified coupon data.
	 */
	public function generate_virtual_coupon_data( $coupon_data, $card_number ) {
		$giftCard = GiftCardFns::get_card_by_code( $card_number );
		if ( ! $giftCard->is_valid() ) {
			return $coupon_data;
		}
		if ( ! empty( $giftCard->card_balance() ) ) {
			return apply_filters(
				'rtsb_temp_coupon_array',
				[
					'discount_type' => 'fixed_cart',
					'coupon_amount' => $giftCard->card_balance(),
					'amount'        => $giftCard->card_balance(),
					'id'            => true,
				],
				$card_number
			);
		}
		return $coupon_data;
	}
	/**
	 * Apply the gift cards discount as a virtual coupon.
	 *
	 * @param WC_Cart $cart WooCommerce cart object.
	 */
	public function apply_gift_cards_discount( $cart ) {
		if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
			return;
		}
		$gift_cards = GiftCardFns::get_gift_cards_from_session();
		if ( empty( $gift_cards ) ) {
			return;
		}

		if ( GiftCardFns::cart_has_gift_card() ) {
			WC()->session->set( 'rtsb_gift_card_code', [] );
			return;
		}

		// Get the cart total before tax.
		$cart_total = $cart->get_total( 'edit' );
		// Store applied gift card amounts.
		$cart->applied_gift_cards_amounts = [];
		foreach ( $gift_cards as $code => $amount ) {
			$discount                                  = min( $amount, $cart_total );
			$cart->applied_gift_cards_amounts[ $code ] = $discount;
			$cart_total                               -= $discount;
		}
		// Ensure total is non-negative.
		$cart->total = max( 0, $cart_total );
		// Get the updated cart totals.
		$cart_totals         = $cart->get_totals();
		$cart_contents_total = $cart_totals['cart_contents_total'];
		$shipping_total      = $cart_totals['shipping_total'];
		// Get tax rates.
		$tax_rates      = WC_Tax::get_rates();
		$tax_rate       = reset( $tax_rates ); // Get the first tax rate.
		$tax_percentage = isset( $tax_rate['rate'] ) ? floatval( $tax_rate['rate'] ) / 100 : 0;
		// Recalculate taxes after the gift card discount.
		$new_cart_total = $cart->total;
		$taxable_amount = $new_cart_total / ( 1 + $tax_percentage );
		$new_tax        = $new_cart_total - $taxable_amount;
		// Apply new tax to cart items.
		foreach ( $cart->get_cart_contents() as $cart_item_key => $item ) {
			$line_tax_data = $item['line_tax_data'];
			foreach ( $line_tax_data['total'] as $tax_key => $tax_value ) {
				$line_tax_data['total'][ $tax_key ] = $new_tax;
			}
			$item['line_tax_data']                 = $line_tax_data;
			$cart->cart_contents[ $cart_item_key ] = $item;
		}
		// Recalculate shipping tax.
		if ( $shipping_total > 0 ) {
			$shipping_taxable_amount = $shipping_total / ( 1 + $tax_percentage );
			$new_shipping_tax        = $shipping_total - $shipping_taxable_amount;
			$cart->set_shipping_taxes( [ 'shipping' => $new_shipping_tax ] );
		}
		// Update cart totals.
		$cart->set_cart_contents_total( $new_cart_total );
		$cart->set_cart_contents_tax( $new_tax );
		$cart->set_total_tax( $new_tax + ( isset( $new_shipping_tax ) ? $new_shipping_tax : 0 ) );
	}
	/**
	 * Show gift cards usage on order item totals
	 *
	 * @param array    $total_rows Order totals.
	 * @param WC_Order $order Order.
	 *
	 * @return array
	 */
	public function show_gift_cards_total_applied_to_order( $total_rows, $order ) {
		$gift_cards = $order->get_meta( 'rtsb_applied_gift_cards' );
		if ( $gift_cards ) {
			$row_totals = $total_rows['order_total'];
			unset( $total_rows['order_total'] );

			$gift_cards_message = '';
			foreach ( $gift_cards as $code => $amount ) {
				/**
				 *
				 * Filter the gift card applied message in the order totals.
				 *
				 * @param string the applied gift card message
				 * @param string $amount the gift card amount
				 * @param string $code the gift card code
				 *
				 * @return string
				 */
				$gift_cards_message .= apply_filters( 'rtsb_gift_card_coupon_message', '-' . wc_price( $amount ), $amount, $code );
			}

			$total_rows['gift_cards'] = [
				'label' => esc_html__( 'Gift cards:', 'shopbuilder-pro' ),
				'value' => $gift_cards_message,
			];

			/**
			 * APPLY_FILTERS: rtsb_gift_card_thankyou_table_total_rows
			 *
			 * Filter the gift card displayed on the totals.
			 *
			 * @param array $total_rows the gift card data displayed on the totals
			 * @param string $code the gift card code
			 *
			 * @return array
			 */
			$total_rows = apply_filters( 'rtsb_gift_card_thankyou_table_total_rows', $total_rows, $code );

			$total_rows['order_total'] = $row_totals;
		}

		return $total_rows;
	}
}
