<?php
/**
 * Main FilterHooks class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\GiftCard;

use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use Random\RandomException;

defined( 'ABSPATH' ) || exit();

/**
 * Main FilterHooks class.
 */
class GiftCardFns {
	/**
	 * Free-gift key.
	 *
	 * @var string
	 */
	const GIFT_CARD_KEY = 'rtsb-gift-card';
	/**
	 * Free-gift key.
	 *
	 * @var string
	 */
	const GIFT_CARD_POST_TYPE = 'rtsb_gift_card';
	/**
	 * Free-gift Amount key.
	 *
	 * @var string
	 */
	const GIFT_CARD_AMOUNTS = '_rtsb_gift_card_amounts';

	/**
	 * @return array
	 */
	public static function get_options() {
		return Fns::get_options( 'modules', 'gift_card' );
	}
	/**
	 * @return RtsbGiftCard
	 */
	public static function get_card_by_code( $cardNumber ) {
		return new RtsbGiftCard( [ 'gift_card_number' => $cardNumber ] );
	}
	/**
	 * @return RtsbGiftCard
	 */
	public static function get_card_by_id( $id ) {
		return new RtsbGiftCard( [ 'ID' => $id ] );
	}
	/**
	 * @return array
	 */
	public static function settings_field() {
		$fields = [
			'general_settings'                  => [
				'id'    => 'general_settings',
				'type'  => 'title',
				'label' => esc_html__( 'Settings', 'shopbuilder' ),
				'tab'   => 'general',
			],
			'publish_card'                      => [
				'id'          => 'publish_card',
				'label'       => esc_html__( 'Publish Gift Card by Order Status', 'shopbuilder' ),
				'help'        => esc_html__( 'The gift card will be published automatically when the order reaches the selected status (e.g., Processing or Completed).', 'shopbuilder-pro' ),
				'type'        => 'checkbox',
				'orientation' => 'vertical',
				'value'       => [
					'processing',
					'completed',
				],
				'options'     => [
					[
						'value' => 'processing',
						'label' => 'Processing',
					],
					[
						'value' => 'completed',
						'label' => 'Completed',
					],
				],
				'tab'         => 'general',
			],
			'product_page_pricing_title'        => [
				'id'    => 'product_page_pricing_title',
				'type'  => 'text',
				'label' => esc_html__( 'Pricing Section Title', 'shopbuilder-pro' ),
				'value' => esc_html__( 'Choose Gift Amount', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'This title appears above the pricing options on the product page. Use it to guide customers when selecting a gift amount.', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'card_number_pattern'               => [
				'id'    => 'card_number_pattern',
				'type'  => 'text',
				'label' => esc_html__( 'Card Number Pattern', 'shopbuilder-pro' ),
				'value' => 'ABCD-EFG-***-**-******',
				'help'  => sprintf(
					// Translators: %1$s = pattern, %2$s = example value.
					esc_html__( 'Choose the pattern of new gift cards. If you set %1$s your cards will have a code like: %2$s', 'shopbuilder-pro' ),
					'ABCD-EFG-***-**-******',
					'ABCD-EFG-EMQ-NP-1GEOGQ'
				),
				'tab'   => 'general',
			],
			'card_validity_message'             => [
				'id'    => 'card_validity_message',
				'label' => __( 'Gift Card Validity Message', 'shopbuilder-pro' ),
				'type'  => 'textarea',
				'value' => esc_html__( 'This gift card is valid for {{validity}} days from the date of purchase.', 'shopbuilder-pro' ),
				'help'  => __( 'Use the <code>{{validity}}</code> placeholder to dynamically display the validity period. It will appear on the product page below the price selection.', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'pricing_button_form_style'         => [
				'id'    => 'pricing_button_form_style',
				'type'  => 'title',
				'label' => esc_html__( 'Pricing Title', 'shopbuilder-pro' ),
				'tab'   => 'product',
			],
			'pricing_button_title_font_size'    => [
				'id'    => 'pricing_button_title_font_size',
				'label' => esc_html__( 'Title Font Size (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 10,
				'max'   => 80,
				'unit'  => 'px',
				'value' => 20,
				'tab'   => 'product',
			],
			'pricing_button_title_text_color'   => [
				'id'    => 'pricing_button_title_text_color',
				'label' => esc_html__( 'Text Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'product',
			],
			'pricing_button_title_font_weight'  => [
				'id'      => 'pricing_button_title_font_weight',
				'type'    => 'select',
				'options' => [
					100 => esc_html__( '100 (Thin)', 'shopbuilder-pro' ),
					200 => esc_html__( '200 (Extra Light)', 'shopbuilder-pro' ),
					300 => esc_html__( '300 (Light)', 'shopbuilder-pro' ),
					400 => esc_html__( '400 (Normal)', 'shopbuilder-pro' ),
					500 => esc_html__( '500 (Medium)', 'shopbuilder-pro' ),
					600 => esc_html__( '600 (Semi Bold)', 'shopbuilder-pro' ),
					700 => esc_html__( '700 (Bold)', 'shopbuilder-pro' ),
					800 => esc_html__( '800 (Extra Bold)', 'shopbuilder-pro' ),
					900 => esc_html__( '900 (Black)', 'shopbuilder-pro' ),
				],
				'value'   => 400,
				'label'   => esc_html__( 'Font Weight', 'shopbuilder-pro' ),
				'help'    => esc_html__( 'Choose Font Weight.', 'shopbuilder-pro' ),
				'tab'     => 'product',
			],
			'pricing_button_title_margin'       => [
				'id'    => 'pricing_button_title_margin',
				'type'  => 'text',
				'size'  => 'small',
				'min'   => 0,
				'max'   => 999,
				'label' => esc_html__( 'Title Margin', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Margin Example: 0 15px 15px 15px', 'shopbuilder-pro' ),
				'tab'   => 'product',
			],

			'pricing_button'                    => [
				'id'    => 'pricing_button',
				'type'  => 'title',
				'label' => esc_html__( 'Product Page Pricing Button Style', 'shopbuilder-pro' ),
				'tab'   => 'product',
			],

			'pricing_button_bg_color'           => [
				'id'    => 'pricing_button_bg_color',
				'label' => esc_html__( 'Background Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'product',
			],
			'pricing_button_text_color'         => [
				'id'    => 'pricing_button_text_color',
				'label' => esc_html__( 'Text Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'product',
			],
			'pricing_button_border_color'       => [
				'id'    => 'pricing_button_border_color',
				'label' => esc_html__( 'Border Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'product',
			],
			'pricing_button_hover_bg_color'     => [
				'id'    => 'pricing_button_hover_bg_color',
				'label' => esc_html__( 'Hover Background Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'product',
			],
			'pricing_button_hover_text_color'   => [
				'id'    => 'pricing_button_hover_text_color',
				'label' => esc_html__( 'Hover Text Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'product',
			],
			'pricing_button_hover_border_color' => [
				'id'    => 'pricing_button_hover_border_color',
				'label' => esc_html__( 'Hover Border Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'product',
			],
			'pricing_button_height'             => [
				'id'    => 'pricing_button_height',
				'label' => esc_html__( 'Height (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 20,
				'max'   => 200,
				'unit'  => 'px',
				'value' => 40,
				'tab'   => 'product',
			],
			'pricing_button_font_size'          => [
				'id'    => 'pricing_button_font_size',
				'label' => esc_html__( 'Button Font Size (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 10,
				'max'   => 80,
				'unit'  => 'px',
				'value' => 15,
				'tab'   => 'product',
			],
			'validity_text'                     => [
				'id'    => 'validity_text',
				'type'  => 'title',
				'label' => esc_html__( 'Validity Text Style', 'shopbuilder-pro' ),
				'tab'   => 'product',
			],
			'validity_text_color'               => [
				'id'    => 'validity_text_color',
				'label' => esc_html__( 'Text Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'product',
			],
			'validity_text_font_size'           => [
				'id'    => 'validity_text_font_size',
				'label' => esc_html__( 'Validity Text Font Size (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 10,
				'max'   => 80,
				'unit'  => 'px',
				'value' => 15,
				'tab'   => 'product',
			],
			'validity_text_margin'              => [
				'id'    => 'validity_text_margin',
				'type'  => 'text',
				'size'  => 'small',
				'min'   => 0,
				'max'   => 999,
				'label' => esc_html__( 'Margin', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Margin Example: 0 15px 15px 15px', 'shopbuilder-pro' ),
				'tab'   => 'product',
			],
			// Table Settings.
			'table_style_settings'              => [
				'id'    => 'table_style_settings',
				'type'  => 'title',
				'label' => esc_html__( 'Table Style', 'shopbuilder-pro' ),
				'tab'   => 'table-styles',
			],
			'table_heading_font_size'           => [
				'id'    => 'table_heading_font_size',
				'label' => esc_html__( 'Table Heading Font Size (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 10,
				'max'   => 80,
				'unit'  => 'px',
				'value' => 15,
				'tab'   => 'table-styles',
			],
			'table_heading_padding'             => [
				'id'    => 'table_heading_padding',
				'type'  => 'text',
				'size'  => 'small',
				'min'   => 0,
				'max'   => 999,
				'label' => esc_html__( 'Heading Padding', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Padding Example: 0 15px 15px 15px', 'shopbuilder-pro' ),
				'tab'   => 'table-styles',
			],
			'table_cell_padding'                => [
				'id'    => 'table_cell_padding',
				'type'  => 'text',
				'size'  => 'small',
				'min'   => 0,
				'max'   => 999,
				'label' => esc_html__( 'Cell Padding', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Padding Example: 0 15px 15px 15px', 'shopbuilder-pro' ),
				'tab'   => 'table-styles',
			],
			'table_cell_border_color'           => [
				'id'    => 'table_cell_border_color',
				'label' => esc_html__( 'Border Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'table-styles',
			],
			'style_settings_button'             => [
				'id'    => 'style_settings',
				'type'  => 'title',
				'label' => esc_html__( 'Table Button Style', 'shopbuilder-pro' ),
				'tab'   => 'table-styles',
			],
			'button_bg_color'                   => [
				'id'    => 'button_bg_color',
				'label' => esc_html__( 'Background Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'table-styles',
			],
			'button_text_color'                 => [
				'id'    => 'button_text_color',
				'label' => esc_html__( 'Text Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'table-styles',
			],
			'button_border_color'               => [
				'id'    => 'button_border_color',
				'label' => esc_html__( 'Border Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'table-styles',
			],
			'button_hover_bg_color'             => [
				'id'    => 'button_hover_bg_color',
				'label' => esc_html__( 'Hover Background Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'table-styles',
			],
			'button_hover_text_color'           => [
				'id'    => 'button_hover_text_color',
				'label' => esc_html__( 'Hover Text Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'table-styles',
			],
			'button_hover_border_color'         => [
				'id'    => 'button_hover_border_color',
				'label' => esc_html__( 'Hover Border Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'table-styles',
			],
			'button_height'                     => [
				'id'    => 'button_height',
				'label' => esc_html__( 'Height (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 20,
				'max'   => 200,
				'unit'  => 'px',
				'value' => 40,
				'tab'   => 'table-styles',
			],
			'button_width'                      => [
				'id'    => 'button_width',
				'label' => esc_html__( 'Width (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 50,
				'max'   => 200,
				'unit'  => 'px',
				'value' => 100,
				'tab'   => 'table-styles',
			],
			'button_font_size'                  => [
				'id'    => 'button_font_size',
				'label' => esc_html__( 'Button Font Size (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 10,
				'max'   => 80,
				'unit'  => 'px',
				'value' => 15,
				'tab'   => 'table-styles',
			],
			'next_prev_button'                  => [
				'id'    => 'next_prev_button',
				'type'  => 'title',
				'label' => esc_html__( 'Next Prev Button Style', 'shopbuilder-pro' ),
				'tab'   => 'table-styles',
			],
			'nxt_prv_button_bg_color'           => [
				'id'    => 'nxt_prv_button_bg_color',
				'label' => esc_html__( 'Background Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'table-styles',
			],
			'nxt_prv_button_text_color'         => [
				'id'    => 'nxt_prv_button_text_color',
				'label' => esc_html__( 'Text Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'table-styles',
			],
			'nxt_prv_button_border_color'       => [
				'id'    => 'nxt_prv_button_border_color',
				'label' => esc_html__( 'Border Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'table-styles',
			],
			'nxt_prv_button_hover_bg_color'     => [
				'id'    => 'nxt_prv_button_hover_bg_color',
				'label' => esc_html__( 'Hover Background Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'table-styles',
			],
			'nxt_prv_button_hover_text_color'   => [
				'id'    => 'nxt_prv_button_hover_text_color',
				'label' => esc_html__( 'Hover Text Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'table-styles',
			],
			'nxt_prv_button_hover_border_color' => [
				'id'    => 'nxt_prv_button_hover_border_color',
				'label' => esc_html__( 'Hover Border Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'table-styles',
			],
			'nxt_prv_button_height'             => [
				'id'    => 'nxt_prv_button_height',
				'label' => esc_html__( 'Height (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 20,
				'max'   => 200,
				'unit'  => 'px',
				'value' => 40,
				'tab'   => 'table-styles',
			],
			'nxt_prv_button_width'              => [
				'id'    => 'nxt_prv_button_width',
				'label' => esc_html__( 'Width (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 50,
				'max'   => 250,
				'unit'  => 'px',
				'value' => 150,
				'tab'   => 'table-styles',
			],
			'nxt_prv_button_font_size'          => [
				'id'    => 'nxt_prv_button_font_size',
				'label' => esc_html__( 'Next Prev Button Font Size (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 10,
				'max'   => 80,
				'unit'  => 'px',
				'value' => 15,
				'tab'   => 'table-styles',
			],
			'redeem_form_style'                 => [
				'id'    => 'redeem_form_style',
				'type'  => 'title',
				'label' => esc_html__( 'Redeem Form Style', 'shopbuilder-pro' ),
				'tab'   => 'redeem-form-styles',
			],
			'form_button_title_font_size'       => [
				'id'    => 'form_button_title_font_size',
				'label' => esc_html__( 'Title Font Size (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 10,
				'max'   => 80,
				'unit'  => 'px',
				'value' => 20,
				'tab'   => 'redeem-form-styles',
			],
			'form_button_title_margin'          => [
				'id'    => 'form_button_title_margin',
				'type'  => 'text',
				'size'  => 'small',
				'min'   => 0,
				'max'   => 999,
				'label' => esc_html__( 'Title Margin', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Margin Example: 0 15px 15px 15px', 'shopbuilder-pro' ),
				'tab'   => 'redeem-form-styles',
			],
			'form_field_height'                 => [
				'id'    => 'form_field_height',
				'label' => esc_html__( 'Height (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 20,
				'max'   => 200,
				'unit'  => 'px',
				'value' => 45,
				'tab'   => 'redeem-form-styles',
			],
			'form_button_width'                 => [
				'id'    => 'form_button_width',
				'label' => esc_html__( 'Form Button Width (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 50,
				'max'   => 250,
				'unit'  => 'px',
				'value' => 150,
				'tab'   => 'redeem-form-styles',
			],
			'form_button_width_font_size'       => [
				'id'    => 'form_button_width_font_size',
				'label' => esc_html__( 'Button Font Size (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 10,
				'max'   => 80,
				'unit'  => 'px',
				'value' => 15,
				'tab'   => 'redeem-form-styles',
			],
			'form_button_bg_color'              => [
				'id'    => 'form_button_bg_color',
				'label' => esc_html__( 'Background Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'redeem-form-styles',
			],
			'form_button_text_color'            => [
				'id'    => 'form_button_text_color',
				'label' => esc_html__( 'Text Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'redeem-form-styles',
			],
			'form_button_border_color'          => [
				'id'    => 'form_button_border_color',
				'label' => esc_html__( 'Border Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'redeem-form-styles',
			],
			'form_button_hover_bg_color'        => [
				'id'    => 'form_button_hover_bg_color',
				'label' => esc_html__( 'Hover Background Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'redeem-form-styles',
			],
			'form_button_hover_text_color'      => [
				'id'    => 'form_button_hover_text_color',
				'label' => esc_html__( 'Hover Text Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'redeem-form-styles',
			],
			'form_button_hover_border_color'    => [
				'id'    => 'form_button_hover_border_color',
				'label' => esc_html__( 'Hover Border Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'redeem-form-styles',
			],
		];
		return $fields;
	}

	/**
	 * Get_gift_cards_from_session
	 *
	 * @return array|string
	 */
	public static function get_gift_cards_from_session() {
		$value = [];
		if ( isset( WC()->session ) ) {
			$value = WC()->session->get( 'rtsb_gift_card_code', [] );
		}
		return $value;
	}

	/**
	 * Empty_gift_cards_session
	 *
	 * @return void
	 */
	public static function empty_gift_cards_session() {
		if ( isset( WC()->session ) ) {
			WC()->session->__unset( 'rtsb_gift_card_code' );
		}
	}

	/**
	 * @param $order_id
	 * @return string
	 */
	public static function get_order_edit_link( $order_id ) {
		if ( ! $order_id ) {
			return '<span>' . esc_html__( 'Created Manually', 'shopbuilder' ) . '</span>';
		}
		// Generate the correct edit link based on storage type.
		$order = wc_get_order( $order_id ); // Works with HPOS and legacy storage.
		if ( ! $order ) {
			return $order_id . '<span hidden="">(' . esc_html__( 'Order Not Exist', 'shopbuilder' ) . ')</span>';
		}
		if ( 'shop_order' === get_post_type( $order_id ) ) {
			// Legacy storage (orders stored as posts).
			$order_url = admin_url( 'post.php?post=' . $order_id . '&action=edit' );
		} else {
			// HPOS (orders stored in custom tables).
			$order_url = admin_url( 'admin.php?page=wc-orders&action=edit&id=' . $order_id );
		}
		return '<a href="' . $order_url . '">#' . $order_id . '</a>';
	}

	/**
	 * Generate a code based on a dynamic pattern.
	 * * = random uppercase letter or digit
	 *
	 * @param string $pattern The pattern to use for code generation.
	 * @return string
	 */
	public static function generate_card_number() {
		// Get the pattern from options.
		$options = self::get_options();
		$pattern = $options['card_number_pattern'] ?? '';
		// Get current timestamp (e.g., seconds since Unix epoch).
		$timestamp = time();
		// Convert the timestamp into an array of characters and shuffle it.
		$timestamp_digits = str_split( (string) $timestamp );
		shuffle( $timestamp_digits ); // Shuffle the timestamp digits.
		// Create a pool of characters (letters and shuffled timestamp digits).
		$chars_digits = implode( '', $timestamp_digits ); // Rebuild shuffled timestamp string.
		$chars_all    = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ' . $chars_digits;
		$code         = '';
		// If no pattern is provided, generate a unique code using the timestamp-based logic.
		if ( empty( $pattern ) ) {
			// Generate the fallback code using shuffled timestamp.
			$code_length = 12; // Default length for fallback code.
			for ( $i = 0; $i < $code_length; $i++ ) {
				$code .= $chars_all[ random_int( 0, strlen( $chars_all ) - 1 ) ];
			}
			return strtoupper( $code ); // Return the generated code in uppercase.
		}
		// If Provided Pattrn.
		$length = strlen( $pattern );
		// Loop through each character in the pattern and generate accordingly.
		for ( $i = 0; $i < $length; $i++ ) {
			$char = $pattern[ $i ];
			if ( '*' === $char ) {
				$code .= $chars_all[ random_int( 0, strlen( $chars_all ) - 1 ) ];
			} else {
				$code .= $char;
			}
		}
		return $code;
	}

	/**
	 * Check if the WooCommerce cart contains a gift card product.
	 *
	 * @param WC_Cart|null $cart Optional. The WooCommerce cart object. Defaults to WC()->cart.
	 * @return bool True if a gift card is found in the cart, false otherwise.
	 */
	public static function cart_has_gift_card( $cart = null ) {
		if ( ! $cart ) {
			$cart = WC()->cart;
		}
		if ( ! $cart || $cart->is_empty() ) {
			return false;
		}
		foreach ( $cart->get_cart() as $cart_item ) {
			if (
				isset( $cart_item['data'], $cart_item['rtsb_gift_card_amount'] ) &&
				$cart_item['data'] instanceof GiftCardWCProduct
			) {
				return true;
			}
		}
		return false;
	}
}
