<?php
/**
 * Customize My Account Module Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\GiftCard;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\BuilderFunPro;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Modules\GiftCard\Email\GiftCardEmail;
use RadiusTheme\SBPRO\Modules\GiftCard\Email\GiftCardUsesActivity;
use RadiusTheme\SBPRO\Modules\GiftCard\Email\GiftCardValidityEmail;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Module Class.
 */
class GiftCardInit {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Asset Handle
	 *
	 * @var string
	 */
	private $handle = 'rtsb-gift-card';

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		$this->init();
		GiftCardAjax::instance();
		if ( is_admin() ) {
			GiftCardAdmin::instance();
		} else {
			giftCardFrontEnd::instance();
		}
		GiftCardCPT::instance();
		GiftCardCartCheckout::instance();
		GiftCardUserAccount::instance();
	}
	/**
	 * @return void
	 */
	public function init() {
		$this->run_cron();
		// Set the correct class for Gift Card products.
		add_filter( 'woocommerce_product_class', [ $this, 'set_gift_card_product_class' ], 10, 2 );
		// Add the "Gift card" type to the product type list.
		add_filter( 'product_type_selector', [ $this, 'add_gift_card_product_type' ] );
		add_filter( 'woocommerce_is_purchasable', [ $this, 'custom_product_purchasable' ], 10, 2 );
		add_filter( 'woocommerce_order_item_get_formatted_meta_data', [ $this, 'format_gift_card_meta' ], 16, 2 );
		add_filter( 'woocommerce_email_classes', [ $this, 'rtsb_register_gift_card_email' ] );
		add_action( 'template_redirect', [ $this, 'maybe_flush_on_404' ] );
		add_action( 'rtsb_check_gift_card_expiry', [ $this, 'process_expiring_gift_cards' ] );

		// Builder Support.
		add_filter( 'rtsb/builder/register/page/types', [ $this, 'register_custom_template' ], 99 );
		add_filter( 'rtsb/builder/set/current/page/type/external', [ $this, 'builder_page_type' ], 16 );
		add_filter( 'rtsb/editor/enable/custom_endpoint', [ $this, 'enable_custom_endpoints' ] );
		/**
		 * Public scripts.
		 */
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_public_scripts' ], 99 );
		// Ajax Disable.
		add_filter( 'woocommerce_product_add_to_cart_url', [ $this, 'modify_add_to_cart_url' ], 10, 2 );
		add_filter( 'rtsb/elementor/render/cart_attributes', [ $this, 'change_add_to_cart_args' ], 10, 2 );
		add_filter( 'woocommerce_loop_add_to_cart_args', [ $this, 'change_add_to_cart_args' ], 10, 2 );
	}

	/**
	 * Modify the "Add to Cart" URL for pre-order products.
	 *
	 * @param string     $permalink The original permalink URL.
	 * @param WC_Product $product The product object.
	 *
	 * @return string
	 */
	public function modify_add_to_cart_url( $permalink, $product ) {
		if ( $product instanceof GiftCardWCProduct ) {
			$permalink = esc_url( get_permalink( $product->get_id() ) );
		}

		return $permalink;
	}

	/**
	 * Modify the "Add to Cart" button arguments for pre-order products in the product loop.
	 *
	 * @param array      $args The original arguments for the "Add to Cart" button.
	 * @param WC_Product $product The product object.
	 *
	 * @return array
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function change_add_to_cart_args( $args, $product ) {
		if ( $product instanceof GiftCardWCProduct ) {
			$args['href']  = esc_url( get_permalink( $product->get_id() ) );
			$args['class'] = str_replace(
				[ 'ajax_add_to_cart', 'rtsb-mini-cart', 'rtsb-add-to-cart-btn', 'variable-product' ],
				[ '', '', 'rtsb-pre-order-btn', 'variable-product rtsb-pre-order-btn' ],
				$args['class']
			);
			if ( ! empty( $args['title'] ) ) {
				$args['title'] = esc_html__( 'Select Gift Card', 'shopbuilder-pro' );
			}
		}

		return $args;
	}

	/**
	 * @return void
	 */
	public function enqueue_public_scripts() { // phpcs:ignore Generic.Metrics.CyclomaticComplexity.MaxExceeded
		// Enqueue assets.
		$this->handle = Fns::enqueue_module_assets(
			$this->handle,
			'gift-card',
			[
				'context' => rtsbpro(),
				'version' => RTSBPRO_VERSION,
			]
		);

		$options = GiftCardFns::get_options();

		$dynamic_css  = ':is(.rtsb-gift-card-table, .rtsb-giftcard-nav) .button{';
		$dynamic_css .= 'display: inline-flex !important; align-items: center;justify-content: center;';
		$dynamic_css .= '}';

		if ( ! empty( $options['pricing_button_title_font_size'] ) ) {
			$dynamic_css .= '.gift-card-amount-title{';
			$dynamic_css .= 'font-size:' . $options['pricing_button_title_font_size'] . 'px ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['pricing_button_title_font_weight'] ) ) {
			$dynamic_css .= '.gift-card-amount-title{';
			$dynamic_css .= 'font-weight:' . $options['pricing_button_title_font_weight'];
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['pricing_button_title_text_color'] ) ) {
			$dynamic_css .= '.gift-card-amount-title{';
			$dynamic_css .= 'color:' . $options['pricing_button_title_text_color'] . ' ! important;';
			$dynamic_css .= '}';
		}

		if ( ! empty( $options['pricing_button_title_margin'] ) ) {
			$dynamic_css .= '.gift-card-amount-title{';
			$dynamic_css .= 'margin:' . $options['pricing_button_title_margin'] . ' ! important;';
			$dynamic_css .= '}';
		}

		if ( ! empty( $options['button_bg_color'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-table .button{';
			$dynamic_css .= 'background-color:' . $options['button_bg_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['button_text_color'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-table .button{';
			$dynamic_css .= 'color:' . $options['button_text_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['button_border_color'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-table .button{';
			$dynamic_css .= 'border-color:' . $options['button_border_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		// Hover.
		if ( ! empty( $options['button_hover_bg_color'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-table .button:hover{';
			$dynamic_css .= 'background-color:' . $options['button_hover_bg_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['button_hover_text_color'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-table .button:hover{';
			$dynamic_css .= 'color:' . $options['button_hover_text_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['button_border_color'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-table .button:hover{';
			$dynamic_css .= 'border-color:' . $options['button_border_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['button_font_size'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-table .button{';
			$dynamic_css .= 'font-size:' . $options['button_font_size'] . 'px ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['button_height'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-table .button{';
			$dynamic_css .= 'height:' . $options['button_height'] . 'px ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['button_width'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-table .button{';
			$dynamic_css .= 'width:' . $options['button_width'] . 'px ! important;';
			$dynamic_css .= '}';
		}
		// Next Prev Buton.
		if ( ! empty( $options['nxt_prv_button_bg_color'] ) ) {
			$dynamic_css .= '.rtsb-giftcard-nav .button{';
			$dynamic_css .= 'background-color:' . $options['nxt_prv_button_bg_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['nxt_prv_button_text_color'] ) ) {
			$dynamic_css .= '.rtsb-giftcard-nav .button{';
			$dynamic_css .= 'color:' . $options['nxt_prv_button_text_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['nxt_prv_button_border_color'] ) ) {
			$dynamic_css .= '.rtsb-giftcard-nav .button{';
			$dynamic_css .= 'border-color:' . $options['nxt_prv_button_border_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		// Hover.
		if ( ! empty( $options['nxt_prv_button_hover_bg_color'] ) ) {
			$dynamic_css .= '.rtsb-giftcard-nav .button:hover{';
			$dynamic_css .= 'background-color:' . $options['nxt_prv_button_hover_bg_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['button_hover_text_color'] ) ) {
			$dynamic_css .= '.rtsb-giftcard-nav .button:hover{';
			$dynamic_css .= 'color:' . $options['button_hover_text_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['nxt_prv_button_border_color'] ) ) {
			$dynamic_css .= '.rtsb-giftcard-nav .button:hover{';
			$dynamic_css .= 'border-color:' . $options['nxt_prv_button_border_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['nxt_prv_button_font_size'] ) ) {
			$dynamic_css .= '.rtsb-giftcard-nav .button{';
			$dynamic_css .= 'font-size:' . $options['nxt_prv_button_font_size'] . 'px ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['nxt_prv_button_height'] ) ) {
			$dynamic_css .= '.rtsb-giftcard-nav .button{';
			$dynamic_css .= 'height:' . $options['nxt_prv_button_height'] . 'px ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['nxt_prv_button_width'] ) ) {
			$dynamic_css .= '.rtsb-giftcard-nav .button{';
			$dynamic_css .= 'width:' . $options['nxt_prv_button_width'] . 'px ! important;';
			$dynamic_css .= '}';
		}
		// Table Style Settings.
		if ( ! empty( $options['table_heading_font_size'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-table th{';
			$dynamic_css .= 'font-size:' . $options['table_heading_font_size'] . 'px ! important;';
			$dynamic_css .= '}';
		}

		if ( ! empty( $options['table_heading_padding'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-table th{';
			$dynamic_css .= 'padding:' . $options['table_heading_padding'];
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['table_cell_padding'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-table td{';
			$dynamic_css .= 'padding:' . $options['table_cell_padding'];
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['table_cell_border_color'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-table :is(td, th), table.shop_table.rtsb-gift-card-table{';
			$dynamic_css .= 'border-color:' . $options['table_cell_border_color'] . ' ! important;';
			$dynamic_css .= '}';
		}

		// Redeem Form.
		if ( ! empty( $options['form_button_title_font_size'] ) ) {
			$dynamic_css .= '#rtsb-redeem-gift-card-form h2  {';
			$dynamic_css .= 'font-size:' . $options['form_button_title_font_size'] . 'px ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['form_button_title_margin'] ) ) {
			$dynamic_css .= '#rtsb-redeem-gift-card-form h2  {';
			$dynamic_css .= 'margin:' . $options['form_button_title_margin'] . '! important;';
			$dynamic_css .= '}';
		}

		if ( ! empty( $options['form_field_height'] ) ) {
			$dynamic_css .= '#rtsb-redeem-gift-card-form :is(#rtsb-redeem-gift-card-number, #rtsb-redeem-button){';
			$dynamic_css .= 'height:' . $options['form_field_height'] . 'px ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['form_button_width'] ) ) {
			$dynamic_css .= '#rtsb-redeem-gift-card-form #rtsb-redeem-button{';
			$dynamic_css .= 'width:' . $options['form_button_width'] . 'px ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['form_button_width_font_size'] ) ) {
			$dynamic_css .= '#rtsb-redeem-gift-card-form #rtsb-redeem-button{';
			$dynamic_css .= 'font-size:' . $options['form_button_width_font_size'] . 'px ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['form_button_bg_color'] ) ) {
			$dynamic_css .= '#rtsb-redeem-gift-card-form #rtsb-redeem-button{';
			$dynamic_css .= 'background-color:' . $options['form_button_bg_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['form_button_text_color'] ) ) {
			$dynamic_css .= '#rtsb-redeem-gift-card-form #rtsb-redeem-button{';
			$dynamic_css .= 'color:' . $options['form_button_text_color'] . ' ! important;';
			$dynamic_css .= '}';
		}

		if ( ! empty( $options['form_button_border_color'] ) ) {
			$dynamic_css .= '#rtsb-redeem-gift-card-form #rtsb-redeem-button{';
			$dynamic_css .= 'border-color:' . $options['form_button_border_color'] . ' ! important;';
			$dynamic_css .= '}';
		}

		if ( ! empty( $options['form_button_hover_bg_color'] ) ) {
			$dynamic_css .= '#rtsb-redeem-gift-card-form #rtsb-redeem-button:hover{';
			$dynamic_css .= 'background-color:' . $options['form_button_hover_bg_color'] . ' ! important;';
			$dynamic_css .= '}';
		}

		if ( ! empty( $options['form_button_hover_text_color'] ) ) {
			$dynamic_css .= '#rtsb-redeem-gift-card-form #rtsb-redeem-button:hover{';
			$dynamic_css .= 'color:' . $options['form_button_hover_text_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['form_button_hover_border_color'] ) ) {
			$dynamic_css .= '#rtsb-redeem-gift-card-form #rtsb-redeem-button:hover{';
			$dynamic_css .= 'border-color:' . $options['form_button_hover_border_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		// Pricing Button.
		if ( ! empty( $options['pricing_button_bg_color'] ) ) {
			$dynamic_css .= '.gift-card-option .gift-card-button{';
			$dynamic_css .= 'background-color:' . $options['pricing_button_bg_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['pricing_button_text_color'] ) ) {
			$dynamic_css .= '.gift-card-option .gift-card-button{';
			$dynamic_css .= 'color:' . $options['pricing_button_text_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['pricing_button_border_color'] ) ) {
			$dynamic_css .= '.gift-card-option .gift-card-button{';
			$dynamic_css .= 'border-color:' . $options['pricing_button_border_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['pricing_button_hover_bg_color'] ) ) {
			$dynamic_css .= '.gift-card-option .gift-card-button:hover, .gift-card-option input[type=radio]:checked + .gift-card-button{';
			$dynamic_css .= 'background-color:' . $options['pricing_button_hover_bg_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['pricing_button_hover_text_color'] ) ) {
			$dynamic_css .= '.gift-card-option .gift-card-button:hover, .gift-card-option input[type=radio]:checked + .gift-card-button{';
			$dynamic_css .= 'color:' . $options['pricing_button_hover_text_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['pricing_button_hover_border_color'] ) ) {
			$dynamic_css .= '.gift-card-option .gift-card-button:hover, .gift-card-option input[type=radio]:checked + .gift-card-button {';
			$dynamic_css .= 'border-color:' . $options['pricing_button_hover_border_color'] . ' ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['pricing_button_height'] ) ) {
			$dynamic_css .= '.gift-card-option .gift-card-button{';
			$dynamic_css .= 'height:' . $options['pricing_button_height'] . 'px ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['pricing_button_font_size'] ) ) {
			$dynamic_css .= '.gift-card-option .gift-card-button{';
			$dynamic_css .= 'font-size:' . $options['pricing_button_font_size'] . 'px ! important;';
			$dynamic_css .= '}';
		}
		// Validity Text.
		if ( ! empty( $options['validity_text_color'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-validity{';
			$dynamic_css .= 'color:' . $options['validity_text_color'] . '! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['validity_text_font_size'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-validity{';
			$dynamic_css .= 'font-size:' . $options['validity_text_font_size'] . 'px ! important;';
			$dynamic_css .= '}';
		}
		if ( ! empty( $options['validity_text_margin'] ) ) {
			$dynamic_css .= '.rtsb-gift-card-validity{';
			$dynamic_css .= 'margin:' . $options['form_button_title_margin'];
			$dynamic_css .= '}';
		}

		if ( ! empty( $dynamic_css ) ) {
			wp_add_inline_style( $this->handle, $dynamic_css );
		}
	}
	/**
	 * Checks if any custom endpoints are active for the My Account page.
	 *
	 * @param bool $bool Existing value.
	 *
	 * @return bool
	 */
	public static function enable_custom_endpoints( $bool ) {

		if ( BuilderFunPro::is_account_custom_endpoint( 'rtsb-gift-cards' ) ) {
			return true;
		}
		if ( BuilderFunPro::is_account_custom_endpoint( 'rtsb-view-card' ) ) {
			return true;
		}
		return $bool;
	}

	/**
	 * Registers custom templates for custom endpoints on the My Account page.
	 *
	 * @param array $type An array of existing templates.
	 * @return array
	 */
	public function builder_page_type( $type ) {
		if ( BuilderFunPro::is_account_custom_endpoint( 'rtsb-gift-cards' ) ) {
			$type = 'myaccount-rtsb-gift-cards';
		}
		if ( BuilderFunPro::is_account_custom_endpoint( 'rtsb-view-card' ) ) {
			$type = 'myaccount-rtsb-view-card';
		}
		return $type;
	}

	/**
	 * Registers custom templates for custom endpoints on the My Account page.
	 *
	 * @param array $templates An array of existing templates.
	 * @return array
	 */
	public function register_custom_template( $templates ) {
		$templates['myaccount-rtsb-gift-cards'] = esc_html__( 'My Account - Gift Card', 'shopbuilder-pro' );
		$templates['myaccount-rtsb-view-card']  = esc_html__( 'My Account - Gift Card Details', 'shopbuilder-pro' );
		return $templates;
	}
	/**
	 * Run Cron Function
	 *
	 * @return void
	 */
	public function run_cron() {
		$action = 'rtsb_check_gift_card_expiry';
		if ( ! wp_next_scheduled( $action ) ) {
			// ‘hourly’, ‘twicedaily’, and ‘daily’.
			wp_schedule_event( time(), 'twicedaily', $action );
			FnsPro::add_to_scheduled_hook_list( $action );
		}
	}
	/**
	 * Add custom cron job schedule.
	 *
	 * @param array $schedules Existing schedules.
	 *
	 * @return array
	 */
	public function test_custom_cron_job_recurrence( $schedules ) {
		if ( ! isset( $schedules['10sec'] ) ) {
			$schedules['10sec'] = [
				'display'  => __( 'Every 10 Seconds', 'shopbuilder-pro' ),
				'interval' => 10,
			];
		}
		return $schedules;
	}
	/**
	 * Processes gift cards that are expiring in a set number of days
	 * and sends reminder emails to the recipients.
	 *
	 * @return void
	 */
	public function process_expiring_gift_cards() {
		// Get all WC email instances.
		$wc_emails = WC()->mailer()->get_emails();
		$email     = $wc_emails['GiftCardValidity'];
		if ( empty( $email ) || ! $email->is_enabled() ) {
			return;
		}

		$days_before = $email->get_option( 'send_before_expiration_days' ); // You can replace this with a dynamic setting.
		$target_date = gmdate( 'Y-m-d', strtotime( "+$days_before days" ) );
		$current     = gmdate( 'Y-m-d', FnsPro::currentTimestamp() );
		// Get the gift card posts using meta query by order ID.
		$gift_cards = get_posts(
			[
				'post_type'      => GiftCardFns::GIFT_CARD_POST_TYPE,
				'posts_per_page' => -1, // Retrieve all posts.
				'post_status'    => 'publish', // Get posts with draft status.
				'orderby'        => 'ID',
				'order'          => 'ASC',
				'fields'         => 'ids', // Only return post IDs for performance.
				'meta_query'     => [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
					'relation' => 'AND', // Ensures both conditions are met.
					[
						'key'     => 'expiration_date',
						'value'   => '',
						'compare' => '!=', // Ensure expiration date is not empty.
					],
					[
						'key'     => 'expiration_date',
						'value'   => [ $current, $target_date ],
						'compare' => 'BETWEEN',
						'type'    => 'DATE',
					],
					[
						'key'     => 'rtsb_gift_card_remaining_balance',
						'value'   => 0,
						'compare' => '>',
						'type'    => 'NUMERIC',
					],
					[
						'key'     => 'rtsb_gift_card_as_virtual',
						'value'   => 'on',
						'compare' => '==',
					],
					[
						'key'     => 'rtsb_gift_card_recipient_email',
						'value'   => '',
						'compare' => '!=',
					],
					[
						'key'     => 'rtsb_gift_card_reminder_email_send', // Replace with the actual meta key you want to check for non-existence.
						'compare' => 'NOT EXISTS', // Check that this meta key does not exist.
					],
				],
			]
		);
		if ( empty( $gift_cards ) ) {
			return;
		}
		foreach ( $gift_cards as $gift_id ) {
			do_action( 'rtsb/gift/card/reminder', $gift_id, $current, $target_date );
		}
	}
	/**
	 * Flush rewrite rules if custom page results in 404.
	 */
	public function maybe_flush_on_404() {
		// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized, WordPress.Security.ValidatedSanitizedInput.InputNotValidated, WordPress.Security.ValidatedSanitizedInput.MissingUnslash
		$current_uri      = untrailingslashit( $_SERVER['REQUEST_URI'] );
		$my_account_uri   = untrailingslashit( parse_url( wc_get_page_permalink( 'myaccount' ), PHP_URL_PATH ) ); // phpcs:ignore WordPress.WP.AlternativeFunctions.parse_url_parse_url
		$is_under_account = strpos( $current_uri, $my_account_uri ) === 0;
		if ( is_404() && $is_under_account ) {
			flush_rewrite_rules();
			// Optional: Redirect to same URL after flush to re-resolve.
			$view_order_url = wc_get_endpoint_url( 'rtsb-view-card', '', wc_get_page_permalink( 'myaccount' ) );
			wp_safe_redirect( $view_order_url );
			exit;
		}
	}

	/**
	 * Register the custom Gift Card email class with WooCommerce.
	 *
	 * This adds the GiftCardEmail class to WooCommerce's email system,
	 * enabling it to send gift card emails to recipients.
	 *
	 * @param array $emails Existing WooCommerce email classes.
	 * @return array Modified email classes with GiftCardEmail added.
	 */
	public function rtsb_register_gift_card_email( $emails ) {
		$emails['GiftCardEmail']    = new GiftCardEmail();
		$emails['GiftCardValidity'] = new GiftCardValidityEmail();
		$emails['GiftCardActivity'] = new GiftCardUsesActivity();
		return $emails;
	}
	/**
	 * Adds formatted gift card metadata to order item display.
	 *
	 * Retrieves and formats custom gift card fields (sender, recipient, message)
	 * for clear display in customer-facing areas like order emails and views.
	 *
	 * @param array  $meta_data Existing metadata for the line item.
	 * @param object $item      WooCommerce order item object.
	 * @return array Updated metadata including formatted gift card details.
	 */
	public function format_gift_card_meta( $meta_data, $item ) {
		$fields = [
			'rtsb_gift_card_sender_name'     => esc_html__( 'Sender Name', 'shopbuilder-pro' ),
			'rtsb_gift_card_recipient_email' => esc_html__( 'Recipient Email', 'shopbuilder-pro' ),
			'rtsb_gift_card_recipient_name'  => esc_html__( 'Recipient Name', 'shopbuilder-pro' ),
			'rtsb_gift_card_message'         => esc_html__( 'Card Message', 'shopbuilder-pro' ),
		];
		// Remove any existing gift card keys from meta_data.
		$meta_data = array_filter(
			$meta_data,
			function ( $meta ) use ( $fields ) {
				return ! in_array( $meta->key, array_keys( $fields ), true );
			}
		);
		foreach ( $fields as $meta_key => $label ) {
			$value = $item->get_meta( $meta_key, true );
			if ( $value ) {
				$meta_data[] = (object) [
					'key'           => $meta_key,
					'value'         => '',
					'display_key'   => $label,
					'display_value' => wp_strip_all_tags( $value ),
				];
			}
		}

		return $meta_data;
	}
	/**
	 * @param bool   $purchasable purchasable.
	 * @param  object $product Product.
	 * @return mixed|true
	 */
	public function custom_product_purchasable( $purchasable, $product ) {
		if ( 'rtsb-gift-card' === $product->get_type() ) {
			return true;
		}
		return $purchasable;
	}
	/**
	 * Set the correct WooCommerce product class for Gift Card products.
	 *
	 * @param string $classname    The default product class name.
	 * @param string $product_type The product type being checked.
	 *
	 * @return string Modified product class name if it's a Gift Card.
	 */
	public function set_gift_card_product_class( $classname, $product_type ) {
		if ( GiftCardFns::GIFT_CARD_KEY === $product_type ) {
			return GiftCardWCProduct::class; // Use the correct class name.
		}
		return $classname;
	}

	/**
	 * Add the "Gift card" type to the WooCommerce product type selector.
	 *
	 * @param array $types The current product types.
	 *
	 * @return array Modified product types array.
	 */
	public function add_gift_card_product_type( $types ) {
		$types[ GiftCardFns::GIFT_CARD_KEY ] = esc_html__( 'Gift Card', 'shopbuilder-pro' );
		return $types;
	}
}
