<?php
/**
 * Customize My Account Module Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\GiftCard;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Module Class.
 */
class RtsbGiftCard {
	/**
	 * @var array
	 */
	private $cache = [];

	/**
	 * Expiration
	 *
	 * @var int the timestamp for gift card valid use
	 */
	public $expiration = 0;

	/**
	 * ID
	 *
	 * @var int the gift card id
	 */
	public $ID = 0;
	/**
	 * Gift_card_number
	 *
	 * @var string the gift card code
	 */
	public $gift_card_number = '';

	/**
	 * Total_amount
	 *
	 * @var float the gift card amount
	 */
	public $total_balance = 0.00;

	/**
	 * Total_balance
	 *
	 * @var float the gift card current balance
	 */
	protected $current_balance = 0.00;
	/**
	 * Default gift card data.
	 *
	 * @var array
	 */
	protected $defaults = [
		'card_number'       => '',
		'product_name'      => '',
		'recipient_email'   => '',
		'card_amount'       => '',
		'sender_name'       => '',
		'recipient_name'    => '',
		'card_message'      => '',
		'remaining_balance' => '',
		'expiration_date'   => '',
		'order_id'          => '',
		'virtual'           => '',
		'card_note'         => '',
	];
	/**
	 * Module Class Constructor.
	 */
	public function __construct( $args = [] ) {
		$post = false;
		if ( 0 < ( $args['ID'] ?? 0 ) ) {
			$post = get_post( $args['ID'] );
		} elseif ( isset( $args['gift_card_number'] ) ) {
			$post = $this->card_by_code( $args['gift_card_number'] );
		}
		if ( $post && $post->ID > 0 ) {
			$this->ID               = $post->ID;
			$this->gift_card_number = $post->post_title;
			$current_balance        = $this->get_card_data( 'remaining_balance' );
			if ( ! empty( $current_balance ) ) {
				$this->current_balance = $current_balance;
			}
			$total_balance = $this->get_card_data( 'card_amount' );
			if ( ! empty( $total_balance ) ) {
				$this->total_balance = $total_balance;
			}
		}
	}
	/**
	 * Retrieves all gift card data or a specific value by key.
	 *
	 * If the gift card post does not exist, returns default values.
	 *
	 * @param string|null $key Optional. Specific data key to retrieve.
	 * @return mixed Array of all data or the value of the requested key.
	 */
	public function get_card_data( $key = null ) {
		if ( ! $this->exists() ) {
			return $this->defaults[ $key ] ?? $this->defaults;
		}
		$cache_key = 'rtsb_get_gift_card_data';
		if ( ! empty( $this->cache[ $cache_key ] ) ) {
			return $this->cache[ $cache_key ][ $key ] ?? $this->cache[ $cache_key ];
		}
		$order_id                  = get_post_meta( $this->ID, 'order_id', true );
		$expiration_date           = get_post_meta( $this->ID, 'expiration_date', true );
		$remaining_balance         = get_post_meta( $this->ID, 'rtsb_gift_card_remaining_balance', true );
		$card_amount               = get_post_meta( $this->ID, 'rtsb_gift_card_amount', true );
		$virtual                   = get_post_meta( $this->ID, 'rtsb_gift_card_as_virtual', true );
		$email                     = get_post_meta( $this->ID, 'rtsb_gift_card_recipient_email', true );
		$sender_name               = get_post_meta( $this->ID, 'rtsb_gift_card_sender_name', true );
		$recipient_name            = get_post_meta( $this->ID, 'rtsb_gift_card_recipient_name', true );
		$card_message              = get_post_meta( $this->ID, 'rtsb_gift_card_message', true );
		$card_note                 = get_post_meta( $this->ID, 'rtsb_gift_card_note', true );
		$product_id                = get_post_meta( $this->ID, 'product_id', true );
		$reminder_email            = get_post_meta( $this->ID, 'rtsb_gift_card_reminder_email_send', true );
		$product_name              = $product_id ? get_the_title( $product_id ) : '';
		$card_number               = get_the_title( $this->ID );
		$data                      = wp_parse_args(
			[
				'card_number'       => $card_number,
				'product_name'      => $product_name,
				'product_id'        => $product_id,
				'recipient_email'   => $email,
				'card_amount'       => $card_amount,
				'sender_name'       => $sender_name,
				'recipient_name'    => $recipient_name,
				'card_message'      => $card_message,
				'remaining_balance' => $remaining_balance,
				'expiration_date'   => $expiration_date,
				'order_id'          => $order_id,
				'virtual'           => $virtual,
				'card_note'         => $card_note,
				'reminder_send'     => $reminder_email,
			],
			$this->defaults
		);
		$this->cache[ $cache_key ] = $data;
		return $data[ $key ] ?? $data;
	}
	/**
	 * @return int[]|\WP_Post|false
	 */
	private function card_by_code( $card_number ) {
		if ( empty( $card_number ) ) {
			return [];
		}
		$post = get_posts(
			[
				'title'       => $card_number,
				'post_type'   => GiftCardFns::GIFT_CARD_POST_TYPE,
				'post_status' => 'publish',
			]
		);
		if ( is_array( $post ) && isset( $post[0] ) ) {
			$post = $post[0];
		}
		if ( $post && is_object( $post ) ) {
			return $post;
		}
		return [];
	}
	/**
	 * Check if the gift card is expired
	 */
	public function is_expired() {
		$expiration = $this->get_card_data( 'expiration_date' );
		if ( ! $expiration ) {
			return false;
		}
		return current_time( 'timestamp' ) > strtotime( $expiration );
	}
	/**
	 * Update and store the new balance
	 *
	 * @param float $new_amount new_amount.
	 */
	public function update_balance( $new_amount ) {
		if ( $this->exists() ) {
			update_post_meta( $this->ID, 'rtsb_gift_card_remaining_balance', $new_amount );
		}
	}

	/**
	 * @param $info
	 * @return void
	 */
	public function update_card_info( $info = [] ) {
		if ( ! $this->exists() || ! is_array( $info ) || empty( $info ) ) {
			return;
		}
		foreach ( $info as $key => $value ) {
			update_post_meta( $this->ID, $key, $value );
		}
	}
	/**
	 * @return int
	 */
	public function card_total_amount() {
		if ( $this->exists() ) {
			return $this->total_balance;
		}
		return 0;
	}
	/**
	 * @return int
	 */
	public function card_balance() {
		if ( $this->exists() ) {
			return $this->current_balance;
		}
		return 0;
	}

	/**
	 * Check if the user is registered as the gift card owner
	 *
	 * @param int $user_id user_id.
	 *
	 * @return bool
	 * @since  1.0.0
	 */

	/**
	 * Retrieve the gift card code
	 *
	 * @return array
	 * @since  1.0.0
	 */
	public function add_card_activity( $order_id, $amount, $new_balance ) {
		if ( $this->exists() ) {
			$order         = wc_get_order( $order_id );
			$customer_user = $order->get_customer_id();
			$order_date    = $order->get_date_modified();
			$status        = in_array( $order->get_status(), [ 'refunded', 'cancelled', 'failed' ], true ) ? __( 'Refunded', 'shopbuilder-pro' ) : __( 'Redeemed', 'shopbuilder-pro' );
			if ( ! empty( $customer_user ) ) {
				$user          = get_user( $customer_user );
				$customer_user = $user->display_name;
			} else {
				$customer_user = $order->get_billing_email();
			}
			$format = get_option( 'date_format' ) . ' ' . get_option( 'time_format' );
			$data   = [
				'order'           => $order_id,
				'user'            => $customer_user,
				'uses_amount'     => $amount,
				'current_balance' => $new_balance,
				'status'          => $status,
				'date'            => $order_date->date( $format ),
			];
			add_post_meta( $this->ID, 'rtsb_card_activities', $data );
		}
	}
	/**
	 * Retrieve the gift card code
	 *
	 * @return array
	 * @since  1.0.0
	 */
	public function get_card_activity() {
		$data = get_post_meta( $this->ID, 'rtsb_card_activities' );
		if ( empty( $data ) ) {
			return [];
		}
		return $data;
	}
	/**
	 * Retrieve the gift card code
	 *
	 * @return string
	 * @since  1.0.0
	 */
	public function get_code() {
		if ( $this->exists() ) {
			return $this->gift_card_number;
		}
	}
	/**
	 * The gift card exists
	 *
	 * @return bool
	 * @since  1.0.0
	 */
	public function is_valid() {
		return $this->exists() && ! $this->is_expired();
	}
	/**
	 * The gift card exists
	 *
	 * @return bool
	 * @since  1.0.0
	 */
	public function is_virtual() {
		return $this->exists() && 'on' === $this->get_card_data( 'virtual' );
	}

	/**
	 * The gift card exists
	 *
	 * @return bool
	 * @since  1.0.0
	 */
	public function exists() {
		return $this->ID > 0;
	}
}
