<?php
/**
 * Customize My Account Module Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\GiftCard;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Back-order Module Class.
 */
class giftCardFrontEnd {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		/**
		 * Show the gift card product frontend template
		 */
		add_action( 'woocommerce_rtsb-gift-card_add_to_cart', [ $this, 'show_gift_card_product_template' ] );
		add_action( 'rtsb_before_add_to_cart_button', [ $this, 'show_gift_card_pricing' ] );
		/**
		 * Gift Card Pricing.
		 */
		add_filter( 'woocommerce_add_cart_item_data', [ $this, 'add_gift_card_amount_to_cart' ] );
		add_action( 'woocommerce_before_calculate_totals', [ $this, 'set_gift_card_price' ] );

		/**
		 * Set the price when a gift card product is added to the cart.
		 */
		add_filter( 'woocommerce_add_cart_item', [ $this, 'set_price_in_cart' ], 10, 1 );

		add_action( 'woocommerce_proceed_to_checkout', [ $this, 'apply_gift_card_form' ] );
		add_action( 'woocommerce_review_order_before_payment', [ $this, 'apply_gift_card_form' ] );

		/**
		 * Show gift card amount usage on cart totals - cart page
		 */
		add_action( 'woocommerce_cart_totals_before_order_total', [ $this, 'show_gift_card_amount_on_cart_totals' ] );
		/**
		 * Show gift card amount usage on cart totals - checkout page
		 */
		add_action( 'woocommerce_review_order_before_order_total', [ $this, 'show_gift_card_amount_on_cart_totals' ] );
		/**
		 * Cart Button
		 */
		add_filter( 'woocommerce_product_add_to_cart_text', [ $this, 'modify_gift_card_button_text' ], 10, 2 );
		/**
		 * Virtual gift card implementaion.
		 */
		add_action( 'woocommerce_checkout_create_order_line_item', [ $this, 'add_gift_card_meta_to_order_item' ], 10, 3 );
	}

	/**
	 * @param $item
	 * @param $cart_item_key
	 * @param $values
	 * @param $order
	 * @return void
	 */
	public function add_gift_card_meta_to_order_item( $item, $cart_item_key, $values ) {
		if ( isset( $values['rtsb_gift_card_recipient_email'] ) ) {
			// Add it as order item meta.
			$item->add_meta_data( 'rtsb_gift_card_recipient_email', $values['rtsb_gift_card_recipient_email'], true );
		}
		// Repeat for other keys if needed.
		if ( isset( $values['rtsb_gift_card_sender_name'] ) ) {
			$item->add_meta_data( 'rtsb_gift_card_sender_name', $values['rtsb_gift_card_sender_name'], true );
		}
		if ( isset( $values['rtsb_gift_card_recipient_name'] ) ) {
			$item->add_meta_data( 'rtsb_gift_card_recipient_name', $values['rtsb_gift_card_recipient_name'], true );
		}
		if ( isset( $values['rtsb_gift_card_message'] ) ) {
			$item->add_meta_data( 'rtsb_gift_card_message', $values['rtsb_gift_card_message'], true );
		}
	}
	/**
	 * Modifies the Add to Cart button text for gift card products.
	 *
	 * @param string     $text The default button text.
	 * @param WC_Product $product The WooCommerce product object.
	 * @return string Modified button text for gift cards.
	 */
	public function modify_gift_card_button_text( $text, $product ) {
		if ( $product && method_exists( $product, 'get_type' ) && $product->get_type() === GiftCardFns::GIFT_CARD_KEY ) {
			return esc_html__( 'Buy Gift Card', 'shopbuilder-pro' );
		}
		return $text;
	}
	/**
	 * Show gift card amount usage on cart totals
	 */
	/**
	 * Show gift card amount usage on cart totals - cart page.
	 */
	public function show_gift_card_amount_on_cart_totals() {
		$gift_cards = GiftCardFns::get_gift_cards_from_session();
		if ( empty( $gift_cards ) ) {
			return;
		}
		foreach ( $gift_cards as $code => $current_balance ) {
			// Apply filters to customize the gift card label.
			$label  = apply_filters( 'rtsb_cart_totals_gift_card_label', esc_html__( 'Gift Card:', 'shopbuilder-pro' ) . ' ' . esc_html( $code ), $code );
			$amount = isset( WC()->cart->applied_gift_cards_amounts[ $code ] ) ? - WC()->cart->applied_gift_cards_amounts[ $code ] : 0;
			// Display formatted price.
			$value = wc_price( -$amount ) . ' <a href="' . esc_url(
				add_query_arg( 'remove_gift_card_code', urlencode( $code ), wc_get_cart_url() )
			) . '" class="rtsb-remove-gift-card" data-gift-card-code="' . esc_attr( $code ) . '">' .
				apply_filters( 'rtsb_remove_gift_card_text', esc_html__( '[Remove]', 'shopbuilder-pro' ) ) . '</a>';

			?>
			<tr class="rtsb-gift-card-applied">
				<th><?php echo wp_kses_post( $label ); ?></th>
				<td>-<?php echo wp_kses_post( $value ); ?></td>
			</tr>
			<?php
		}
	}
	/**
	 * @return void
	 */
	public function apply_gift_card_form() {
		if ( GiftCardFns::cart_has_gift_card() ) {
            WC()->session->set( 'rtsb_gift_card_code', [] );
			return;
		}
		$data = [];
		Fns::load_template( 'gift-card/apply-gift-card', $data, false, '', rtsbpro()->get_plugin_template_path() );
	}

	/**
	 * Set the real amount for the gift card product
	 *
	 * @param array $cart_item Cart Line Item.
	 *
	 * @since 1.5.0
	 * @return mixed
	 */
	public function set_price_in_cart( $cart_item ) {
		if ( isset( $cart_item['data'] ) ) {
			if ( $cart_item['data'] instanceof GiftCardWCProduct && isset( $cart_item['rtsb_gift_card_amount'] ) ) {
				$cart_item['data']->update_meta_data( 'price', $cart_item['rtsb_gift_card_amount'] );
				$cart_item['data']->save_meta_data();
				$cart_item['data']->set_price( $cart_item['rtsb_gift_card_amount'] );
			}
		}
		return $cart_item;
	}
	/**
	 * Set the gift card price in the cart based on the selected amount.
	 *
	 * @param WC_Cart $cart_object WooCommerce cart object.
	 * @return void
	 */
	public function set_gift_card_price( $cart_object ) {
		foreach ( $cart_object->get_cart() as $cart_item ) {
			if ( isset( $cart_item['rtsb_gift_card_amount'] ) ) {
				$cart_item['data']->set_price( floatval( $cart_item['rtsb_gift_card_amount'] ) );
			}
		}
	}

	/**
	 * Add the selected gift card amount to cart item data.
	 *
	 * @param array $cart_item_data The cart item data.
	 * @return array Modified cart item data with gift card amount.
	 */
	public function add_gift_card_amount_to_cart( $cart_item_data ) {
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
		if ( isset( $_POST['rtsb_gift_card_amount'] ) ) {
			$cart_item_data['rtsb_gift_card_amount'] = sanitize_text_field( wp_unslash( $_POST['rtsb_gift_card_amount'] ) );  // phpcs:ignore WordPress.Security.NonceVerification.Missing
		}
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
		if ( isset( $_POST['rtsb_gift_card_recipient_name'] ) ) {
			$cart_item_data['rtsb_gift_card_recipient_name'] = sanitize_text_field( wp_unslash( $_POST['rtsb_gift_card_recipient_name'] ) );  // phpcs:ignore WordPress.Security.NonceVerification.Missing
		}
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
		if ( isset( $_POST['rtsb_gift_card_recipient_email'] ) ) {
			$cart_item_data['rtsb_gift_card_recipient_email'] = sanitize_text_field( wp_unslash( $_POST['rtsb_gift_card_recipient_email'] ) );  // phpcs:ignore WordPress.Security.NonceVerification.Missing
		}
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
		if ( isset( $_POST['rtsb_gift_card_sender_name'] ) ) {
			$cart_item_data['rtsb_gift_card_sender_name'] = sanitize_text_field( wp_unslash( $_POST['rtsb_gift_card_sender_name'] ) );  // phpcs:ignore WordPress.Security.NonceVerification.Missing
		}
        // phpcs:ignore WordPress.Security.NonceVerification.Missing
		if ( isset( $_POST['rtsb_gift_card_message'] ) ) {
			$cart_item_data['rtsb_gift_card_message'] = sanitize_text_field( wp_unslash( $_POST['rtsb_gift_card_message'] ) );  // phpcs:ignore WordPress.Security.NonceVerification.Missing
		}

		return $cart_item_data;
	}
	/**
	 * @return mixed|true
	 */
	public function show_gift_card_pricing() {
		global $product;
		$get_post_meta = get_post_meta( $product->get_id(), GiftCardFns::GIFT_CARD_AMOUNTS, true );
		if ( empty( $get_post_meta ) || ! is_array( $get_post_meta ) ) {
			return;
		}
		$virtual       = $product && $product->is_virtual() ? 'on' : '';
		$options       = GiftCardFns::get_options();
		$title         = ! empty( $options['product_page_pricing_title'] ) ? $options['product_page_pricing_title'] : __( 'Choose Gift Amount', 'shopbuilder-pro' );
		$get_post_meta = array_unique( $get_post_meta );
		?>
		 <div class="gift-card-amount-title"> <?php echo esc_html( $title ); ?></div>
		<div class="gift-card-amount-options">
			<?php
			$first = true; // Flag to mark the first item.
			foreach ( $get_post_meta as $key => $value ) {
				?>
				<label class="gift-card-option">
					<input <?php echo $first ? 'checked' : ''; ?> name="rtsb_gift_card_amount" value="<?php echo esc_attr( $value ); ?>"  type="radio" hidden >
					<span class="gift-card-button"><?php Fns::print_html( wc_price( $value ), true ); ?></span>
				</label>
				<?php
				$first = false; // Set the flag to false after the first item.
			}
			?>
		</div>
		<?php $validity = (int) get_post_meta( $product->get_id(), '_rtsb_gift_card_validity', true ); ?>
		<div class="rtsb-gift-card-validity">
			<span class="gift-card-validity-label"><?php esc_html_e( 'Validity: ', 'shopbuilder-pro' ); ?></span>
			<span class="gift-card-validity-text" >
			<?php
			$card_message = $options['card_validity_message'] ?? '';
			$message      = str_replace( '{{validity}}', $validity, $card_message );
			if ( $validity ) {
				echo esc_html( $message );
			} else {
				esc_html_e( 'Lifetime', 'shopbuilder-pro' );
			}
			?>
			</span>
		</div>
		<?php if ( $virtual ) { ?>
			<div class="gift-card-virtual-options">
				<div class="gift-card-section-title"> <?php esc_html_e( 'Delivery Info', 'shopbuilder-pro' ); ?> </div>
				<div class="gift-card-section-subtitle" > <?php esc_html_e( 'Recipient Info', 'shopbuilder-pro' ); ?> </div>
				<div class="rtsb-field-wrapper">
					<label for="rtsb_gift_card_recipient_name" class="field-label">
						<?php esc_html_e( 'Recipient Name', 'shopbuilder-pro' ); ?>
					</label>
					<input type="text" id="rtsb_gift_card_recipient_name" name="rtsb_gift_card_recipient_name" class="rtsb-input-field" value="" placeholder="<?php esc_html_e( 'Recipient Name', 'shopbuilder-pro' ); ?>"/>
				</div>
				<div class="rtsb-field-wrapper">
					<label for="rtsb_gift_card_recipient_email" class="field-label"><?php esc_html_e( 'Recipient\'s email', 'shopbuilder-pro' ); ?></label>
					<input type="text" id="rtsb_gift_card_recipient_email" name="rtsb_gift_card_recipient_email" class="rtsb-input-field" value="" placeholder="<?php esc_html_e( 'Email', 'shopbuilder-pro' ); ?>">
				</div>
				<div class="gift-card-section-subtitle" > <?php esc_html_e( 'Your Info', 'shopbuilder-pro' ); ?> </div>
				<div class="rtsb-field-wrapper">
					<label for="rtsb_gift_card_sender_name" class="field-label"><?php esc_html_e( 'Sender Name', 'shopbuilder-pro' ); ?> </label>
					<input type="text" id="rtsb_gift_card_sender_name" name="rtsb_gift_card_sender_name" class="rtsb-input-field" value="" placeholder="<?php esc_html_e( 'Sender Name', 'shopbuilder-pro' ); ?>">
				</div>
				<div class="rtsb-field-wrapper">
					<label class="field-label"><?php esc_html_e( 'Sender Message', 'shopbuilder-pro' ); ?></label>
					<textarea name="rtsb_gift_card_message" class="rtsb-input-field" value="" placeholder="<?php esc_html_e( 'Message', 'shopbuilder-pro' ); ?>"></textarea>
				</div>
			</div>
			<?php
		}
	}


	/**
	 * Show the gift card product frontend template
	 */
	public function show_gift_card_product_template() {
		global $product;
		if ( 'rtsb-gift-card' === $product->get_type() ) {
			$data = [];
			Fns::load_template( 'gift-card/add-to-cart', $data, false, '', rtsbpro()->get_plugin_template_path() );
		}
	}
}
