<?php
/**
 * Main MiniCart class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\MiniCart;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Main FilterHooks class.
 */
class MiniCart {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * @var array|mixed
	 */
	private array $options;

	/**
	 * Asset Handle
	 *
	 * @var string
	 */
	private $handle = 'rtsb-mini-cart';

	/**
	 * Class constructor
	 */
	private function __construct() {
		$this->options = Fns::get_options( 'modules', 'mini_cart' );
		MiniCartHooks::instance();
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_public_scripts' ], 99 );
		add_action( 'wp_footer', [ $this, 'render' ] );
	}


	/**
	 * Public CSS
	 *
	 * @return void
	 */
	public function enqueue_public_scripts() {
		// Enqueue assets.
		$this->handle = Fns::enqueue_module_assets(
			$this->handle,
			'mini-cart',
			[
				'context' => rtsbpro(),
				'version' => RTSBPRO_VERSION,
			]
		);

		if ( ! apply_filters( 'rtsb/module/minicart/show_button', true ) ) {
			return;
		}

		wp_enqueue_script( 'wc-cart-fragments' );

		$opt         = $this->options;
		$dynamic_css = '';

		// Float button style.
		$pre = '#rtsb-cart-float-menu';

		if ( ! empty( $opt['mini_cart_float_btn_radius'] ) ) {
			$dynamic_css .= "$pre {border-radius:{$opt['mini_cart_float_btn_radius']};overflow:hidden}";
		}
		if ( ! empty( $opt['mini_cart_float_bg'] ) ) {
			$dynamic_css .= "{$pre} {--rtsb-float-bg:{$opt['mini_cart_float_bg']};}";
		}
		if ( ! empty( $opt['mini_cart_float_bg_hover'] ) ) {
			$dynamic_css .= "{$pre} {--rtsb-float-bg-hover:{$opt['mini_cart_float_bg_hover']};}";
		}
		if ( ! empty( $opt['mini_cart_btn_width'] ) ) {
			$dynamic_css .= "$pre {min-width:{$opt['mini_cart_btn_width']}px}";
		}

		// MiniCart Wrapper.

		$pre = '#rtsb-side-content-area-id';

		if ( ! empty( $opt['mini_cart_primary'] ) ) {
			$dynamic_css .= "body {--rtsb-drawer-primary:{$opt['mini_cart_primary']}}";
		}

		if ( ! empty( $opt['mini_cart_secondary'] ) ) {
			$dynamic_css .= "body {--rtsb-drawer-secondary:{$opt['mini_cart_secondary']}}";
		}

		if ( ! empty( $dynamic_css ) ) {
			wp_add_inline_style( $this->handle, $dynamic_css );
		}
	}


	/**
	 * Mini cart markup
	 *
	 * @return void
	 */
	public function render() {
		add_filter( 'woocommerce_widget_cart_is_hidden', '__return_true' );

		$opt = $this->options;

		// Check include page.
		$include_page         = Fns::multiselect_settings_field_value( $opt['mini_cart_include_page'] ?? [] );
		$include_archive_page = Fns::multiselect_settings_field_value( $opt['mini_cart_include_archive'] ?? [] );
		$include_details_page = Fns::multiselect_settings_field_value( $opt['mini_cart_include_details_page'] ?? [] );
		$exclude_page         = Fns::multiselect_settings_field_value( $opt['mini_cart_exclude_page'] ?? [] );

		$post_id   = get_queried_object_id();
		$post_type = get_post_type( $post_id );
		$obj_id    = Fns::wpml_object_id( $post_id, $post_type );
		if ( $obj_id ) {
			if ( in_array( $obj_id, $include_page, true ) ) {
				$include_page[] = $post_id;
			}
			if ( in_array( $obj_id, $exclude_page, true ) ) {
				$exclude_page[] = $post_id;
			}
		}

		if (
			( ! empty( $exclude_page ) && ( is_page() && is_page( $exclude_page ) ) ) ||
				// phpcs:ignore Generic.CodeAnalysis.RequireExplicitBooleanOperatorPrecedence.MissingParentheses, WordPress.PHP.StrictInArray.MissingTrueStrict
				( ( is_404() && in_array( 'error', $exclude_page ) ) ) && ( ! empty( $include_page ) && ! ( is_page() && is_page( $include_page ) ) ) ||
				is_cart() ||
				is_checkout()
		) {
			return;
		}

		if (
			// phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
			( ! empty( $include_page ) && ( ( is_page() && ! is_page( $include_page ) ) || ( is_404() && ! in_array( 'error', $include_page ) ) ) ) ||
			// phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
			( ! empty( $include_archive_page ) && ( ( is_archive() && ! is_post_type_archive( $include_archive_page ) ) || ( is_home() && ! in_array( 'post', $include_archive_page ) ) ) ) ||
			( ! empty( $include_details_page ) && is_single() && ! is_singular( $include_details_page ) )
		) {
			return;
		}

		$count_label          = is_object( WC()->cart ) && ( WC()->cart->get_cart_contents_count() < 2 ) ? __( 'Item', 'shopbuilder-pro' ) : __( 'Items', 'shopbuilder-pro' );
		$cart_drawer_classes  = ' ' . ( $opt['mini_cart_drawer_style'] ?? 'style1' );
		$cart_drawer_classes .= ' ' . ( $opt['mini_cart_open_style'] ?? 'open-always' );

		$rtsb_float_classes  = ' ' . ( $opt['mini_cart_position'] ?? 'left_center' );
		$rtsb_float_classes .= ' ' . ( $opt['mini_cart_float_btn_style'] ?? 'style1' );

		if ( ! empty( $opt['mini_cart_show_on_mobile'] ) && 'on' !== $opt['mini_cart_show_on_mobile'] ) {
			$rtsb_float_classes  .= ' rtrb-hide-mobile';
			$cart_drawer_classes .= ' rtrb-hide-mobile';
		}

		$default_image_path = rtsbpro()->get_assets_uri( 'images/mini-cart-loading.gif' );
		if ( ! empty( $opt['mini_cart_loading_image'] ) ) {
			$loading_json    = json_decode( stripslashes( $opt['mini_cart_loading_image'] ), true );
			$loading_img_src = $loading_json['source'] ?? $default_image_path;
		} else {
			$loading_img_src = $default_image_path;
		}

		$has_ovelay      = Fns::get_options_by_default_val( $opt, 'mini_cart_overlay_visibility', 'on' );
		$has_flying_Cart = Fns::get_options_by_default_val( $opt, 'mini_cart_flying_animation', 'on' );

		?>
		<?php
		if ( empty( $opt['hide_mini_cart'] ) ) {
			?>

			<div id="rtsb-cart-float-menu" class="rtsb-cart-float-menu <?php echo esc_attr( $rtsb_float_classes . ( 'on' === $has_flying_Cart ? ' rtsb-flying-cart' : '' ) ); ?>">
				<div class="rtsb-cart-float-inner">
					<span class="cart-icon">
						<span class="cart-icon-svg"></span>
						<span class="cart-number-wrapper">
							<span class="rtsb-cart-icon-num">
								<?php Fns::print_html( WC()->cart->get_cart_contents_count() ); ?>
							</span>
							<span class="item-label"><?php echo esc_html( $count_label ); ?></span>
						</span>
					</span>
					<span class="rtsb-cart-icon-total">
						<?php
						// phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
						echo wc_price( WC()->cart->get_cart_contents_total() );
						?>
					</span>
				</div>
			</div>
		<?php } ?>

		<!-- Minicart Drawer -->
		<div class="rtsb-drawer-container rtsb-minicart-drawer <?php echo esc_attr( $cart_drawer_classes ); ?>">
			<span class="close"></span>
			<div id="rtsb-side-content-area-id">
				<img class="loading-cart" src="<?php echo esc_url( $loading_img_src ); ?>" alt="<?php echo esc_attr__( 'Loadding...', 'shopbuilder-pro' ); ?>">
			</div>
		</div>

		<?php if ( 'on' === $has_ovelay ) : ?>
			<div class="drawer-overlay"></div>
			<?php
		endif;
	}
}
