<?php
/**
 * Main MiniCart class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\MiniCart;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Main FilterHooks class.
 */
class MiniCartHooks {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * @var array|mixed
	 */
	private static $options;

	/**
	 * Class constructor
	 */
	private function __construct() {

		self::$options = Fns::get_options( 'modules', 'mini_cart' );

		add_filter( 'woocommerce_loop_add_to_cart_args', [ __CLASS__, 'woocommerce_loop_add_to_cart_args' ] );

		// Load Template Ajax.
		add_action( 'wp_ajax_nopriv_rtsb_mini_cart_template', [ __CLASS__, 'rtsb_mini_cart_template' ] );
		add_action( 'wp_ajax_rtsb_mini_cart_template', [ __CLASS__, 'rtsb_mini_cart_template' ] );

		// Applly coupon Ajax.
		add_action( 'wp_ajax_rtsb_apply_coupon', [ __CLASS__, 'rtsb_apply_coupon_ajax_handler' ] );
		add_action( 'wp_ajax_nopriv_rtsb_apply_coupon', [ __CLASS__, 'rtsb_apply_coupon_ajax_handler' ] );

		// Cart count updated.
		add_filter( 'woocommerce_add_to_cart_fragments', [ __CLASS__, 'cart_count_update' ] );

		add_filter( 'body_class', [ __CLASS__, 'body_classes' ] );

		add_action( 'rtrb_minicart_quantity', [ __CLASS__, 'rtrb_minicart_quantity' ], 10, 3 );
		add_action( 'rtsb_minicart_extra_fields', [ __CLASS__, 'rtsb_minicart_extra_fields' ] );
	}


	public static function rtsb_apply_coupon_ajax_handler() {
		// Check if the coupon code is provided
		if ( isset( $_POST['coupon_code'] ) ) {
			$couponCode = sanitize_text_field( $_POST['coupon_code'] );

			// Apply the coupon
			$result = WC()->cart->apply_coupon( $couponCode );

			if ( $result ) {
				wp_send_json_success();
			} else {
				wp_send_json_error();
			}
		} else {
			wp_send_json_error();
		}
	}


	/**
	 * Added new class in Add to cart button for mini-cart
	 *
	 * @param array $args argument.
	 *
	 * @return array
	 */
	public static function woocommerce_loop_add_to_cart_args( $args ) {
		$args['class'] .= ' rtsb-mini-cart';

		return $args;
	}

	/**
	 * Added body classes
	 *
	 * @param array $classes body classes.
	 *
	 * @return array
	 */
	public static function body_classes( $classes ) {

		$opt = self::$options;

		if ( ! empty( $opt['mini_cart_open_style'] ) && 'open-always' == $opt['mini_cart_open_style'] ) {
			$classes[] = 'rtsb-ajax-sidebar';
		}

		if ( ! empty( $opt['mini_cart_open_style'] ) ) {
			$classes[] = "rtsb-{$opt['mini_cart_open_style']}";
		}

		return $classes;
	}


	/**
	 * Cart fragments update
	 *
	 * @param $fragments
	 *
	 * @return mixed
	 */
	public static function cart_count_update( $fragments ) {
		$count_label                            = WC()->cart->get_cart_contents_count() < 2 ? __( 'Item', 'shopbuilder-pro' ) : __( 'Items', 'shopbuilder-pro' );
		$number                                 = '<span class="rtsb-cart-icon-num">' . WC()->cart->get_cart_contents_count() . ' <span class="items">' . $count_label . '<span></span>';
		$total                                  = '<span class="rtsb-cart-icon-total">' . wc_price( WC()->cart->get_cart_contents_total() ) . '</span>';
		$fragments['span.rtsb-cart-icon-num']   = $number;
		$fragments['span.rtsb-cart-icon-total'] = $total;

		return $fragments;
	}

	/**
	 * Load minicart template by ajax
	 *
	 * @return void
	 */
	public static function rtsb_mini_cart_template() {
		if ( ! Fns::verify_nonce() ) {
			wp_send_json_error();
		}
		$param = $_POST['param'] ?? [];
		if ( isset( $param['cartItemKey'] ) && isset( $param['quantity'] ) ) {
			$product_id    = absint( $param['product_id'] ?? 0 );
			$_product      = wc_get_product( $product_id );
			$max_available = $_product->get_stock_quantity();
			$calculet_max  = $max_available && $max_available < $param['quantity'] ? $max_available : $param['quantity'];
			WC()->cart->set_quantity( $param['cartItemKey'], $calculet_max );
		}

		$htmlData = Fns::load_template( 'mini-cart/mini-cart', [], true, '', rtsbpro()->get_plugin_template_path() );
		wp_send_json(
			[
				'status' => 'OK',
				'data'   => $htmlData,
			]
		);
	}


	/**
	 * Mini-cart quantity markup
	 *
	 * @param $args
	 * @param $product
	 * @param $echo
	 *
	 * @return false|string|void
	 */
	public static function rtsb_woocommerce_quantity_input( $args = [], $product = null, $echo = true ) {
		if ( is_null( $product ) ) {
			$product = $GLOBALS['product'];
		}
		$defaults = [
			'input_id'     => uniqid( 'quantity_' ),
			'input_name'   => 'quantity',
			'input_value'  => '1',
			'classes'      => apply_filters( 'woocommerce_quantity_input_classes', [ 'input-text', 'qty', 'text' ], $product ),
			'max_value'    => apply_filters( 'woocommerce_quantity_input_max', - 1, $product ),
			'min_value'    => apply_filters( 'woocommerce_quantity_input_min', 0, $product ),
			'step'         => apply_filters( 'woocommerce_quantity_input_step', 1, $product ),
			'pattern'      => apply_filters( 'woocommerce_quantity_input_pattern', has_filter( 'woocommerce_stock_amount', 'intval' ) ? '[0-9]*' : '' ),
			'inputmode'    => apply_filters( 'woocommerce_quantity_input_inputmode', has_filter( 'woocommerce_stock_amount', 'intval' ) ? 'numeric' : '' ),
			'product_name' => $product ? $product->get_title() : '',
			'placeholder'  => apply_filters( 'woocommerce_quantity_input_placeholder', '', $product ),
			// When autocomplete is enabled in firefox, it will overwrite actual value with what user entered last. So we default to off.
			// See @link https://github.com/woocommerce/woocommerce/issues/30733.
			'autocomplete' => apply_filters( 'woocommerce_quantity_input_autocomplete', 'off', $product ),
			'readonly'     => false,
		];

		$args = apply_filters( 'woocommerce_quantity_input_args', wp_parse_args( $args, $defaults ), $product );

		// Apply sanity to min/max args - min cannot be lower than 0.
		$args['min_value'] = max( $args['min_value'], 0 );
		$args['max_value'] = 0 < $args['max_value'] ? $args['max_value'] : '';

		// Max cannot be lower than min if defined.
		if ( '' !== $args['max_value'] && $args['max_value'] < $args['min_value'] ) {
			$args['max_value'] = $args['min_value'];
		}
		$type = $args['min_value'] > 0 && $args['min_value'] === $args['max_value'] ? 'hidden' : 'number';
		$type = $args['readonly'] && 'hidden' !== $type ? 'text' : $type;

		$args['type'] = apply_filters( 'woocommerce_quantity_input_type', $type );

		ob_start();
		extract( $args )
		?>

		<div class="quantity">
			<input
					type="<?php echo esc_attr( $type ); ?>"
				<?php echo $readonly ? 'readonly="readonly"' : ''; ?>
					id="<?php echo esc_attr( $input_id ); ?>"
					class="<?php echo esc_attr( join( ' ', (array) $classes ) ); ?>"
					name="<?php echo esc_attr( $input_name ); ?>"
					value="<?php echo esc_attr( $input_value ); ?>"
					aria-label="<?php esc_attr_e( 'Product quantity', 'shopbuilder-pro' ); ?>"
					size="4"
					min="<?php echo esc_attr( $min_value ); ?>"
					max="<?php echo esc_attr( 0 < $max_value ? $max_value : '' ); ?>"
				<?php if ( ! $readonly ) : ?>
					step="<?php echo esc_attr( $step ); ?>"
					placeholder="<?php echo esc_attr( $placeholder ); ?>"
					inputmode="<?php echo esc_attr( $inputmode ); ?>"
					autocomplete="<?php echo esc_attr( isset( $autocomplete ) ? $autocomplete : 'on' ); ?>"
				<?php endif; ?>
			/>
		</div>

		<?php
		if ( $echo ) {
			//phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
			echo ob_get_clean();
		} else {
			return ob_get_clean();
		}
	}

	public static function rtrb_minicart_quantity( $_product, $cart_item_key, $cart_item ) {
		if ( $_product->is_sold_individually() ) {
			$min_quantity = 1;
			$max_quantity = 1;
		} else {
			$min_quantity = 0;
			$max_quantity = $_product->get_max_purchase_quantity();
		}

		$max_quantity       = apply_filters( 'rtsb/minicart/quantity/max', $max_quantity, $_product, $cart_item_key, $cart_item );
		$quantity_value     = ( $max_quantity < 0 || $max_quantity >= $cart_item['quantity'] ) ? $cart_item['quantity'] : $max_quantity;
		$product_quantity   = self::rtsb_woocommerce_quantity_input(
			[
				'input_name'   => "cart[{$cart_item_key}][qty]",
				'input_value'  => sprintf( '%d', $quantity_value ),
				'max_value'    => $max_quantity,
				'min_value'    => $min_quantity,
				'product_name' => $_product->get_name(),
			],
			$_product,
			false
		);
		$increment_disabled = $max_quantity > 0 && $quantity_value >= $max_quantity ? 'disabled' : null;
		?>
		<button type="button" class="button decrement">
			<svg width="8" height="2" viewBox="0 0 8 2" fill="none"
				 xmlns="http://www.w3.org/2000/svg">
				<rect width="8" height="2" rx="1" fill="#323846"/>
			</svg>
		</button>
		<?php echo apply_filters( 'rtsb_woocommerce_cart_item_quantity', $product_quantity, $cart_item_key, $cart_item ); // PHPCS: XSS ok. ?>
		<button type="button" class="button increment <?php echo esc_attr( $increment_disabled ); ?>" <?php echo esc_attr( $increment_disabled ); ?>>
			<svg width="8" height="8" viewBox="0 0 8 8" fill="none"
				 xmlns="http://www.w3.org/2000/svg">
				<path fill-rule="evenodd" clip-rule="evenodd"
					  d="M4 0C4.21217 0 4.41566 0.0842856 4.56569 0.234315C4.71571 0.384344 4.8 0.587827 4.8 0.8V3.2H7.2C7.41217 3.2 7.61566 3.28429 7.76569 3.43431C7.91571 3.58434 8 3.78783 8 4C8 4.21217 7.91571 4.41566 7.76569 4.56569C7.61566 4.71571 7.41217 4.8 7.2 4.8H4.8V7.2C4.8 7.41217 4.71571 7.61566 4.56569 7.76569C4.41566 7.91571 4.21217 8 4 8C3.78783 8 3.58434 7.91571 3.43431 7.76569C3.28429 7.61566 3.2 7.41217 3.2 7.2V4.8H0.8C0.587827 4.8 0.384344 4.71571 0.234315 4.56569C0.0842856 4.41566 0 4.21217 0 4C0 3.78783 0.0842856 3.58434 0.234315 3.43431C0.384344 3.28429 0.587827 3.2 0.8 3.2H3.2V0.8C3.2 0.587827 3.28429 0.384344 3.43431 0.234315C3.58434 0.0842856 3.78783 0 4 0Z"
					  fill="#323846"/>
			</svg>
		</button>
		<?php
	}

	/**
	 * Minicart extra fields markup.
	 *
	 * @param $_product
	 *
	 * @return void
	 */
	public static function rtsb_minicart_extra_fields() {
		foreach ( WC()->cart->get_coupons() as $code => $coupon ) :
			?>
			<tr class="cart-discount coupon-<?php echo esc_attr( sanitize_title( $code ) ); ?>">
				<th><?php wc_cart_totals_coupon_label( $coupon ); ?></th>
				<td data-title="<?php echo esc_attr( wc_cart_totals_coupon_label( $coupon, false ) ); ?>"><?php wc_cart_totals_coupon_html( $coupon ); ?></td>
			</tr>
		<?php endforeach; ?>

		<?php if ( WC()->cart->get_shipping_total() && 0 != WC()->cart->get_shipping_total() ) : ?>
			<tr class="order-total">
				<th><?php esc_html_e( 'Shipping', 'shopbuilder-pro' ); ?></th>
				<td data-title="<?php esc_attr_e( 'Shipping', 'shopbuilder-pro' ); ?>"><?php echo wc_price( WC()->cart->get_shipping_total() ); ?></td>
			</tr>
		<?php endif; ?>

		<?php foreach ( WC()->cart->get_fees() as $fee ) : ?>
			<tr class="fee">
				<th><?php echo esc_html( $fee->name ); ?></th>
				<td data-title="<?php echo esc_attr( $fee->name ); ?>"><?php wc_cart_totals_fee_html( $fee ); ?></td>
			</tr>
		<?php endforeach; ?>

		<?php
		if ( wc_tax_enabled() && ! WC()->cart->display_prices_including_tax() ) {
			$taxable_address = WC()->customer->get_taxable_address();
			$estimated_text  = '';

			if ( WC()->customer->is_customer_outside_base() && ! WC()->customer->has_calculated_shipping() ) {
				/* translators: %s location. */
				$estimated_text = sprintf( ' <small>' . esc_html__( '(estimated for %s)', 'shopbuilder-pro' ) . '</small>', WC()->countries->estimated_for_prefix( $taxable_address[0] ) . WC()->countries->countries[ $taxable_address[0] ] );
			}

			if ( 'itemized' === get_option( 'woocommerce_tax_total_display' ) ) {
				foreach ( WC()->cart->get_tax_totals() as $code => $tax ) { // phpcs:ignore WordPress.WP.GlobalVariablesOverride.Prohibited
					?>
					<tr class="tax-rate tax-rate-<?php echo esc_attr( sanitize_title( $code ) ); ?>">
						<th><?php echo esc_html( $tax->label ) . $estimated_text; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?></th>
						<td data-title="<?php echo esc_attr( $tax->label ); ?>"><?php echo wp_kses_post( $tax->formatted_amount ); ?></td>
					</tr>
					<?php
				}
			} else {
				?>
				<tr class="tax-total">
					<th><?php echo esc_html( WC()->countries->tax_or_vat() ) . $estimated_text; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped ?></th>
					<td data-title="<?php echo esc_attr( WC()->countries->tax_or_vat() ); ?>"><?php wc_cart_totals_taxes_total_html(); ?></td>
				</tr>
				<?php
			}
		}
		?>

		<tr class="order-total">
			<th><?php esc_html_e( 'Total', 'shopbuilder-pro' ); ?></th>
			<td data-title="<?php esc_attr_e( 'Total', 'shopbuilder-pro' ); ?>"><?php wc_cart_totals_order_total_html(); ?></td>
		</tr>
		<?php
	}
}




