<?php
/**
 * Main MiniCart class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\MultiStepsCheckout;

use RadiusTheme\SB\Controllers\Hooks\ActionHooks;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Main FilterHooks class.
 */
class MultiStepsHooks {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * @var array|mixed
	 */
	private array $options;

	/**
	 * Class constructor
	 */
	private function __construct() {

		$this->options = Fns::get_options( 'modules', 'multi_step_checkout' );
		// Split the `Order` and the `Payment` tabs.
		remove_action( 'woocommerce_checkout_order_review', 'woocommerce_order_review' );
		remove_action( 'woocommerce_checkout_order_review', 'woocommerce_checkout_payment', 20 );
		add_action( 'rtrb-checkout-woocommerce_order_review', 'woocommerce_order_review', 20 );
		add_action( 'rtrb-checkout-woocommerce_checkout_payment', 'woocommerce_checkout_payment', 10 );

		// Split the `woocommerce_before_checkout_form`.
		remove_action( 'woocommerce_before_checkout_form', 'woocommerce_checkout_login_form' );
		remove_action( 'woocommerce_before_checkout_form', 'woocommerce_checkout_coupon_form' );
		remove_action( 'woocommerce_before_checkout_form', [ ActionHooks::class, 'woocommerce_checkout_coupon_form' ] );

		add_action( 'rtrb_checkout_woocommerce_checkout_login_form', [ __CLASS__, 'woocommerce_checkout_login_form' ], 10 );
		add_action( 'rtrb_checkout_woocommerce_checkout_coupon_form', [ ActionHooks::class, 'woocommerce_checkout_coupon_form' ], 10 );

		add_action(
			'template_redirect',
			function () {
				remove_action( 'woocommerce_checkout_billing', [ WC()->checkout(), 'checkout_form_shipping' ] );
				add_action( 'woocommerce_checkout_shipping', [ WC()->checkout(), 'checkout_form_shipping' ] );
			}
		);

		add_filter( 'woocommerce_checkout_fields', [ __CLASS__, 'woocommerce_checkout_fields' ] );

		// Add the content functions to the steps.
		add_action( 'rtrb_step_content_login', 'rtrb_step_content_login', 10 );
		add_action( 'rtrb_step_content_shipping', [ __CLASS__, 'rtrb_step_content_shipping' ], 10 );
		add_action( 'rtrb_step_content_billing', [ __CLASS__, 'rtrb_step_content_billing' ], 10 );

		if ( class_exists( 'WooCommerce_Germanized' ) ) {
			// Plugin support - Germanized for WooCommerce.
			add_action( 'rtrb_step_content_order', [ __CLASS__, 'rtrb_step_content_order_germanized' ], 10 );
			add_action( 'rtrb_step_content_payment', [ __CLASS__, 'rtrb_step_content_payment_germanized' ], 10 );
			add_action( 'rtrb-checkout-woocommerce_order_review', 'woocommerce_gzd_template_render_checkout_checkboxes', 10 );
			add_filter( 'wc_gzd_checkout_params', [ __CLASS__, 'wc_germanized_checkout_params' ] );
			add_filter( 'wp_loaded', [ __CLASS__, 'woocommerce_review_order_after_payment' ] );
		} elseif ( class_exists( 'Woocommerce_German_Market' ) ) {
			// Plugin support - WooCommerce German Market.
			add_action( 'rtrb_step_content_order', [ __CLASS__, 'rtrb_step_content_order_german_market' ], 10 );
			add_action( 'rtrb_step_content_payment', [ __CLASS__, 'rtrb_step_content_payment_german_market' ], 10 );
		} else {
			// Multi step Checkout.
			add_action( 'rtrb_step_content_order', [ __CLASS__, 'rtrb_step_content_order' ], 10 );
			add_action( 'rtrb_step_content_payment', [ __CLASS__, 'rtrb_step_content_payment' ], 10 );
		}

		add_action( 'rtrb_step_checkout_button', [ __CLASS__, 'rtrb_step_checkout_button_html' ], 11, 2 );
	}

	/**
	 * woocommerce checkout fields
	 *
	 * @param $fields
	 *
	 * @return array
	 */
	public static function woocommerce_checkout_fields( $fields ) {
		$fields['billing']['billing_address_2']['label_class']   = 'rtrb-custom-address-2';
		$fields['shipping']['shipping_address_2']['label_class'] = 'rtrb-custom-address-2';

		return $fields;
	}

	public static function rtrb_step_content_billing() {
		do_action( 'woocommerce_checkout_before_customer_details' );
		do_action( 'woocommerce_checkout_billing' );
	}

	public static function rtrb_step_content_shipping() {
		echo '<h3>' . esc_html__( 'Shipping Information', 'shopbuilder-pro' ) . '</h3>';
		do_action( 'woocommerce_checkout_shipping' );
		do_action( 'woocommerce_checkout_after_customer_details' );
	}

	/**
	 * Germanized for WooCommerce plugin support
	 *
	 * @return void
	 */
	public static function rtrb_step_content_payment() {
		echo '<h3 id="payment_heading">' . esc_html__( 'Payment Information', 'shopbuilder-pro' ) . '</h3>';
		do_action( 'rtrb-checkout-woocommerce_checkout_payment' );
	}

	/**
	 * Checkout payment
	 *
	 * @return void
	 */
	public static function rtrb_step_content_order() {
		do_action( 'woocommerce_checkout_before_order_review_heading' );
		echo '<h3 id="order_review_heading">' . esc_html__( 'Your order', 'shopbuilder-pro' ) . '</h3>';
		do_action( 'woocommerce_checkout_before_order_review' );
		echo '<div id="order_review" class="woocommerce-checkout-review-order">';
		do_action( 'woocommerce_checkout_order_review' );
		do_action( 'rtrb-checkout-woocommerce_order_review' );
		echo '</div>';
	}

	/**
	 * Germanized for WooCommerce plugin support
	 *
	 * @return void
	 */
	public static function rtrb_step_content_payment_germanized() {
		echo '<h3 id="payment_heading">' . esc_html__( 'Choose a Payment Gateway', 'shopbuilder-pro' ) . '</h3>';
		do_action( 'rtrb-checkout-woocommerce_checkout_payment' );
	}

	/**
	 * WooCommerce German Market - plugin support
	 *
	 * @return void
	 */
	public static function rtrb_step_content_order_germanized() {
		do_action( 'woocommerce_checkout_before_order_review_heading' );
		echo '<h3 id="order_review_heading">' . esc_html__( 'Your order', 'shopbuilder-pro' ) . '</h3>';
		do_action( 'woocommerce_checkout_before_order_review' );
		echo '<div id="order_review" class="woocommerce-checkout-review-order">';
		do_action( 'rtrb-checkout-woocommerce_order_review' );
		if ( function_exists( 'woocommerce_gzd_template_order_submit' ) ) {
			woocommerce_gzd_template_order_submit();
		}
		echo '</div>';
	}

	/**
	 * WooCommerce German Market - plugin support
	 *
	 * @return void
	 */
	public static function rtrb_step_content_order_german_market() {
		do_action( 'woocommerce_checkout_before_order_review_heading' );
		echo '<h3 id="order_review_heading">' . esc_html__( 'Your order', 'shopbuilder-pro' ) . '</h3>';
		do_action( 'woocommerce_checkout_before_order_review' );
		echo '<div id="order_review" class="woocommerce-checkout-review-order">';
		do_action( 'rtrb-checkout-woocommerce_order_review' );
		do_action( 'woocommerce_checkout_order_review' );
		echo '</div>';
	}

	/**
	 * Payment Information
	 *
	 * @return void
	 */
	public static function rtrb_step_content_payment_german_market() {
		echo '<h3 id="payment_heading">' . esc_html__( 'Payment Information', 'shopbuilder-pro' ) . '</h3>';
		do_action( 'rtrb-checkout-woocommerce_checkout_payment' );
	}


	/**
	 * @param $params
	 *
	 * @return mixed
	 */
	public static function wc_germanized_checkout_params( $params ) {
		$params['adjust_heading'] = false;

		return $params;
	}

	/**
	 * Remove Germanized WooCommerce Terms and Conditions.
	 *
	 * @return void
	 */
	public static function woocommerce_review_order_after_payment() {
		remove_action( 'woocommerce_review_order_after_payment', 'woocommerce_gzd_template_render_checkout_checkboxes', 10 );
	}

	/**
	 * Woocommerce checkout login markup.
	 *
	 * @return void
	 */
	public static function woocommerce_checkout_login_form() {
		?>
		<div class="rtrb-checkout-step-item rtrb-checkout-step-login">
			<div id="checkout_login" class="shopbuilder-checkout-login woocommerce_checkout_login">
				<?php
				woocommerce_login_form(
					[
						'message'  => apply_filters(
							'woocommerce_checkout_logged_in_message',
							sprintf(
								"<div class='checkout-login-text'><p>%s</p> <h3>%s</h3> </div>",
								esc_html__( "If you've previously shopped with us, kindly provide your information in the fields below. For new customers, please continue to the Billing & Shipping section.", 'shopbuilder-pro' ),
								esc_html__( 'User Login?', 'shopbuilder-pro' )
							)
						),
						'redirect' => wc_get_page_permalink( 'checkout' ),
						'hidden'   => false,
					]
				);
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * Step checkout button html
	 *
	 * @return void
	 */
	public static function rtrb_step_checkout_button_html( $options ) {
		$btn_wrapper_class = 'rtrb-checkout-nav-wrapper';
		$back_to_cart      = $options['show_back_to_cart_button'] ?? false;
		$checkout_layout   = $options['checkout_layout'] ?? 'form-checkout';
		if ( ! $back_to_cart ) {
			$btn_wrapper_class .= ' rtrb-checkout-no-back-to-cart';
		}
		$left_arrow_icon      = '<svg xmlns="http://www.w3.org/2000/svg" width="17" height="14" viewBox="0 0 21 18" fill="none"><path d="M2.05859 9L9.34431 2M2.05859 9L9.34431 16M2.05859 9H19.0586" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"></path></svg>';
		$right_arrow_icon     = '<svg xmlns="http://www.w3.org/2000/svg" width="17" height="18" viewBox="0 0 21 18" fill="none"><path d="M19.0586 9L11.7729 2M19.0586 9L11.7729 16M19.0586 9H2.05859" stroke="currentColor" stroke-width="2.5" stroke-linecap="round" stroke-linejoin="round"></path></svg>';
		$show_login_step      = ! is_user_logged_in() && ( $options['login_step_visibility'] ?? '' );
		$back_to_top_btn_text = ! empty( $options['back_to_top_btn_text'] ) ? $options['back_to_top_btn_text'] : esc_html__( 'Back to cart', 'shopbuilder-pro' );
		$previous_btn_text    = ! empty( $options['previous_btn_text'] ) ? $options['previous_btn_text'] : esc_html__( 'Previous', 'shopbuilder-pro' );
		$next_btn_text        = ! empty( $options['next_btn_text'] ) ? $options['next_btn_text'] : esc_html__( 'Next', 'shopbuilder-pro' );
		$skip_login_btn_text  = ! empty( $options['skip_login_btn_text'] ) ? $options['skip_login_btn_text'] : esc_html__( 'Skip Login', 'shopbuilder-pro' );
		$place_order_btn_text = ! empty( $options['place_order_btn_text'] ) ? $options['place_order_btn_text'] : esc_html__( 'Place Order', 'shopbuilder-pro' );
		?>

		<div class="<?php echo esc_attr( apply_filters( 'rtrb_checkout_btn_wrapper_class', $btn_wrapper_class ) ); ?>">
			<?php if ( $back_to_cart ) : ?>

				<button data-href="<?php echo wc_get_cart_url(); ?>" id="rtrb-checkout-back-to-cart" class="button" type="button">
					<?php
					if ( 'layout2' === $checkout_layout ) {
						?>
							<span class="rtsb-checkout-icon"><?php Fns::print_html( $left_arrow_icon ); ?></span>
						<?php
					}
					?>
					<?php echo esc_html( $back_to_top_btn_text ); ?>
				</button>
			<?php endif; ?>

			<button id="rtrb-checkout-prev" class="button button-inactive rtrb-checkout-nav-button" type="button">
				<?php
				if ( 'layout2' === $checkout_layout ) {
					?>
					<span class="rtsb-checkout-icon"><?php Fns::print_html( $left_arrow_icon ); ?></span>
					<?php
				}
				?>
				<?php echo esc_html( $previous_btn_text ); ?>
			</button>

			<?php if ( $show_login_step ) : ?>
				<button id="rtrb-checkout-next" class="button button-active rtrb-checkout-nav-button" type="button">
					<?php echo esc_html( $next_btn_text ); ?>
					<?php
					if ( 'layout2' === $checkout_layout ) {
						?>
						<span class="rtsb-checkout-icon"><?php Fns::print_html( $right_arrow_icon ); ?></span>
						<?php
					}
					?>
				</button>
				<button id="rtrb-checkout-skip-login" class="button button-active current rtrb-checkout-nav-button" type="button">
					<?php echo esc_html( $skip_login_btn_text ); ?>
				</button>
			<?php else : ?>
				<button id="rtrb-checkout-next" class="button button-active current rtrb-checkout-nav-button" type="button">
					<?php echo esc_html( $next_btn_text ); ?>
					<?php
					if ( 'layout2' === $checkout_layout ) {
						?>
						<span class="rtsb-checkout-icon"><?php Fns::print_html( $right_arrow_icon ); ?></span>
						<?php
					}
					?>
				</button>
			<?php endif; ?>

			<?php echo apply_filters( 'woocommerce_order_button_html', '<button type="submit" class="button alt' . esc_attr( wc_wp_theme_get_element_class_name( 'button' ) ? ' ' . wc_wp_theme_get_element_class_name( 'button' ) : '' ) . '" name="woocommerce_checkout_place_order" id="place_order2" value="' . esc_attr( $place_order_btn_text ) . '" data-value="' . esc_attr( $place_order_btn_text ) . '">' . esc_html( $place_order_btn_text ) . '</button>' ); ?>
		</div>
		<?php
	}
}
