<?php
/**
 * Partial Pay Single Product Page Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\PartialPay\Frontend;

use Exception;
use RadiusTheme\SBPRO\Modules\AddOns\Frontend\Product as AddOnsProduct;
use RadiusTheme\SBPRO\Modules\BulkDiscount\BulkDiscountFns;
use RadiusTheme\SBPRO\Modules\PreOrder\Frontend\Product;
use WC_Product;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\PreOrder\PreOrderFns;
use RadiusTheme\SBPRO\Modules\PartialPay\PartialPayFns;

defined( 'ABSPATH' ) || exit();

/**
 * Partial Pay Single Product Page Class.
 */
class ProductPage {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Options.
	 *
	 * @var array|mixed
	 */
	private array $options;

	/**
	 * User restriction.
	 *
	 * @var bool
	 */
	private bool $restriction = false;

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		/**
		 * Get the options.
		 */
		$this->options = PartialPayFns::get_settings_data();

		$has_user_restriction = 'specific' === ( $this->options['user_role_management'] ?? null );
		$user_roles           = $this->options['roles'];

		if ( $has_user_restriction && ( ! FnsPro::is_visible_for_user( $user_roles ) ) ) {
			$this->restriction = true;
		}

		/**
		 * Hooks.
		 */
		$this->init_hooks();
	}

	/**
	 * Init hooks.
	 *
	 * @return void
	 */
	private function init_hooks() {
		if ( $this->restriction ) {
			return;
		}

		/**
		 * Actions.
		 */
		add_action( 'woocommerce_before_add_to_cart_button', [ $this, 'render_partial_payment' ], 99 );
		add_action( 'rtsb/after/shop/product/title', [ $this, 'display_partial_pay_note' ], 11 );

		add_action( 'wp_ajax_rtsb_get_partial_pay_data', [ $this, 'adjust_partial_pay' ] );
		add_action( 'wp_ajax_nopriv_rtsb_get_partial_pay_data', [ $this, 'adjust_partial_pay' ] );

		/**
		 * Filters.
		 */
		add_filter( 'woocommerce_loop_add_to_cart_link', [ $this, 'add_partial_pay_note' ], 11, 2 );
		add_filter( 'rtsb/elementor/render/cart_attributes', [ $this, 'change_add_to_cart_args' ], 10, 2 );
		add_filter( 'woocommerce_loop_add_to_cart_args', [ $this, 'change_add_to_cart_args' ], 10, 2 );
		add_filter( 'woocommerce_product_add_to_cart_url', [ $this, 'modify_add_to_cart_url' ], 10, 2 );
		add_filter( 'woocommerce_available_variation', [ $this, 'partial_pay_variation_data' ] );
		add_filter( 'woocommerce_post_class', [ $this, 'add_custom_class' ], 10, 2 );

		// Bulk Discount.
		add_filter( 'rtsb/module/bulk_discounts/global/disable', [ $this, 'is_bulk_discounts' ], 99, 2 );
		add_filter( 'rtsb/module/bxgy_bogo/global/disable', [ $this, 'is_bulk_discounts' ], 99, 2 );
		add_filter( 'rtsb/calculate/pre_order/price', [ $this, 'disable_pre_order' ], 99, 2 );
	}

	/**
	 * Checks if the product has bulk discounts.
	 *
	 * @param mixed      $is        The current state or flag determining discount applicability.
	 * @param WC_Product $product   The WooCommerce product object.
	 *
	 * @return bool
	 */
	public function is_bulk_discounts( $is, $product ) {
		if ( PartialPayFns::product_has_partial_pay( $product, 'variation' === $product->get_type() ) ) {
			return true;
		}

		return $is;
	}

	/**
	 * Checks if the product has pre-order.
	 *
	 * @param mixed      $is        The current state or flag determining discount applicability.
	 * @param WC_Product $product   The WooCommerce product object.
	 *
	 * @return bool
	 */
	public function disable_pre_order( $is, $product ) {
		if ( PartialPayFns::product_has_partial_pay( $product, 'variation' === $product->get_type() ) ) {
			return false;
		}

		return $is;
	}

	/**
	 * Display partial payment in product details page.
	 *
	 * @param int|null        $product_id The product ID.
	 * @param WC_Product|null $product The product object.
	 * @param int|null        $variation_id The variation ID.
	 *
	 * @return void
	 */
	public function render_partial_payment( $product_id = null, $product = null, $variation_id = null ) {
		if ( PartialPayFns::is_partial_pay_disabled( $product ) ) {
			return;
		}

		$product    = empty( $product ) ? self::get_product()['product'] : $product;
		$product_id = empty( $product_id ) ? self::get_product()['product_id'] : $product_id;

		if ( ! $product instanceof WC_Product || ! PartialPayFns::product_has_partial_pay( $product, 'variation' === $product->get_type() ) ) {
			return;
		}

		if ( Fns::is_module_active( 'bulk_discounts' ) && BulkDiscountFns::product_has_bulk_discounts( $product ) ) {
			return;
		}

		$cache_key    = 'rtsb_partial_pay_' . $product_id;
		$cached_rules = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false === $cached_rules ) {
			$cached_rules = PartialPayFns::get_partial_pay_rules( $product_id );

			wp_cache_set( $cache_key, $cached_rules, 'shopbuilder', 12 * HOUR_IN_SECONDS );
			Cache::set_data_cache_key( $cache_key );
		}

		if ( $this->has_rules( $cached_rules ) ) {
			$partial_amount = PartialPayFns::get_partial_amount( $product_id, $variation_id );
			$partial_price  = $partial_amount[0]['amount'] ?? 0;
			$full_price     = FnsPro::get_module_compatible_price( $product );

			if ( ! PartialPayFns::is_valid_partial_payment( $partial_amount, $full_price ) ) {
				return;
			}

			$args = [
				'full_price'    => $full_price,
				'installment'   => $partial_amount,
				'partial_price' => wc_price( $partial_price ),
				'labels'        => $this->options['labels'],
				'default_type'  => $this->options['default_payment_type'],
			];

			// Nonce verification.
			wp_nonce_field( rtsb()->nonceText, rtsb()->nonceId );

			// Load template.
			Fns::load_template( 'partial-pay/radio-input', $args, false, '', rtsbpro()->get_plugin_template_path() );
		}
	}

	/**
	 * Displays the partial payment note in the product loop.
	 *
	 * @return void
	 */
	public function display_partial_pay_note() {
		global $product;

		if ( PartialPayFns::is_partial_pay_disabled( $product ) ) {
			return;
		}

		$note = $this->add_partial_pay_note( '', $product );
		Fns::print_html( $note, true );
	}

	/**
	 * Ajax response to adjust partial payment.
	 *
	 * @return void
	 */
	public function adjust_partial_pay() {
		if ( ! Fns::verify_nonce() ) {
			wp_send_json_error( [ 'message' => esc_html__( 'Security check failed.', 'shopbuilder-pro' ) ] );
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( empty( $_REQUEST['product_id'] ) ) {
			wp_send_json_error( [ 'message' => __( 'Invalid product ID', 'shopbuilder-pro' ) ] );
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$product_id = sanitize_text_field( wp_unslash( $_REQUEST['product_id'] ) );
		$product    = wc_get_product( $product_id );

		ob_start();
		$this->render_partial_payment( $product->get_parent_id(), $product, $product->get_id() );
		$html = ob_get_clean();

		wp_send_json_success( [ 'html' => $html ] );
	}

	/**
	 * Add a note to the "Add to Cart" button in the product loop.
	 *
	 * @param string     $button The HTML markup for the "Add to Cart" button.
	 * @param WC_Product $product The product object.
	 *
	 * @return string
	 */
	public function add_partial_pay_note( $button, $product ) {
		if ( PartialPayFns::is_partial_pay_disabled( $product )
			|| ! ( PartialPayFns::product_has_partial_pay( $product ) && 'on' === $this->options['display_in_loop'] ) ) {
			return $button;
		}

		if ( Fns::is_module_active( 'pre_order' ) && PreOrderFns::is_on_pre_order( $product ) ) {
			return $button;
		}

		return '<div class="rtsb-partial-pay-avail-note"><span>' . esc_html__( 'Partial Pay Available', 'shopbuilder-pro' ) . '</span></div>' . $button;
	}

	/**
	 * Modify the "Add to Cart" button arguments for partial pay products in the product loop.
	 *
	 * @param array      $args The original arguments for the "Add to Cart" button.
	 * @param WC_Product $product The product object.
	 *
	 * @return array
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function change_add_to_cart_args( $args, $product ) {
		if ( PartialPayFns::is_partial_pay_disabled( $product ) || ! PartialPayFns::product_has_partial_pay( $product ) ) {
			return $args;
		}

		$args['href']  = esc_url( get_permalink( $product->get_id() ) );
		$args['class'] = str_replace(
			[ 'ajax_add_to_cart', 'rtsb-mini-cart', 'rtsb-add-to-cart-btn', 'variable-product' ],
			[ '', '', 'rtsb-pre-order-btn', 'variable-product rtsb-pre-order-btn' ],
			$args['class']
		);

		return $args;
	}

	/**
	 * Modify the "Add to Cart" URL for pre-order products.
	 *
	 * @param string     $permalink The original permalink URL.
	 * @param WC_Product $product The product object.
	 *
	 * @return string
	 */
	public function modify_add_to_cart_url( $permalink, $product ) {
		if ( PartialPayFns::is_partial_pay_disabled( $product ) ) {
			return $permalink;
		}

		if ( PartialPayFns::product_has_partial_pay( $product ) ) {
			$permalink = esc_url( get_permalink( $product->get_id() ) );
		}

		return $permalink;
	}

	/**
	 * Adds pre-order data to product variations.
	 *
	 * @param array $variations The variation data.
	 *
	 * @return array
	 */
	public function partial_pay_variation_data( $variations ) {
		if ( PartialPayFns::product_has_partial_pay( wc_get_product( $variations['variation_id'] ), true ) ) {
			$variations['rtsb_product_has_partial_pay'] = 'yes';
		}

		return $variations;
	}

	/**
	 * Add custom class.
	 *
	 * @param array  $classes Classes.
	 * @param object $product The product object.
	 *
	 * @return array
	 */
	public function add_custom_class( $classes, $product ) {
		if ( PartialPayFns::product_has_partial_pay( $product ) ) {
			$classes[] = 'rtsb-product-has-partial-pay';
		}

		return $classes;
	}

	/**
	 * Check if there is any partial pay.
	 *
	 * @param array $rules The array of rules.
	 *
	 * @return bool
	 */
	private function has_rules( $rules ) {
		return is_array( $rules ) && count( $rules ) > 0;
	}

	/**
	 * Get product.
	 *
	 * @return array
	 */
	public static function get_product() {
		global $product;

		if ( ! $product instanceof WC_Product ) {
			return [];
		}

		return [
			'product'    => $product,
			'product_id' => $product->is_type( 'variation' ) ? $product->get_parent_id() : $product->get_id(),
		];
	}
}
