<?php
/**
 * Partial Pay Expiration On Hold Email Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\PartialPay\Notification;

use WC_Email;
use WC_Order;
use Exception;

defined( 'ABSPATH' ) || exit();

/**
 * Partial Pay Expiration On Hold Email Class.
 */
class PartialPayOnHoldEmail extends WC_Email {
	/**
	 * Email content.
	 *
	 * @var string
	 */
	public $email_content;

	/**
	 * Class Constructor.
	 */
	public function __construct() {
		$this->id             = 'rtsb_partial_pay_expiration_on_hold';
		$this->title          = esc_html__( 'ShopBuilder: Partial Payment Expiration On Hold', 'shopbuilder-pro' );
		$this->description    = esc_html__( 'Sends an email to the customer when a partial payment has expired and the order is placed on hold.', 'shopbuilder-pro' );
		$this->heading        = esc_html__( 'Order On Hold – Complete Your Payment', 'shopbuilder-pro' );
		$this->subject        = esc_html__( '[{site_title}]: Order #{order_number} on Hold – Complete Your Payment', 'shopbuilder-pro' );
		$this->customer_email = true;
		$this->email_content  = $this->get_option(
			'email_content',
			sprintf(
				/* translators: line break */
				__( 'Hi {customer_name},%1$s%1$sYour {order_number} is currently on hold because the remaining balance was not paid by the due date.%1$s%1$sTo resume processing your order, please complete the remaining balance payment as soon as possible.%1$s%1$s{remaining_amount}%1$s%1$s{balance_pay_link}%1$s%1$sAfter completing the payment, please contact the store administrator to process your order.%1$s%1$sIf you’ve already completed the payment or have any questions, kindly contact our support team.%1$s%1$sThank you,%1$s{site_title}', 'shopbuilder-pro' ),
				"\n"
			)
		);

		$this->template_html = 'emails/partial-pay-on-hold.php';

		add_action( 'rtsb/module/partial_pay/on_hold_email', [ $this, 'trigger' ], 10, 1 );

		parent::__construct();

		$this->email_type = 'html';
	}

	/**
	 * Trigger the email notification for a new order.
	 *
	 * @param WC_Order $order The order object.
	 *
	 * @return void
	 * @throws Exception When `WC_Data_Store::load` validation fails.
	 */
	public function trigger( $order ) {
		if ( ! $order ) {
			return;
		}

		if ( $order->get_meta( '_rtsb_partial_due_paid' ) ) {
			return;
		}

		$this->object    = $order;
		$customer_name   = $this->object->get_formatted_billing_full_name();
		$this->recipient = $this->object->get_billing_email();
		$order_number    = $this->object->get_order_number();

		$this->placeholders = array_merge(
			$this->placeholders,
			[
				'{customer_name}' => $customer_name,
				'{order_number}'  => $order_number,
				'{order_date}'    => wc_format_datetime( $this->object->get_date_created() ),
			]
		);

		if ( ! $this->is_enabled() || ! $this->get_recipient() ) {
			return;
		}

		$this->send( $this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments() );
	}

	/**
	 * Get the HTML content for the email.
	 *
	 * @return string The HTML content.
	 */
	public function get_content_html() {
		$balance_pay_link = add_query_arg(
			[
				'rtsb_partial_pay_order' => 'true',
			],
			$this->object->get_checkout_payment_url()
		);

		$installments   = $this->object->get_meta( '_rtsb_partial_installments' );
		$original_total = $this->object->get_meta( '_rtsb_partial_total_price' ) ?: $this->object->get_total();
		$paid           = $installments[0]['amount'] ?? '';
		$total_quantity = 0;
		$due            = floatval( $original_total ) - floatval( $paid );

		foreach ( $this->object->get_items() as $item ) {
			$total_quantity += $item->get_quantity();
			$per_item_due    = floatval( $original_total ) - floatval( $paid );
			$due             = $per_item_due * $total_quantity;
		}

		return wc_get_template_html(
			$this->template_html,
			[
				'order'            => $this->object,
				'email_heading'    => $this->get_heading(),
				'sent_to_admin'    => true,
				'plain_text'       => false,
				'email'            => $this,
				'balance_pay_link' => $balance_pay_link,
				'remaining_amount' => $due,
				'store_email'      => $this->get_from_address(),
			],
			'',
			rtsbpro()->get_plugin_template_path()
		);
	}

	/**
	 * Initialize the form fields for the plugin settings.
	 *
	 * @return void
	 */
	public function init_form_fields() {
		$this->placeholders = array_merge(
			$this->placeholders,
			[
				'{customer_name}'    => __( 'Customer Name', 'shopbuilder-pro' ),
				'{order_number}'     => __( 'Order Number', 'shopbuilder-pro' ),
				'{order_date}'       => __( 'Order Date', 'shopbuilder-pro' ),
				'{order_link}'       => __( 'Order Link', 'shopbuilder-pro' ),
				'{balance_pay_link}' => __( 'Remaining Balance Payment Link', 'shopbuilder-pro' ),
				'{remaining_amount}' => __( 'Remaining Amount', 'shopbuilder-pro' ),
			]
		);

		/* translators: %s: list of placeholders */
		$placeholder_text  = sprintf( __( 'Available placeholders: %s', 'shopbuilder-pro' ), '<code>' . implode( '</code>, <code>', array_keys( $this->placeholders ) ) . '</code>' );
		$this->form_fields = [
			'enabled'       => [
				'title'   => __( 'Enable/Disable', 'shopbuilder-pro' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable this email notification', 'shopbuilder-pro' ),
				'default' => 'yes',
			],
			'subject'       => [
				'title'       => __( 'Subject', 'shopbuilder-pro' ),
				'type'        => 'text',
				'desc_tip'    => true,
				'description' => $placeholder_text,
				'placeholder' => $this->get_default_subject(),
				'default'     => $this->get_default_subject(),
			],
			'heading'       => [
				'title'       => __( 'Email heading', 'shopbuilder-pro' ),
				'type'        => 'text',
				'desc_tip'    => true,
				'description' => $placeholder_text,
				'placeholder' => $this->get_default_heading(),
				'default'     => $this->get_default_heading(),
			],
			'email_content' => [
				'title'       => __( 'Email Content', 'shopbuilder-pro' ),
				'description' => __( 'Text to appear in the main email body.', 'shopbuilder-pro' ) . ' ' . $placeholder_text . '.',
				'css'         => 'width:800px; height: 185px;',
				'placeholder' => '',
				'type'        => 'textarea',
				'default'     => $this->email_content,
			],
		];
	}
}
