<?php
/**
 * Partial Payment Functions Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\PartialPay;

use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SB\Elementor\Helper\RenderHelpers;
use RadiusTheme\SBPRO\Modules\CurrencySwitcher\CsFns;
use WC_Order;

/**
 * Partial Payment Functions Class.
 *
 * @package RadiusTheme\SB
 */
class PartialPayFns {
	/**
	 * Returns an array of settings fields.
	 *
	 * @return array
	 */
	public static function settings_field() {
		$payment_gateways = CsFns::search_payment_gateways();
		$placeholder      = ! empty( $payment_gateways ) ? esc_html__( 'Choose payment methods...', 'shopbuilder-pro' ) : esc_html__( 'No active payment methods found.', 'shopbuilder-pro' );

		return [
			'partial_pay_settings_intro'       => [
				'id'    => 'partial_pay_settings_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Partial Pay Settings', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'partial_pay_in_product_loop'      => [
				'id'    => 'partial_pay_in_product_loop',
				'type'  => 'switch',
				'label' => esc_html__( 'Show Partial Pay Availability Note', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Display partial pay availability note in shop, archive, related, upsell, and cross-sell sections', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'default_payment_type'             => [
				'id'      => 'default_payment_type',
				'type'    => 'select',
				'label'   => __( 'Default payment type', 'shopbuilder-pro' ),
				'help'    => esc_html__( 'Select a payment type that you want to set by default.', 'shopbuilder-pro' ),
				'value'   => 'full',
				'options' => [
					'partial' => esc_html__( 'Partial Payment', 'shopbuilder-pro' ),
					'full'    => esc_html__( 'Full Payment', 'shopbuilder-pro' ),
				],
				'tab'     => 'general',
			],
			'restrict_payment_method'          => [
				'id'              => 'restrict_payment_method',
				'label'           => esc_html__( 'Restrict Payment Method in Checkout', 'shopbuilder-pro' ),
				'help'            => esc_html__( 'Payment method selection will be hidden on the first checkout, but for the second installment, all payment methods will appear', 'shopbuilder-pro' ),
				'type'            => 'search_and_multi_select',
				'placeholder'     => $placeholder,
				'func_with_param' => [ CsFns::class, 'search_payment_gateways', [] ],
				'options'         => $payment_gateways,
				'tab'             => 'general',
			],
			'balanced_payment_date'            => [
				'id'    => 'balanced_payment_date',
				'label' => __( 'Due Date After Payment (Days)', 'shopbuilder-pro' ),
				'type'  => 'number',
				'value' => 30,
				'help'  => esc_html__( 'Set the number of days after the initial payment when the remaining balance is due.', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'days_to_send_mail'                => [
				'id'    => 'send_mail',
				'label' => __( 'Send Reminder Before Due (Days)', 'shopbuilder-pro' ),
				'type'  => 'number',
				'value' => 5,
				'help'  => esc_html__( 'Specify how many days before the due date the customer should receive a reminder email.', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'disable_coupons'                  => [
				'id'    => 'disable_coupons',
				'type'  => 'switch',
				'label' => __( 'Disable Coupons?', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Automatically disable coupons if any product in the cart has a partial pay product.', 'shopbuilder-pro' ),
				'value' => 'on',
				'tab'   => 'general',
			],
			'partial_pay_label_intro'          => [
				'id'    => 'partial_pay_label_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Label Settings', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'partial_payment_label'            => [
				'id'    => 'partial_payment_label',
				'label' => __( 'Partial Payment Input Label', 'shopbuilder-pro' ),
				'type'  => 'text',
				'value' => __( 'Pay Partial Amount', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter the label for the partial payment option.', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'full_payment_label'               => [
				'id'    => 'full_payment_label',
				'label' => __( 'Full Payment Input Label', 'shopbuilder-pro' ),
				'type'  => 'text',
				'value' => __( 'Pay Full Amount', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter the label for the full payment option.', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'first_installment_label_single'   => [
				'id'    => 'first_installment_label_single',
				'label' => __( 'Partial Pay Value Label', 'shopbuilder-pro' ),
				'type'  => 'text',
				'value' => __( 'Partial Pay', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter the label for the partial payment value.', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'remaining_label_single'           => [
				'id'    => 'remaining_label_single',
				'label' => __( 'Remaining Amount Value Label', 'shopbuilder-pro' ),
				'type'  => 'text',
				'value' => __( 'Remaining Balance', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter the label for the remaining payment value.', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			// TODO: Will be implemented later.
			/*
			'first_installment_label'          => [
				'id'    => 'first_installment_label',
				'label' => __( 'First Installment Label for Multiple Installment', 'shopbuilder-pro' ),
				'type'  => 'text',
				'value' => __( 'First Installment', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter the label for the first installment payment option for multiple installment.', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'other_installment_label'          => [
				'id'    => 'other_installment_label',
				'label' => __( 'Other Installment Labels for Multiple Installment', 'shopbuilder-pro' ),
				'type'  => 'text',
				'value' => __( 'Second Installment', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter the label for the other installment payment option for multiple installment. You can add multiple labels separated by commas.', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			*/
			'to_pay_label'                     => [
				'id'    => 'to_pay_label',
				'label' => __( 'Balance Payment Button Label', 'shopbuilder-pro' ),
				'type'  => 'text',
				'value' => __( 'Pay Balance', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter the label for the balance payment option.', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'partial_pay_user_settings_intro'  => [
				'id'    => 'partial_pay_user_settings_intro',
				'type'  => 'title',
				'label' => esc_html__( 'User Access and Permissions', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'partial_pay_user_role_management' => [
				'id'      => 'partial_pay_user_role_management',
				'label'   => __( 'Specify Users Roles', 'shopbuilder-pro' ),
				'help'    => esc_html__( 'Determine which user groups can see partial payment.', 'shopbuilder-pro' ),
				'type'    => 'select',
				'value'   => 'all',
				'options' => [
					'all'      => esc_html__( 'All users', 'shopbuilder-pro' ),
					'specific' => esc_html__( 'Only users with a specific role', 'shopbuilder-pro' ),
				],
				'tab'     => 'general',
			],
			'partial_pay_roles'                => [
				'id'         => 'partial_pay_roles',
				'label'      => esc_html__( 'Select User Roles', 'shopbuilder-pro' ),
				'help'       => esc_html__( 'Please choose user roles.', 'shopbuilder-pro' ),
				'type'       => 'select',
				'value'      => [ 'administrator', 'customer' ],
				'options'    => FnsPro::get_all_user_roles(),
				'multiple'   => true,
				'tab'        => 'general',
				'dependency' => [
					'rules' => [
						[
							'item'     => 'modules.partial_pay.partial_pay_user_role_management',
							'value'    => 'specific',
							'operator' => '==',
						],
					],
				],
			],
			'rules_field_intro'                => [
				'id'   => 'rules_field_intro',
				'type' => 'description',
				'text' => esc_html__( 'To add new partial pay rules, simply click on the \'Add New\' button below.', 'shopbuilder-pro' ),
				'tab'  => 'rules',
			],
			'partial_pay_rules'                => [
				'id'     => 'partial_pay_rules',
				'type'   => 'repeaters',
				'label'  => '',
				'tab'    => 'rules',
				'repeat' => [
					'title'                 => [
						'id'          => 'title',
						'label'       => esc_html__( 'Rule Title', 'shopbuilder-pro' ),
						'type'        => 'text',
						'placeholder' => esc_html__( 'Partial Pay Rule', 'shopbuilder-pro' ),
						'value'       => esc_html__( 'Untitled Rule', 'shopbuilder-pro' ),
						'help'        => esc_html__( 'The title of the partial pay.', 'shopbuilder-pro' ),
					],
					'apply_for'             => [
						'id'      => 'apply_for',
						'type'    => 'select',
						'value'   => 'product',
						'label'   => esc_html__( 'Applicable For', 'shopbuilder-pro' ),
						'options' => [
							'product'     => esc_html__( 'Products', 'shopbuilder-pro' ),
							'product_cat' => esc_html__( 'Product Categories', 'shopbuilder-pro' ),
							'product_tag' => esc_html__( 'Product Tag', 'shopbuilder-pro' ),
						],
						'help'    => esc_html__( 'Specify whether the partial pay applies to selected products, categories or tags.', 'shopbuilder-pro' ),
					],
					'applicable_products'   => [
						'id'              => 'applicable_products',
						'type'            => 'search_and_multi_select',
						'label'           => esc_html__( 'Applicable Products', 'shopbuilder-pro' ),
						'help'            => esc_html__( 'Choose specific products to include. Leave blank to apply the discount to all products.', 'shopbuilder-pro' ),
						'placeholder'     => esc_html__( 'Search Products', 'shopbuilder-pro' ),
						'func_with_param' => [ Fns::class, 'get_post_types', [ 'post_type' => 'product' ] ],
						'options'         => Fns::get_post_types( null, [ 'post_type' => 'product' ] ),
						'dependency'      => [
							'rules' => [
								[
									'item'     => 'modules.partial_pay.partial_pay_rules.apply_for',
									'value'    => 'product',
									'operator' => '==',
								],
							],
						],
					],
					'applicable_categories' => [
						'id'              => 'applicable_categories',
						'type'            => 'search_and_multi_select',
						'label'           => esc_html__( 'Applicable Categories', 'shopbuilder-pro' ),
						'help'            => esc_html__( 'Choose specific categories to include. Leave blank to apply the discount to all categories.', 'shopbuilder-pro' ),
						'placeholder'     => esc_html__( 'Search Category', 'shopbuilder-pro' ),
						'func_with_param' => [
							Fns::class,
							'products_category_query',
						],
						'options'         => Fns::products_category_query(),
						'dependency'      => [
							'rules' => [
								[
									'item'     => 'modules.partial_pay.partial_pay_rules.apply_for',
									'value'    => 'product_cat',
									'operator' => '==',
								],
							],
						],
					],

					'applicable_tags'       => [
						'id'              => 'applicable_tags',
						'type'            => 'search_and_multi_select',
						'label'           => esc_html__( 'Applicable Tags', 'shopbuilder-pro' ),
						'help'            => esc_html__( 'Choose specific tags to include. Leave blank to apply the discount to all tags.', 'shopbuilder-pro' ),
						'placeholder'     => esc_html__( 'Search Tags', 'shopbuilder-pro' ),
						'func_with_param' => [
							Fns::class,
							'products_tags_query',
						],
						'options'         => Fns::products_tags_query(),
						'dependency'      => [
							'rules' => [
								[
									'item'     => 'modules.partial_pay.partial_pay_rules.apply_for',
									'value'    => 'product_tag',
									'operator' => '==',
								],
							],
						],
					],
					'exclude_product'       => [
						'id'              => 'exclude_product',
						'type'            => 'search_and_multi_select',
						'isPro'           => ! rtsb()->has_pro(),
						'label'           => esc_html__( 'Exclude Products', 'shopbuilder-pro' ),
						'help'            => esc_html__( 'Choose specific products to exclude. Leave blank to exclude none.', 'shopbuilder-pro' ),
						'placeholder'     => esc_html__( 'Search Products', 'shopbuilder-pro' ),
						'func_with_param' => [ Fns::class, 'get_post_types', [ 'post_type' => 'product' ] ],
						'options'         => Fns::get_post_types( null, [ 'post_type' => 'product' ] ),
						'dependency'      => [
							'rules' => [
								[
									'item'     => 'modules.partial_pay.partial_pay_rules.apply_for',
									'value'    => 'product',
									'operator' => '!=',
								],
							],
						],
					],
					'amount_type'           => [
						'id'      => 'amount_type',
						'type'    => 'select',
						'label'   => __( 'Amount Type', 'shopbuilder-pro' ),
						'help'    => esc_html__( 'Choose how you want to received the partial payment.', 'shopbuilder-pro' ),
						'value'   => 'fixed',
						'options' => [
							'percent' => esc_html__( 'Percentage', 'shopbuilder-pro' ),
							'fixed'   => esc_html__( 'Fixed', 'shopbuilder-pro' ),
						],
					],
					// 'partial_amount'        => [
					// 'id'    => 'partial_amount',
					// 'label' => __( 'Partial Pay Amount', 'shopbuilder-pro' ),
					// 'type'  => 'text',
					// 'value' => 50,
					// 'help'  => __( 'Enter the partial payment amount based on the amount type you chose above.<br /> Also, you can separate multiple amounts with commas. Multiple amounts will serve as multiple installments.', 'shopbuilder-pro' ),
					// ],
					'partial_amount'        => [
						'id'    => 'partial_amount',
						'label' => __( 'Partial Pay Amount', 'shopbuilder-pro' ),
						'type'  => 'number',
						'value' => 50,
						'help'  => __( 'Enter the partial payment amount based on the amount type you chose above.', 'shopbuilder-pro' ),
					],
				],
			],
			'partial_pay_notification_intro'   => [
				'id'   => 'partial_pay_notification_intro',
				'type' => 'description',
				'text' => sprintf(
				/* translators: %s: link */
					__( 'You can manage the below emails from the <strong><a href="%s" target="_blank">WooCommerce->Settings->Emails</a></strong> tab.', 'shopbuilder-pro' ),
					esc_url( admin_url( 'admin.php?page=wc-settings&tab=email' ) )
				),
				'tab'  => 'notification',
			],
			'partial_pay_received_email'       => [
				'id'    => 'partial_pay_received_email',
				'type'  => 'switch',
				'label' => __( 'New Partial Payment Received Email (For Admin)', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option to send an email to admin when an order contains a partial-pay product is received.', 'shopbuilder-pro' ),
				'tab'   => 'notification',
				'value' => 'on',
			],
			'partial_pay_confirmation_email'   => [
				'id'    => 'partial_pay_confirmation_email',
				'type'  => 'switch',
				'label' => __( 'Partial Payment Confirmation Email (For Customer)', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option to send an email to customers when an order contains a partial-pay product is placed.', 'shopbuilder-pro' ),
				'tab'   => 'notification',
				'value' => 'on',
			],
			'partial_pay_expiration_email'     => [
				'id'    => 'partial_pay_expiration_email',
				'type'  => 'switch',
				'label' => __( 'Partial Payment Balance Expiration Reminder Email', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option to send an email to customers when a partial payment balance dues is about to expire.', 'shopbuilder-pro' ),
				'tab'   => 'notification',
				'value' => 'on',
			],
			'partial_pay_on_hold_email'        => [
				'id'    => 'partial_pay_on_hold_email',
				'type'  => 'switch',
				'label' => __( 'Partial Payment Expiration On-Hold Notification Email', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option to send an email to customers when their partial payment has expired and the order has been placed on hold.', 'shopbuilder-pro' ),
				'tab'   => 'notification',
				'value' => 'on',
			],
			'pp_product_showcase'              => [
				'id'    => 'pp_product_showcase',
				'type'  => 'title',
				'label' => esc_html__( 'Shop / Archive Text Styles', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pp_note_font_size'                => [
				'id'    => 'pp_note_font_size',
				'label' => esc_html__( 'Font Size (px)', 'shopbuilder-pro' ),
				'type'  => 'slider',
				'min'   => 10,
				'max'   => 50,
				'unit'  => 'px',
				'value' => 15,
				'tab'   => 'styles',
			],
			'pp_note_font_color'               => [
				'id'    => 'pp_note_font_color',
				'label' => esc_html__( 'Text Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'pp_note_bg_color'                 => [
				'id'    => 'pp_note_bg_color',
				'label' => esc_html__( 'Background Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'pp_note_radius'                   => [
				'id'    => 'pp_note_radius',
				'type'  => 'text',
				'label' => esc_html__( 'Border Radius', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Border Radius. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pp_note_padding'                  => [
				'id'    => 'pp_note_padding',
				'type'  => 'text',
				'label' => esc_html__( 'Padding', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Padding. Example: 15px', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pp_note_margin'                   => [
				'id'    => 'pp_note_margin',
				'type'  => 'text',
				'label' => esc_html__( 'Margin', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Margin. Example: 0 15px 15px 15px', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pp_text_style_intro'              => [
				'id'    => 'pp_text_style_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Partial Pay Form Styles', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pp_form_primary_color'            => [
				'id'    => 'pp_form_primary_color',
				'type'  => 'color',
				'label' => esc_html__( 'Field Primary Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the primary color for the fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pp_form_background_color'         => [
				'id'    => 'pp_form_background_color',
				'type'  => 'color',
				'label' => esc_html__( 'Field Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the background color for the fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pp_radio_label_color'             => [
				'id'    => 'pp_radio_label_color',
				'type'  => 'color',
				'label' => esc_html__( 'Field Label Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the label color for the radio fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pp_radio_price_color'             => [
				'id'    => 'pp_radio_price_color',
				'type'  => 'color',
				'label' => esc_html__( 'Field Price Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the label color for the prices.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pp_field_text_color'              => [
				'id'    => 'pp_field_text_color',
				'label' => esc_html__( 'Field Text Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the text color for the fields.', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'pp_form_border_color'             => [
				'id'    => 'pp_form_border_color',
				'type'  => 'color',
				'label' => esc_html__( 'Field Border Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Select the border color for the fields.', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pp_radius'                        => [
				'id'    => 'pp_radius',
				'type'  => 'text',
				'label' => esc_html__( 'Border Radius', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Border Radius. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pp_margin'                        => [
				'id'    => 'pp_margin',
				'type'  => 'text',
				'label' => esc_html__( 'Margin', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enter Margin. Ex. 10px | 5px 10px | 0 5px 5px 0', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pp_badge_style_intro'             => [
				'id'    => 'pp_badge_style_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Partial Pay Badge', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'pp_badge_text_color'              => [
				'id'    => 'pp_badge_text_color',
				'label' => esc_html__( 'Badge Text Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose color for Partial Pay badge text', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'pp_badge_bg_color'                => [
				'id'    => 'pp_badge_bg_color',
				'label' => esc_html__( 'Badge Background Color', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Choose background color for Partial Pay badge text', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
		];
	}

	/**
	 * Get partial pay settings data.
	 *
	 * @return array
	 */
	public static function get_settings_data() {
		$settings    = Fns::get_options( 'modules', 'partial_pay' ) ?? [];
		$cache_key   = 'rtsb_partial_pay_settings_data_' . md5( wp_json_encode( $settings ) );
		$cached_data = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false !== $cached_data ) {
			return $cached_data;
		}

		$rules = RenderHelpers::get_data( $settings, 'partial_pay_rules', [] );

		if ( ! empty( $rules ) && is_string( $rules ) ) {
			$rules = json_decode( $rules, true );
		}

		$settings_data = [
			'display_in_loop'         => RenderHelpers::get_data( $settings, 'partial_pay_in_product_loop', '' ),
			'default_payment_type'    => RenderHelpers::get_data( $settings, 'default_payment_type', 'full' ),
			'restrict_payment_method' => RenderHelpers::get_data( $settings, 'restrict_payment_method', [] ),
			'payment_date'            => RenderHelpers::get_data( $settings, 'balanced_payment_date', 30 ),
			'days_to_send_mail'       => RenderHelpers::get_data( $settings, 'days_to_send_mail', 5 ),
			'disable_coupons'         => RenderHelpers::get_data( $settings, 'disable_coupons', '' ),
			'user_role_management'    => RenderHelpers::get_data( $settings, 'partial_pay_user_role_management', 'all' ),
			'roles'                   => RenderHelpers::get_data( $settings, 'partial_pay_roles', [ 'administrator', 'customer' ] ),
			'rules'                   => $rules,
			'new_email'               => RenderHelpers::get_data( $settings, 'partial_pay_received_email', '' ),
			'confirm_email'           => RenderHelpers::get_data( $settings, 'partial_pay_confirmation_email', '' ),
			'reminder_email'          => RenderHelpers::get_data( $settings, 'partial_pay_expiration_email', '' ),
			'on_hold_email'           => RenderHelpers::get_data( $settings, 'partial_pay_on_hold_email', '' ),
			'labels'                  => [
				'partial_amount'         => RenderHelpers::get_data( $settings, 'partial_payment_label', __( 'Pay Partial Amount', 'shopbuilder-pro' ) ),
				'full_amount'            => RenderHelpers::get_data( $settings, 'full_payment_label', __( 'Pay Full Amount', 'shopbuilder-pro' ) ),
				'single_label'           => RenderHelpers::get_data( $settings, 'first_installment_label_single', __( 'Partial Pay', 'shopbuilder-pro' ) ),
				'remaining_label_single' => RenderHelpers::get_data( $settings, 'remaining_label_single', __( 'Remaining Balance', 'shopbuilder-pro' ) ),
				'first_installment'      => RenderHelpers::get_data( $settings, 'first_installment_label', __( 'First Installment', 'shopbuilder-pro' ) ),
				'other_installment'      => RenderHelpers::get_data( $settings, 'other_installment_label', __( 'Second Installment', 'shopbuilder-pro' ) ),
				'to_pay'                 => RenderHelpers::get_data( $settings, 'to_pay_label', __( 'Pay Balance', 'shopbuilder-pro' ) ),
			],
		];

		wp_cache_set( $cache_key, $settings_data, 'shopbuilder', 12 * HOUR_IN_SECONDS );
		Cache::set_data_cache_key( $cache_key );

		return $settings_data;
	}

	/**
	 * Checks if a product has partial pay.
	 *
	 * @param object $product The product object to check.
	 * @param bool   $variation Variation check.
	 *
	 * @return bool
	 */
	public static function product_has_partial_pay( $product, $variation = false ) {
		$product_id    = $variation ? $product->get_parent_id() : $product->get_id();
		$cache_key     = 'rtsb_product_partial_pay_for_' . $product_id;
		$cached_result = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false !== $cached_result ) {
			return $cached_result;
		}

		if ( apply_filters( 'rtsb/module/partial_pay/global/disable', false, $product, $product_id ) ) {
			return false;
		}

		$partial_pay_rules = self::get_partial_pay_rules( $product_id );
		$has_partial_pay   = ! empty( $partial_pay_rules );

		wp_cache_set( $cache_key, $has_partial_pay, 'shopbuilder', 12 * HOUR_IN_SECONDS );
		Cache::set_data_cache_key( $cache_key );

		return $has_partial_pay;
	}

	/**
	 * Get global and product partial Pay rules.
	 *
	 * @param number $product_id .
	 * @return array;
	 */
	public static function get_partial_pay_rules( $product_id ) {
		$cache_key   = 'rtsb_partial_pay_rules_' . $product_id;
		$cached_data = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false !== $cached_data ) {
			return $cached_data;
		}

		if ( self::is_global_partial_pay_disabled( $product_id ) ) {
			return [];
		}

		if ( self::is_global_partial_pay_overridden( $product_id ) ) {
			$partial_pay = self::get_partial_pay_meta_data( $product_id )['partial_pay'];
		} else {
			$global_rules = self::get_global_partial_pay_rules();
			$partial_pay  = self::get_relevant_partial_pay_rules( $global_rules, $product_id );

			foreach ( $global_rules as $rule ) {
				if ( ! empty( $rule['exclude_product'] ) ) {
					$excluded_product_ids = array_column( $rule['exclude_product'], 'value' );

					if ( in_array( $product_id, $excluded_product_ids, true ) ) {
						return [];
					}
				}
			}
		}

		wp_cache_set( $cache_key, $partial_pay, 'shopbuilder', 12 * HOUR_IN_SECONDS );
		Cache::set_data_cache_key( $cache_key );

		return $partial_pay;
	}

	/**
	 * Filters and retrieves relevant partial Pay based on product ID and grouping settings.
	 *
	 * @param array $settings An array of settings.
	 * @param int   $product_id The ID of the product.
	 *
	 * @return array
	 */
	public static function get_relevant_partial_pay_rules( $settings, $product_id ) {
		$relevant_partial_pay = [];

		foreach ( $settings as $setting ) {
			$is_relevant_partial_pay = self::is_relevant_partial_pay( $setting, $product_id );

			if ( $is_relevant_partial_pay ) {
				$relevant_partial_pay[] = $setting;
			}
		}

		return array_values( $relevant_partial_pay );
	}

	/**
	 * Determines if a setting is relevant based on the product ID and rules settings.
	 *
	 * @param array $setting The setting array.
	 * @param int   $product_id The ID of the product.
	 *
	 * @return bool
	 */
	private static function is_relevant_partial_pay( $setting, $product_id ) {
		$group_product = $setting['applicable_products'] ?? [];
		$group_cat     = $setting['applicable_categories'] ?? [];
		$group_tag     = $setting['applicable_tags'] ?? [];
		$group_by      = $setting['apply_for'] ?? 'product';
		$exclude       = $setting['exclude_product'] ?? [];

		if ( 'product' !== $group_by && in_array( $product_id, $exclude, true ) ) {
			return false;
		}

		if ( 'product' === $group_by ) {
			$group_product_ids = self::get_group_ids( $group_product );

			return self::is_relevant_by_product( $group_product_ids, $product_id );
		} elseif ( 'product_cat' === $group_by ) {
			$group_cat_ids = self::get_group_ids( $group_cat );

			return self::is_relevant_by_taxonomy( 'product_cat', $group_cat_ids, $product_id );
		} elseif ( 'product_tag' === $group_by ) {
			$group_cat_ids = self::get_group_ids( $group_tag );

			return self::is_relevant_by_taxonomy( 'product_tag', $group_cat_ids, $product_id );
		}

		return false;
	}

	/**
	 * Get group product IDs.
	 *
	 * @param array $ids The group id array.
	 *
	 * @return array
	 */
	public static function get_group_ids( array $ids ) {
		$group_ids = [];

		foreach ( $ids as $id ) {
			$group_ids[] = $id['value'] ?? null;
		}

		return $group_ids;
	}

	/**
	 * Checks if the setting is relevant based on product ID for 'product' grouping.
	 *
	 * @param array $group_product The product group array.
	 * @param int   $product_id The ID of the product.
	 *
	 * @return bool
	 */
	private static function is_relevant_by_product( $group_product, $product_id ) {
		return empty( $group_product )
			|| ( in_array( $product_id, $group_product, true ) && ( wp_doing_ajax() || ! is_admin() ) );
	}

	/**
	 * Checks if the setting is relevant based on product ID for taxonomy grouping.
	 *
	 * @param string $taxonomy The taxonomy name.
	 * @param array  $group_tax_ids The taxonomy group ids array.
	 * @param int    $product_id The ID of the product.
	 *
	 * @return bool
	 */
	private static function is_relevant_by_taxonomy( $taxonomy, $group_tax_ids, $product_id ) {
		if ( empty( $group_tax_ids ) ) {
			return true;
		}

		$taxonomy      = ! empty( $taxonomy ) ? $taxonomy : 'product_cat';
		$product_terms = get_the_terms( $product_id, $taxonomy );

		if ( is_array( $product_terms ) ) {
			$product_term_ids = wp_list_pluck( $product_terms, 'term_id' );

			return ! empty( array_intersect( $product_term_ids, $group_tax_ids ) );
		}

		return false;
	}

	/**
	 * Get Product partial pay data.
	 *
	 * @param number $product_id .
	 * @return array
	 */
	public static function get_partial_pay_meta_data( $product_id ) {
		if ( ! $product_id ) {
			return [];
		}

		$cache_key   = 'rtsb_partial_pay_meta_data_' . $product_id;
		$cached_data = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false !== $cached_data ) {
			return $cached_data;
		}

		$product = wc_get_product( $product_id );

		if ( ! $product ) {
			return [];
		}

		$meta_mapping = self::product_meta_data_mapping();
		$product_meta = [];

		foreach ( $meta_mapping as $meta_key => $custom_field_id ) {
			$product_meta[ $meta_key ] = $product->get_meta( $custom_field_id );
		}

		wp_cache_set( $cache_key, $product_meta, 'shopbuilder', 12 * HOUR_IN_SECONDS );
		Cache::set_data_cache_key( $cache_key );

		return $product_meta;
	}

	/**
	 * Maps product meta-keys to their corresponding custom field IDs.
	 *
	 * @return array
	 */
	public static function product_meta_data_mapping() {
		return [
			'disable'     => '_rtsb_disable_partial_pay',
			'override'    => '_rtsb_enable_partial_pay',
			'partial_pay' => '_rtsb_product_partial_pay',
		];
	}

	/**
	 * Check if the global partial pay is disabled for a specific product.
	 *
	 * @param int $product_id The ID of the product to check.
	 *
	 * @return bool
	 */
	public static function is_global_partial_pay_disabled( $product_id ) {
		return 'on' === ( self::get_partial_pay_meta_data( $product_id )['disable'] ?? 'off' );
	}

	/**
	 * Check if the global partial pay settings are overridden for a specific product.
	 *
	 * @param int $product_id The ID of the product to check.
	 *
	 * @return bool
	 */
	public static function is_global_partial_pay_overridden( $product_id ) {
		return 'on' === ( self::get_partial_pay_meta_data( $product_id )['override'] ?? 'off' );
	}

	/**
	 * Get global partial pay rules.
	 *
	 * @return mixed
	 */
	public static function get_global_partial_pay_rules() {
		return self::get_settings_data()['rules'];
	}

	/**
	 * Get product partial pay rules.
	 *
	 * @param number $product_id The ID of the product.
	 * @param number $variation_id The ID of the product variation.
	 *
	 * @return array
	 */
	public static function get_partial_amount( $product_id, $variation_id = null ) {
		$rules = self::get_partial_pay_rules( $product_id );

		if ( empty( $rules ) ) {
			return [
				'amount'   => 0,
				'due_date' => '',
			];
		}

		$rule = self::is_global_partial_pay_overridden( $product_id ) ? $rules : reset( $rules );

		$amount_type = $rule['amount_type'];
		$amounts     = explode( ',', trim( $rule['partial_amount'] ) );
		$amounts     = array_map( 'floatval', $amounts );
		$product     = ! empty( $variation_id ) ? wc_get_product( $variation_id ) : wc_get_product( $product_id );
		$full_price  = $product->get_price();

		$installments = [];
		$total_paid   = 0;

		$labels       = self::get_settings_data()['labels'];
		$first_label  = $labels['first_installment'] ?? esc_html__( 'First Installment', 'shopbuilder-pro' );
		$other_labels = $labels['other_installment'] ?? '';
		$other_labels = explode( ',', trim( $other_labels ) );
		$payment_date = self::get_settings_data()['payment_date'] ?? 30;
		$payment_date = ! empty( $rule['due_date'] ) ? $rule['due_date'] : $payment_date;

		foreach ( $amounts as $index => $value ) {
			$amount = ( 'percent' === $amount_type ) ? ( $full_price * ( $value / 100 ) ) : $value;

			if ( 0 === $index ) {
				$title = $first_label;
			} else {
				$title = isset( $other_labels[ $index - 1 ] ) ? trim( $other_labels[ $index - 1 ] ) : __( 'Installment', 'shopbuilder-pro' ) . ' ' . ( $index + 1 );
			}

			// TODO: Will be implemented in the future.
			/*
			if ( ( count( $amounts ) - 1 ) === $index && 'percent' === $amount_type ) {
				$amount = $full_price >= $total_paid ? $full_price - $total_paid : max( 0, $full_price - ( $total_paid - $amount ) );
			} else {
				$total_paid += $amount;
			}
			*/

			$total_paid += $amount;

			if ( $amount > 0 ) {
				$installments[] = [
					'title'    => $title,
					'amount'   => $amount,
					'due_date' => gmdate( 'Y-m-d', strtotime( '+' . ( absint( $payment_date ) * ( $index ) ) . ' days' ) ),
				];
			}
		}

		// If the total paid is less than the full price, adjust the last installment.
		if ( count( $amounts ) > 1 && $total_paid < $full_price ) {
			$remaining_amount                      = $full_price - $total_paid;
			$last_index                            = count( $installments ) - 1;
			$installments[ $last_index ]['amount'] = 'percent' === $amount_type ? $full_price - $total_paid : floatval( $installments[ $last_index ]['amount'] ) + $remaining_amount;
		} else {
			$installments[0]['due_date'] = gmdate( 'Y-m-d', strtotime( '+' . ( absint( $payment_date ) ) . ' days' ) );
		}

		return $installments;
	}

	/**
	 * Get formatted installment text.
	 *
	 * @param string $type The type of installment.
	 * @param array  $data The installment data.
	 *
	 * @return string
	 */
	public static function get_formatted_installment_text( $type, $data ) {
		switch ( $type ) {
			case 'multiple_installments':
				return sprintf(
					'<span><b>%s</b>: %s%s</span>',
					$data['title'],
					wc_price( $data['amount'] ),
					$data['index'] > 0 ? ', ' . esc_html__( 'due on', 'shopbuilder-pro' ) . ' ' . gmdate( 'F j, Y', strtotime( $data['due_date'] ) ) : ''
				);

			case 'single_installment':
				return sprintf(
					'<span><b>%s</b>: %s</span>',
					$data['label'],
					wc_price( floatval( $data['amount'] ) )
				);

			case 'remaining_amount':
				return sprintf(
					'<span><b>%s</b>: %s %s %s</span>',
					$data['label'],
					wc_price( floatval( $data['full_price'] ) - floatval( $data['amount'] ) ),
					esc_html__( 'will be due on', 'shopbuilder-pro' ),
					esc_html( gmdate( 'F j, Y', strtotime( $data['due_date'] ) ) )
				);

			default:
				return '';
		}
	}

	/**
	 * Check if the partial payment is valid.
	 *
	 * @param array $partial_amount The partial payment amount.
	 * @param float $full_price The full price of the product.
	 *
	 * @return bool
	 */
	public static function is_valid_partial_payment( $partial_amount, $full_price ) {
		if ( empty( $partial_amount ) || ! is_array( $partial_amount ) ) {
			return false;
		}

		$partial_price = $partial_amount[0]['amount'] ?? 0;

		if ( 0 === (int) $partial_price ) {
			return false;
		}

		if ( count( $partial_amount ) <= 1
			&& ( 0 === ( (int) $full_price - (int) $partial_price ) || (int) $full_price < (int) $partial_price ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Display a notice related to the product's partial payment status.
	 *
	 * @param string $type  Notice Type.
	 *
	 * @return void
	 */
	public static function partial_pay_product_notice( $type = 'cart_mix' ) {
		$message = '';

		if ( 'cart_mix' === $type ) {
			$message = esc_html__( 'Sorry, you cannot add Partial-Pay and other products in the same order. Please check-out separately.', 'shopbuilder-pro' );
		} elseif ( 'cart_multi' === $type ) {
			$message = esc_html__( 'Sorry, you can only add one Partial-Pay product at a time. Please complete your current Partial-Pay first.', 'shopbuilder-pro' );
		}

		wc_add_notice( '<strong>' . $message . '</strong>', 'error' );
	}

	/**
	 * Check if an order contains partial pay products.
	 *
	 * @param WC_Order $order The order object.
	 *
	 * @return bool
	 */
	public static function order_contains_partial_pay_products( $order ) {
		if ( ! $order instanceof WC_Order ) {
			return false;
		}

		foreach ( $order->get_items() as $item ) {
			$product_id = $item->get_product_id();
			$product    = wc_get_product( $product_id );

			if ( $product && self::product_has_partial_pay( $product, 'variation' === $product->get_type() ) ) {
				return true;
			}
		}

		return false;
	}

	/**
	 * Check if partial pay is disabled globally for a product.
	 *
	 * @param object $product The product object.
	 *
	 * @return bool
	 */
	public static function is_partial_pay_disabled( $product ) {
		return apply_filters( 'rtsb/module/partial_pay/global/disable', false, $product );
	}
}
