<?php
/**
 * Pre-order Cart Page Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\PreOrder\Frontend;

use WC_Cart;
use Exception;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\PreOrder\PreOrderFns;

defined( 'ABSPATH' ) || exit();

/**
 * Pre-order Cart Page Class.
 */
class CartPage {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Prevent mixed products.
	 *
	 * @var bool
	 */
	private $prevent_mix = false;

	/**
	 * Prevent multiple products.
	 *
	 * @var bool
	 */
	private $prevent_multiple = false;

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		$this
			->init_data()
			->init_hooks();
	}

	/**
	 * Initialize pre-order settings data.
	 *
	 * @return CartPage
	 */
	private function init_data() {
		$settings = PreOrderFns::get_pre_order_settings_data();

		$this->prevent_mix      = 'on' === $settings['prevent_mixing'];
		$this->prevent_multiple = 'on' === $settings['prevent_multiple'];

		return $this;
	}

	/**
	 * Initialize the hooks.
	 *
	 * @return void
	 */
	private function init_hooks() {
		/**
		 * Actions.
		 */
		add_action( 'woocommerce_after_cart_item_name', [ $this, 'display_pre_order_cart_text' ] );

		/**
		 * Filters.
		 */
		add_filter( 'woocommerce_check_cart_items', [ $this, 'display_pre_order_limit_notice' ], 99, 4 );
		add_filter( 'rtsb/minicart/quantity/max', [ $this, 'pre_order_cart_limit' ], 99, 2 );
		add_filter( 'woocommerce_add_to_cart_validation', [ $this, 'validate_pre_order_cart' ], 10, 5 );
		add_filter( 'woocommerce_before_calculate_totals', [ $this, 'validate_pre_order_cart_totals' ] );
		add_filter( 'woocommerce_cart_item_name', [ $this, 'pre_order_checkout_message' ], 10, 2 );
		add_filter( 'woocommerce_order_button_text', [ $this, 'pre_order_checkout_button' ], 10, 2 );
	}

	/**
	 * Pre-order cart message.
	 *
	 * @param array $cart_item Cart Item.
	 *
	 * @return void
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function display_pre_order_cart_text( $cart_item ) {
		if ( ! is_cart() ) {
			return;
		}
		$product_id = $cart_item['variation_id'] ?: $cart_item['product_id'];
		$product    = $cart_item['data'];
		$message    = '';

		$is_on_pre_order = $cart_item['variation_id']
			? PreOrderFns::variation_id_is_on_pre_order( $cart_item['variation_id'] )
			: PreOrderFns::is_on_pre_order( $product );

		if ( ! $is_on_pre_order ) {
			return;
		}

		if ( PreOrderFns::is_on_pre_order( $product ) && PreOrderFns::is_pre_order_allowed( $product_id ) ) {
			$message = PreOrderFns::get_pre_order_message( $product_id );
		}

		if ( ! empty( $message ) ) {
			echo '<span class="rtsb-pre-order-text-badge">' . esc_html__( 'Pre-Order', 'shopbuilder-pro' ) . '</span>';
			echo '<div class="stock available-on-pre-order rtsb-pre-order-message preorder_notification">
					<span class="wc-item-meta-label rtsb-stock-text">' . wp_kses_post( $message ) . '
					</span>
				</div>';
		}
	}

	/**
	 * Display pre-order limit notice and adjust cart item quantity if necessary.
	 *
	 * @return void
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function display_pre_order_limit_notice() {
		// Get cart contents.
		$cart = WC()->cart->get_cart();

		foreach ( $cart as $cart_item_key => $cart_item ) {
			// Get product data.
			$product_id   = $cart_item['product_id'];
			$variation_id = $cart_item['variation_id'];
			$id           = $variation_id ?: $product_id;
			$quantity     = $cart_item['quantity'];
			$product      = $cart_item['data'];

			if ( ! $product ) {
				continue;
			}

			$is_on_pre_order = $variation_id
				? PreOrderFns::variation_id_is_on_pre_order( $variation_id )
				: PreOrderFns::is_on_pre_order( $product );

			if ( ! $is_on_pre_order ) {
				continue;
			}

			$amount     = PreOrderFns::get_pre_order_limit( $id );
			$validation = PreOrderFns::is_pre_order_allowed( $id, $variation_id );

			// Check if the cart quantity exceeds the available pre-order amount.
			if ( $quantity > $amount ) {
				$validation = false;
			}

			if ( ! $validation ) {
				WC()->cart->set_quantity( $cart_item_key, $amount );

				// Display error notice.
				PreOrderFns::pre_order_product_notice( $product, $amount );
			}
		}
	}

	/**
	 * Filters the cart limit for a product when it is on pre-order.
	 *
	 * @param int    $limit   The original cart limit for the product.
	 * @param object $product The product object to check for pre-order status.
	 *
	 * @return int
	 */
	public function pre_order_cart_limit( $limit, $product ) {
		$type            = $product->get_type();
		$product_id      = $product->get_id();
		$is_on_pre_order = 'variation' === $type
			? PreOrderFns::variation_id_is_on_pre_order( $product_id )
			: PreOrderFns::is_on_pre_order( $product );

		if ( ! $is_on_pre_order ) {
			return $limit;
		}

		return PreOrderFns::get_pre_order_limit( $product->get_id() );
	}

	/**
	 * Perform cart validation for pre-ordered products.
	 *
	 * @param bool  $passed_validation Whether the validation has passed.
	 * @param int   $product_id        The ID of the product being added to the cart.
	 * @param int   $quantity          The quantity of the product being added to the cart.
	 * @param int   $variation_id      The ID of the product variation being added to the cart.
	 * @param array $variation         The variation attributes of the product being added to the cart.
	 *
	 * @return bool
	 * @throws Exception If there is an error in determining the pre-order status.
	 */
	public function validate_pre_order_cart( $passed_validation, $product_id, $quantity, $variation_id = '', $variation = [] ) {
		$id      = $variation_id ?: $product_id;
		$product = wc_get_product( $id );

		if ( ! $product ) {
			return $passed_validation;
		}

		if ( $this->cart_contains_mixed_products() ) {
			PreOrderFns::pre_order_product_notice( $product, 0, 'cart_mix' );

			return true;
		}

		if ( $this->cart_contains_multiple_pre_order_products() ) {
			PreOrderFns::pre_order_product_notice( $product, 0, 'cart_multi' );

			return true;
		}

		if ( ! PreOrderFns::is_on_pre_order( $product ) ) {
			return $passed_validation;
		}

		$amount = PreOrderFns::get_pre_order_limit( $id );

		if ( empty( $amount ) || $amount < 0 ) {
			return true;
		}

		$cart_item_key     = WC()->cart->find_product_in_cart( WC()->cart->generate_cart_id( $product_id, $variation_id, $variation ) );
		$passed_validation = PreOrderFns::is_pre_order_allowed( $id, $variation_id );

		if ( $quantity > $amount ) {
			$passed_validation = false;
		}

		if ( $cart_item_key ) {
			$cart_item = WC()->cart->get_cart_item( $cart_item_key );

			$quantity = $cart_item['quantity'] + $quantity;

			// Check if the cart quantity exceeds the available pre-order amount.
			if ( $quantity > $amount ) {
				$passed_validation = false;
			}
		}

		if ( ! $passed_validation ) {
			PreOrderFns::pre_order_product_notice( $product, $amount );

			$passed_validation = false;
		}

		return $passed_validation;
	}

	/**
	 * Perform cart calculations for pre-order products.
	 *
	 * @param WC_Cart $cart The cart object.
	 *
	 * @return void
	 */
	public function validate_pre_order_cart_totals( $cart ) {
		if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
			return;
		}

		if ( $this->cart_contains_mixed_products() ) {
			foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {
				$product = $cart_item['data'];

				if ( ! PreOrderFns::is_on_pre_order( $product ) ) {
					$cart->remove_cart_item( $cart_item_key );
					PreOrderFns::pre_order_product_notice( $product, 0, 'cart_mix' );
					break;
				}
			}
		}

		if ( $this->cart_contains_multiple_pre_order_products() ) {
			$preorder_found = false;

			foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {
				$product = $cart_item['data'];

				if ( PreOrderFns::is_on_pre_order( $product ) ) {
					if ( $preorder_found ) {
						$cart->remove_cart_item( $cart_item_key );
						PreOrderFns::pre_order_product_notice( $product, 0, 'cart_multi' );
					} else {
						$preorder_found = true;
					}
				}
			}
		}
	}

	/**
	 * Pre-order checkout message.
	 *
	 * @param string $name Name.
	 * @param array  $cart_item Cart Item.
	 *
	 * @return string
	 */
	public function pre_order_checkout_message( $name, $cart_item ) {
		if ( ! is_checkout() ) {
			return $name;
		}
		$product = $cart_item['data'];
		$text    = '<span class="rtsb-checkout-text"><span class="rtsb-pre-order-text-badge">' . esc_html__( 'Pre-Order', 'shopbuilder-pro' ) . '</span></span>';

		if ( ! PreOrderFns::is_on_pre_order( $product ) ) {
			return $name;
		}

		return $name . $text;
	}

	/**
	 * Change the "Place Order" button label on the WooCommerce checkout page.
	 *
	 * @param string $button_text The original button text.
	 *
	 * @return string
	 */
	public function pre_order_checkout_button( $button_text ) {
		$custom_label = PreOrderFns::get_pre_order_settings_data()['checkout_label'];

		if ( ! $this->cart_contains_pre_order_products() || empty( $custom_label ) ) {
			return $button_text;
		}

		return $custom_label;
	}

	/**
	 * Check the contents of the cart for mixed and multiple pre-order products.
	 *
	 * @return bool[]
	 */
	private function analyze_cart_content() {
		$has_pre_order   = false;
		$has_regular     = false;
		$pre_order_count = 0;

		foreach ( WC()->cart->get_cart() as $cart_item ) {
			$product = $cart_item['data'];

			if ( PreOrderFns::is_on_pre_order( $product ) ) {
				$has_pre_order = true;
				$pre_order_count++;
			} else {
				$has_regular = true;
			}
		}

		return [
			'cart_contains_pre_order' => $has_pre_order,
			'cart_contains_mixed'     => $has_pre_order && $has_regular,
			'cart_contains_multiple'  => $pre_order_count > 1,
		];
	}

	/**
	 * Check if the cart contains pre-order products.
	 *
	 * @return bool
	 */
	private function cart_contains_pre_order_products() {
		return $this->analyze_cart_content()['cart_contains_pre_order'];
	}

	/**
	 * Check if the cart contains mixed products (pre-order and regular).
	 *
	 * @return bool
	 */
	private function cart_contains_mixed_products() {
		return $this->prevent_mix && $this->analyze_cart_content()['cart_contains_mixed'];
	}

	/**
	 * Check if the cart contains multiple pre-order products.
	 *
	 * @return bool
	 */
	private function cart_contains_multiple_pre_order_products() {
		return $this->prevent_multiple && $this->analyze_cart_content()['cart_contains_multiple'];
	}
}
