<?php
/**
 * Pre-order Release Email Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\PreOrder\Notification;

use WC_Email;
use WC_Order;
use Exception;

defined( 'ABSPATH' ) || exit();

/**
 * Pre-order Release Email Class.
 */
class PreOrderReleaseEmail extends WC_Email {
	/**
	 * Email content.
	 *
	 * @var string
	 */
	public $email_content;

	/**
	 * Class Constructor.
	 */
	public function __construct() {
		$this->id             = 'rtsb_pre_order_released';
		$this->title          = esc_html__( 'ShopBuilder: Pre-Order Product Released', 'shopbuilder-pro' );
		$this->description    = esc_html__( 'This email is sent to customer when a pre-order has been released.', 'shopbuilder-pro' );
		$this->heading        = esc_html__( 'The Pre-Order product has been released', 'shopbuilder-pro' );
		$this->subject        = esc_html__( '[{site_title}]: Your Pre-Order product is now available #{order_number}', 'shopbuilder-pro' );
		$this->customer_email = true;
		$this->email_content  = $this->get_option(
			'email_content',
			sprintf(
				/* translators: line break */
				__( 'Hi {customer_name}! %1$s%2$sWe are pleased to inform you that the product you pre-ordered is now available in our shop. %1$sYou can view your order here: {order_link} %1$s%2$sThank you for your purchase.%1$s%2$sBest regards,%1$s{site_title}', 'shopbuilder-pro' ),
				"\n",
				"\n"
			)
		);
		$this->template_html = 'emails/pre-order-available.php';

		add_action( 'rtsb/module/pre_order/release_email', [ $this, 'trigger' ], 10, 1 );

		parent::__construct();

		$this->email_type = 'html';
	}

	/**
	 * Trigger the email notification for a new order.
	 *
	 * @param WC_Order $order The order object.
	 *
	 * @return void
	 * @throws Exception When `WC_Data_Store::load` validation fails.
	 */
	public function trigger( $order ) {
		if ( ! $order ) {
			return;
		}

		$this->object    = $order;
		$customer_name   = $this->object->get_formatted_billing_full_name();
		$this->recipient = $this->object->get_billing_email();
		$order_number    = $this->object->get_order_number();
		$items           = $this->object->get_items();

		foreach ( $items as $item ) {
			$product_id = $item->get_variation_id() ?: $item->get_product_id();

			// Add the meta key to indicate the notification has been sent.
			update_post_meta( $product_id, '_rtsb_preorder_availability_notified', 'yes' );
		}

		$this->placeholders = array_merge(
			$this->placeholders,
			[
				'{customer_name}' => $customer_name,
				'{order_number}'  => $order_number,
				'{order_date}'    => wc_format_datetime( $this->object->get_date_created() ),
			]
		);

		if ( ! $this->is_enabled() || ! $this->get_recipient() ) {
			return;
		}

		$this->send( $this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments() );

		// Add order meta.
		wc_update_order_item_meta( $order_number, '_rtsb_preorder_availability_notified', 'yes' );
	}

	/**
	 * Get the HTML content for the email.
	 *
	 * @return string The HTML content.
	 */
	public function get_content_html() {
		return wc_get_template_html(
			$this->template_html,
			[
				'order'         => $this->object,
				'email_heading' => $this->get_heading(),
				'sent_to_admin' => true,
				'plain_text'    => false,
				'email'         => $this,
				'store_email'   => $this->get_from_address(),
			],
			'',
			rtsbpro()->get_plugin_template_path()
		);
	}

	/**
	 * Initialize the form fields for the plugin settings.
	 *
	 * @return void
	 */
	public function init_form_fields() {
		$this->placeholders = array_merge(
			$this->placeholders,
			[
				'{customer_name}' => __( 'Customer Name', 'shopbuilder-pro' ),
				'{order_number}'  => __( 'Order Number', 'shopbuilder-pro' ),
				'{order_date}'    => __( 'Order Date', 'shopbuilder-pro' ),
				'{order_link}'    => __( 'Order Link', 'shopbuilder-pro' ),
			]
		);

		/* translators: %s: list of placeholders */
		$placeholder_text  = sprintf( __( 'Available placeholders: %s', 'shopbuilder-pro' ), '<code>' . implode( '</code>, <code>', array_keys( $this->placeholders ) ) . '</code>' );
		$this->form_fields = [
			'enabled'       => [
				'title'   => __( 'Enable/Disable', 'shopbuilder-pro' ),
				'type'    => 'checkbox',
				'label'   => __( 'Enable this email notification', 'shopbuilder-pro' ),
				'default' => 'yes',
			],
			'subject'       => [
				'title'       => __( 'Subject', 'shopbuilder-pro' ),
				'type'        => 'text',
				'desc_tip'    => true,
				'description' => $placeholder_text,
				'placeholder' => $this->get_default_subject(),
				'default'     => $this->get_default_subject(),
			],
			'heading'       => [
				'title'       => __( 'Email heading', 'shopbuilder-pro' ),
				'type'        => 'text',
				'desc_tip'    => true,
				'description' => $placeholder_text,
				'placeholder' => $this->get_default_heading(),
				'default'     => $this->get_default_heading(),
			],
			'email_content' => [
				'title'       => __( 'Email Content', 'shopbuilder-pro' ),
				'description' => __( 'Text to appear in the main email body.', 'shopbuilder-pro' ) . ' ' . $placeholder_text . '.',
				'css'         => 'width:800px; height: 185px;',
				'placeholder' => '',
				'type'        => 'textarea',
				'default'     => $this->email_content,
			],
		];
	}
}
