<?php
/**
 * Pre-order Module Admin Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\PreOrder;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Helpers\AdminInputsFns;

defined( 'ABSPATH' ) || exit();

/**
 * Pre-order Module Admin Class.
 */
class PreOrderAdmin {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Global settings.
	 *
	 * @var array
	 */
	private $data = [];

	/**
	 * Class Constructor.
	 */
	private function __construct() {
		$this->get_data();
		$this->product_meta_boxes();
		$this->admin_column();
	}

	/**
	 * Adds meta-boxes for pre-order settings in WooCommerce.
	 *
	 * @return void
	 */
	public function product_meta_boxes() {
		$this
			->simple_products_support()
			->variable_products_support();
	}

	/**
	 * Sets up support for single products in the back order settings.
	 *
	 * @return PreOrderAdmin
	 */
	public function simple_products_support() {
		add_action( 'woocommerce_product_options_stock_status', [ $this, 'add_simple_meta_box' ] );
		add_action( 'woocommerce_process_product_meta', [ $this, 'save_simple_meta_box' ] );

		return $this;
	}

	/**
	 * Sets up support for variable products in the back order settings.
	 *
	 * @return void
	 */
	public function variable_products_support() {
		add_action( 'woocommerce_product_after_variable_attributes', [ $this, 'add_variation_meta_box' ], 10, 3 );
		add_action( 'woocommerce_save_product_variation', [ $this, 'save_variation_meta_box' ], 10, 2 );
	}

	/**
	 * Get global preorder settings.
	 *
	 * @return void
	 */
	private function get_data() {
		$limit        = absint( Fns::get_option( 'modules', 'pre_order', 'pre_order_max_allowed_order' ) );
		$auto_convert = Fns::get_option( 'modules', 'pre_order', 'pre_order_auto_convert' ) ?? '';

		$this->data = [
			// Global Settings.
			'global_limit'        => $limit,
			'global_auto_convert' => 'on' === $auto_convert ? 'yes' : 'no',

			// Placeholders.
			'limit_placeholder'   => $limit ? sprintf(
				'%s (%s)',
				esc_html__( 'Override global limit', 'shopbuilder-pro' ),
				'Set as ' . esc_html( $limit )
			) : esc_html__( 'Override global limit (no limit is set)', 'shopbuilder-pro' ),
		];
	}

	/**
	 * Adds back order meta-fields to the product settings.
	 *
	 * @return void
	 */
	public function add_simple_meta_box() {
		$product_id   = get_the_id();
		$product      = wc_get_product( $product_id );
		$status       = 'preorder' !== $product->get_stock_status() ? 'hidden' : '';
		$product_meta = PreOrderFns::get_pre_order_meta_data( $product_id );

		if ( empty( $product_meta['preorder_auto_convert'] ) ) {
			$product_meta['preorder_auto_convert'] = $this->data['global_auto_convert'];
		}

		if ( 'variable' === $product->get_type() ) {
			$status = 'hidden';
		}

		echo '<div id="rtsb_pre_order_options_group" class="options_group ' . esc_attr( $status ) . '">';
		Fns::print_html( $this->render_preorder_meta_boxes( $product_id, $status, $product_meta ), true );
		echo '</div>';
	}

	/**
	 * Add variation meta-box with preorder limit and availability date inputs.
	 *
	 * @param int    $loop           The loop index of the variation.
	 * @param array  $variation_data The variation data.
	 * @param object $variation      The variation object.
	 *
	 * @return void
	 */
	public function add_variation_meta_box( $loop, $variation_data, $variation ) {
		$preorders    = get_post_meta( $variation->ID, '_stock_status', true );
		$status       = 'preorder' !== $preorders ? 'hidden' : '';
		$product_meta = PreOrderFns::get_pre_order_meta_data( $variation->ID );

		if ( empty( $product_meta['preorder_auto_convert'] ) ) {
			$product_meta['preorder_auto_convert'] = $this->data['global_auto_convert'];
		}

		echo sprintf( '<div id="rtsb_pre_order_variation_options_group_%s" class="%s">', esc_attr( $loop ), esc_attr( $status ) );
		Fns::print_html( $this->render_preorder_meta_boxes( $variation->ID, $status, $product_meta, true, $loop ), true );
		echo '</div>';
	}

	/**
	 * Saves meta-box data for the specified post-ID.
	 *
	 * @param int $post_id The ID of the post being saved.
	 *
	 * @return void
	 */
	public function save_simple_meta_box( $post_id ) {
		// Check if the nonce is set and valid.
		if (
			! isset( $_REQUEST['_wpnonce'] ) &&
			! isset( $_REQUEST['post_ID'] ) &&
			! wp_verify_nonce( sanitize_text_field( wp_unslash( $_REQUEST['_wpnonce'] ) ), 'update-post_' . sanitize_text_field( wp_unslash( $_REQUEST['post_ID'] ) ) ) ) {
			return;
		}

		$meta_mapping = PreOrderFns::product_meta_data_mapping();

		foreach ( $meta_mapping as $meta_key => $custom_field_id ) {
			$sanitized_value = ! empty( $_REQUEST[ $custom_field_id ] ) ? sanitize_text_field( wp_unslash( $_REQUEST[ $custom_field_id ] ) ) : null;

			if ( 'preorder_limit' === $meta_key && ( ! empty( $sanitized_value ) ) ) {
				$sanitized_value = strlen( trim( $sanitized_value ) ) <= 0 ? '' : intval( $sanitized_value );
			}

			if ( 'preorder_auto_convert' === $meta_key ) {
				$sanitized_value = ! empty( $sanitized_value ) ? $sanitized_value : 'no';
			}

			// Update post meta.
			update_post_meta( $post_id, $custom_field_id, $sanitized_value );
		}
	}

	/**
	 * Save variation meta-box data.
	 *
	 * @param int $variation_id The ID of the variation being saved.
	 * @param int $loop The loop index of the variation.
	 *
	 * @return void
	 */
	public function save_variation_meta_box( $variation_id, $loop ) {
		// Check security.
		check_ajax_referer( 'save-variations', 'security' );

		$meta_mapping = PreOrderFns::product_meta_data_mapping();

		foreach ( $meta_mapping as $meta_key => $custom_field_id ) {
			$sanitized_value = ! empty( $_REQUEST[ $custom_field_id ][ $loop ] ) ? sanitize_text_field( wp_unslash( $_REQUEST[ $custom_field_id ][ $loop ] ) ) : null;

			if ( 'preorder_limit' === $meta_key && ( ! empty( $sanitized_value ) ) ) {
				$sanitized_value = strlen( trim( $sanitized_value ) ) <= 0 ? '' : intval( $sanitized_value );
			}

			if ( 'preorder_auto_convert' === $meta_key ) {
				$sanitized_value = ! empty( $sanitized_value ) ? $sanitized_value : 'no';
			}

			// Update post meta.
			update_post_meta( $variation_id, $custom_field_id, $sanitized_value );
		}
	}

	/**
	 * Renders the pre-order meta-boxes for a product.
	 *
	 * @param int    $product_id   The ID of the product.
	 * @param string $classes      Additional CSS classes to add to the meta-box wrapper.
	 * @param array  $product_meta An associative array of product metadata for pre-order settings.
	 * @param bool   $variation    Whether the product is a variation.
	 * @param int    $loop         The loop index for variations.
	 *
	 * @return string
	 */
	private function render_preorder_meta_boxes( $product_id, $classes = '', $product_meta = [], $variation = false, $loop = null ) {
		$html = '<div class="rtsb-group-heading"><h2>' . esc_html__( 'Pre-Order Settings', 'shopbuilder-pro' ) . '</h2></div>';

		ob_start();

		// Limit.
		woocommerce_wp_text_input(
			[
				'id'                => '_rtsb_max_preorder_allowed' . ( $variation ? $loop : '' ),
				'name'              => '_rtsb_max_preorder_allowed' . ( $variation ? "[$loop]" : '' ),
				'label'             => esc_html__( 'Pre-Order Limit', 'shopbuilder-pro' ),
				'desc_tip'          => true,
				'description'       => esc_html__( 'Set the maximum number of items a customer can place on pre-order. Leave empty to apply global limit. Enter -1 for no limit.', 'shopbuilder-pro' ),
				'placeholder'       => $this->data['limit_placeholder'],
				'type'              => 'number',
				'value'             => $product_meta['preorder_limit'],
				'wrapper_class'     => $variation ? 'form-row form-row-first' : '',
				'custom_attributes' => [
					'step' => 'any',
				],
			]
		);

		// Date.
		AdminInputsFns::generate_date_picker_input(
			$product_id,
			[
				'id'            => '_rtsb_preorder_avail_date' . ( $variation ? $loop : '' ),
				'name'          => '_rtsb_preorder_avail_date' . ( $variation ? "[$loop]" : '' ),
				'label'         => esc_html__( 'Availability Date', 'shopbuilder-pro' ),
				'desc_tip'      => true,
				'value'         => $product_meta['preorder_availability'],
				'description'   => esc_html__( 'Enter the estimated date when pre-ordered items will be available for shipment.', 'shopbuilder-pro' ),
				'wrapper_class' => $variation ? 'form-row form-row-last' : '',
			],
			true
		);

		// Pricing Options.
		woocommerce_wp_select(
			[
				'id'            => '_rtsb_preorder_pricing_options' . ( $variation ? $loop : '' ),
				'name'          => '_rtsb_preorder_pricing_options' . ( $variation ? "[$loop]" : '' ),
				'label'         => esc_html__( 'Pricing Options', 'shopbuilder-pro' ),
				'options'       => [
					'regular'             => esc_html__( 'Regular Product Price', 'shopbuilder-pro' ),
					'fixed'               => esc_html__( 'Fixed Pre-Order Price', 'shopbuilder-pro' ),
					'discount_percentage' => esc_html__( 'Discount by Percentage (%) of Regular Price', 'shopbuilder-pro' ),
					'discount_fixed'      => esc_html__( 'Discount by Fixed Amount of Regular Price', 'shopbuilder-pro' ),
					'increase_percentage' => esc_html__( 'Increase by Percentage (%) of Regular Price', 'shopbuilder-pro' ),
					'increase_fixed'      => esc_html__( 'Increase by Fixed Amount of Regular Price', 'shopbuilder-pro' ),
				],
				'value'         => $product_meta['preorder_price_option'],
				'desc_tip'      => 'true',
				'description'   => esc_html__( 'Choose how the pre-order price is calculated relative to the regular selling price.', 'shopbuilder-pro' ),
				'wrapper_class' => $variation ? 'form-row form-row-first' : '',
			]
		);

		// Pre-Order Price.
		woocommerce_wp_text_input(
			[
				'id'                => '_rtsb_preorder_price' . ( $variation ? $loop : '' ),
				'name'              => '_rtsb_preorder_price' . ( $variation ? "[$loop]" : '' ),
				'label'             => esc_html__( 'Pre-Order Price', 'shopbuilder-pro' ),
				'desc_tip'          => true,
				'description'       => esc_html__( 'Enter the pre-order price for this product.', 'shopbuilder-pro' ),
				'type'              => 'number',
				'value'             => $product_meta['preorder_price'],
				'wrapper_class'     => $variation ? 'hidden form-row form-row-last' : 'hidden',
				'custom_attributes' => [
					'step' => 'any',
					'min'  => '0',
				],
			]
		);

		// % Discount.
		woocommerce_wp_text_input(
			[
				'id'                => '_rtsb_preorder_percentage_discount' . ( $variation ? $loop : '' ),
				'name'              => '_rtsb_preorder_percentage_discount' . ( $variation ? "[$loop]" : '' ),
				'label'             => esc_html__( 'Discount % By', 'shopbuilder-pro' ),
				'desc_tip'          => true,
				'description'       => esc_html__( 'Enter the pre-order discount percentage for this product.', 'shopbuilder-pro' ),
				'type'              => 'number',
				'value'             => $product_meta['preorder_percent_discount'],
				'wrapper_class'     => $variation ? 'hidden form-row form-row-last' : 'hidden',
				'custom_attributes' => [
					'step' => 'any',
					'min'  => '0',
				],
			]
		);

		// Fixed Discount.
		woocommerce_wp_text_input(
			[
				'id'                => '_rtsb_preorder_fixed_discount' . ( $variation ? $loop : '' ),
				'name'              => '_rtsb_preorder_fixed_discount' . ( $variation ? "[$loop]" : '' ),
				'label'             => esc_html__( 'Discount Amount By', 'shopbuilder-pro' ),
				'desc_tip'          => true,
				'description'       => esc_html__( 'Enter the pre-order discount amount for this product.', 'shopbuilder-pro' ),
				'type'              => 'number',
				'value'             => $product_meta['preorder_fixed_discount'],
				'wrapper_class'     => $variation ? 'hidden form-row form-row-last' : 'hidden',
				'custom_attributes' => [
					'step' => 'any',
					'min'  => '0',
				],
			]
		);

		// % Increase.
		woocommerce_wp_text_input(
			[
				'id'                => '_rtsb_preorder_percentage_increase' . ( $variation ? $loop : '' ),
				'name'              => '_rtsb_preorder_percentage_increase' . ( $variation ? "[$loop]" : '' ),
				'label'             => esc_html__( 'Increase % By', 'shopbuilder-pro' ),
				'desc_tip'          => true,
				'description'       => esc_html__( 'Enter the pre-order increase percentage for this product.', 'shopbuilder-pro' ),
				'type'              => 'number',
				'value'             => $product_meta['preorder_percent_increase'],
				'wrapper_class'     => $variation ? 'hidden form-row form-row-last' : 'hidden',
				'custom_attributes' => [
					'step' => 'any',
					'min'  => '0',
				],
			]
		);

		// Fixed Increase.
		woocommerce_wp_text_input(
			[
				'id'                => '_rtsb_preorder_fixed_increase' . ( $variation ? $loop : '' ),
				'name'              => '_rtsb_preorder_fixed_increase' . ( $variation ? "[$loop]" : '' ),
				'label'             => esc_html__( 'Increase Amount By', 'shopbuilder-pro' ),
				'desc_tip'          => true,
				'description'       => esc_html__( 'Enter the pre-order increase amount for this product.', 'shopbuilder-pro' ),
				'type'              => 'number',
				'value'             => $product_meta['preorder_fixed_increase'],
				'wrapper_class'     => $variation ? 'hidden form-row form-row-last' : 'hidden',
				'custom_attributes' => [
					'step' => 'any',
					'min'  => '0',
				],
			]
		);

		// Auto Convert Checkbox.
		echo '<div class="form-row rtsb-preorder-auto-convert options_group form-row-first">';
		woocommerce_wp_checkbox(
			[
				'id'            => '_rtsb_preorder_auto_convert' . ( $variation ? $loop : '' ),
				'name'          => '_rtsb_preorder_auto_convert' . ( $variation ? "[$loop]" : '' ),
				'label'         => esc_html__( 'Auto Convert Product', 'shopbuilder-pro' ),
				'value'         => $product_meta['preorder_auto_convert'],
				'description'   => esc_html__( 'Enable', 'shopbuilder-pro' ),
				'wrapper_class' => $classes,
			]
		);
		Fns::print_html( wc_help_tip( __( 'Enable this option to automatically convert this product to standard product once the available date is reached.', 'shopbuilder-pro' ) ) );
		echo '</div>';

		$html .= ob_get_clean();

		return $html;
	}

	/**
	 * Registers a custom column to display product status in the admin product list.
	 *
	 * @return void
	 */
	private function admin_column() {
		add_action( 'manage_product_posts_custom_column', [ $this, 'manage_product_custom_column' ], 20, 2 );
	}

	/**
	 * Outputs a custom "Pre-Order" status for products in the admin product list.
	 *
	 * @param string $column column.
	 * @param int    $product_id product id.
	 *
	 * @return void
	 */
	public function manage_product_custom_column( $column, $product_id ) {
		if ( 'name' === $column ) {
			$product   = wc_get_product( $product_id );
			$pre_order = false;

			if ( $product ) {
				if ( $product->is_type( 'variable' ) ) {
					$variations = $product->get_children();

					foreach ( $variations as $variation_id ) {
						$variation = wc_get_product( $variation_id );

						if ( 'preorder' === $variation->get_stock_status() ) {
							$pre_order = true;
							break;
						}
					}
				} else {
					$pre_order = 'preorder' === $product->get_stock_status();
				}

				if ( $pre_order ) {
					echo '<span class="rtsb-pre-order-text-badge admin-column">' . esc_html__( 'Pre-Order', 'shopbuilder-pro' ) . '</span>';
				}
			}
		}
	}
}
