<?php
/**
 * Pre-order Module Common Class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO\Modules\PreOrder;

use WP_Query;
use DateTime;
use WC_Order;
use Exception;
use DateTimeZone;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\PreOrder\Notification\PreOrderCancelEmail;
use RadiusTheme\SBPRO\Modules\PreOrder\Notification\PreOrderReleaseEmail;
use RadiusTheme\SBPRO\Modules\PreOrder\Notification\PreOrderReceivedEmail;
use RadiusTheme\SBPRO\Modules\PreOrder\Notification\OrderConfirmationEmail;

defined( 'ABSPATH' ) || exit();

/**
 * Pre-order Module Common Class.
 */
class PreOrderCommon {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Module Class Constructor.
	 */
	private function __construct() {
		$this->add_stock_status();
		$this->add_notification_emails();
		$this->add_notification_schedule();
	}

	/**
	 * Add stock status option.
	 *
	 * @return void
	 */
	public function add_stock_status() {
		add_filter( 'woocommerce_product_stock_status_options', [ $this, 'add_preorder_stock_status' ] );
	}

	/**
	 * Add Pre-Order stock status option.
	 *
	 * @param array $status_options The array of existing stock status options.
	 *
	 * @return array
	 */
	public function add_preorder_stock_status( $status_options ) {
		$status_options['preorder'] = esc_html__( 'On Pre-Order', 'shopbuilder-pro' );

		return $status_options;
	}

	/**
	 * Add Pre-Order emails.
	 *
	 * @return void
	 */
	public function add_notification_emails() {
		add_filter( 'woocommerce_email_classes', [ $this, 'register_pre_order_emails' ] );
		add_filter( 'woocommerce_locate_core_template', [ $this, 'locate_templates' ], 10, 2 );
		add_action( 'woocommerce_order_status_rtsb-preordered_to_cancelled', [ $this, 'pre_order_cancel_action' ] );

		// Disable default emails.
		$settings = PreOrderFns::get_pre_order_settings_data();

		if ( 'on' === $settings['new_email'] ) {
			add_filter( 'woocommerce_email_enabled_new_order', [ $this, 'disable_default_emails' ], 10, 2 );
		}

		if ( 'on' === $settings['confirm_email'] ) {
			add_filter( 'woocommerce_email_enabled_customer_processing_order', [ $this, 'disable_default_emails' ], 10, 2 );
		}
	}

	/**
	 * Add Pre-Order availability check scheduling.
	 *
	 * @return void
	 */
	public function add_notification_schedule() {
		/**
		 * Pre-Order availability check scheduling.
		 */
		register_activation_hook( RTSBPRO_FILE, [ $this, 'start_availability_check_schedule' ] );
		register_deactivation_hook( RTSBPRO_FILE, [ $this, 'end_availability_check_schedule' ] );

		// Manually start the schedule.
		$this->start_availability_check_schedule();

		/**
		 * Pre-Order availability check.
		 */
		add_action( 'rtsb/module/pre_order/availability_check', [ $this, 'availability_check' ] );
	}

	/**
	 * Schedule the availability check.
	 *
	 * @return void
	 */
	public function start_availability_check_schedule() {
		if ( ! wp_next_scheduled( 'rtsb/module/pre_order/availability_check' ) ) {
			wp_schedule_event( time(), 'twicedaily', 'rtsb/module/pre_order/availability_check' );
		}
	}

	/**
	 * Clear the scheduled availability check.
	 *
	 * @return void
	 */
	public function end_availability_check_schedule() {
		wp_clear_scheduled_hook( 'rtsb/module/pre_order/availability_check' );
	}

	/**
	 * Check pre-order availability.
	 *
	 * @return void
	 * @throws Exception If there is an error.
	 */
	public function availability_check() {
		$timezone_string = get_option( 'timezone_string' );

		if ( empty( $timezone_string ) ) {
			$gmt_offset = get_option( 'gmt_offset' );
			$gmt_offset = intval( $gmt_offset );
			$timezone   = new DateTimeZone( sprintf( 'Etc/GMT%+d', -$gmt_offset ) );
		} else {
			$timezone = new DateTimeZone( $timezone_string );
		}

		$current_time         = new DateTime( 'now', $timezone );
		$twelve_hours_ago     = ( clone $current_time )->modify( '-24 hours' );
		$current_time_str     = $current_time->format( 'Y-m-d\TH:i' );
		$twelve_hours_ago_str = $twelve_hours_ago->format( 'Y-m-d\TH:i' );

		$args = [
			'post_type'      => [ 'product', 'product_variation' ],
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			'meta_query'     => [
				'relation' => 'AND',
				[
					'key'     => '_rtsb_preorder_avail_date',
					'value'   => [ $twelve_hours_ago_str, $current_time_str ],
					'compare' => 'BETWEEN',
					'type'    => 'DATETIME',
				],
				[
					'key'     => '_rtsb_preorder_availability_notified',
					'compare' => 'NOT EXISTS',
				],
			],
			'posts_per_page' => -1,
		];

		$products = new WP_Query( $args );

		if ( $products->have_posts() ) {
			while ( $products->have_posts() ) {
				$products->the_post();

				$this->maybe_trigger_release_email( get_the_ID() );
			}
		}

		wp_reset_postdata();
	}

	/**
	 * Send Pre-Order release emails to customers.
	 *
	 * @param int $product_id The ID of the product being released.
	 *
	 * @return void
	 */
	private function maybe_trigger_release_email( $product_id ) {
		$product    = wc_get_product( $product_id );
		$product_id = 'variable' === $product->get_type() ? $product->get_parent_id() : $product_id;
		$settings   = PreOrderFns::get_pre_order_settings_data();

		$orders = wc_get_orders(
			[
				'limit'      => -1,
				'status'     => [ 'wc-rtsb-preordered', 'processing', 'on-hold' ],
				// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
				'meta_query' => [
					[
						'key'     => '_rtsb_pre_order_product_id',
						'value'   => $product_id,
						'compare' => '=',
					],
				],
			]
		);

		foreach ( $orders as $order ) {
			WC()->mailer();

			if ( 'on' === $settings['release_email'] ) {
				do_action( 'rtsb/module/pre_order/release_email', $order );
			}
		}
	}

	/**
	 * Registers the custom pre-order email classes with WooCommerce.
	 *
	 * @param array $email_classes An array of existing WooCommerce email classes.
	 *
	 * @return array
	 */
	public function register_pre_order_emails( $email_classes ) {
		$settings = PreOrderFns::get_pre_order_settings_data();

		if ( 'on' === $settings['new_email'] ) {
			$email_classes['RTSB_Email_New_Pre_Order_Received'] = new PreOrderReceivedEmail();
		}

		if ( 'on' === $settings['confirm_email'] ) {
			$email_classes['RTSB_Email_Pre_Order_Confirmed'] = new OrderConfirmationEmail();
		}

		if ( 'on' === $settings['release_email'] ) {
			$email_classes['RTSB_Email_Pre_Order_Released'] = new PreOrderReleaseEmail();
		}

		if ( 'on' === $settings['cancel_email'] ) {
			$email_classes['RTSB_Email_Pre_Order_Cancelled'] = new PreOrderCancelEmail();
		}

		return $email_classes;
	}

	/**
	 * Locates custom email templates.
	 *
	 * @param string $core_file The path to the core template file.
	 * @param string $template The requested template file.
	 *
	 * @return string
	 */
	public function locate_templates( $core_file, $template ) {
		$custom_templates = [
			'emails/pre-order-received.php',
			'emails/pre-order-confirmed.php',
			'emails/pre-order-available.php',
			'emails/pre-order-cancel.php',
		];

		if ( in_array( $template, $custom_templates, true ) ) {
			$core_file = rtsbpro()->get_plugin_template_path() . $template;
		}

		return $core_file;
	}

	/**
	 * Perform actions when a pre-order is canceled.
	 *
	 * @param int $order_id The ID of the canceled order.
	 *
	 * @return void
	 */
	public function pre_order_cancel_action( $order_id ) {
		$settings = PreOrderFns::get_pre_order_settings_data();
		WC()->mailer();

		if ( 'on' === $settings['cancel_email'] ) {
			do_action( 'rtsb/module/pre_order/cancel_email', $order_id );
		}
	}

	/**
	 * Disables default emails for orders containing pre-order products.
	 *
	 * @param bool     $enabled Whether the email is enabled. Default is true.
	 * @param WC_Order $order   The WooCommerce order object.
	 *
	 * @return bool
	 */
	public function disable_default_emails( $enabled, $order ) {
		if ( ! $order || ! PreOrderFns::order_contains_pre_order_products( $order ) ) {
			return $enabled;
		}

		if ( true === apply_filters( 'rtsb/pre_order/enabled_default_emails', false, $order ) ) {
			return $enabled;
		}

		return false;
	}
}
