<?php
/**
 * Main FilterHooks class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\ProductSizeChart;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use WC_Product;

defined( 'ABSPATH' ) || exit();

/**
 * Main FilterHooks class.
 */
class SizeChartFrontEnd {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * @var array|mixed
	 */
	private array $options;

	/**
	 * Asset Handle
	 *
	 * @var string
	 */
	private $handle = 'rtsb-product-size-chart';

	/**
	 * Class constructor.
	 */
	private function __construct() {
		$this->options = Fns::get_options( 'modules', 'product_size_chart' );

		add_action( 'rtsb/modules/product_size_chart/frontend/display', [ $this, 'frontend_display' ] );
		// Add do_action( 'rtsb/modules/product_size_chart/frontend/display' ); for display anywhere inside product query.
		add_action( 'rtsb/modules/product_size_chart/print_size_chart', [ $this, 'print_size_chart' ] );
		// ShortCode.
		add_shortcode( 'rtsb_product_size_chart', [ $this, 'product_size_chart_shortcode' ] );

		// Template.
		add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_public_scripts' ], 99 );

		$this->attach_button();
	}

	/**
	 * @return void
	 */
	public function enqueue_public_scripts() {
		// Enqueue assets.
		$this->handle = Fns::enqueue_module_assets(
			$this->handle,
			'product-size-chart',
			[
				'context' => rtsbpro(),
				'version' => RTSBPRO_VERSION,
			]
		);

		if ( ! apply_filters( 'rtsb/module/product_size_chart/show_button', true ) ) {
			return;
		}

		$pre         = '.rtsb-size-chart-wrapper ';
		$dynamic_css = '';

		$dynamic_css .= $pre . ' .rtsb-size-chart-button {';
		$dynamic_css .= 'justify-content: center;align-items: center;';
		$dynamic_css .= '}';

		if ( ! empty( $this->options['size_chart_wrapper_padding'] ) ) {
			$dynamic_css .= $pre . '.rtsb-size-chart-button,' . $pre . '.rtsb-size-chart-image{';
			$dynamic_css .= 'padding:' . $this->options['size_chart_wrapper_padding'] . ';';
			$dynamic_css .= '}';
		}

		if ( ! empty( $this->options['size_chart_wrapper_margin'] ) ) {
			$dynamic_css .= $pre . '.rtsb-size-chart-button,' . $pre . '.rtsb-size-chart-image{';
			$dynamic_css .= 'margin:' . $this->options['size_chart_wrapper_margin'] . ';';
			$dynamic_css .= '}';
		}

		$dynamic_css_property = 'display: inline-flex;margin: 0;align-items: center;text-decoration: none; transition: 0.3s all;';
		if ( ! empty( $this->options['size_chart_button_height'] ) && absint( $this->options['size_chart_button_height'] ) ) {
			$dynamic_css_property .= 'height:' . absint( $this->options['size_chart_button_height'] ) . 'px;';
		}

		if ( ! empty( $this->options['size_chart_button_background_color'] ) ) {
			$dynamic_css_property .= 'background-color:' . $this->options['size_chart_button_background_color'] . ';';
		}
		if ( ! empty( $this->options['size_chart_button_text_color'] ) ) {
			$dynamic_css_property .= 'color:' . $this->options['size_chart_button_text_color'] . ';';
		}
		if ( ! empty( $dynamic_css_property ) ) {
			$dynamic_css .= $pre . ' .rtsb-size-chart-popups-view{';
			$dynamic_css .= $dynamic_css_property;
			$dynamic_css .= '}';
		}

		$dynamic_css_hover_property = '';

		if ( ! empty( $this->options['size_chart_button_background_hover_color'] ) ) {
			$dynamic_css_hover_property .= 'background-color:' . $this->options['size_chart_button_background_hover_color'] . ';';
		}
		if ( ! empty( $this->options['size_chart_button_text_hover_color'] ) ) {
			$dynamic_css_hover_property .= 'color:' . $this->options['size_chart_button_text_hover_color'];
		}

		if ( ! empty( $dynamic_css_hover_property ) ) {
			$dynamic_css .= $pre . ' .rtsb-size-chart-popups-view:hover{';
			$dynamic_css .= $dynamic_css_hover_property;
			$dynamic_css .= '}';
		}

		if ( ! empty( $dynamic_css ) ) {
			wp_add_inline_style( $this->handle, $dynamic_css );
		}
	}

	/**
	 * Add the "Add to Wishlist" button. Needed to use in wp_head hook.
	 *
	 * @return void
	 * @since 1.0.0
	 */
	public function attach_button() {

		$product_hook_priority = $this->options['product_hook_priority'] ?? null;
		$product_hook_priority = is_int( $product_hook_priority ) ? $product_hook_priority : null;

		$positions = apply_filters(
			'rtsb/module/product_size_chart/button_positions',
			[
				'before_add_to_cart' => [
					'hooks'    => [
						'woocommerce_before_add_to_cart_button',
					],
					'priority' => $product_hook_priority ?? 20,
				],
				// astra_woo_single_add_to_cart_before.
				'after_add_to_cart'  => [
					'hooks'    => [
						'woocommerce_single_product_summary',
					],
					'priority' => $product_hook_priority ?? 31,
				],
				'after_thumbnail'    => [
					'hooks'    => [
						'woocommerce_before_single_product_summary',
					],
					'priority' => $product_hook_priority ?? 21,
				],
				'after_summary'      => [
					'hooks'    => [
						'woocommerce_after_single_product_summary',
					],
					'priority' => $product_hook_priority ?? 11,
				],
				'after_short_desc'   => [
					'hooks'    => [
						'woocommerce_single_product_summary',
					],
					'priority' => $product_hook_priority ?? 21,
				],
				'custom'             => [
					'hooks'    => explode( ',', trim( $this->options['product_custom_hook_name'] ?? '' ) ),
					'priority' => $product_hook_priority,
				],
			]
		);

		// Add the link "Add to wishlist".

		$product_button_position = $this->options['product_size_chart_position'] ?? 'after_add_to_cart';

		if ( 'shortcode' !== $product_button_position && isset( $positions[ $product_button_position ]['hooks'] ) ) {
			foreach ( $positions[ $product_button_position ]['hooks'] as $hook ) {
				add_action( trim( $hook ), [ $this, 'frontend_display' ], $positions[ $product_button_position ]['priority'] ?? '' );
			}
		}
	}

	/**
	 * Print "Add to compare" button
	 *
	 * @return void
	 */
	public function frontend_display() {

		if ( ! apply_filters( 'rtsb/module/product_size_chart/show_button', true ) ) {
			return;
		}

		global $product;

		if ( $product instanceof WC_Product ) {
			do_action( 'rtsb/modules/product_size_chart/print_size_chart', $product->get_id() );
		}
	}


	/**
	 * Wishlist button Shortcode callable function
	 *
	 * @return string [HTML]
	 */
	public function product_size_chart_shortcode() {
		global $product;
		if ( ! $product instanceof WC_Product ) {
			return '';
		}
		ob_start();

		do_action( 'rtsb/modules/product_size_chart/frontend/display' );

		return ob_get_clean();
	}

	/**
	 * Print size chart
	 *
	 * @param  int $product_id Product ID.
	 *
	 * @return void
	 */
	public function print_size_chart( $product_id = 0 ) {
		global $woocommerce_loop;

		global $product;

		// product object.
		$current_product = $product_id ? wc_get_product( $product_id ) : false;
		$current_product = $current_product ?: $product;

		if ( ! $current_product instanceof WC_Product ) {
			return;
		}

		$chart = SizeChartFns::get_size_chart_for_current_product();

		if ( is_array( $chart ) && ! count( $chart ) ) {
			return;
		}

		$classes = [ 'rtsb-size-chart' ];

		// phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
		if ( ! in_array( ( $woocommerce_loop['name'] ?? '' ), [ 'related', 'up-sells', 'cross-sells' ] ) ) {
			$classes[] = ! empty( $this->options['show_size_chart_product_page'] ) ? 'rtsb-size-chart-button-product' : '';
		}

		if ( 'popups' === ( $this->options['size_chart_button_action'] ?? 'general' ) ) {
			$classes[] = 'rtsb-size-chart-popups-view';
		} else {
			$classes[] = 'rtsb-size-chart-popups-general';
		}

		$classes       = apply_filters( 'rtsb/product_size_chart/parent/class', $classes, $current_product, $woocommerce_loop );
		$buttonText    = $this->options['size_chart_button_text'] ?? esc_html__( 'Size Chart', 'shopbuilder-pro' );
		$attachment_id = absint( $chart['size_chart_image']['id'] ?? 0 );

		if ( ! $attachment_id ) {
			return;
		}
		$image_attr = wp_get_attachment_image_src( $attachment_id, 'full' );

		do_action( 'rtdb/before/product_size_chart/button' );

		$image = '<img src="' . esc_url( $image_attr[0] ?? '#' ) . '" alt="' . esc_attr( get_post_meta( $attachment_id, '_wp_attachment_image_alt', true ) ) . '" />';
		?>
		<div class="rtsb-size-chart-wrapper">

			<?php
			$button_action           = $this->options['size_chart_button_action'] ?? 'general';
			$product_button_position = $this->options['product_size_chart_position'] ?? '';
			if ( 'product_tab' === $product_button_position ) {
				$button_action = 'general';
			}

			if ( 'popups' === $button_action ) {
				// phpcs:ignore WordPress.PHP.StrictInArray.MissingTrueStrict
				if ( in_array( rtsb()->current_theme, [ 'storefront' ] ) ) {
					$classes[] = 'button';
				}
				$data_image_for_modal = '<div class="rtsb-size-chart">' . $image . '</div>';
				?>
				<button class="rtsb-size-chart-button <?php echo esc_attr( implode( ' ', $classes ) ); ?>" data-image="<?php echo esc_attr( $data_image_for_modal ); ?>" data-title="<?php echo esc_attr( '<h2>' . __( 'Size Chart', 'shopbuilder-pro' ) . '</h2>' ); ?>" >
					<?php echo esc_html( $buttonText ); ?>
				</button>
			<?php } else { ?>
				<div class="rtsb-size-chart-image <?php echo esc_attr( implode( ' ', $classes ) ); ?>" >
					<?php Fns::print_html( $image ); ?>
				</div>
			<?php } ?>
		</div>
		<!-- End Size Chart Wrapper -->
		<?php
		do_action( 'rtdb/after/product_size_chart/button' );
	}
}
