<?php
/**
 * Main FilterHooks class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\QuickCheckout;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Traits\SingletonTrait;

defined( 'ABSPATH' ) || exit();

/**
 * Main FilterHooks class.
 */
class Checkout {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * @var array|mixed
	 */
	private array $options;

	/**
	 *
	 */
	private function __construct() {
		$this->options = Fns::get_options( 'modules', 'quick_checkout' );

		CheckoutFrontEnd::instance();
		add_action( 'template_redirect', [ $this, 'handle_quick_checkout_button' ] );
		add_action( 'wp_ajax_checkout_page_as_modal', [ $this, 'checkout_page_as_modal' ] );
		add_action( 'wp_ajax_nopriv_checkout_page_as_modal', [ $this, 'checkout_page_as_modal' ] );
	}

	/**
	 * Add to Cart via AJAX
	 *
	 * @return void
	 * @throws \Exception
	 */
	public function checkout_page_as_modal() {

		if ( ! Fns::verify_nonce() || ! apply_filters( 'rtsb/module/quick_checkout/show_button', true ) ) {
			wp_send_json_error( esc_html__( 'Invalid request', 'shopbuilder-pro' ) );

			return;
		}
		if ( 'popups' !== ( $this->options['quick_checkout_button_action'] ?? 'redirect' ) ) {
			wp_send_json_error( esc_html__( 'Invalid request', 'shopbuilder-pro' ) );

			return;
		}

		$product_id   = absint( $_POST['product_id'] ?? 0 );
		$variation_id = absint( $_POST['variation_id'] ?? 0 );
		$quantity     = absint( $_POST['quantity'] ?? 1 );
		$attributes   = isset( $_POST['variation'] ) ? $_POST['variation'] : '{}';
		if ( ! $product_id ) {
			// Return a response.
			wp_send_json(
				[
					'html' => esc_html__( 'No Product Found', 'shopbuilder-pro' ),
				]
			);
		}
		$product = wc_get_product( $product_id );
		if ( $product->is_type( 'variable' ) && ! $variation_id ) {
			wp_send_json(
				[
					'html' => esc_html__( 'No Variation Selected', 'shopbuilder-pro' ),
				]
			);
		}

		if ( ! empty( $this->options['quick_checkout_clear_cart'] ) && ! WC()->cart->is_empty() ) {
			// Clear the cart.
			WC()->cart->empty_cart();
		}

		$variation = json_decode( stripslashes( $attributes ), true );
		// Add the product to the cart.
		WC()->cart->add_to_cart( $product_id, $quantity, $variation_id, $variation );

		ob_start();
        // Get checkout object.
        $checkout = WC()->checkout();

        ?>
		<div class="rtsb-notice rtsb-quick-checkout-coupon-form rtsb-checkout-coupon-form">
			<?php woocommerce_checkout_coupon_form(); ?>
		</div>
		<?php
		// If checkout registration is disabled and not logged in, the user cannot check out.
		if ( ! $checkout->is_registration_enabled() && $checkout->is_registration_required() && ! is_user_logged_in() ) {
			echo esc_html( apply_filters( 'woocommerce_checkout_must_be_logged_in_message', __( 'You must be logged in to checkout.', 'shopbuilder-pro' ) ) );
		} else {
			?>
			<form name="checkout" method="post" class="checkout woocommerce-checkout" action="<?php echo esc_url( wc_get_checkout_url() ); ?>" enctype="multipart/form-data">
				<?php if ( $checkout->get_checkout_fields() ) : ?>
					<div id="rtsb-customer_details">
						<div class="rtsb-col rtsb-col-1">
							<?php do_action( 'woocommerce_checkout_billing' ); ?>
							<?php do_action( 'woocommerce_checkout_shipping' ); ?>
						</div>
						<div class="rtsb-col rtsb-col-2">
							<h3 id="rtsb-order_review_heading"><?php esc_html_e( 'Your order', 'shopbuilder-pro' ); ?></h3>
							<div id="rtsb-order_review" class="woocommerce-checkout-review-order">
								<?php do_action( 'woocommerce_checkout_order_review' ); ?>
							</div>
						</div>
					</div>
				<?php endif; ?>
			</form>
			<?php
		}
		do_action( 'woocommerce_after_checkout_form', $checkout );
		$checkout_form = ob_get_clean();

		// Return a response
		wp_send_json(
			[
				'html' => $checkout_form,
			]
		);
	}


	/**
	 * @return void
	 */
	public function handle_quick_checkout_button() {
		if ( ! apply_filters( 'rtsb/module/quick_checkout/show_button', true ) ) {
			return;
		}
		if ( 'popups' === ( $this->options['quick_checkout_button_action'] ?? 'redirect' ) ) {
			return;
		}
		if ( ! boolval( $_GET['quick_checkout'] ?? false ) ) {
			return;
		}
		$product_id   = intval( $_GET['quick_checkout'] ?? 0 );
		$variation_id = intval( $_GET['variation_id'] ?? 0 );
		$quantity     = intval( $_GET['quantity'] ?? 0 );
		$attributes   = isset( $_GET['variation'] ) ? $_GET['variation'] : '{}';
		if ( ! empty( $this->options['quick_checkout_clear_cart'] ) && ! WC()->cart->is_empty() ) {
			// Clear the cart.
			WC()->cart->empty_cart();
		}
		$variation = json_decode( stripslashes( $attributes ), true );
		// Add the product to the cart.
		WC()->cart->add_to_cart( $product_id, $quantity, $variation_id, $variation );
		// Redirect to the checkout page.
		wp_safe_redirect( wc_get_checkout_url() );
		exit;
	}
}
