<?php
/**
 * Smart Coupons Module Admin Columns Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\SmartCoupons\Admin;

use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\SmartCoupons\SmartCouponsFns;

defined( 'ABSPATH' ) || exit();

/**
 * Smart Coupons Module Admin Columns Class.
 */
class AdminColumns {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Class Constructor.
	 */
	private function __construct() {
		/**
		 * Actions.
		 */
		add_action( 'manage_shop_coupon_posts_custom_column', [ $this, 'populate_coupon_columns' ], 20, 2 );
		add_action( 'admin_footer', [ $this, 'enqueue_coupon_admin_script' ], 20 );

		/**
		 * Filters
		 */
		add_filter( 'manage_shop_coupon_posts_columns', [ $this, 'add_coupon_columns' ], 20 );
		add_filter( 'post_row_actions', [ $this, 'add_custom_actions' ], 10, 2 );
	}

	/**
	 * Adds custom columns to the WooCommerce coupon list table in the admin area.
	 *
	 * @param array $columns The existing columns.
	 *
	 * @return array
	 */
	public function add_coupon_columns( $columns ) {
		$extra_columns   = array_flip( SmartCouponsFns::get_settings_data()['admin_columns'] );
		$default_columns = array_flip( array_keys( $columns ) );

		$new_columns = [
			'cb'                => $columns['cb'],
			'coupon_code'       => $columns['coupon_code'],
			'type'              => $columns['type'],
			'coupon_groups'     => esc_html__( 'Coupon Groups', 'shopbuilder-pro' ),
			'amount'            => $columns['amount'],
			'description'       => $columns['description'],
			'products'          => $columns['products'],
			'usage'             => $columns['usage'],
			'start_date'        => esc_html__( 'Valid from', 'shopbuilder-pro' ),
			'expiry_date'       => $columns['expiry_date'],
			'total_sales'       => esc_html__( 'Total Sales', 'shopbuilder-pro' ),
			'used_in_orders'    => esc_html__( 'Applied in Orders', 'shopbuilder-pro' ),
			'allowed_user_role' => esc_html__( 'Eligible User Roles', 'shopbuilder-pro' ),
			'allowed_locations' => esc_html__( 'Allowed Locations', 'shopbuilder-pro' ),
			'url_coupon'        => esc_html__( 'Shareable Link', 'shopbuilder-pro' ),
		];

		foreach ( array_keys( $new_columns ) as $column_key ) {
			if ( isset( $default_columns[ $column_key ] ) ) {
				continue;
			}

			if ( ! isset( $extra_columns[ $column_key ] ) ) {
				unset( $new_columns[ $column_key ] );
			}
		}

		return $new_columns;
	}

	/**
	 * Populates custom columns with data.
	 *
	 * @param string $column  The name of the column to display data for.
	 * @param int    $coupon_id The ID of the coupon post being processed.
	 *
	 * @return void
	 */
	public function populate_coupon_columns( $column, $coupon_id ) {
		switch ( $column ) {
			case 'used_in_orders':
				$order_count = $this->get_coupon_used_orders( $coupon_id );

				Fns::print_html( $order_count );
				break;

			case 'coupon_groups':
				$groups = $this->get_coupon_groups( $coupon_id );

				Fns::print_html( $groups );
				break;

			case 'allowed_user_role':
				$roles = $this->get_allowed_user_roles( $coupon_id );

				echo esc_html( $roles );
				break;

			case 'start_date':
				$start_date = $this->get_coupon_start_date( $coupon_id );

				echo esc_html( $start_date );
				break;

			case 'allowed_locations':
				$locations = $this->get_allowed_locations( $coupon_id );

				echo esc_html( $locations );
				break;

			case 'total_sales':
				$total_sales = $this->get_total_sales_by_coupon( $coupon_id );

				Fns::print_html( $total_sales );
				break;

			case 'url_coupon':
				$url_button = $this->get_url_coupon( $coupon_id );

				Fns::print_html( $url_button, true );
				break;
		}
	}

	/**
	 * Add custom actions to the coupon post-row actions.
	 *
	 * @param array  $actions An array of existing row action links.
	 * @param object $post    The current post-object.
	 *
	 * @return array
	 */
	public function add_custom_actions( $actions, $post ) {
		if ( 'shop_coupon' !== $post->post_type ) {
			return $actions;
		}

		// phpcs:ignore WordPress.WP.Capabilities.Unknown
		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			return $actions;
		}

		$coupon_code     = esc_js( $post->post_title );
		$copy_button     = '<a href="#" class="rtsb-copy-coupon-code" aria-label="Copy coupon code \'' . $coupon_code . '\' to the clipboard" data-code="' . $coupon_code . '">' . esc_html__( 'Copy to Clipboard', 'shopbuilder-pro' ) . '</a>';
		$actions['copy'] = $copy_button;

		return $actions;
	}

	/**
	 * Retrieve the orders in which a specific coupon was used.
	 *
	 * @param int $coupon_id The ID of the coupon.
	 *
	 * @return string
	 */
	private function get_coupon_used_orders( $coupon_id ) {
		global $wpdb;

		// Query to get the order IDs where the coupon was used.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$order_ids = $wpdb->get_col(
			$wpdb->prepare(
				"
                    SELECT DISTINCT order_id 
                    FROM {$wpdb->prefix}woocommerce_order_items 
                    WHERE order_item_type = 'coupon' AND order_item_name = %s",
				get_the_title( $coupon_id )
			)
		);

		$total_orders = count( $order_ids );

		if ( $total_orders > 0 ) {
			$coupon_code     = rawurlencode( get_the_title( $coupon_id ) );
			$view_orders_url = admin_url( "admin.php?page=wc-orders&s=$coupon_code&status=all" );
			$btn_text        = $total_orders > 1 ? __( 'View Orders', 'shopbuilder-pro' ) : __( 'View Order', 'shopbuilder-pro' );
			$icon            = '<span class="icon" style="margin-left: 8px;"><svg height="12" width="12" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" xml:space="preserve" fill="currentColor"><g><style>.st0{fill:currentColor}</style><path class="st0" d="M96 0v416h416V0H96zm376 376H136V40h336v336z"/><path class="st0" d="M40 472V96H0v416h416v-40h-40z"/><path class="st0" d="m232.812 312.829 117.859-117.86v84.797h40V126.688H237.594v40h84.796L204.531 284.547z"/></g></svg></span>';

			return sprintf(
				'<a href="%s" target="_blank">%s</a>',
				esc_url( $view_orders_url ),
				esc_html( $btn_text ) . $icon
			);
		}

		return '–';
	}

	/**
	 * Calculate the total sales amount associated with a specific coupon.
	 *
	 * @param int $coupon_id The ID of the coupon.
	 *
	 * @return string
	 */
	private function get_total_sales_by_coupon( $coupon_id ) {
		global $wpdb;

		// Query to sum up all order totals where the coupon was used.
		// phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching
		$total_sales = $wpdb->get_col(
			$wpdb->prepare(
				"
                    SELECT SUM({$wpdb->prefix}wc_order_product_lookup.product_net_revenue)
                    FROM {$wpdb->prefix}wc_order_coupon_lookup
                    INNER JOIN {$wpdb->prefix}wc_order_product_lookup
                    ON {$wpdb->prefix}wc_order_coupon_lookup.order_id = {$wpdb->prefix}wc_order_product_lookup.order_id
                    WHERE {$wpdb->prefix}wc_order_coupon_lookup.coupon_id = %d",
				$coupon_id
			)
		)[0] ?? 0;

		return $total_sales ? wc_price( $total_sales ) : '–';
	}

	/**
	 * Retrieve the shareable link for a specific coupon.
	 *
	 * @param int $coupon_id The ID of the coupon.
	 *
	 * @return string
	 */
	private function get_url_coupon( $coupon_id ) {
		$coupon_code = get_the_title( $coupon_id );
		$html        = '';

		if ( ! empty( $coupon_code ) ) {
			$store_url     = wc_get_page_permalink( 'shop' );
			$shareable_url = add_query_arg( 'rtsb_apply_coupon', $coupon_code, $store_url );

			// Display the shareable link with a copy button.
			$html .= '<div class="rtsb-shareable-coupon-link" style="display: flex; align-items: center;">';
			$html .= '<input type="hidden" readonly value="' . esc_url( $shareable_url ) . '" class="shareable-coupon-link" />';
			$html .= '<button type="button" class="button rtsb-copy-coupon-link" aria-label="Copy coupon link to the clipboard"><span class="dashicons dashicons-admin-page"></span> ' . esc_html__( 'Copy Link', 'shopbuilder-pro' ) . '</button>';
			$html .= '</div>';
		} else {
			$html .= '–';
		}

		return $html;
	}

	/**
	 * Retrieve and format the coupon groups associated with a given coupon.
	 *
	 * @param int $coupon_id The ID of the coupon.
	 *
	 * @return string
	 */
	private function get_coupon_groups( $coupon_id ) {
		$terms = get_the_terms( $coupon_id, 'rtsb_coupon_group' );

		if ( ! $terms || is_wp_error( $terms ) ) {
			return '–';
		}

		$links = [];

		foreach ( $terms as $term ) {
			$link    = sprintf(
				'<a href="%s">%s</a>',
				esc_url( admin_url( 'edit.php?rtsb_coupon_group=' . $term->slug . '&post_type=shop_coupon' ) ),
				esc_html( $term->name )
			);
			$links[] = $link;
		}

		return implode( ', ', $links );
	}

	/**
	 * Retrieve and format the allowed user roles for a given coupon.
	 *
	 * @param int $coupon_id The ID of the coupon.
	 *
	 * @return string
	 */
	private function get_allowed_user_roles( $coupon_id ) {
		$role_keys = get_post_meta( $coupon_id, 'rtsb_user_role_based_coupon', true );
		$role_keys = is_array( $role_keys ) ? $role_keys : [];

		$all_roles  = FnsPro::get_all_user_roles();
		$role_names = array_map(
			function ( $role_key ) use ( $all_roles ) {
				return $all_roles[ $role_key ] ?? $role_key;
			},
			$role_keys
		);

		return ! empty( $role_names ) ? implode( ', ', $role_names ) : esc_html__( 'No restrictions', 'shopbuilder-pro' );
	}

	/**
	 * Retrieve the start date for a given coupon and format it.
	 *
	 * @param int $coupon_id The ID of the coupon.
	 *
	 * @return string
	 */
	private function get_coupon_start_date( $coupon_id ) {
		$start_date = get_post_meta( $coupon_id, 'rtsb_coupon_start', true );

		return $start_date ? gmdate( 'F j, Y', $start_date ) : '–';
	}

	/**
	 * Retrieve the locations for a given coupon and format it.
	 *
	 * @param int $coupon_id The ID of the coupon.
	 *
	 * @return string
	 */
	private function get_allowed_locations( $coupon_id ) {
		$locations = get_post_meta( $coupon_id, 'rtsb_coupon_locations', true );

		return is_array( $locations ) ? $this->convert_location_codes_to_names( $locations ) : '–';
	}

	/**
	 * Convert location codes to their corresponding names.
	 *
	 * @param array $locations An array of location codes.
	 *
	 * @return string
	 */
	private function convert_location_codes_to_names( $locations ) {
		$countries = WC()->countries->get_countries();
		$states    = WC()->countries->get_states();

		$location_names = array_map(
			function ( $location ) use ( $countries, $states ) {
				if ( false !== strpos( $location, '::' ) ) {
					  list( $country_code, $state_code ) = explode( '::', $location );

					  return $states[ $country_code ][ $state_code ] ?? $state_code;
				} else {
					return $countries[ $location ] ?? $location;
				}
			},
			$locations
		);

		return implode( ', ', $location_names );
	}

	/**
	 * Enqueue JavaScript for the coupon admin interface.
	 *
	 * @return void
	 */
	public function enqueue_coupon_admin_script() {
		$screen = get_current_screen();

		if ( 'shop_coupon' !== $screen->post_type ) {
			return;
		}
		?>
		<script type="text/javascript">
			(function($) {
				var post = $('#post');

				if (post.length) {
					var title = post.find('#title');
					var couponCode = title.val();
					var copyButton = ' <button type="button" class="rtsb-copy-coupon-code button" id="rtsb-coupon-group-filter" aria-label="Copy coupon code \'' + couponCode + '\' to the clipboard"><span class="dashicons dashicons-admin-page"></span> Copy Coupon Code</button>';

					$('.post-type-shop_coupon').find('#title').after(copyButton);

					if ( couponCode ) {
						$('.rtsb-copy-coupon-code').data('code', couponCode);
					}

					title.on('input', function() {
						var updatedCouponCode = $(this).val();

						$('.rtsb-copy-coupon-code').data('code', updatedCouponCode);
					});

					$(document).on('click', '.generate-coupon-code', function() {
						var updatedCouponCode = $('#title').val();

						$('.rtsb-copy-coupon-code').data('code', updatedCouponCode);

						setTimeout(function() {
							if (updatedCouponCode) {
								updateShareableLink(updatedCouponCode);
							}
						}, 300);
					});
				}

				$(document).on('click', '.rtsb-copy-coupon-code', function(e) {
					e.preventDefault();

					var couponCode = $(this).data('code');

					if ( couponCode ) {
						rtsbCopyCouponCode(couponCode);
					} else {
						alert('Generate a coupon code first.');
					}
				});

				function rtsbCopyCouponCode( couponCode ) {
					var $tempInput = $('<input>');

					$('body').append($tempInput);
					$tempInput.val(couponCode).select();
					document.execCommand('copy');
					$tempInput.remove();

					alert('Coupon code copied: ' + couponCode);
				}

				function updateShareableLink(newCode) {
					var storeUrl = '<?php echo esc_url( home_url( '/' ) ); ?>';
					var newLink = storeUrl + '?apply_coupon=' + encodeURIComponent(newCode);

					$('.shareable-coupon-link').val(newLink);
					$('#shareable_coupon_link').val(newLink);
				}

				$('.rtsb-copy-coupon-link').on('click', function() {
					var couponLinkInput = $(this).closest('.rtsb-shareable-coupon-link').find('.shareable-coupon-link').val();
					var couponLinkTextarea = $(this).closest('.rtsb-shareable-coupon-link').find('#shareable_coupon_link');

					try {
						if ( couponLinkTextarea.length ) {
							couponLinkTextarea.select();
							document.execCommand('copy');

							alert('<?php echo esc_js( __( 'Coupon link copied to clipboard!', 'shopbuilder-pro' ) ); ?>');
						} else {
							var $tempInput = $('<input>');

							$('body').append($tempInput);
							$tempInput.val(couponLinkInput).select();
							document.execCommand('copy');
							$tempInput.remove();

							alert('<?php echo esc_js( __( 'Coupon link copied to clipboard!', 'shopbuilder-pro' ) ); ?>');
						}
					} catch (err) {
						console.error('Unable to copy the coupon link', err);
					}
				});
			})(jQuery);
		</script>
		<?php
	}
}
