<?php
/**
 * Smart Coupons Module Custom Fields Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\SmartCoupons\Admin;

use WP_Post;
use WC_Coupon;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Helpers\AdminInputsFns;
use RadiusTheme\SBPRO\Modules\CurrencySwitcher\CsFns;
use RadiusTheme\SBPRO\Modules\SmartCoupons\SmartCouponsFns;

defined( 'ABSPATH' ) || exit();

/**
 * Smart Coupons Module Admin Class.
 */
class CouponFields {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Class Constructor.
	 */
	private function __construct() {
		/**
		 * Actions
		 */
		add_action( 'woocommerce_coupon_options', [ $this, 'custom_coupon_fields' ] );
		add_action( 'woocommerce_coupon_options_usage_restriction', [ $this, 'coupon_conditions_fields' ] );
		add_action( 'woocommerce_coupon_data_panels', [ $this, 'gift_product_panel' ], 10, 2 );
		add_action( 'woocommerce_coupon_options_save', [ $this, 'save_coupon_metadata' ] );
		add_action( 'add_meta_boxes', [ $this, 'add_shareable_coupon_link_metabox' ] );

		/**
		 * Filters
		 */
		add_filter( 'woocommerce_coupon_data_tabs', [ $this, 'custom_coupon_data_tab' ] );
		add_filter( 'woocommerce_coupon_discount_types', [ $this, 'add_volume_discount' ] );
	}

	/**
	 * Custom coupon fields.
	 *
	 * @param int $coupon_id Coupon ID.
	 *
	 * @return void
	 */
	public function custom_coupon_fields( $coupon_id ) {
		global $post;

		$coupon_id = $coupon_id ?? $post->ID;
		?>
		<div id="rtsb_smart_coupons" class="options_group rtsb-coupons-field">
			<?php
			$start_date            = get_post_meta( $coupon_id, 'rtsb_coupon_start', true );
			$start_date_formatted  = ! empty( $start_date ) ? gmdate( 'Y-m-d', $start_date ) : '';
			$auto_apply            = get_post_meta( $coupon_id, 'rtsb_coupon_auto_apply', true );
			$auto_apply_status     = ! empty( $auto_apply ) && 'on' === $auto_apply ? 'on' : 'off';
			$first_purchase        = get_post_meta( $coupon_id, 'rtsb_new_user_coupon', true );
			$first_purchase_status = ! empty( $first_purchase ) && 'on' === $first_purchase ? 'on' : 'off';

			woocommerce_wp_text_input(
				[
					'id'                => 'rtsb_coupon_start',
					'value'             => esc_attr( $start_date_formatted ),
					'label'             => __( 'Coupon start date', 'shopbuilder-pro' ),
					'placeholder'       => 'YYYY-MM-DD',
					'description'       => __( 'The coupon will start at 00:00:00 of this date.', 'shopbuilder-pro' ),
					'desc_tip'          => true,
					'class'             => 'date-picker',
					'custom_attributes' => [
						'pattern' => '[0-9]{4}-(0[1-9]|1[012])-(0[1-9]|1[0-9]|2[0-9]|3[01])',
					],
				]
			);

			?>
			<div class="rtsb-group-heading">
				<h2>Extra Settings</h2>
			</div>
			<?php

			$fields = AdminInputsFns::generate_switch_input(
				[
					'id'            => 'rtsb_coupon_auto_apply',
					'label'         => esc_html__( 'Auto apply coupon?', 'shopbuilder-pro' ),
					'desc_tip'      => true,
					'aria_label'    => esc_html__( 'Automatically apply this coupon', 'shopbuilder-pro' ),
					'description'   => esc_html__( 'Switch on to automatically apply this coupon if valid. If multiple coupon is set to auto apply, only the first 5 of these coupons will be applied.', 'shopbuilder-pro' ),
					'checked'       => esc_attr( $auto_apply_status ),
					'value'         => 'yes',
					'input_class'   => 'switch-input',
					'wrapper_class' => 'form-field auto-apply',
				]
			);

			$fields .= AdminInputsFns::generate_switch_input(
				[
					'id'            => 'rtsb_new_user_coupon',
					'label'         => esc_html__( 'First purchase coupon?', 'shopbuilder-pro' ),
					'desc_tip'      => true,
					'aria_label'    => esc_html__( 'First purchase coupon', 'shopbuilder-pro' ),
					'description'   => esc_html__( 'Switch on to use this coupon as a coupon that will only valid for the new user\'s first order.', 'shopbuilder-pro' ),
					'checked'       => esc_attr( $first_purchase_status ),
					'value'         => 'yes',
					'input_class'   => 'switch-input',
					'wrapper_class' => 'form-field new-user-coupon',
				]
			);

			if ( get_current_screen()->id !== 'marketing_page_rtsb-smart-coupons' ) {
				$show_frontend        = get_post_meta( $coupon_id, 'rtsb_show_on_frontend', true );
				$show_frontend_status = ! empty( $show_frontend ) && 'on' === $show_frontend ? 'on' : 'off';

				$fields .= AdminInputsFns::generate_switch_input(
					[
						'id'            => 'rtsb_show_on_frontend',
						'label'         => esc_html__( 'Show on frontend?', 'shopbuilder-pro' ),
						'desc_tip'      => true,
						'aria_label'    => esc_html__( 'Display coupon on frontend', 'shopbuilder-pro' ),
						'description'   => esc_html__( 'Switch on to include this coupon in the coupon list to display on the frontend.', 'shopbuilder-pro' ),
						'checked'       => esc_attr( $show_frontend_status ),
						'value'         => 'yes',
						'input_class'   => 'switch-input',
						'wrapper_class' => 'form-field new-user-coupon',
					]
				);
			}

			Fns::print_html( $fields, true );
			?>
			<div class="rtsb-group-heading rtsb-show-on-percentage">
				<h2>Volume Discounts</h2>
			</div>
			<?php
			$volume_type    = get_post_meta( $coupon_id, 'rtsb_volume_coupon_type', true );
			$volume_coupons = get_post_meta( $coupon_id, 'rtsb_volume_coupon', true );

			$fields = AdminInputsFns::generate_select_input(
				[
					'id'            => 'rtsb_volume_coupon_type',
					'label'         => esc_html__( 'Choose Discount Type', 'shopbuilder-pro' ),
					'desc_tip'      => true,
					'aria_label'    => esc_html__( 'Choose Discount Type', 'shopbuilder-pro' ),
					'description'   => esc_html__( '\'Per Product\' requires choosing specific products, while \'Total Cart Item\' applies based on the total quantity in the cart.', 'shopbuilder-pro' ),
					'help_text'     => __( '— For \'Per Product\', select products in the "Usage Restrictions" tab. If none are selected, the discount won’t apply. <br />— \'Total Cart Item\' applies to the cart’s total quantity, ignoring selected products in "Usage Restrictions".', 'shopbuilder-pro' ),
					'value'         => $volume_type ?? '',
					'label_class'   => 'field-label',
					'placeholder'   => esc_html__( 'Choose discount type...', 'shopbuilder-pro' ),
					'options'       => [
						'per_product' => esc_html__( 'Per Product (discount(%) to each product individually)', 'shopbuilder-pro' ),
						'total_cart'  => esc_html__( 'Total Cart Item (discount(%) based on total cart quantity)', 'shopbuilder-pro' ),
					],
					'input_class'   => 'short rtsb-volume-coupon-type',
					'wrapper_class' => 'form-field volume-coupons rtsb-show-on-percentage',
				]
			);

			$fields .= AdminInputsFns::generate_textarea_input(
				[
					'id'            => 'rtsb_volume_coupon',
					'label'         => esc_html__( 'Volume Discount Tiers', 'shopbuilder-pro' ),
					'desc_tip'      => true,
					'aria_label'    => esc_html__( 'Volume discounts', 'shopbuilder-pro' ),
					'description'   => esc_html__( 'Enter \'Quantity:Discount%\' pairs separated by | (pipe).', 'shopbuilder-pro' ),
					'help_text'     => __( 'Enter \'Quantity:Discount%\' pairs separated by | (pipe).<br /> <u>For Example</u>: 10:5|20:10 means 10:5 for 5% off above 10 items, 20:10 for 10% off above 20, etc.', 'shopbuilder-pro' ),
					'placeholder'   => esc_html__( 'Example: 10:5|20:10|30:15 (Quantity:Discount%)', 'shopbuilder-pro' ),
					'value'         => $volume_coupons ?? '',
					'input_class'   => 'short rtsb-volume-coupon',
					'wrapper_class' => 'form-field volume-coupons rtsb-show-on-percentage',
				]
			);

			Fns::print_html( $fields, true );
			?>
			<script type="text/javascript">
				jQuery(document).ready(function($) {
					var $discountType = $('select#discount_type'),
						$volumeCoupons = $('.rtsb-show-on-percentage'),
						$couponAmount  = $('#general_coupon_data .coupon_amount_field'),
						$giftProductSwitch = $('#rtsb_enable_gift_products'),
						$giftProductSection = $('.rtsb-show-gift-products');

					function toggleVolumeDiscountField() {
						if ($discountType.val() === 'rtsb_volume_discount') {
							$volumeCoupons.stop(true, true).fadeIn();
							$couponAmount.hide();
						} else {
							$volumeCoupons.hide();
							$couponAmount.show();
						}
					}

					function toggleGiftProductField() {
						if ($giftProductSwitch.is(':checked')) {
							$giftProductSection.stop(true, true).fadeIn();
						} else {
							$giftProductSection.hide();
						}
					}

					toggleVolumeDiscountField();
					toggleGiftProductField();

					$discountType.on('change', toggleVolumeDiscountField);
					$giftProductSwitch.on('change', toggleGiftProductField);
				});
			</script>
		</div>
		<?php
	}

	/**
	 * Custom coupon fields.
	 *
	 * @param int $coupon_id Coupon ID.
	 *
	 * @return void
	 */
	public function coupon_conditions_fields( $coupon_id ) {
		global $post;

		$coupon_id = $coupon_id ?? $post->ID;
		?>
		<div id="rtsb_smart_coupons_conditions" class="options_group rtsb-coupons-conditions-field">
			<?php

			$min_qty         = get_post_meta( $coupon_id, 'rtsb_min_product_qty', true );
			$max_qty         = get_post_meta( $coupon_id, 'rtsb_max_product_qty', true );
			$user_role       = get_post_meta( $coupon_id, 'rtsb_user_role_based_coupon', true );
			$shipping_method = get_post_meta( $coupon_id, 'rtsb_shipping_based_coupon', true );
			$payment_method  = get_post_meta( $coupon_id, 'rtsb_payment_based_coupon', true );
			$locations       = get_post_meta( $coupon_id, 'rtsb_coupon_locations', true );
			?>
			<div class="rtsb-group-heading">
				<h2>Extra Restrictions</h2>
			</div>
			<?php
			$fields = AdminInputsFns::generate_number_input(
				[
					'id'            => 'rtsb_min_product_qty',
					'label'         => esc_html__( 'Minimum cart quantity', 'shopbuilder-pro' ),
					'desc_tip'      => true,
					'aria_label'    => esc_html__( 'Minimum cart quantity', 'shopbuilder-pro' ),
					'description'   => esc_html__( 'This field allows you to set the minimum total cart quantity allowed to use the coupon.', 'shopbuilder-pro' ),
					'value'         => $min_qty ?? '',
					'placeholder'   => esc_html__( 'No minimum', 'shopbuilder-pro' ),
					'min'           => '0',
					'step'          => '1',
					'input_class'   => 'rtsb-min-product-qty',
					'label_class'   => 'field-label',
					'wrapper_class' => 'form-field minimum-product-qty',
				]
			);

			$fields .= AdminInputsFns::generate_number_input(
				[
					'id'            => 'rtsb_max_product_qty',
					'label'         => esc_html__( 'Maximum cart quantity', 'shopbuilder-pro' ),
					'desc_tip'      => true,
					'aria_label'    => esc_html__( 'Maximum cart quantity', 'shopbuilder-pro' ),
					'description'   => esc_html__( 'This field allows you to set the maximum total cart product quantity allowed to use the coupon.', 'shopbuilder-pro' ),
					'value'         => $max_qty ?? '',
					'placeholder'   => esc_html__( 'No maximum', 'shopbuilder-pro' ),
					'min'           => '0',
					'step'          => '1',
					'input_class'   => 'rtsb-max-product-qty',
					'label_class'   => 'field-label',
					'wrapper_class' => 'form-field maximum-product-qty',
				]
			);

			$fields .= AdminInputsFns::generate_select_input(
				[
					'id'            => 'rtsb_user_role_based_coupon',
					'label'         => esc_html__( 'Allowed user roles', 'shopbuilder-pro' ),
					'desc_tip'      => true,
					'aria_label'    => esc_html__( 'Select user role', 'shopbuilder-pro' ),
					'description'   => esc_html__( 'Select the user role who can use this coupon.', 'shopbuilder-pro' ),
					'value'         => $user_role ?? '',
					'input_class'   => 'rtsb-select2-input wc-enhanced-select',
					'label_class'   => 'field-label',
					'placeholder'   => esc_html__( 'Select user roles...', 'shopbuilder-pro' ),
					'multiple'      => true,
					'options'       => FnsPro::get_all_user_roles(),
					'wrapper_class' => 'form-field user-based-coupon',
				]
			);

			$fields .= AdminInputsFns::generate_select_input(
				[
					'id'            => 'rtsb_shipping_based_coupon',
					'label'         => __( 'Shipping methods', 'shopbuilder-pro' ),
					'desc_tip'      => true,
					'aria_label'    => esc_html__( 'Select shipping method', 'shopbuilder-pro' ),
					'description'   => esc_html__( 'Select the shipping method to apply this coupon. This coupon will be applied only if these shipping methods are selected.', 'shopbuilder-pro' ),
					'value'         => $shipping_method ?? '',
					'input_class'   => 'rtsb-select2-input wc-enhanced-select',
					'label_class'   => 'field-label',
					'placeholder'   => esc_html__( 'Select shipping methods...', 'shopbuilder-pro' ),
					'multiple'      => true,
					'options'       => FnsPro::get_all_shipping_methods(),
					'wrapper_class' => 'form-field shipping-based-coupon',
				]
			);

			$payment_methods = [];

			foreach ( CsFns::search_payment_gateways() as $p_method ) {
				$payment_methods[ $p_method['value'] ] = $p_method['label'];
			}

			$fields .= AdminInputsFns::generate_select_input(
				[
					'id'            => 'rtsb_payment_based_coupon',
					'label'         => __( 'Payment methods', 'shopbuilder-pro' ),
					'desc_tip'      => true,
					'aria_label'    => esc_html__( 'Select payment method', 'shopbuilder-pro' ),
					'description'   => esc_html__( 'Select the payment method to apply this coupon. This coupon will be applied only if these payment methods are selected.', 'shopbuilder-pro' ),
					'value'         => $payment_method ?? '',
					'input_class'   => 'rtsb-select2-input wc-enhanced-select',
					'label_class'   => 'field-label',
					'placeholder'   => ! empty( CsFns::search_payment_gateways() ) ? esc_html__( 'Choose payment methods...', 'shopbuilder-pro' ) : esc_html__( 'No active payment methods found.', 'shopbuilder-pro' ),
					'multiple'      => true,
					'options'       => $payment_methods,
					'wrapper_class' => 'form-field shipping-based-coupon',
				]
			);

			$fields .= AdminInputsFns::generate_location_select_input( 'rtsb_coupon_locations', $locations ?? [] );

			Fns::print_html( $fields, true );
			?>
		</div>
		<?php
	}

	/**
	 * Custom coupon data tab.
	 *
	 * @param  array $tabs Data tabs.
	 *
	 * @return array
	 */
	public function custom_coupon_data_tab( $tabs ) {
		$tabs['rtsb_coupon_gift_products'] = [
			'label'  => esc_html__( 'Gift / BOGO Products', 'shopbuilder-pro' ),
			'target' => 'rtsb_gift_products_panel',
			'class'  => 'rtsb-gift-panel',
		];

		return $tabs;
	}

	/**
	 * Add volume discount type.
	 *
	 * @param array $type Discount type.
	 *
	 * @return array
	 */
	public function add_volume_discount( $type ) {
		$type['rtsb_volume_discount'] = esc_html__( 'Volume Discount', 'shopbuilder-pro' );

		return $type;
	}

	/**
	 * Panel for Smart Coupons additional data fields
	 *
	 * @param integer   $coupon_id The coupon id.
	 * @param WC_Coupon $coupon The coupon object.
	 */
	public function gift_product_panel( $coupon_id, $coupon = null ) {
		?>
		<div id="rtsb_gift_products_panel" class="panel woocommerce_options_panel">
			<div class="options_group">
				<div class="rtsb-group-heading">
					<h2>Gift / BOGO Products</h2>
				</div>
				<?php
				$gift_products = get_post_meta( $coupon_id, 'rtsb_enable_gift_products', true );
				$gift_status   = ! empty( $gift_products ) && 'on' === $gift_products ? 'on' : 'off';
				$fields        = AdminInputsFns::generate_switch_input(
					[
						'id'            => 'rtsb_enable_gift_products',
						'label'         => esc_html__( 'Enable Gift Products?', 'shopbuilder-pro' ),
						'desc_tip'      => true,
						'aria_label'    => esc_html__( 'Switch on to enable gift products.', 'shopbuilder-pro' ),
						'description'   => esc_html__( 'Switch on to enable gift products.', 'shopbuilder-pro' ),
						'checked'       => esc_attr( $gift_status ),
						'value'         => 'yes',
						'input_class'   => 'switch-input',
						'wrapper_class' => 'form-field enable-gift-products',
					]
				);

				Fns::print_html( $fields, true );
				?>
				<p class="form-field free-products rtsb-show-gift-products">
					<label for="rtsb_free_product_ids"><?php esc_html_e( 'Choose gift product(s)', 'shopbuilder-pro' ); ?></label>
					<select class="wc-product-search" multiple="multiple" style="width: 50%;" id="rtsb_free_product_ids" name="rtsb_free_product_ids[]" data-placeholder="<?php esc_attr_e( 'Select gift products&hellip;', 'shopbuilder-pro' ); ?>" data-action="woocommerce_json_search_products_and_variations">
						<?php
						if ( is_object( $coupon ) ) {
							$product_ids = get_post_meta( $coupon_id, 'rtsb_free_product_ids', true );

							if ( ! empty( $product_ids ) ) {
								foreach ( $product_ids as $product_id ) {
									$product = wc_get_product( $product_id );

									if ( is_object( $product ) ) {
										echo '<option value="' . esc_attr( $product_id ) . '"' . selected( true, true, false ) . '>' . esc_html( wp_strip_all_tags( $product->get_formatted_name() ) ) . '</option>';
									}
								}
							}
						} else {
							echo '<option value="">' . esc_html__( 'No product available', 'shopbuilder-pro' ) . '</option>';
						}
						?>
					</select>
					<?php
					Fns::print_html( wc_help_tip( __( 'Select free products. These free products will be added to the cart when the coupon is applied.', 'shopbuilder-pro' ) ), true );
					?>
				</p>
				<?php
				global $post;

				$coupon_id = $coupon_id ?? $post->ID;
				$free_qty  = get_post_meta( $coupon_id, 'rtsb_free_product_qty', true );
				$free_msg  = get_post_meta( $coupon_id, 'rtsb_free_product_msg', true );

				$fields = AdminInputsFns::generate_number_input(
					[
						'id'            => 'rtsb_free_product_qty',
						'label'         => esc_html__( 'Gift Product Quantity', 'shopbuilder-pro' ),
						'desc_tip'      => true,
						'aria_label'    => esc_html__( 'Free product quantity', 'shopbuilder-pro' ),
						'description'   => esc_html__( 'Enter the free product quantity to be added to the cart.', 'shopbuilder-pro' ),
						'value'         => $free_qty ?? '',
						'min'           => '0',
						'step'          => '1',
						'input_class'   => 'rtsb-addon-input-price',
						'label_class'   => 'field-label',
						'wrapper_class' => 'form-field free-product-qty rtsb-show-gift-products',
					]
				);

				$fields .= AdminInputsFns::generate_wysiwyg_editor(
					[
						'id'            => 'rtsb_free_product_msg',
						'label'         => esc_html__( 'Custom Gift Notice', 'shopbuilder-pro' ),
						'desc_tip'      => true,
						'aria_label'    => esc_html__( 'Custom gift notice', 'shopbuilder-pro' ),
						'description'   => esc_html__( 'Custom gift notice for your users. This notice will be displayed when the coupon is applied. Use {{gifts}} placeholder to display the gift products.', 'shopbuilder-pro' ),
						'placeholder'   => esc_html__( 'Enter the custom notice', 'shopbuilder-pro' ),
						'value'         => ! empty( $free_msg ) ? $free_msg : esc_html__( 'Congratulations! You have received the following complimentary gift(s) with your order: {{gifts}}', 'shopbuilder-pro' ),
						'input_class'   => 'rtsb-free-product-msg',
						'wrapper_class' => 'form-field free-product-msg rtsb-show-gift-products',
						'tag'           => 'div',
					]
				);

				Fns::print_html( $fields, true );
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * Renders the content for the shareable coupon link metabox.
	 *
	 * @param WP_Post $post The current post-object.
	 */
	public function render_shareable_coupon_link( $post ) {
		// Ensure the coupon has a code before displaying the metabox content.
		if ( empty( $post->post_title ) ) {
			return;
		}

		$coupon_code   = $post->post_title;
		$store_url     = wc_get_page_permalink( 'shop' );
		$shareable_url = add_query_arg( 'rtsb_apply_coupon', $coupon_code, $store_url );

		?>
		<div class="inside rtsb-shareable-coupon-link">
			<p><?php esc_html_e( 'Copy and share the link below to automatically apply this coupon to the user\'s cart:', 'shopbuilder-pro' ); ?></p>
			<label>
				<textarea readonly id="shareable_coupon_link" style="width: 100%; height: auto;"><?php echo esc_url( $shareable_url ); ?></textarea>
			</label>
			<input type="hidden" readonly value="<?php echo esc_url( $shareable_url ); ?>" class="shareable-coupon-link" />
			<p>
				<button type="button" class="button rtsb-copy-coupon-link" aria-label="Copy coupon link to the clipboard" data-code="<?php echo esc_attr( $coupon_code ); ?>">
					<span class="dashicons dashicons-admin-page"></span> <?php esc_html_e( 'Copy Link', 'shopbuilder-pro' ); ?>
				</button>
			</p>
		</div>
		<?php
	}

	/**
	 * Save coupon custom fields.
	 *
	 * @param int $post_id The post id.
	 *
	 * @return void
	 */
	public function save_coupon_metadata( $post_id ) {
		if ( empty( $post_id ) ) {
			return;
		}

		$metadata_list = self::coupon_metadata_list();

		foreach ( $metadata_list as $meta_key => $meta_type ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( isset( $_REQUEST[ $meta_key ] ) ) {
				switch ( $meta_type ) {
					case 'switch':
						$sanitized_value = SmartCouponsFns::sanitize_switch( $meta_key );
						break;

					case 'date':
						$sanitized_value = SmartCouponsFns::sanitize_date( $meta_key );
						break;

					case 'text':
						$sanitized_value = SmartCouponsFns::sanitize_text( $meta_key );
						break;

					case 'array_number':
						$sanitized_value = SmartCouponsFns::sanitize_array_number( $meta_key );
						break;

					case 'array':
						$sanitized_value = SmartCouponsFns::sanitize_array( $meta_key );
						break;

					case 'number':
						$sanitized_value = SmartCouponsFns::sanitize_number( $meta_key );
						break;

					case 'textarea':
						$sanitized_value = SmartCouponsFns::sanitize_textarea( $meta_key );
						break;

					default:
						$sanitized_value = '';
						break;
				}

				update_post_meta( $post_id, $meta_key, $sanitized_value );
			} else {
				switch ( $meta_type ) {
					case 'switch':
					case 'array':
					case 'array_number':
						delete_post_meta( $post_id, $meta_key );
						break;

					default:
						break;
				}
			}
		}
	}

	/**
	 * Add shareable coupon link metabox.
	 *
	 * @return void
	 */
	public function add_shareable_coupon_link_metabox() {
		global $pagenow, $typenow, $post;

		// Check if we're on the coupon edit page.
		if ( 'post.php' === $pagenow && 'shop_coupon' === $typenow && ! empty( $post->post_title ) ) {
			add_meta_box(
				'rtsb_shareable_coupon_link',
				esc_html__( 'Shareable Coupon Link', 'shopbuilder-pro' ),
				[ $this, 'render_shareable_coupon_link' ],
				'shop_coupon',
				'side',
				'core'
			);
		}
	}

	/**
	 * Coupon metadata list.
	 *
	 * @return string[]
	 */
	public static function coupon_metadata_list() {
		return [
			'rtsb_coupon_auto_apply'      => 'switch',
			'rtsb_new_user_coupon'        => 'switch',
			'rtsb_show_on_frontend'       => 'switch',
			'rtsb_coupon_start'           => 'date',
			'rtsb_enable_gift_products'   => 'switch',
			'rtsb_free_product_ids'       => 'array_number',
			'rtsb_free_product_qty'       => 'number',
			'rtsb_free_product_msg'       => 'textarea',
			'rtsb_user_role_based_coupon' => 'array',
			'rtsb_shipping_based_coupon'  => 'array',
			'rtsb_payment_based_coupon'   => 'array',
			'rtsb_coupon_locations'       => 'array',
			'rtsb_min_product_qty'        => 'number',
			'rtsb_max_product_qty'        => 'number',
			'rtsb_volume_coupon_type'     => 'text',
			'rtsb_volume_coupon'          => 'textarea',
		];
	}
}
