<?php
/**
 * Smart Coupon Management Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\SmartCoupons\Frontend;

use Exception;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\SmartCoupons\RTSB_Coupon;
use RadiusTheme\SBPRO\Modules\SmartCoupons\SmartCouponsFns;

defined( 'ABSPATH' ) || exit();

/**
 * Smart Coupon Management Class.
 */
class CouponManagement {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Class Constructor.
	 */
	private function __construct() {
		/**
		 * Actions.
		 */
		add_action( 'woocommerce_before_calculate_totals', [ $this, 'handle_auto_coupons_apply' ], 5 );
		add_action( 'woocommerce_applied_coupon', [ $this, 'handle_manual_coupon_apply' ] );
		add_action( 'woocommerce_removed_coupon', [ $this, 'handle_coupon_removal' ] );
		add_action( 'woocommerce_before_cart', [ $this, 'display_custom_message' ] );
		add_action( 'woocommerce_checkout_create_order_line_item', [ $this, 'add_custom_order_items' ], 10, 3 );

		/**
		 * Filters
		 */
		add_filter( 'woocommerce_coupon_is_valid', [ $this, 'validate_manual_coupons' ], 10, 2 );
		add_filter( 'woocommerce_coupon_message', [ $this, 'disable_woocommerce_coupon_message' ], 10, 3 );
		add_filter( 'woocommerce_cart_totals_coupon_html', [ $this, 'add_custom_coupon_class' ], 10, 2 );
		add_filter( 'woocommerce_cart_item_name', [ $this, 'display_custom_label' ], 10, 2 );
		/**
		 * This is rendering duplicate labels. Need to check if this is necessary.
		 * add_filter( 'rtsb/element/cart/table/item_name', [ $this, 'display_custom_label' ], 10, 2 );
		 */
		add_filter( 'rtsb/calculate/countdown/campaign', [ $this, 'disable_modules' ], 11, 2 );
		add_filter( 'rtsb/calculate/pre_order/price', [ $this, 'disable_modules' ], 11, 2 );
		add_filter( 'woocommerce_order_item_get_formatted_meta_data', [ $this, 'format_custom_meta' ], 10, 2 );
		add_filter( 'woocommerce_hidden_order_itemmeta', [ $this, 'hide_custom_meta' ], 10, 1 );
	}

	/**
	 * Handles auto-applying coupons.
	 *
	 * @param object $cart Cart object.
	 *
	 * @return void
	 */
	public function handle_auto_coupons_apply( $cart ) {
		if ( $this->should_skip_auto_apply() ) {
			return;
		}

		if ( did_action( 'woocommerce_before_calculate_totals' ) >= 2 ) {
			return;
		}

		$applied_coupons = [];
		// Apply first-purchase coupon.
		$first_purchase_coupon = $this->get_first_purchase_coupon();
		$coupon_id             = $this->get_first_purchase_coupon( true );
		$is_auto_apply         = 'on' === get_post_meta( $coupon_id, 'rtsb_coupon_auto_apply', true );

		if ( $is_auto_apply && $this->can_apply_coupon( $coupon_id ) && $first_purchase_coupon && ! $cart->has_discount( $first_purchase_coupon ) && $this->is_first_purchase() ) {
			$applied = $cart->apply_coupon( $first_purchase_coupon );
			if ( $applied ) {
				$applied_coupons[] = $first_purchase_coupon;
			}
		}

		$auto_apply_coupons = $this->get_auto_apply_coupons();
		$removed_coupons    = $this->get_removed_coupons();
		$coupon_counter     = 0;
		$auto_apply_limit   = $this->get_auto_apply_limit();

		foreach ( $auto_apply_coupons as $coupon_id ) {
			if ( $coupon_counter > $auto_apply_limit ) {
				break;
			}

			$coupon_code = sanitize_text_field( get_post_field( 'post_title', $coupon_id ) );

			if ( empty( $coupon_code ) || in_array( strtolower( $coupon_code ), $removed_coupons, true ) || $first_purchase_coupon === $coupon_code ) {
				continue;
			}
			if ( ! $cart->has_discount( $coupon_code ) && $this->can_apply_coupon( $coupon_id ) ) {
				$applied = $cart->apply_coupon( $coupon_code );
				if ( $applied ) {
					$applied_coupons[] = $coupon_code;
				}
			}

			$coupon_counter++;
		}

		/**
		 * Action hook for: after auto-apply.
		 */
		do_action( 'rtsb/module/smart_coupons/after_auto_apply', $cart );

		if ( ! empty( array_keys( $applied_coupons ) ) ) {
			$this->trigger_merged_notice( $applied_coupons );
			WC()->session->set( 'rtsb_auto_apply_coupons_status', true );
		} else {
			WC()->session->set( 'rtsb_auto_apply_coupons_status', false );
		}
	}

	/**
	 * Validate manually applied coupon and check restrictions.
	 *
	 * @param string $coupon_code Coupon code.
	 *
	 * @return void
	 * @throws Exception Exception.
	 */
	public function handle_manual_coupon_apply( $coupon_code ) {
		if ( did_action( 'woocommerce_applied_coupon' ) >= 2 ) {
			return;
		}

		$cart   = WC()->cart;
		$coupon = $this->coupon( $coupon_code );

		if ( ! $coupon->get_id() || ( $this->is_auto_applied_coupon( $coupon->get_id() ) && ! WC()->session->get( 'rtsb_auto_apply_coupons_status' ) ) ) {
			return;
		}

		// Check if it's a first-purchase coupon.
		if ( $coupon->rtsb_is_first_purchase() && ! $this->is_first_purchase() ) {
			SmartCouponsFns::trigger_notice( esc_html__( 'This coupon is only valid for first-time purchases.', 'shopbuilder-pro' ), 'error' );
			$cart->remove_coupon( $coupon_code );

			return;
		}

		// Check if coupon meets custom conditions.
		$coupon_id = $coupon->get_id();

		if ( ! $this->can_apply_coupon( $coupon_id ) ) {
			$message = esc_html__( 'This coupon is not applicable to your cart.', 'shopbuilder-pro' );
			$type    = 'error';

			$cart->remove_coupon( $coupon_code );
			SmartCouponsFns::trigger_notice( $message, $type );
		}

		// Add free products to the cart if applicable.
		if ( ! empty( $coupon->rtsb_get_gift_products() ) ) {
			FreeGift::add_free_gifts_to_cart( $coupon );
		}
	}

	/**
	 * Handles coupon removal.
	 *
	 * @param string $coupon_code Coupon code.
	 *
	 * @return void
	 */
	public function handle_coupon_removal( $coupon_code ) {
		$removed_coupons   = $this->get_removed_coupons();
		$removed_coupons[] = $coupon_code;

		WC()->session->set( 'rtsb_removed_coupons', array_unique( $removed_coupons ) );

		/**
		 * Action hook for: after coupon removal.
		 */
		do_action( 'rtsb/module/smart_coupons/after_coupon_removal', $coupon_code, $this );
	}

	/**
	 * Displays a custom message.
	 *
	 * @return void
	 */
	public function display_custom_message() {
		$applied_coupons = WC()->cart->get_applied_coupons();

		FreeGift::display_gift_product_message( $applied_coupons );
		VolumeDiscount::display_volume_discount_message( $applied_coupons );
	}

	/**
	 * Add custom metadata to order line items.
	 *
	 * @param object $item Line item.
	 * @param string $cart_item_key Cart item key.
	 * @param array  $values Cart item values.
	 *
	 * @return void
	 */
	public function add_custom_order_items( $item, $cart_item_key, $values ) {
		FreeGift::add_gift_meta_to_order_items( $item, $cart_item_key, $values );
		VolumeDiscount::add_volume_discount_to_order_items( $item, $cart_item_key, $values );
	}

	/**
	 * Validates the first-purchase coupon on manual entry.
	 *
	 * @param bool   $is_valid Whether the coupon is valid.
	 * @param object $coupon The WooCommerce coupon object.
	 *
	 * @throws Exception Throws Exception for invalid coupons.
	 * @return bool
	 */
	public function validate_manual_coupons( $is_valid, $coupon ) {
		$coupon_id = $coupon->get_id();

		if ( ! $this->can_apply_coupon( $coupon_id ) ) {
			throw new Exception( esc_html__( 'Sorry, this coupon is not valid for your cart.', 'shopbuilder-pro' ) );
		}

		// Add free products to the cart if applicable.
		if ( ! empty( $coupon->get_meta( 'rtsb_free_product_ids' ) ) ) {
			$rtsb_coupon = $this->coupon( $coupon_id );

			FreeGift::add_free_gifts_to_cart( $rtsb_coupon );
		}

		return $is_valid;
	}

	/**
	 * Disable WooCommerce coupon message.
	 *
	 * @param string $msg    The original coupon message.
	 * @param int    $msg_id The message ID.
	 * @param object $coupon The coupon object.
	 *
	 * @return string
	 */
	public function disable_woocommerce_coupon_message( $msg, $msg_id, $coupon ) {
		if (
			( $this->is_auto_applied_coupon( $coupon->get_id() ) && ! WC()->session->get( 'rtsb_auto_apply_coupons_status' ) )
			|| ! empty( Fns::getSession( 'rtsb_pending_url_coupon' ) )
		) {
			return '';
		}

		return $msg;
	}

	/**
	 * Add a custom class to zero-amount coupons in cart totals.
	 *
	 * @param string $coupon_html The coupon HTML.
	 * @param object $coupon      The coupon object.
	 *
	 * @return string
	 */
	public function add_custom_coupon_class( $coupon_html, $coupon ) {
		if ( 0 === absint( $coupon->get_amount() ) || SmartCouponsFns::coupon_includes_volume_discount( $coupon ) ) {
			$coupon_html = str_replace( '<span class="woocommerce-Price-amount amount', '<span class="woocommerce-Price-amount amount rtsb-coupon-zero-amount', $coupon_html );
		}

		return $coupon_html;
	}

	/**
	 * Appends a custom label to the product name if conditions are met.
	 *
	 * @param string $product_name Product name.
	 * @param array  $cart_item    Cart item data.
	 *
	 * @return string
	 */
	public function display_custom_label( $product_name, $cart_item ) {
		if ( FreeGift::is_free_gift( $cart_item ) ) {
			$svg_icon     = FreeGift::get_free_gift_icon_svg();
			$product_name = '<span class="rtsb-free-gift-wrapper"><span class="rtsb-free-gift-label">' . $svg_icon . '</span>' . $product_name . '</span>';
		}

		$applied_coupons = WC()->cart->get_applied_coupons();

		foreach ( $applied_coupons as $coupon_code ) {
			$coupon        = $this->coupon( $coupon_code );
			$discount_type = get_post_meta( $coupon->get_id(), 'rtsb_volume_coupon_type', true );

			if ( SmartCouponsFns::coupon_includes_volume_discount( $coupon ) && 'per_product' === $discount_type ) {
				$associated_product_ids = $coupon->get_product_ids();

				if ( ! empty( $associated_product_ids ) && in_array( $cart_item['product_id'], $associated_product_ids, true ) ) {
					$volume_rules     = SmartCouponsFns::get_volume_discount_rules( $coupon );
					$discount_percent = SmartCouponsFns::get_applicable_discount( $cart_item['quantity'], $volume_rules );

					if ( $discount_percent > 0 ) {
						$regular_price     = $cart_item['data']->get_regular_price();
						$sale_price        = $cart_item['data']->get_sale_price();
						$product_price     = ! empty( $sale_price ) ? $sale_price : $regular_price;
						$per_item_discount = ( $product_price * $discount_percent ) / 100;
						$total_discount    = $per_item_discount * $cart_item['quantity'];

						$price_text = sprintf(
							'<span class="rtsb-volume-product-name">%s: %s</span>',
							esc_html__( 'Product Price', 'shopbuilder-pro' ),
							wc_price( $product_price )
						);

						$discount_text = sprintf(
							'<span class="rtsb-volume-discount">%s: %s (%d%%)</span>',
							esc_html__( 'Per Product Discount', 'shopbuilder-pro' ),
							wc_price( $per_item_discount ),
							absint( $discount_percent )
						);

						$total_discount_text = sprintf(
							'<span class="rtsb-volume-total-discount">%s: %s</span>',
							esc_html__( 'Total Discount', 'shopbuilder-pro' ),
							wc_price( $total_discount )
						);

						$product_name .= '<span class="rtsb-coupon-discount-meta">' . $price_text . $discount_text . $total_discount_text . '</span>';
					}
				}
			}
		}

		return $product_name;
	}

	/**
	 * Disables the modules.
	 *
	 * @return bool
	 */
	public function disable_modules() {
		return false;
	}

	/**
	 * Add the custom metadata to order line items.
	 *
	 * @param array  $meta_data Array of meta-data.
	 * @param object $item Line item.
	 *
	 * @return array
	 */
	public function format_custom_meta( $meta_data, $item ) {
		$free_gift       = $item->get_meta( '_rtsb_coupon_free_gift' );
		$volume_discount = $item->get_meta( '_rtsb_volume_discount' );

		if ( $free_gift ) {
			$meta_data = FreeGift::format_free_gift_meta( $meta_data, $free_gift );
		}

		if ( $volume_discount ) {
			$meta_data = VolumeDiscount::format_volume_discount_meta( $meta_data, $volume_discount, $item );
		}

		return $meta_data;
	}

	/**
	 * Hides the custom meta-data from the admin order details view.
	 *
	 * @param array $hidden_meta_data Array of meta-data keys that should be hidden.
	 *
	 * @return array
	 */
	public function hide_custom_meta( $hidden_meta_data ) {
		$hidden_meta_data[] = '_rtsb_volume_discount';
		$hidden_meta_data[] = '_rtsb_coupon_free_gift';

		return $hidden_meta_data;
	}

	/**
	 * Determines whether auto-apply should be skipped.
	 *
	 * @return bool
	 */
	private function should_skip_auto_apply() {
		return is_admin() || wp_doing_ajax() || ! WC()->cart || WC()->cart->is_empty();
	}

	/**
	 * Retrieves the auto-apply coupons.
	 *
	 * @return array
	 */
	private function get_auto_apply_coupons() {
		$wc_session         = WC()->session;
		$auto_apply_coupons = $wc_session->get( 'rtsb_auto_apply_coupons', [] );

		if ( empty( $auto_apply_coupons ) ) {
			$cache_key = 'rtsb_coupons_data_' . md5( wp_json_encode( $wc_session ) );
			$coupons   = wp_cache_get( $cache_key, 'shopbuilder' );

			if ( false === $coupons ) {
				$coupons = SmartCouponsFns::get_coupons(
					[
						'posts_per_page' => $this->get_auto_apply_limit(),
						'meta_query'     => [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
							[
								'key'     => 'rtsb_coupon_auto_apply',
								'value'   => 'on',
								'compare' => '==',
							],
						],
					]
				);
				wp_cache_set( $cache_key, $coupons, 'shopbuilder', HOUR_IN_SECONDS );
				Cache::set_data_cache_key( $cache_key );
			}

			$auto_apply_coupons = array_unique( array_map( 'absint', $coupons ) );
			$wc_session->set( 'rtsb_auto_apply_coupons', $auto_apply_coupons );
		}

		return $auto_apply_coupons;
	}

	/**
	 * Retrieves the removed coupons.
	 *
	 * @return array
	 */
	private function get_removed_coupons() {
		return WC()->session->get( 'rtsb_removed_coupons', [] );
	}

	/**
	 * Retrieves the auto-apply limit.
	 *
	 * @return int
	 */
	private function get_auto_apply_limit() {
		return absint( apply_filters( 'rtsb/modules/coupons/auto_apply_limit', 5 ) );
	}

	/**
	 * Checks if a coupon can be applied to the cart.
	 *
	 * @param int $coupon_id Coupon ID.
	 *
	 * @return bool
	 */
	private function can_apply_coupon( $coupon_id ) {
		$coupon      = $this->coupon( $coupon_id );
		$coupon_type = $coupon->get_discount_type();

		if ( $coupon->rtsb_is_first_purchase() && ! $this->is_first_purchase() ) {
			return false;
		}
		if ( 'fixed_product' === $coupon_type ) {
			$eligible_products = $coupon->get_product_ids();
			if ( empty( $eligible_products ) ) {
				return false;
			}
			foreach ( WC()->cart->get_cart() as $cart_item ) {
				if ( in_array( $cart_item['product_id'], $eligible_products, true ) ) {
					return $coupon->rtsb_validate_coupon_conditions();
				}
			}

			return false;
		}

		return $coupon->rtsb_validate_coupon_conditions();
	}

	/**
	 * Check if the coupon is being auto-applied.
	 *
	 * @param string $coupon_code The coupon code.
	 *
	 * @return bool
	 */
	private function is_auto_applied_coupon( $coupon_code ) {
		$auto_apply_coupons = $this->get_auto_apply_coupons();
		$coupon_codes       = array_map( 'absint', $auto_apply_coupons );

		return in_array( absint( $coupon_code ), $coupon_codes, true );
	}

	/**
	 * Checks if the user is making their first purchase.
	 *
	 * @return bool
	 */
	private function is_first_purchase() {
		$current_user = wp_get_current_user();

		if ( ! $current_user->exists() ) {
			return true;
		}

		$orders = wc_get_orders(
			[
				'customer_id' => $current_user->ID,
				'limit'       => 1,
			]
		);

		return empty( $orders );
	}

	/**
	 * Retrieves the first-purchase coupon code.
	 *
	 * @param bool $get_id Get the coupon ID.
	 *
	 * @return string|null
	 */
	private function get_first_purchase_coupon( $get_id = false ) {
		$coupon_query = SmartCouponsFns::get_coupons(
			[
				'posts_per_page' => 1,
				'meta_query'     => [ // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
					[
						'key'     => 'rtsb_new_user_coupon',
						'value'   => 'on',
						'compare' => '==',
					],
				],
			]
		);
		if ( $get_id && ! empty( $coupon_query ) ) {
			return $coupon_query[0];
		}
		return ! empty( $coupon_query ) ? get_post_field( 'post_title', $coupon_query[0] ) : null;
	}

	/**
	 * Displays a merged notice for multiple coupons.
	 *
	 * @param array $coupons Applied coupon codes.
	 *
	 * @return void
	 */
	private function trigger_merged_notice( $coupons ) {
		$coupon_list = implode( ', ', array_map( 'esc_html', $coupons ) );

		$message = sprintf(
		/* translators: %s: coupon codes */
			esc_html__( 'The following coupons were automatically applied: %s', 'shopbuilder-pro' ),
			$coupon_list
		);

		SmartCouponsFns::trigger_notice( $message );
	}

	/**
	 * Creates a new RTSB_Coupon object from the given coupon data.
	 *
	 * @param mixed $coupon The coupon data to create the RTSB_Coupon object from.
	 *
	 * @return RTSB_Coupon
	 */
	private function coupon( $coupon ) {
		return new RTSB_Coupon( $coupon );
	}
}
