<?php
/**
 * FreeGift Model Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\SmartCoupons\Frontend;

use Exception;
use RadiusTheme\SBPRO\Helpers\FnsPro;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Modules\SmartCoupons\RTSB_Coupon;
use RadiusTheme\SBPRO\Modules\SmartCoupons\SmartCouponsFns;

defined( 'ABSPATH' ) || exit();

/**
 * FreeGift Model Class.
 */
class FreeGift {
	/**
	 * Singleton Trait.
	 */
	use SingletonTrait;

	/**
	 * Free-gift key.
	 *
	 * @var string
	 */
	private static $free_gift_cart_key = 'rtsb_coupon_free_gift';

	/**
	 * Class Constructor.
	 */
	private function __construct() {
		/**
		 * Actions.
		 */
		add_action( 'rtsb/module/smart_coupons/after_coupon_removal', [ $this, 'remove_on_coupon_removal' ] );
		add_action( 'rtsb/module/smart_coupons/after_auto_apply', [ $this, 'remove_free_gift_if_cart_empty' ], 20 );

		/**
		 * Filters
		 */
		add_filter( 'woocommerce_add_cart_item_data', [ $this, 'add_gift_item_data' ] );
		add_filter( 'woocommerce_get_cart_item_from_session', [ $this, 'get_gift_item_from_session' ] );
		add_filter( 'woocommerce_cart_item_price', [ $this, 'hide_free_gift_price' ], 10, 2 );
		add_filter( 'woocommerce_cart_item_subtotal', [ $this, 'hide_free_gift_price' ], 10, 2 );
		add_filter( 'woocommerce_order_item_get_subtotal', [ $this, 'adjust_free_gift_price_display' ], 10, 2 );
		add_filter( 'woocommerce_order_item_get_total', [ $this, 'adjust_free_gift_price_display' ], 10, 2 );
		add_filter( 'woocommerce_cart_item_key', [ $this, 'free_gift_item_key' ], 10, 4 );
		add_filter( 'woocommerce_cart_subtotal', [ $this, 'adjust_cart_subtotal' ], 10, 3 );
		add_filter( 'woocommerce_calculated_total', [ $this, 'adjust_cart_total' ], 10, 2 );
		add_filter( 'woocommerce_coupon_get_discount_amount', [ $this, 'adjust_coupon_discount' ], 5, 5 );
		add_filter( 'woocommerce_cart_item_quantity', [ $this, 'disable_quantity_selector_for_free_gifts' ], 10, 3 );
	}

	/**
	 * Removes gift items from the cart when a coupon is removed.
	 *
	 * @return void
	 */
	public function remove_on_coupon_removal() {
		$cart    = WC()->cart;
		$trigger = false;

		foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {
			if ( self::is_free_gift( $cart_item ) ) {
				WC()->cart->remove_cart_item( $cart_item_key );
				$trigger = true;
			}
		}

		if ( ! $trigger ) {
			return;
		}

		SmartCouponsFns::trigger_notice( esc_html__( 'Gifts have been removed from cart.', 'shopbuilder-pro' ), 'notice' );
	}

	/**
	 * Removes gift items from the cart if the cart is empty.
	 *
	 * @param object $cart The WooCommerce cart object.
	 *
	 * @return void
	 */
	public function remove_free_gift_if_cart_empty( $cart ) {
		if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
			return;
		}

		$cart_items           = $cart->get_cart();
		$non_free_items_count = 0;

		foreach ( $cart_items as $cart_item_key => $cart_item ) {
			if ( ! self::is_free_gift( $cart_item ) ) {
				$non_free_items_count++;
			}
		}

		if ( 0 === $non_free_items_count ) {
			SmartCouponsFns::trigger_notice( esc_html__( 'You need at least one non-gift item in your cart.', 'shopbuilder-pro' ), 'error' );
			WC()->cart->empty_cart();
		}
	}

	/**
	 * Adds or updates free item data in the cart.
	 *
	 * @param array $cart_item_data Cart item data.
	 *
	 * @return array
	 */
	public function add_gift_item_data( $cart_item_data ) {
		if ( self::is_free_gift( $cart_item_data ) ) {
			$cart_item_data[ self::$free_gift_cart_key ] = true;
		}

		return $cart_item_data;
	}

	/**
	 * Get the cart item data from the session and update it with gift product data.
	 *
	 * @param array $cart_item The current cart item data.
	 *
	 * @return array
	 */
	public function get_gift_item_from_session( $cart_item ) {
		if ( isset( $cart_item_session_data[ self::$free_gift_cart_key ] ) ) {
			$cart_item[ self::$free_gift_cart_key ] = true;
		}

		return $cart_item;
	}

	/**
	 * Hides the price of a gift cart item by displaying a price of zero.
	 *
	 * @param string $price     Price HTML.
	 * @param array  $cart_item Cart item data.
	 *
	 * @return string
	 */
	public function hide_free_gift_price( $price, $cart_item ) {
		if ( self::is_free_gift( $cart_item ) ) {
			return '<span>' . wc_price( 0 ) . '</span>';
		}

		return $price;
	}

	/**
	 * Adjusts the order item price for the gift items.
	 *
	 * @param string $price Order item price.
	 * @param object $item Order item data.
	 *
	 * @return string
	 */
	public function adjust_free_gift_price_display( $price, $item ) {
		if ( $item->get_meta( '_rtsb_coupon_free_gift' ) ) {
			$price = 0;
		}

		return $price;
	}

	/**
	 * Modifies the cart item key for gift items to prevent conflicts with paid items.
	 *
	 * @param string $cart_item_key  Original cart item key.
	 * @param int    $product_id     Product ID.
	 * @param int    $variation_id   Variation ID.
	 * @param array  $cart_item_data Cart item data.
	 *
	 * @return string
	 */
	public function free_gift_item_key( $cart_item_key, $product_id, $variation_id, $cart_item_data ) {
		if ( self::is_free_gift( $cart_item_data ) ) {
			return md5( $cart_item_key . '-free-gift' );
		}

		return $cart_item_key;
	}

	/**
	 * Adjusts the cart subtotal by subtracting the total value of gift items.
	 *
	 * @param string $cart_subtotal Original cart subtotal.
	 * @param bool   $compound      Compound flag.
	 * @param object $cart          Cart object.
	 *
	 * @return string Adjusted cart subtotal.
	 */
	public function adjust_cart_subtotal( $cart_subtotal, $compound, $cart ) {
		$free_gift_total = 0;

		foreach ( $cart->get_cart() as $cart_item ) {
			if ( self::is_free_gift( $cart_item ) ) {
				$free_product_qty   = $cart_item['quantity'];
				$free_product_price = $cart_item['data']->get_price();
				$free_gift_total   += $free_product_qty * $free_product_price;
			}
		}

		if ( $free_gift_total > 0 ) {
			$cart_subtotal = wc_price( max( 0, $cart->subtotal - $free_gift_total ) );
		}

		return $cart_subtotal;
	}

	/**
	 * Adjusts the coupon discount by subtracting the value of gift items.
	 *
	 * @param float  $discount    Original coupon discount.
	 * @param float  $discounting_amount Discount amount.
	 * @param array  $cart_item   Cart item data.
	 * @param bool   $single      Single flag.
	 * @param object $coupon      Coupon object.
	 *
	 * @return float
	 */
	public function adjust_coupon_discount( $discount, $discounting_amount, $cart_item, $single, $coupon ) {
		$free_gift_total = 0;

		if ( self::is_free_gift( $cart_item ) ) {
			$free_gift_total += $cart_item['data']->get_price();
		}

		if ( $free_gift_total > 0 ) {
			$discount = max( 0, $discount - $free_gift_total );
		}

		/**
		 * Filters the adjusted coupon discount.
		 */
		return apply_filters( 'rtsb/module/smart_coupons/adjust_coupon_discount', $discount, $discounting_amount, $cart_item, $single, $coupon );
	}

	/**
	 * Adjusts the cart total by subtracting the total value of gift items.
	 *
	 * @param float  $total Original cart total.
	 * @param object $cart Cart object.
	 *
	 * @return float
	 */
	public function adjust_cart_total( $total, $cart ) {
		$free_gift_total = 0;

		foreach ( $cart->get_cart() as $cart_item ) {
			if ( self::is_free_gift( $cart_item ) ) {
				$free_product_qty   = $cart_item['quantity'];
				$free_product_price = $cart_item['data']->get_price();
				$free_gift_total   += $free_product_qty * $free_product_price;
			}
		}

		return max( 0, $total - $free_gift_total );
	}

	/**
	 * Disables the quantity selector for gift items in the cart.
	 *
	 * @param string $product_quantity Original quantity HTML.
	 * @param string $cart_item_key    Cart item key.
	 * @param array  $cart_item        Cart item data.
	 *
	 * @return string
	 */
	public function disable_quantity_selector_for_free_gifts( $product_quantity, $cart_item_key, $cart_item ) {
		if ( self::is_free_gift( $cart_item ) ) {
			$applied_coupons = WC()->cart->get_applied_coupons();

			foreach ( $applied_coupons as $coupon_code ) {
				$coupon           = new RTSB_Coupon( $coupon_code );
				$free_product_qty = $coupon->rtsb_gift_products_qty();

				return '<span class="rtsb-no-quantity">' . esc_html( $free_product_qty ) . '</span>';
			}
		}

		return $product_quantity;
	}

	/**
	 * Add free products to the cart when a coupon is applied.
	 *
	 * @param object $coupon The coupon code being applied.
	 *
	 * @return void
	 * @throws Exception Throws Exception for invalid coupons.
	 */
	public static function add_free_gifts_to_cart( $coupon ) {
		$free_product_ids = array_filter( array_map( 'absint', $coupon->rtsb_get_gift_products() ) );
		$free_product_qty = absint( $coupon->rtsb_gift_products_qty() );
		$cart             = WC()->cart;

		if ( ! $coupon->rtsb_has_gift_products() ) {
			foreach ( $cart->get_cart() as $cart_item_key => $cart_item ) {
				if ( self::is_free_gift( $cart_item ) ) {
					$cart->remove_cart_item( $cart_item_key );
				}
			}

			return;
		}

		foreach ( $free_product_ids as $product_id ) {
			$cart_item_data  = [ 'rtsb_coupon_free_gift' => true ];
			$product_cart_id = $cart->generate_cart_id( $product_id, 0, [], $cart_item_data );

			if ( ! $cart->find_product_in_cart( $product_cart_id ) ) {
				$cart->add_to_cart( $product_id, $free_product_qty, 0, [], $cart_item_data );
			}
		}
	}

	/**
	 * Displays a custom message for gift products.
	 *
	 * @param array $applied_coupons Array of applied coupons.
	 *
	 * @return void
	 */
	public static function display_gift_product_message( $applied_coupons ) {
		if ( empty( $applied_coupons ) ) {
			return;
		}

		foreach ( $applied_coupons as $coupon_code ) {
			$coupon = new RTSB_Coupon( $coupon_code );

			if ( ! $coupon->rtsb_has_gift_products() ) {
				return;
			}

			$gift_products  = $coupon->rtsb_get_gift_products();
			$custom_message = $coupon->rtsb_gift_products_msg();

			if ( ! empty( $gift_products ) && ! empty( $custom_message ) ) {
				$gift_product_names = [];

				foreach ( $gift_products as $product_id ) {
					$gift_product_names[] = get_the_title( $product_id );
				}

				$custom_message = str_replace( '{{gifts}}', implode( ', ', $gift_product_names ), $custom_message );
				$icon_svg       = self::get_free_gift_icon_svg();

				FnsPro::custom_notice( $custom_message, 'rtsb-free-gift-message', $icon_svg );
			}
		}
	}

	/**
	 * Add the gift item metadata to order line items.
	 *
	 * @param object $item Line item.
	 * @param string $cart_item_key Cart item key.
	 * @param array  $values Cart item values.
	 *
	 * @return void
	 */
	public static function add_gift_meta_to_order_items( $item, $cart_item_key, $values ) {
		if ( isset( $values[ self::$free_gift_cart_key ] ) ) {
			$item->add_meta_data( '_rtsb_coupon_free_gift', true, true );
		}
	}

	/**
	 * Add the gift item metadata to order line items.
	 *
	 * @param array  $meta_data Array of meta-data.
	 * @param string $meta_value Meta-data value.
	 *
	 * @return array
	 */
	public static function format_free_gift_meta( $meta_data, $meta_value ) {
		$meta_data[] = (object) [
			'key'           => esc_html__( 'Gift Item', 'shopbuilder-pro' ),
			'value'         => esc_html( $meta_value ),
			'display_key'   => esc_html__( 'Gift Item', 'shopbuilder-pro' ),
			'display_value' => esc_html__( '100% Discount', 'shopbuilder-pro' ),
		];

		return $meta_data;
	}

	/**
	 * Get SVG icon for volume discount message.
	 *
	 * @return string
	 */
	public static function get_free_gift_icon_svg() {
		return '<svg width="24" height="24" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" class="no-fill"><path d="M12 7v13m0-13H8.464c-.52 0-1.02-.21-1.389-.586A2.019 2.019 0 0 1 6.5 5c0-.53.207-1.04.575-1.414A1.947 1.947 0 0 1 8.465 3C11.214 3 12 7 12 7Zm0 0h3.536c.52 0 1.02-.21 1.389-.586.368-.375.575-.884.575-1.414 0-.53-.207-1.04-.575-1.414A1.947 1.947 0 0 0 15.535 3C12.786 3 12 7 12 7Zm-7 5h14v5.8c0 1.12 0 1.68-.218 2.108a2 2 0 0 1-.874.874C17.48 21 16.92 21 15.8 21H8.2c-1.12 0-1.68 0-2.108-.218a2 2 0 0 1-.874-.874C5 19.48 5 18.92 5 17.8V12Zm-.4 0h14.8c.56 0 .84 0 1.054-.109a1 1 0 0 0 .437-.437C21 11.24 21 10.96 21 10.4V8.6c0-.56 0-.84-.109-1.054a1 1 0 0 0-.437-.437C20.24 7 19.96 7 19.4 7H4.6c-.56 0-.84 0-1.054.109a1 1 0 0 0-.437.437C3 7.76 3 8.04 3 8.6v1.8c0 .56 0 .84.109 1.054a1 1 0 0 0 .437.437C3.76 12 4.04 12 4.6 12Z" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>';
	}

	/**
	 * Checks if a cart item is a free gift.
	 *
	 * @param array $cart_item Cart item data.
	 *
	 * @return bool
	 */
	public static function is_free_gift( $cart_item ) {
		return isset( $cart_item[ self::$free_gift_cart_key ] ) && true === $cart_item[ self::$free_gift_cart_key ];
	}
}
