<?php
/**
 * RTSB Coupons Class.
 *
 * @package RadiusTheme\SB
 */

namespace RadiusTheme\SBPRO\Modules\SmartCoupons;

use RadiusTheme\SBPRO\Modules\SmartCoupons\Frontend\FreeGift;
use WC_Coupon;

defined( 'ABSPATH' ) || exit();

/**
 * RTSB Coupons Class.
 */
class RTSB_Coupon extends WC_Coupon {
	/**
	 * Check if the coupon is for first purchase.
	 *
	 * @return bool
	 */
	public function rtsb_is_first_purchase() {
		return $this->get_meta( 'rtsb_new_user_coupon' ) === 'on';
	}

	/**
	 * Check if the coupon has gift products.
	 *
	 * @return bool
	 */
	public function rtsb_has_gift_products() {
		return 'on' === $this->get_meta( 'rtsb_enable_gift_products' );
	}

	/**
	 * Get the gift products associated with the coupon.
	 *
	 * @return array
	 */
	public function rtsb_get_gift_products() {
		return $this->get_meta( 'rtsb_free_product_ids' ) ?? [];
	}

	/**
	 * Get the gift products quantity associated with the coupon.
	 *
	 * @return string
	 */
	public function rtsb_gift_products_qty() {
		return ! empty( $this->get_meta( 'rtsb_free_product_qty' ) ) ? $this->get_meta( 'rtsb_free_product_qty' ) : 1;
	}

	/**
	 * Get the gift products message associated with the coupon.
	 *
	 * @return string
	 */
	public function rtsb_gift_products_msg() {
		return $this->get_meta( 'rtsb_free_product_msg' );
	}

	/**
	 * Get volume discount tiers associated with the coupon.
	 *
	 * @return array|mixed
	 */
	public function rtsb_get_volume_discount_tiers() {
		return $this->get_meta( 'rtsb_volume_coupon' );
	}

	/**
	 * Validate the coupon based on custom conditions.
	 *
	 * @return bool
	 */
	public function rtsb_validate_coupon_conditions() {
		$conditions = [
			'rtsb_is_start_date_valid',
			'rtsb_is_expiration_date_valid',
			'rtsb_is_user_role_valid',
			'rtsb_is_product_quantity_valid',
			'rtsb_is_shipping_method_valid',
			'rtsb_is_payment_method_valid',
		];

		// Include location check only if the user is logged in.
		if ( is_user_logged_in() ) {
			$conditions[] = 'rtsb_is_location_valid';
		}

		// Validate all conditions.
		foreach ( $conditions as $condition ) {
			if ( ! $this->$condition() ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Check if the start date condition is valid.
	 *
	 * @return bool
	 */
	protected function rtsb_is_start_date_valid() {
		$start_date = $this->rtsb_get_date_starts();

		return empty( $start_date ) || $start_date <= time();
	}

	/**
	 * Check if the expiration date condition is valid using WooCommerce's get_date_expires.
	 *
	 * @return bool
	 */
	protected function rtsb_is_expiration_date_valid() {
		$expire_date = $this->get_date_expires();

		return empty( $expire_date ) || $expire_date->getTimestamp() >= time();
	}

	/**
	 * Check if the user role condition is valid.
	 *
	 * @return bool
	 */
	protected function rtsb_is_user_role_valid() {
		$allowed_roles = $this->get_meta( 'rtsb_user_role_based_coupon' );

		return empty( $allowed_roles ) || array_intersect( wp_get_current_user()->roles, (array) $allowed_roles );
	}

	/**
	 * Check if the location condition is valid.
	 *
	 * @return bool
	 */
	protected function rtsb_is_location_valid() {
		$allowed_locations = $this->rtsb_get_allowed_locations();

		if ( empty( $allowed_locations ) ) {
			return true;
		}

		$user_country = WC()->customer->get_shipping_country();
		$user_state   = WC()->customer->get_shipping_state();

		if ( empty( $user_country ) && empty( $user_state ) ) {
			return true;
		}

		$user_location = ! empty( $user_state ) ? strtoupper( "$user_country::$user_state" ) : strtoupper( $user_country );

		return in_array( $user_location, (array) $allowed_locations, true );
	}

	/**
	 * Check if the product quantity is within the allowed range.
	 *
	 * @return bool
	 */
	protected function rtsb_is_product_quantity_valid() {
		$cart_qty = 0;

		foreach ( WC()->cart->get_cart() as $cart_item ) {
			if ( FreeGift::is_free_gift( $cart_item ) ) {
				continue;
			}

			$cart_qty += $cart_item['quantity'];
		}

		$min_qty = (int) $this->get_meta( 'rtsb_min_product_qty' ) ?? 0;
		$max_qty = (int) $this->get_meta( 'rtsb_max_product_qty' ) ?? 0;

		if ( ( $min_qty && $cart_qty < $min_qty ) || ( $max_qty && $cart_qty > $max_qty ) ) {
			return false;
		}

		return true;
	}

	/**
	 * Check if the selected shipping method is valid for the coupon.
	 *
	 * @return bool
	 */
	protected function rtsb_is_shipping_method_valid() {
		$allowed_shipping = $this->get_meta( 'rtsb_shipping_based_coupon' );

		if ( empty( $allowed_shipping ) ) {
			return true;
		}

		$chosen_shipping = WC()->session->get( 'chosen_shipping_methods' );

		if ( empty( $chosen_shipping ) ) {
			return false;
		}

		$normalized_shipping = array_map(
			function ( $method ) {
				return explode( ':', $method )[0];
			},
			$chosen_shipping
		);

		return ! empty( array_intersect( $normalized_shipping, (array) $allowed_shipping ) );
	}

	/**
	 * Check if the selected payment method is valid for the coupon.
	 *
	 * @return bool
	 */
	protected function rtsb_is_payment_method_valid() {
		$allowed_payments = $this->get_meta( 'rtsb_payment_based_coupon' );

		if ( empty( $allowed_payments ) ) {
			return true;
		}

		$chosen_payment = WC()->session->get( 'chosen_payment_method' );

		return in_array( $chosen_payment, (array) $allowed_payments, true );
	}

	/**
	 * Get coupon expiration date.
	 *
	 * @return string
	 */
	public function rtsb_get_date_starts() {
		return $this->get_meta( 'rtsb_coupon_start' );
	}

	/**
	 * Get coupon allowed locations.
	 *
	 * @return string
	 */
	public function rtsb_get_allowed_locations() {
		return $this->get_meta( 'rtsb_coupon_locations' );
	}
}
