<?php
/**
 * Smart Coupons Functions Class.
 *
 * @package Rse\SB
 */

namespace RadiusTheme\SBPRO\Modules\SmartCoupons;

use WP_Query;
use WC_Coupon;
use RadiusTheme\SB\Helpers\Fns;
use RadiusTheme\SB\Helpers\Cache;
use RadiusTheme\SB\Elementor\Helper\RenderHelpers;

defined( 'ABSPATH' ) || exit();

/**
 * Sticky add-to-cart Functions Class.
 */
class SmartCouponsFns {
	/**
	 * Returns an array of settings fields.
	 *
	 * @return array
	 */
	public static function settings_field() {
		return [
			'coupon_settings'        => [
				'id'    => 'coupon_settings',
				'type'  => 'title',
				'label' => esc_html__( 'Settings', 'shopbuilder-pro' ),
				'tab'   => 'general',
			],
			'coupon_admin_columns'   => [
				'id'          => 'coupon_admin_columns',
				'label'       => esc_html__( 'Extra Admin Columns', 'shopbuilder-pro' ),
				'type'        => 'checkbox',
				'orientation' => 'vertical',
				'help'        => esc_html__( 'Select extra columns to display in dashboard Coupons table.', 'shopbuilder-pro' ),
				'value'       => [ 'coupon_groups', 'start_date', 'total_sales', 'used_in_orders', 'url_coupon' ],
				'options'     => self::admin_columns(),
				'tab'         => 'general',
			],
			'enable_bulk_generator'  => [
				'id'    => 'enable_bulk_generator',
				'type'  => 'switch',
				'label' => esc_html__( 'Enable Bulk Coupon Generator?', 'shopbuilder-pro' ),
				'help'  => sprintf(
					'%s <a href="%s" target="_blank" rel="noopener noreferrer">%s</a>.',
					/* translators: %s = Coupons > Bulk Coupon Generator */
					esc_html__( 'Enable this option to enable bulk coupon generator. You can find it under', 'shopbuilder-pro' ),
					esc_url( admin_url( 'edit.php?post_type=shop_coupon' ) ),
					esc_html__( 'Marketing > Coupons > Bulk Coupon Generator', 'shopbuilder-pro' )
				),
				'value' => 'on',
				'tab'   => 'general',
			],
			'enable_coupon_exporter' => [
				'id'    => 'enable_coupon_exporter',
				'type'  => 'switch',
				'label' => esc_html__( 'Enable Coupon Exporter?', 'shopbuilder-pro' ),
				'help'  => esc_html__( 'Enable this option to enable coupon exporter. You can export coupons in csv format.', 'shopbuilder-pro' ),
				'value' => 'on',
				'tab'   => 'general',
			],
			'coupon_shortcode_info'  => [
				'id'    => 'coupon_shortcode_info',
				'type'  => 'title',
				'label' => esc_html__( 'Coupon Shortcode', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'coupon_shortcode_attr'  => [
				'type'  => 'raw',
				'label' => esc_html__( 'Shortcode', 'shopbuilder-pro' ),
				'html'  => sprintf(
					/* translators: %s: The shortcode example */
					__( 'You can display the coupon list using the following shortcode. You can place it anywhere on your cart, checkout or product page: <br /><br />%s', 'shopbuilder-pro' ),
					'<code>[rtsb_coupon_list layout="rtsb-coupon-layout1" group="" type="all" include="" exclude="" coupon_limit="10" visibility="active,no_expiry,expired"]</code>'
				) . '<br /><br />' .
					'<strong>' . esc_html__( 'Shortcode Attributes:', 'shopbuilder-pro' ) . '</strong>' .
					'<ul style="margin-top:5px; padding-left: 15px;">' .
					'<li><code>layout</code> - ' . esc_html__( 'Choose the coupon layout from rtsb-coupon-layout1, rtsb-coupon-layout2 or rtsb-coupon-layout3.', 'shopbuilder-pro' ) . '</li>' .
					'<li><code>group</code> - ' . esc_html__( 'Filter coupons by comma seperated Coupon Groups (leave empty for all).', 'shopbuilder-pro' ) . '</li>' .
					'<li><code>type</code> - ' . esc_html__( 'Coupon type (e.g., all, percentage, fixed).', 'shopbuilder-pro' ) . '</li>' .
					'<li><code>include</code> - ' . esc_html__( 'Comma seperated coupon IDs to include.', 'shopbuilder-pro' ) . '</li>' .
					'<li><code>exclude</code> - ' . esc_html__( 'Comma seperated coupon IDs to exclude.', 'shopbuilder-pro' ) . '</li>' .
					'<li><code>coupon_limit</code> - ' . esc_html__( 'Limit the number of coupons displayed (default: 10).', 'shopbuilder-pro' ) . '</li>' .
					'<li><code>visibility</code> - ' . esc_html__( 'Filter coupons by status (active, no_expiry, expired).', 'shopbuilder-pro' ) . '</li>' .
					'</ul>',
				'tab'   => 'styles',
			],
			'sc_style_intro'         => [
				'id'    => 'sc_style_intro',
				'type'  => 'title',
				'label' => esc_html__( 'Shortcode Styles', 'shopbuilder-pro' ),
				'tab'   => 'styles',
			],
			'sc_coupon_bg_color'     => [
				'id'    => 'sc_coupon_bg_color',
				'label' => esc_html__( 'Coupon Card Background Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
			'sc_coupon_text_color'   => [
				'id'    => 'sc_coupon_text_color',
				'label' => esc_html__( 'Coupon Card Text Color', 'shopbuilder-pro' ),
				'type'  => 'color',
				'tab'   => 'styles',
			],
		];
	}

	/**
	 * Get sticky add-to-cart settings data.
	 *
	 * @return array
	 */
	public static function get_settings_data() {
		$settings    = Fns::get_options( 'modules', 'smart_coupons' ) ?? [];
		$cache_key   = 'rtsb_smart_coupons_settings_data_' . md5( wp_json_encode( $settings ) );
		$cached_data = wp_cache_get( $cache_key, 'shopbuilder' );

		if ( false !== $cached_data ) {
			return $cached_data;
		}

		$settings_data = [
			'admin_columns'      => RenderHelpers::get_data( $settings, 'coupon_admin_columns', [ 'coupon_groups', 'start_date', 'total_sales', 'used_in_orders', 'url_coupon' ] ),
			'enable_bulk_coupon' => RenderHelpers::get_data( $settings, 'enable_bulk_generator', '' ),
			'enable_exporter'    => RenderHelpers::get_data( $settings, 'enable_coupon_exporter', '' ),
		];

		wp_cache_set( $cache_key, $settings_data, 'shopbuilder', 12 * HOUR_IN_SECONDS );
		Cache::set_data_cache_key( $cache_key );

		return $settings_data;
	}

	/**
	 * Admin columns.
	 *
	 * @return array
	 */
	public static function admin_columns() {
		return [
			[
				'value' => 'coupon_groups',
				'label' => esc_html__( 'Coupon Groups', 'shopbuilder-pro' ),
			],
			[
				'value' => 'start_date',
				'label' => esc_html__( 'Valid from', 'shopbuilder-pro' ),
			],
			[
				'value' => 'total_sales',
				'label' => esc_html__( 'Total Sales', 'shopbuilder-pro' ),
			],
			[
				'value' => 'used_in_orders',
				'label' => esc_html__( 'Applied in Orders', 'shopbuilder-pro' ),
			],
			[
				'value' => 'allowed_user_role',
				'label' => esc_html__( 'Eligible User Roles', 'shopbuilder-pro' ),
			],
			[
				'value' => 'allowed_locations',
				'label' => esc_html__( 'Allowed Locations', 'shopbuilder-pro' ),
			],
			[
				'value' => 'url_coupon',
				'label' => esc_html__( 'Shareable Link', 'shopbuilder-pro' ),
			],
		];
	}

	/**
	 * Sanitize switch value (on/off).
	 *
	 * @param string $meta_key Meta key.
	 *
	 * @return string
	 */
	public static function sanitize_switch( $meta_key ) {
		$value = isset( $_REQUEST[ $meta_key ] ) ? sanitize_text_field( wp_unslash( $_REQUEST[ $meta_key ] ) ) : 'off'; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
		return 'yes' === $value ? 'on' : $value;
	}

	/**
	 * Sanitize text value.
	 *
	 * @param string $meta_key Meta key.
	 *
	 * @return string
	 */
	public static function sanitize_text( $meta_key ) {
		return isset( $_REQUEST[ $meta_key ] ) ? sanitize_text_field( wp_unslash( $_REQUEST[ $meta_key ] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	}

	/**
	 * Sanitize date value.
	 *
	 * @param string $meta_key Meta key.
	 *
	 * @return string
	 */
	public static function sanitize_date( $meta_key ) {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( isset( $_REQUEST[ $meta_key ] ) ) {
			// phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$date      = sanitize_text_field( wp_unslash( $_REQUEST[ $meta_key ] ) );
			$timestamp = strtotime( $date );

			return $timestamp ?: '';
		}

		return '';
	}

	/**
	 * Sanitize an array of numbers.
	 *
	 * @param string $meta_key Meta key.
	 *
	 * @return array
	 */
	public static function sanitize_array_number( $meta_key ) {
		return isset( $_REQUEST[ $meta_key ] ) && is_array( $_REQUEST[ $meta_key ] ) ? array_map( 'absint', wp_unslash( $_REQUEST[ $meta_key ] ) ) : []; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	}

	/**
	 * Sanitize an array of strings.
	 *
	 * @param string $meta_key Meta key.
	 *
	 * @return array
	 */
	public static function sanitize_array( $meta_key ) {
		return isset( $_REQUEST[ $meta_key ] ) && is_array( $_REQUEST[ $meta_key ] ) ? array_map( 'sanitize_text_field', wp_unslash( $_REQUEST[ $meta_key ] ) ) : []; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	}

	/**
	 * Sanitize number value.
	 *
	 * @param string $meta_key Meta key.
	 *
	 * @return int|string
	 */
	public static function sanitize_number( $meta_key ) {
		return isset( $_REQUEST[ $meta_key ] ) ? absint( wp_unslash( $_REQUEST[ $meta_key ] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	}

	/**
	 * Sanitize textarea (with HTML allowed).
	 *
	 * @param string $meta_key Meta key.
	 *
	 * @return string
	 */
	public static function sanitize_textarea( $meta_key ) {
		return isset( $_REQUEST[ $meta_key ] ) ? wp_kses_post( wp_unslash( $_REQUEST[ $meta_key ] ) ) : ''; // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	}

	/**
	 * Get coupons.
	 *
	 * @param array $args Query arguments.
	 *
	 * @return array
	 */
	public static function get_coupons( $args = [] ) {
		$defaults = [
			'post_status'    => 'publish',
			'posts_per_page' => -1,
			'meta_query'     => [], // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			'tax_query'      => [], // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
			'orderby'        => 'date',
			'order'          => 'DESC',
			'fields'         => 'ids',
		];

		$args = wp_parse_args( $args, $defaults );

		$query_args = [
			'post_type'      => 'shop_coupon',
			'post_status'    => $args['post_status'],
			'posts_per_page' => $args['posts_per_page'],
			'orderby'        => $args['orderby'],
			'order'          => $args['order'],
		];

		if ( ! empty( $args['meta_query'] ) ) {
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_query
			$query_args['meta_query'] = $args['meta_query'];
		}

		if ( ! empty( $args['tax_query'] ) ) {
			// phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_tax_query
			$query_args['tax_query'] = $args['tax_query'];
		}

		$query = new WP_Query( $query_args );

		wp_reset_postdata();

		return 'ids' === $args['fields'] ? wp_list_pluck( $query->posts, 'ID' ) : $query->posts;
	}

	/**
	 * Custom Coupon Message.
	 *
	 * @param string $message Message.
	 * @param string $type Notice type.
	 * @param string $coupon_code Coupon code.
	 *
	 * @return void
	 */
	public static function trigger_notice( $message = [], $type = 'success', $coupon_code = '' ) {
		if ( empty( $message ) ) {
			$message = sprintf(
			/* translators: %s: coupon code */
				esc_html__( 'The coupon "%s" was automatically applied.', 'shopbuilder-pro' ),
				$coupon_code
			);
		}

		wc_add_notice( $message, $type );
	}

	/**
	 * Check if the coupon has volume discount.
	 *
	 * @param object $coupon Coupon object.
	 *
	 * @return bool
	 */
	public static function coupon_includes_volume_discount( $coupon ) {
		if ( ! $coupon instanceof WC_Coupon ) {
			$coupon = new WC_Coupon( $coupon );
		}

		$coupon_id = $coupon->get_id();

		$is_volume_discount = get_post_meta( $coupon_id, 'discount_type', true );
		$rtsb_volume_coupon = get_post_meta( $coupon_id, 'rtsb_volume_coupon', true );

		return 'rtsb_volume_discount' === $is_volume_discount && ! empty( $rtsb_volume_coupon );
	}

	/**
	 * Get volume discount rules from coupon meta.
	 *
	 * @param WC_Coupon $coupon WooCommerce coupon object.
	 *
	 * @return array
	 */
	public static function get_volume_discount_rules( $coupon ) {
		$rules = $coupon->get_meta( 'rtsb_volume_coupon' );

		if ( empty( $rules ) ) {
			return [];
		}

		$rules        = explode( '|', $rules );
		$parsed_rules = [];

		foreach ( $rules as $rule ) {
			list( $quantity, $discount )         = array_map( 'trim', explode( ':', $rule ) );
			$parsed_rules[ intval( $quantity ) ] = floatval( $discount );
		}

		// Sort rules by quantity in ascending order.
		ksort( $parsed_rules );

		return $parsed_rules;
	}

	/**
	 * Get the applicable discount based on quantity and volume discount rules.
	 *
	 * @param int   $quantity The product quantity.
	 * @param array $rules    Volume discount rules.
	 *
	 * @return float
	 */
	public static function get_applicable_discount( $quantity, $rules ) {
		$applicable_discount = 0;

		foreach ( $rules as $min_quantity => $discount ) {
			if ( $quantity >= $min_quantity ) {
				$applicable_discount = $discount;
			} else {
				break;
			}
		}

		return $applicable_discount;
	}

	/**
	 * Check if the URL contains the "rtsb_apply_coupon" parameter.
	 *
	 * @return string
	 */
	public static function get_url_coupon() {
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		if ( is_admin() || empty( $_GET['rtsb_apply_coupon'] ?? '' ) ) {
			return '';
		}

		return sanitize_text_field( wp_unslash( $_GET['rtsb_apply_coupon'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
	}
}
