<?php
/**
 * Main initialization class.
 *
 * @package RadiusTheme\SBPRO
 */

namespace RadiusTheme\SBPRO;

use RadiusTheme\SBPRO\Controllers\Frontend\Ajax\AjaxLookbook;
use RadiusTheme\SBPRO\Helpers\Installation;
use RadiusTheme\SBPRO\Modules\ModuleManager;
use RadiusTheme\SBPRO\Traits\SingletonTrait;
use RadiusTheme\SBPRO\Controllers\Dependencies;
use RadiusTheme\SBPRO\Controllers\Admin\Licensing;
use RadiusTheme\SBPRO\Controllers\Admin\AdminInitPro;
use RadiusTheme\SBPRO\Controllers\AssetsControllerPro;
use RadiusTheme\SBPRO\Controllers\OrderStatusController;
use RadiusTheme\SBPRO\Controllers\Frontend\Ajax\AjaxActions;
use RadiusTheme\SBPRO\Controllers\Frontend\Ajax\AjaxFilters;
use RadiusTheme\SBPRO\Controllers\Frontend\Ajax\AjaxClearCart;
use RadiusTheme\SBPRO\Controllers\Hooks\{
	FilterHooksPro,
	ActionHooksPro,
	BuilderHooksPro,
	ElementorHooks
};

// Do not allow directly accessing this file.
if ( ! defined( 'ABSPATH' ) ) {
	exit( 'This script cannot be accessed directly.' );
}

/**
 * Main initialization class.
 */
final class ShopBuilderPro {
	/**
	 * Singleton
	 */
	use SingletonTrait;

	/**
	 * Nonce id
	 *
	 * @var string
	 */
	public $license_option = 'rtsbpro_license';

	/**
	 * Class Constructor
	 */
	private function __construct() {
		add_action( 'init', [ $this, 'language' ] );
		$this->init();
		// HPOS Declare compatibility.
		add_action(
			'before_woocommerce_init',
			function () {
				if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
					\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', RTSBPRO_FILE, true );
				}
			}
		);
	}

	/**
	 * Load Text Domain
	 *
	 * @return void
	 */
	public function language() {
		load_plugin_textdomain( 'shopbuilder-pro', false, dirname( plugin_basename( RTSBPRO_FILE ) ) . '/languages/' );
	}

	/**
	 * Init
	 *
	 * @return void
	 */
	public function init() {
		if ( ! Dependencies::instance()->check() ) {
			return;
		}
		do_action( 'rtsbpro/before_loaded' );
		if ( is_admin() ) {
			AdminInitPro::instance();
		}
		Licensing::instance();
		// Include File.
		ModuleManager::instance();
		AssetsControllerPro::instance();
		OrderStatusController::instance();
		BuilderHooksPro::instance();
		FilterHooksPro::init_hooks();
		ActionHooksPro::init_hooks();
		ElementorHooks::init_hooks();

		// Ajax.
		AjaxActions::instance();
		AjaxFilters::instance();
		AjaxClearCart::instance();
		AjaxLookbook::instance();

		Installation::init();

		do_action( 'rtsbpro/after_loaded' );
	}

	/**
	 * Get the plugin path.
	 *
	 * @return string
	 */
	public function plugin_path() {
		return untrailingslashit( plugin_dir_path( RTSBPRO_FILE ) );
	}

	/**
	 * Get the plugin path.
	 *
	 * @param string $key Key.
	 *
	 * @return string
	 */
	public function get_license( $key ) {
		$license_option = get_option( $this->license_option );
		return $license_option[ $key ] ?? null;
	}

	/**
	 * Get the plugin path.
	 *
	 * @param array $args Args.
	 *
	 * @return string
	 */
	public function update_license( $args ) {
		$license = [
			'license_key'    => '',
			'license_status' => '',
		];
		$license = wp_parse_args( $args, $license );
		return update_option( $this->license_option, $license );
	}


	/**
	 * Get the plugin templates path.
	 *
	 * @return string
	 */
	public function get_plugin_template_path() {
		return $this->plugin_path() . '/templates/';
	}

	/**
	 * Assets url generate with given assets file
	 *
	 * @param string $file File.
	 *
	 * @return string
	 */
	public function get_assets_uri( $file ) {
		$file = ltrim( $file, '/' );

		return trailingslashit( RTSBPRO_URL . '/assets' ) . $file;
	}

	/**
	 * Assets path.
	 *
	 * @param string $file File.
	 *
	 * @return string
	 */
	public function get_assets_path( $file ) {
		$file = ltrim( $file, '/' );

		return trailingslashit( RTSBPRO_PATH . 'assets' ) . $file;
	}
}
